"""
tsyganenko.tests : tests for the tsyganenko Trace class.

These unit tests were written by John Coxon to provide an easy way to verify
that updates to Geopack-2008 had not resulted in bugs in the tracing. See each
class for the rationale behind the tests within it.

.. moduleauthor:: John Coxon

"""
import unittest
import datetime as dt
import numpy as np
from tsyganenko import Trace


def generate_test(lat, lon, rho, start, end):
    """Generate a string to paste into a new function in the tests below."""
    dates = np.array([dt.datetime(i, 1, 1) for i in np.arange(start, end)])

    trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                  lon=np.ones_like(dates, dtype=float) * lon,
                  rho=np.ones_like(dates, dtype=float) * rho,
                  coords='geo', datetime=dates)

    if lat < 0:
        lat_string = "{}S".format(-1 * lat)
    else:
        lat_string = "{}N".format(lat)

    print("    def test_{}_{}_{}(self):".format(lat_string, lon, rho))
    print("        \"\"\"Test for latitude = {}, longitude = {}, rho = {}\"\"\"".format(lat, lon, rho))
    print("        lat = {}".format(lat))
    print("        lon = {}".format(lon))
    print("        rho = {}\n".format(rho))
    print("        expected = {}")
    print("        expected['lat_n'] = np.array([{}])".format(", ".join(trace.lat_n.astype('str'))))
    print("        expected['lon_n'] = np.array([{}])".format(", ".join(trace.lon_n.astype('str'))))
    print("        expected['rho_n'] = np.array([{}])".format(", ".join(trace.rho_n.astype('str'))))
    print("        expected['lat_s'] = np.array([{}])".format(", ".join(trace.lat_s.astype('str'))))
    print("        expected['lon_s'] = np.array([{}])".format(", ".join(trace.lon_s.astype('str'))))
    print("        expected['rho_s'] = np.array([{}])".format(", ".join(trace.rho_s.astype('str'))))
    print("")
    print("        years = np.arange({}, {})".format(start, end))
    print("        dates = np.array([dt.datetime(i, 1, 1) for i in years])")
    print("")
    print("        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat, ")
    print("                          lon=np.ones_like(dates, dtype=float) * lon, ")
    print("                          rho=np.ones_like(dates, dtype=float) * rho, ")
    print("                          coords='geo', datetime=dates)")
    print("")
    print("        for cnt, year in enumerate(years):")
    print("            for coord in expected:")
    print("                with self.subTest(coord=coord, year=year):")
    print("                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)")
    print("")


class Trace1965to2015(unittest.TestCase):
    """
    The tests herein are benchmarked against the version of Geopack-2008 from
    2020-01-01 containing IGRF-13 coefficients. Because IGRF-13 coefficients
    are definitive to 2015, the returns from the program should be identical
    for 1965--2015 even with versions using later releases of IGRF.
    """

    def test_90S_0_30000(self):
        """Test for latitude = -90, longitude = 0, rho = 30000"""
        lat = -90
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-6.744800649487416, -6.730306988068648, -6.716284609702612, -6.702590080416741, -6.756207065401341, -6.741665592658379, -6.732485818413892, -6.723674874384571, -6.782387180490005, -6.772742953381936, -6.763570009326614, -6.757060839047597, -6.818191032119202, -6.810951031598989, -6.804045710348106, -6.797638992906641, -6.864390431665171, -6.862648733426823, -6.861187072944432, -6.860292318163161, -6.927460398461065, -6.92511764357576, -6.923253001932338, -6.921893794287513, -6.988672553802729, -6.986562025349187, -6.987600214103026, -6.989034553828731, -7.058845917335432, -7.059563087198285, -7.06075154011387, -7.066359125422451, -7.1403915458362235, -7.145329772606132, -7.150609508834549, -7.156456150764086, -7.22965528809911, -7.233452873277642, -7.23769442075222, -7.242420911657874, -7.3159615584513915, -7.328262729146587, -7.34104201308368, -7.35437454234355, -7.436842246382284, -7.449539568049332, -7.465699795625582, -7.482447419470461, -7.568473652066771, -7.584729502291395])
        expected['lon_n'] = np.array([350.3510837397057, 350.3288719645248, 350.3067148308573, 350.28447573491974, 350.37703845855174, 350.3548813248842, 350.3423137768114, 350.329882832522, 350.43236299082884, 350.4199047257827, 350.4075284230067, 350.3975563468185, 350.50271393928, 350.49287846687514, 350.48304299447034, 350.47315288055216, 350.59126051168005, 350.5943204364282, 350.5972710781496, 350.60035832365446, 350.7186572000791, 350.7172365207317, 350.71581584138437, 350.7144498035504, 350.8282953966361, 350.82703864182884, 350.8279402267993, 350.8287598494997, 350.94522823522675, 350.9461844617106, 350.9471133674377, 350.9554461982251, 351.07931850901247, 351.0877333020699, 351.0959022083172, 351.104207718348, 351.2237360288232, 351.227424330975, 351.2310853123701, 351.2348009352786, 351.3544385287806, 351.367552491987, 351.3807757382201, 351.3939716636966, 351.52339008809, 351.53664065507985, 351.55464503373196, 351.5726494123841, 351.70714949752005, 351.72534512146893])
        expected['rho_n'] = np.array([292487.15989990236, 292480.06307373045, 292473.2822021484, 292466.9145019531, 292520.0434814453, 292513.0438720703, 292511.29396972654, 292510.2974975586, 292569.06505126954, 292566.9748901367, 292565.37081298826, 292564.8604248047, 292624.673059082, 292623.43354492186, 292622.8016357422, 292622.4856811523, 292690.14857177733, 292696.9537475586, 292703.8561401367, 292711.5119628906, 292779.6852416992, 292782.7961791992, 292786.49041748047, 292790.52491455077, 292855.14978027344, 292858.69819335936, 292862.7569946289, 292867.4963134765, 292933.0204345703, 292937.0306274414, 292941.5512084961, 292950.6409790039, 293021.0745483398, 293029.7268432617, 293038.8409179687, 293048.7327270508, 293116.12827148434, 293121.66962890624, 293127.50263671874, 293134.1619873047, 293201.970703125, 293212.1784667969, 293222.92092285154, 293234.46541748045, 293308.08283691405, 293319.1655517578, 293333.9425048828, 293349.740234375, 293427.4164550781, 293443.287097168])
        expected['lat_s'] = np.array([-78.96699553695964, -78.94655961096296, -78.92606904345291, -78.90552383442952, -78.88586654999821, -78.8652120579481, -78.84998073609898, -78.83470843311483, -78.82032405472279, -78.80494246871191, -78.78951990156602, -78.77494427787721, -78.76129755991553, -78.74663997395669, -78.73194140686283, -78.7171881982556, -78.70512608418136, -78.69208042286661, -78.67899378041687, -78.66585249645377, -78.65366743897448, -78.64256155138403, -78.6314283430369, -78.62028147431144, -78.6100908320698, -78.5989029822093, -78.58878064185936, -78.57864464113106, -78.56947852726489, -78.55930154540158, -78.54911090315991, -78.53948033643019, -78.5308469773193, -78.52118908983292, -78.51151754196817, -78.50181867334675, -78.49710584281945, -78.49134116315994, -78.48557648350047, -78.47978448308427, -78.47498969028695, -78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.46619240663597, -78.4636788970214, -78.4600588967613])
        expected['lon_s'] = np.array([136.90634712010825, 136.89589693067813, 136.88547406200468, 136.87502387257456, 136.8688903626999, 136.85841285251308, 136.88405338265733, 136.90959829015318, 136.93969210363625, 136.96534629415882, 136.99089120165465, 136.99660123980078, 137.00670991977242, 137.0124062975402, 137.0180343734163, 137.02374441156243, 137.04442622436923, 137.06064109345883, 137.07689694368347, 137.09313913352975, 137.11379362557986, 137.10207302096413, 137.0904207182401, 137.0787820758944, 137.07140547159077, 137.05968486697503, 137.0382380729812, 137.01669565633898, 136.9996065230356, 136.97811874790673, 136.95675391618295, 136.9179721020756, 136.8835616090371, 136.8447934553081, 136.8060936034708, 136.7672161667151, 136.70973329466025, 136.64775615813156, 136.58573804046785, 136.52366528129076, 136.46601848469587, 136.39912361196477, 136.33206481469358, 136.26503333817908, 136.20222291857164, 136.13491823449033, 136.05294230407182, 135.97082976986994, 135.8930065944668, 135.81082575837323])
        expected['rho_s'] = np.array([6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.198860740661])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_0_30000(self):
        """Test for latitude = -60, longitude = 0, rho = 30000"""
        lat = -60
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([5.8205139514774515, 5.816771007812278, 5.8125841018510584, 5.808035195863837, 5.765401155064623, 5.761617230264434, 5.75369421082722, 5.745381870607304, 5.698656546495258, 5.6911296780298954, 5.683247639727696, 5.673753676781388, 5.6255666921869505, 5.616851370806003, 5.607801370155741, 5.598437180803657, 5.546876082759226, 5.535237440413525, 5.523243628230986, 5.510942457535791, 5.459354038734688, 5.449935207869231, 5.440099735464415, 5.430018376249478, 5.380274108524276, 5.370998711631401, 5.360090899526895, 5.34889621947724, 5.297751762972183, 5.287342554677082, 5.276639648247681, 5.264229194525754, 5.211520624700768, 5.2000322465168125, 5.18811356641514, 5.176071942908408, 5.123998580676229, 5.113493749732754, 5.102708881033294, 5.091671295334564, 5.039488650075654, 5.023184988526836, 5.00661494960039, 4.989792193674646, 4.9314828687302565, 4.9156641506125, 4.89793980971632, 4.880072034847558, 4.820205426772404, 4.803280218009107])
        expected['lon_n'] = np.array([354.8852091975703, 354.87425357414156, 354.8632706299562, 354.8523150065275, 354.90796738788475, 354.8970390852127, 354.8912470847966, 354.8854550843804, 354.94640769253357, 354.9406703336308, 354.93496029548464, 354.9313266348462, 354.9945741865605, 354.9909951674354, 354.9873888275537, 354.98383712918525, 355.0535870209894, 355.05645570044084, 355.0593517006489, 355.06219305934366, 355.13202491341787, 355.1318609888778, 355.131669743581, 355.1314238567709, 355.198359710637, 355.19811382382693, 355.19860559744717, 355.19904272955404, 355.2667708853639, 355.2672626589841, 355.26767247033433, 355.27215307442987, 355.34395202298504, 355.3483233440538, 355.3527219858793, 355.35692938240805, 355.4265699911855, 355.42867368944985, 355.43066810468747, 355.4326351991685, 355.5023850909726, 355.5093792046827, 355.5164006391495, 355.52328546983284, 355.59806238086617, 355.60497453230624, 355.6145641179009, 355.6240170997122, 355.70166269019694, 355.7111429927649])
        expected['rho_n'] = np.array([320479.4727416992, 320477.3582763672, 320475.559765625, 320474.07720947266, 320498.1383666992, 320496.0482055664, 320497.38493652345, 320499.37788085936, 320526.96314697264, 320528.12974853517, 320529.3935668945, 320532.28576660156, 320561.54802246095, 320563.54096679686, 320566.0442993164, 320568.7906738281, 320602.91376953124, 320609.5731201172, 320617.3990722656, 320625.5652832031, 320659.7369873047, 320665.15682373045, 320670.69818115234, 320676.40966796875, 320708.5155151367, 320713.57078857423, 320718.91771240236, 320724.5562866211, 320756.7350463867, 320761.6444946289, 320766.7483764648, 320774.7201538086, 320809.7182006836, 320816.79072265624, 320824.32502441405, 320831.7864135742, 320864.9616455078, 320869.7981811523, 320875.04788818356, 320880.85659179685, 320914.05612792965, 320921.90638427733, 320930.0239868164, 320938.2631103516, 320973.8444580078, 320981.2572387695, 320990.5900512695, 320999.9957763672, 321037.4485473633, 321046.14945068356])
        expected['lat_s'] = np.array([-79.90934307636795, -79.90413847222041, -79.89886556618114, -79.89356533938522, -79.88857930129112, -79.8831834518468, -79.88692639551198, -79.89066933917715, -79.89480843381418, -79.89855137747932, -79.90232164190118, -79.90493077416411, -79.9079223970206, -79.91054518966189, -79.91316798230315, -79.91579077494444, -79.92018941676994, -79.9242055680019, -79.92820805885552, -79.93223787046583, -79.93666383304802, -79.93466941781037, -79.9326340214377, -79.93058496468669, -79.92891839852922, -79.92685568139987, -79.92292149243795, -79.91897364309767, -79.91540828435092, -79.91143311425398, -79.90743062340036, -79.89722632078036, -79.88740450875389, -79.87714556462052, -79.8668319789738, -79.8564910725704, -79.84280337347369, -79.82865122151344, -79.81448540917481, -79.80029227607955, -79.78648163357778, -79.77171476459219, -79.7569342352283, -79.74211272472937, -79.72767370482396, -79.71278389243332, -79.69486147605122, -79.6768980785341, -79.6593308319888, -79.6413127929583])
        expected['lon_s'] = np.array([57.82406049443696, 57.91081072708529, 57.99797077108382, 58.08528790943327, 58.16835667011896, 58.256268034926194, 58.31304056730734, 58.369901892147695, 58.42254216008099, 58.48016163591922, 58.53797918724337, 58.57721862402509, 58.61197062652212, 58.651531082194836, 58.6912622925968, 58.731116446403824, 58.774180789120756, 58.82210822652673, 58.870445475282914, 58.91876223347158, 58.96265302907808, 59.02170001445285, 59.08095873569187, 59.14029258901178, 59.19491361180437, 59.25449335193439, 59.30602712922214, 59.3576701895366, 59.4046140797021, 59.45653717777254, 59.508631030572225, 59.55180465631587, 59.590313262856405, 59.63360300181595, 59.67700885399138, 59.720380555220956, 59.7246084423172, 59.73375406561584, 59.74283138702279, 59.751895048051395, 59.75615708609349, 59.71415142269791, 59.67220723100486, 59.630235718555134, 59.58353771519975, 59.541812089560146, 59.52018088045872, 59.498720426086535, 59.47232857513361, 59.45105253586902])
        expected['rho_s'] = np.array([6371.192404937744, 6371.192404937744, 6371.192025184631, 6371.192025184631, 6371.192404937744, 6371.191645431519, 6371.192025184631, 6371.191645431519, 6371.192404937744, 6371.192025184631, 6371.191645431519, 6371.190885925293, 6371.191645431519, 6371.192404937744, 6371.191645431519, 6371.192404937744, 6371.191645431519, 6371.191265678405, 6371.192025184631, 6371.191645431519, 6371.191265678405, 6371.191645431519, 6371.191645431519, 6371.190885925293, 6371.191645431519, 6371.191645431519, 6371.191645431519, 6371.191645431519, 6371.190885925293, 6371.192025184631, 6371.191645431519, 6371.191645431519, 6371.191265678405, 6371.190885925293, 6371.191265678405, 6371.191265678405, 6371.190885925293, 6371.191645431519, 6371.191645431519, 6371.191645431519, 6371.192025184631, 6371.192404937744, 6371.192025184631, 6371.193164443969, 6371.193923950195, 6371.194303703308, 6371.194303703308, 6371.1946834564205, 6371.195822715759, 6371.1965822219845])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_0_30000(self):
        """Test for latitude = -30, longitude = 0, rho = 30000"""
        lat = -30
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([62.178641680205835, 62.2348865804736, 62.291049518471326, 62.34710488098962, 62.402924601981546, 62.458817747507055, 62.52256902567266, 62.5864944736621, 62.65023550654395, 62.71357868090665, 62.777210430761784, 62.84974362465273, 62.9221589977805, 62.994482163354476, 63.06679679119198, 63.13871526805763, 63.20520374453312, 63.271757107805726, 63.33790919746459, 63.404189353170395, 63.469656716364966, 63.526934682744695, 63.58397188495618, 63.640659039972704, 63.69723861950979, 63.753737944324136, 63.81287030706352, 63.871676528270044, 63.93043664569966, 63.98888428197475, 64.04743095599281, 64.11070412091651, 64.17367504996945, 64.23620372427362, 64.29901755897563, 64.3609588370112, 64.4206563979044, 64.47986560027195, 64.5391892083081, 64.5980654389536, 64.6563064620063, 64.72819761811527, 64.79961407607692, 64.87112444913967, 64.94139002022618, 65.01223957248672, 65.08555482303763, 65.15839537544123, 65.23060072025203, 65.3029904801704])
        expected['lon_n'] = np.array([334.27926613207217, 334.31877194623155, 334.3582504396342, 334.3978655368203, 334.4442288609063, 334.4841171656592, 334.54239233965774, 334.6007494759263, 334.66615536741836, 334.72505891882054, 334.7840171117361, 334.8876447418235, 334.99884022151133, 335.10372460640605, 335.2091280856777, 335.3151872631097, 335.4493594991654, 335.57708403664464, 335.70562819682425, 335.8348280551642, 335.972415385805, 336.09462113043486, 336.2173732901983, 336.34072650660875, 336.4723579122932, 336.59696788351096, 336.7207309112716, 336.84498571265243, 336.97779191087415, 337.103330787819, 337.22938875914076, 337.3992692241773, 337.57822018043163, 337.74995845692246, 337.9224617146003, 338.0957845949785, 338.2761562305802, 338.44887805431154, 338.62222825544643, 338.796206833985, 338.9797203566049, 339.16484580386896, 339.3510094398868, 339.53802001936174, 339.73535784486177, 339.92460872638446, 340.13708225108525, 340.3507852098366, 340.5752798674766, 340.79166026038257])
        expected['rho_n'] = np.array([6371.195442962647, 6371.195822715759, 6371.1965822219845, 6371.197721481323, 6371.1984809875485, 6371.199620246887, 6371.194303703308, 6371.196202468872, 6371.199240493775, 6371.195063209533, 6371.197721481323, 6371.1946834564205, 6371.196961975097, 6371.195442962647, 6371.197721481323, 6371.195063209533, 6371.195442962647, 6371.197341728211, 6371.195442962647, 6371.198101234436, 6371.1946834564205, 6371.196961975097, 6371.198101234436, 6371.195442962647, 6371.196961975097, 6371.1965822219845, 6371.197341728211, 6371.195442962647, 6371.1984809875485, 6371.195442962647, 6371.1965822219845, 6371.195822715759, 6371.198860740661, 6371.195063209533, 6371.199620246887, 6371.195063209533, 6371.199620246887, 6371.195822715759, 6371.197721481323, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.195442962647, 6371.199620246887, 6371.195442962647, 6371.198101234436, 6371.198101234436, 6371.196202468872, 6371.196202468872, 6371.1984809875485])
        expected['lat_s'] = np.array([-69.54882036967237, -69.54436708633352, -69.539927463373, -69.53546051965583, -69.53041984004835, -69.5259665567095, -69.52723697189512, -69.52853470783742, -69.52925870788945, -69.53061108534513, -69.53199078355746, -69.53225033074591, -69.53193614204412, -69.53223667036758, -69.53249621755603, -69.53279674587952, -69.53756421792019, -69.54291908622949, -69.5483149356738, -69.55375176625313, -69.55856021942881, -69.56092346488163, -69.5633003707128, -69.5656909369223, -69.56750776724152, -69.56992565420771, -69.56754874837654, -69.56515818216704, -69.56212557817557, -69.55978965347941, -69.55739908726991, -69.55244036993247, -69.54682595443472, -69.5418672370973, -69.53685387824652, -69.53189516090907, -69.51764738630044, -69.5040553098521, -69.49042225226876, -69.47677553430708, -69.46244579742839, -69.4400564373291, -69.41762609609478, -69.39516843410377, -69.37198677206075, -69.34939250628634, -69.32884729726294, -69.30828842786121, -69.28701921878579, -69.26640570787069])
        expected['lon_s'] = np.array([31.391426481973543, 31.40673293590354, 31.422172578522357, 31.4376736928437, 31.452443976923853, 31.468211468622826, 31.45421641101347, 31.44018378736368, 31.425218842892182, 31.411066690931914, 31.396863312552874, 31.363388555430642, 31.328971232202946, 31.295199361851818, 31.261191849974324, 31.226928206002956, 31.201653090979324, 31.17694146656222, 31.152185445915507, 31.127279161107055, 31.101297121504327, 31.109636782480912, 31.117993518930422, 31.126329764812425, 31.133965916304497, 31.142421690496974, 31.146666653066138, 31.150839898649014, 31.154224257382754, 31.158588748262392, 31.162898597628672, 31.173731277652312, 31.18373067459721, 31.194635071607134, 31.205481412009117, 31.216498507140347, 31.21822654500036, 31.2209278848171, 31.22352677179629, 31.226193960667178, 31.22800054570265, 31.191967882736247, 31.155887408445654, 31.11978644358755, 31.082862440934463, 31.04669317418466, 31.024949266961926, 31.003215605022945, 30.980624754343125, 30.958809130134107])
        expected['rho_s'] = np.array([6371.190885925293, 6371.188987159729, 6371.187088394165, 6371.187088394165, 6371.186328887939, 6371.185189628601, 6371.185189628601, 6371.186328887939, 6371.186708641052, 6371.187468147277, 6371.188607406616, 6371.19050617218, 6371.193544197083, 6371.1965822219845, 6371.199620246887, 6371.1965822219845, 6371.193544197083, 6371.191265678405, 6371.190126419067, 6371.188607406616, 6371.186708641052, 6371.186328887939, 6371.185569381713, 6371.184809875488, 6371.184809875488, 6371.184809875488, 6371.185189628601, 6371.184809875488, 6371.185189628601, 6371.187468147277, 6371.188227653503, 6371.190126419067, 6371.191645431519, 6371.193164443969, 6371.195442962647, 6371.197341728211, 6371.198101234436, 6371.195442962647, 6371.192404937744, 6371.190126419067, 6371.187847900391, 6371.185949134827, 6371.183290863037, 6371.184050369263, 6371.1844301223755, 6371.185189628601, 6371.187088394165, 6371.19050617218, 6371.194303703308, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_0_30000(self):
        """Test for latitude = 0, longitude = 0, rho = 30000"""
        lat = 0
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([60.10271838010176, 60.14800253429895, 60.193344745104085, 60.238710861571306, 60.283615940269556, 60.32848345292737, 60.37916345656886, 60.42988444134538, 60.48020244496086, 60.53055459952219, 60.58071550878677, 60.6363303241036, 60.69152849788105, 60.746934992428194, 60.80186337373344, 60.85676784937659, 60.904466475445375, 60.95251685625641, 61.0001915766631, 61.047521372516705, 61.09435939475007, 61.13330854849207, 61.17215866449109, 61.21089949746341, 61.24869776433026, 61.28683071046645, 61.32701612844821, 61.36716398038955, 61.40684908701461, 61.44651370307217, 61.48599561156943, 61.52702968055557, 61.56750879667165, 61.60824233733432, 61.64876072703812, 61.68870879594624, 61.726148477881665, 61.763938207012046, 61.80152815310922, 61.83899686334861, 61.87565790371866, 61.92215270894623, 61.96862702360629, 62.015024498638184, 62.0605511245509, 62.10611190140947, 62.152852593447165, 62.199506200572934, 62.245025996296484, 62.2907131316547])
        expected['lon_n'] = np.array([342.52286732817066, 342.5707332938741, 342.6185719388209, 342.666465225281, 342.7144131532545, 342.7624976850114, 342.824351878135, 342.88623339201536, 342.9481968681657, 343.0103515896128, 343.07261559408664, 343.17810103562834, 343.2839416470069, 343.3901647489789, 343.4968796245712, 343.6040043115137, 343.7242429616626, 343.84489142316164, 343.966031658281, 344.08771830853397, 344.2098967324072, 344.32396089154634, 344.43835289976573, 344.55301811555194, 344.66814778420166, 344.7834960189048, 344.89663127231694, 345.01006705405246, 345.12377604335484, 345.2378128817374, 345.3521229276867, 345.503070108344, 345.6543997795948, 345.80608462068244, 345.9581519523636, 346.11068373690836, 346.2602648797316, 346.41003726785164, 346.5600555427818, 346.7103470252788, 346.86093903609924, 347.0189076512233, 347.1772314361842, 347.33596503249527, 347.49521772318326, 347.6548802252215, 347.8329569172621, 348.0115253829229, 348.19074954674403, 348.3704654841854])
        expected['rho_n'] = np.array([6371.195442962647, 6371.193923950195, 6371.193923950195, 6371.1965822219845, 6371.198860740661, 6371.195063209533, 6371.193164443969, 6371.194303703308, 6371.197721481323, 6371.196202468872, 6371.194303703308, 6371.193544197083, 6371.196202468872, 6371.1984809875485, 6371.194303703308, 6371.193544197083, 6371.194303703308, 6371.197721481323, 6371.197721481323, 6371.195063209533, 6371.193544197083, 6371.194303703308, 6371.1965822219845, 6371.199240493775, 6371.196202468872, 6371.1946834564205, 6371.193923950195, 6371.195442962647, 6371.1984809875485, 6371.196961975097, 6371.194303703308, 6371.194303703308, 6371.195063209533, 6371.196961975097, 6371.199240493775, 6371.196202468872, 6371.194303703308, 6371.193923950195, 6371.195063209533, 6371.197721481323, 6371.1984809875485, 6371.194303703308, 6371.194303703308, 6371.196961975097, 6371.197341728211, 6371.1946834564205, 6371.1946834564205, 6371.1984809875485, 6371.195822715759, 6371.1946834564205])
        expected['lat_s'] = np.array([-63.749702896386395, -63.73598787653302, -63.72224553592295, -63.708434893421185, -63.694979420756255, -63.681100476362786, -63.67293157011545, -63.66476266386812, -63.656976248214306, -63.648698058940255, -63.64043353004453, -63.634259039034845, -63.62838507634865, -63.62200567966386, -63.61565360373575, -63.60916492402421, -63.611596471368756, -63.61371383001145, -63.61576288676247, -63.61782560389179, -63.620134207831256, -63.61876816999725, -63.61744311329829, -63.61615903773432, -63.61516183011551, -63.613823113038166, -63.60426084820017, -63.59468492298379, -63.58539586571257, -63.57569699709117, -63.56602544922643, -63.560083184648505, -63.55445510877243, -63.548417221546146, -63.542365673941504, -63.536437069741936, -63.51988069119383, -63.50294182205218, -63.48592099064052, -63.4689821214989, -63.452343780680735, -63.422249967197644, -63.39206053106619, -63.36178913266471, -63.33169531918162, -63.30121901510503, -63.27582437177094, -63.250238483140066, -63.22503508510272, -63.19940821533683])
        expected['lon_s'] = np.array([30.765050323759183, 30.753367285183877, 30.74171156736525, 30.72986460424986, 30.715855886262165, 30.70393379106591, 30.66356395797655, 30.622866275807027, 30.579887310454723, 30.538755911272908, 30.497388870564723, 30.43202054511312, 30.364135294952362, 30.297937101516595, 30.23132226654146, 30.16429762021613, 30.104878389531603, 30.04732380049049, 29.989291098207477, 29.93097152797932, 29.869995014164033, 29.84832282392758, 29.826531105380656, 29.80471889626622, 29.780485385091016, 29.758362402369343, 29.73228815521282, 29.70609437974582, 29.677506622974732, 29.651022564468008, 29.624456543691245, 29.603911334667846, 29.580941408489092, 29.560054690007195, 29.539058688498574, 29.518165139827506, 29.488638232045542, 29.461320890460076, 29.433863529996625, 29.406471056330286, 29.37667777117069, 29.303803067821242, 29.23081908144507, 29.157787283744707, 29.082224900956856, 29.008906235311354, 28.94519081563889, 28.88113388650793, 28.81476152324833, 28.75057140542855])
        expected['rho_s'] = np.array([6371.185569381713, 6371.185569381713, 6371.194303703308, 6371.190885925293, 6371.184050369263, 6371.186708641052, 6371.198860740661, 6371.186708641052, 6371.1844301223755, 6371.191645431519, 6371.191645431519, 6371.182911109924, 6371.186708641052, 6371.198860740661, 6371.185189628601, 6371.184050369263, 6371.190885925293, 6371.193923950195, 6371.185189628601, 6371.182911109924, 6371.189746665955, 6371.197721481323, 6371.187088394165, 6371.182151603699, 6371.185189628601, 6371.195063209533, 6371.190126419067, 6371.1825313568115, 6371.183670616149, 6371.195442962647, 6371.190126419067, 6371.183290863037, 6371.182911109924, 6371.188227653503, 6371.198101234436, 6371.186708641052, 6371.18101234436, 6371.184050369263, 6371.195442962647, 6371.188987159729, 6371.181771850585, 6371.186328887939, 6371.191645431519, 6371.181392097473, 6371.187847900391, 6371.189746665955, 6371.18101234436, 6371.189746665955, 6371.188607406616, 6371.18101234436])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_0_30000(self):
        """Test for latitude = 30, longitude = 0, rho = 30000"""
        lat = 30
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([65.43793111250775, 65.46715066177707, 65.49630361670198, 65.52540534520811, 65.55462489447743, 65.58364466071352, 65.61616831399382, 65.64857243896365, 65.68109950733856, 65.71339605682896, 65.74561918178578, 65.78222216554789, 65.81884222478291, 65.85520102928217, 65.89144030547098, 65.92757200618034, 65.95776656495367, 65.98766230295082, 66.01742143716456, 66.04710543929744, 66.07678602633572, 66.10029553745892, 66.12361892592722, 66.14686035212547, 66.17009153303998, 66.19309561016459, 66.21833315914779, 66.24350923642845, 66.26864945521598, 66.29360867399049, 66.31845177954912, 66.3453114984602, 66.37214901925647, 66.39863820040507, 66.42497540984465, 66.45110429851452, 66.4752780455345, 66.49919736800788, 66.52294422820472, 66.54650325819938, 66.56997008064026, 66.59947308276013, 66.62879166977241, 66.65790022846771, 66.68693707017673, 66.71570241186623, 66.7449322064193, 66.77389220850016, 66.80271219170302, 66.83118725035281])
        expected['lon_n'] = np.array([345.4761318422575, 345.51653924138725, 345.5570012820304, 345.59749064343026, 345.6395919294742, 345.68013593238743, 345.73311087959, 345.7861677890627, 345.8408366231795, 345.8939754949222, 345.94714168742155, 346.0489934683248, 346.15270306068226, 346.2551012567191, 346.3577726603227, 346.46071727149314, 346.5745901853356, 346.6871790236141, 346.8001230317294, 346.9133129266549, 347.028497236818, 347.13551264073374, 347.2427466107029, 347.3501718259689, 347.4594548526891, 347.56728987930524, 347.67476973608456, 347.78244083816065, 347.89202439320434, 348.00007798587393, 348.10837746535367, 348.25386049467494, 348.40122865620714, 348.547258100662, 348.69350611117034, 348.84002732924546, 348.98671247186064, 349.1318403313451, 349.2770774738562, 349.42247854090743, 349.5697374194129, 349.7228429398479, 349.8761943470931, 350.02984628266165, 350.18546531271124, 350.3396363426568, 350.5120029965512, 350.6847794617958, 350.85963230454814, 351.0331191094665])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.197721481323, 6371.197341728211, 6371.196961975097, 6371.196202468872, 6371.195822715759, 6371.195442962647, 6371.195822715759, 6371.195063209533, 6371.195822715759, 6371.195442962647, 6371.195442962647, 6371.196202468872, 6371.196202468872, 6371.196961975097, 6371.196961975097, 6371.197721481323, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.197341728211, 6371.196961975097, 6371.1965822219845, 6371.196202468872, 6371.196202468872, 6371.195822715759, 6371.196202468872, 6371.195822715759, 6371.195822715759, 6371.195822715759, 6371.195822715759, 6371.196202468872, 6371.195822715759, 6371.195822715759, 6371.196202468872, 6371.196202468872, 6371.195822715759, 6371.196961975097, 6371.1965822219845, 6371.197721481323, 6371.197721481323, 6371.198860740661, 6371.198860740661, 6371.2])
        expected['lat_s'] = np.array([-66.9342922572832, -66.9211509733201, -66.9079823686003, -66.89475912236716, -66.88753278222529, -66.87425489447878, -66.87171406410755, -66.86911859222292, -66.87249270567293, -66.86992455454498, -66.86732908266038, -66.8715364791891, -66.88169980067408, -66.88587987644615, -66.89003263146151, -66.89417172609853, -66.91216244437234, -66.92425187920327, -66.9363413140342, -66.9484853903785, -66.9664761086523, -66.96837490124156, -66.97024637307413, -66.97214516566339, -66.97991792093887, -66.98185769466315, -66.97704924148746, -66.97224078831178, -66.97325165630895, -66.96841588237658, -66.96353912730919, -66.96177693850333, -66.96584773124866, -66.96405822168612, -66.96229603288023, -66.96057482520939, -66.9551106738734, -66.94381354098618, -66.93251640809899, -66.92116463369842, -66.91552289744399, -66.89929436797604, -66.88297021585973, -66.86645481844661, -66.85538991199121, -66.83857398625463, -66.82524145699477, -66.81179964470817, -66.80369904035254, -66.79001134125585])
        expected['lon_s'] = np.array([44.82232071117037, 44.81469480496205, 44.80715427611836, 44.79956935104506, 44.77807474572703, 44.77042493385662, 44.73033513852322, 44.69004385260932, 44.63564822605934, 44.59492322313314, 44.55398989943725, 44.48811272489248, 44.408032171968664, 44.341222676602186, 44.27397263403424, 44.20636400653487, 44.12982832179029, 44.06659101535973, 44.00293023720062, 43.93897234581262, 43.8605105477221, 43.824604243255344, 43.788513523681, 43.752337426742024, 43.70204332878863, 43.66549840163448, 43.620586492747066, 43.57562677253546, 43.51662759848488, 43.47133319900394, 43.42585779951, 43.37719953186283, 43.31459060283591, 43.26545080685225, 43.21626661463899, 43.16705510166905, 43.09032475653312, 43.02722063879138, 42.96407553991461, 42.90085872405155, 42.823862001538004, 42.698835388780914, 42.5738395118751, 42.44860799344291, 42.30981854950826, 42.18424210652299, 42.0651611734384, 41.945981202610845, 41.81330819307794, 41.693728656183936])
        expected['rho_s'] = np.array([6371.1946834564205, 6371.186708641052, 6371.196202468872, 6371.187468147277, 6371.187468147277, 6371.196961975097, 6371.187468147277, 6371.189366912841, 6371.1946834564205, 6371.188607406616, 6371.187088394165, 6371.1965822219845, 6371.190126419067, 6371.186708641052, 6371.199240493775, 6371.186708641052, 6371.191265678405, 6371.192025184631, 6371.193164443969, 6371.192404937744, 6371.195063209533, 6371.187468147277, 6371.186328887939, 6371.190126419067, 6371.188227653503, 6371.185189628601, 6371.193544197083, 6371.188987159729, 6371.185569381713, 6371.188227653503, 6371.193923950195, 6371.193544197083, 6371.186328887939, 6371.1946834564205, 6371.191265678405, 6371.184809875488, 6371.19050617218, 6371.187088394165, 6371.190126419067, 6371.187088394165, 6371.184050369263, 6371.193923950195, 6371.185949134827, 6371.187847900391, 6371.1844301223755, 6371.192404937744, 6371.188227653503, 6371.184809875488, 6371.187468147277, 6371.185949134827])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_0_30000(self):
        """Test for latitude = 60, longitude = 0, rho = 30000"""
        lat = 60
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([76.22452272717602, 76.24011007263564, 76.25567009733858, 76.27116011260252, 76.28655877408632, 76.30206671859683, 76.31980130477677, 76.33755552775057, 76.35516717052545, 76.37280357273606, 76.39044253626763, 76.41581498148693, 76.44099959650406, 76.46617994265296, 76.49128003407911, 76.51627852644135, 76.53793193365759, 76.55963742106624, 76.58120801223879, 76.60273506095538, 76.6240708643752, 76.64201974774035, 76.65988666883545, 76.67766992011323, 76.69530376100292, 76.71294016321353, 76.73292273540405, 76.75281822268265, 76.77255149296846, 76.7922463434402, 76.8118669156047, 76.83533459181922, 76.8584821029164, 76.88153655268613, 76.90437841281795, 76.92706659369343, 76.94859108108908, 76.97000030904249, 76.99121658415184, 77.01223307622797, 77.03298916734198, 77.05780153704936, 77.08237655768306, 77.10676887075647, 77.13087858475296, 77.15481498269928, 77.17950526277522, 77.20393428811542, 77.22800045965596, 77.25188819250444])
        expected['lon_n'] = np.array([340.27259320421825, 340.30699003687846, 340.34138686953867, 340.3759203059822, 340.40886913853836, 340.44332061271194, 340.49350884273315, 340.5436151104844, 340.59227337813155, 340.64254357042284, 340.69281376271414, 340.81671339425816, 340.93946555402164, 341.0641848082661, 341.1893138738607, 341.31482543004887, 341.45066423226206, 341.58847012895626, 341.72693172381076, 341.86583045077214, 342.00371830973637, 342.139775678003, 342.2763248198899, 342.41322913161366, 342.54893133004344, 342.68660062295424, 342.83118206730506, 342.9761733230061, 343.11990782389984, 343.2658006645713, 343.41204867507963, 343.61389442543185, 343.81478394930025, 344.0178591337031, 344.2215900162662, 344.4258399932063, 344.6313194041969, 344.8389571549653, 345.0470320378406, 345.2555713735794, 345.4627993129976, 345.6836329892425, 345.905177005161, 346.1272674362131, 346.3483196785114, 346.57177614739754, 346.82615971284554, 347.0814175425072, 347.33582842871186, 347.5926981830177])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.198101234436, 6371.197341728211, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.197721481323, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([30.62138845046814, 30.633068073948863, 30.644453999295273, 30.655539396318204, 30.598172637479266, 30.60987958171667, 30.619878978661568, 30.629495885012943, 30.570783578907516, 30.581110824932573, 30.5911033916883, 30.59953867531327, 30.539822331399847, 30.548995275455177, 30.557819879862834, 30.566296144622818, 30.506941800735405, 30.51631965046583, 30.525321839791907, 30.534037161172847, 30.475099458824808, 30.484135799096734, 30.492878441234346, 30.501231762589278, 30.44229406024123, 30.451391872215694, 30.458570401033377, 30.465496212851768, 30.405247114183084, 30.41293107699935, 30.420280360546286, 30.427137870472976, 30.367284922776157, 30.374962055403252, 30.382352320085204, 30.389435226254506, 30.330115033312943, 30.33801756218265, 30.345646883485557, 30.352873223627427, 30.294119936386977, 30.298402464996578, 30.3024049558502, 30.306141069326202, 30.24423906487837, 30.248808461433114, 30.25230551828816, 30.255584009089766, 30.193682004641943, 30.19782109927897])
        expected['lon_s'] = np.array([5.858789420969782, 5.833867768235699, 5.808931174462805, 5.783936524081967, 5.8172106443973295, 5.792285149681838, 5.7631689070237195, 5.733999730399534, 5.763115119284006, 5.733972836529677, 5.704847629248273, 5.671130827297328, 5.6956614517014295, 5.661977093149042, 5.6282607180849205, 5.594490555281085, 5.613712842039616, 5.574726122257183, 5.5356979944529074, 5.496592600133646, 5.515691089713471, 5.478723971491246, 5.441669341470281, 5.404569888332887, 5.42570847004043, 5.388593222090581, 5.350346724059448, 5.312016983097807, 5.331913751036876, 5.293646762438233, 5.255316594589768, 5.212105829692506, 5.227009302461473, 5.183812197942551, 5.140489161810806, 5.09718362803881, 5.113842459424471, 5.072350340877113, 5.03083986619636, 4.989284995286003, 5.0058601568543315, 4.958679344501999, 4.911492555734143, 4.864199045260505, 4.87509789274173, 4.827863719536507, 4.776903251252218, 4.725902228719733, 4.73304703347839, 4.682092968496448])
        expected['rho_s'] = np.array([343096.6202758789, 343103.4497558594, 343110.1091064453, 343116.52541503904, 343096.28001708986, 343103.0851928711, 343109.3556762695, 343115.5532470703, 343095.0891113281, 343101.6755493164, 343107.99464111327, 343114.02208251954, 343092.95034179685, 343099.05069580075, 343105.2482666015, 343111.2027954101, 343089.8394042969, 343095.745324707, 343101.4325073242, 343107.0467773437, 343085.7806030273, 343091.4677856445, 343097.22788085934, 343102.79354248045, 343081.21141357423, 343087.6034179687, 343095.28354492184, 343101.7241577148, 343083.3987915039, 343089.6449707031, 343095.5994995117, 343101.09224853513, 343081.18710937497, 343086.99581298826, 343092.63438720704, 343098.12713623047, 343077.63869628904, 343083.2772705078, 343088.69710693357, 343094.0683349609, 343073.36115722655, 343078.7566894531, 343083.8119628906, 343088.79432373046, 343067.7225830078, 343073.06950683595, 343078.1004760742, 343082.86409912107, 343061.3548828125, 343066.4344604492])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_0_30000(self):
        """Test for latitude = 90, longitude = 0, rho = 30000"""
        lat = 90
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([85.85865951139581, 85.86528650243801, 85.87190196753599, 85.87851871329444, 85.88420015002342, 85.8908100655927, 85.90044148609607, 85.91007376037308, 85.91877072562062, 85.92836287253625, 85.9379588614333, 85.96213773109514, 85.98538257238799, 86.00949442081861, 86.03358236358713, 86.05763700918344, 86.08356654170693, 86.1103968054272, 86.13721853141102, 86.1640359885266, 86.18989380406381, 86.21774176597042, 86.2455671028754, 86.27336896100512, 86.30023209501078, 86.32797290832478, 86.3591271086765, 86.39022794817534, 86.4203657310012, 86.45135792780354, 86.48229420243206, 86.52514318074671, 86.56698299861151, 86.60962258893943, 86.65214841392898, 86.694575414619, 86.73572773125514, 86.77764396186127, 86.81943212642047, 86.86104868247676, 86.90162747666605, 86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361])
        expected['lon_n'] = np.array([264.32083499216316, 264.29329566942965, 264.2659749127496, 264.2384082692594, 264.21455724867775, 264.1870998882143, 264.14551769654724, 264.104126750177, 264.06604161536495, 264.0240769331044, 263.98205760933047, 263.83900612735374, 263.69863207953165, 263.5525753143201, 263.40509786976116, 263.2560904628282, 263.09041739432035, 262.91824198572266, 262.7439901996173, 262.5673068661674, 262.39215349509163, 262.2075744629612, 262.0200447891093, 261.829564473536, 261.6406960826068, 261.4440412760238, 261.226103599987, 261.0038765651515, 260.7826877190698, 260.55292015539055, 260.3186719876158, 260.0591247991553, 259.79821157286085, 259.5255777419506, 259.24644157095054, 258.96017468245697, 258.6797817566496, 258.3867119997427, 258.0864295630723, 257.77827874847793, 257.4688984998331, 257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([48.18526771248012, 48.19166076954325, 48.19770890205329, 48.20333014774022, 48.14152376594077, 48.1477802192205, 48.15184076668208, 48.155423200901744, 48.09146872460833, 48.09570002679916, 48.099562498774795, 48.10218870651067, 48.03728141882804, 48.040600890764665, 48.04341151360812, 48.04581915529055, 47.979409225990516, 47.9809630940267, 47.98201494315888, 47.98257501867082, 47.91573820254766, 47.91704276867913, 47.91777018382574, 47.918009240446686, 47.8512475564044, 47.852056933821046, 47.85141831113365, 47.8502230280289, 47.78217385332806, 47.78152498535691, 47.78046972113014, 47.77903196630985, 47.71076422555558, 47.709855810395965, 47.70833609330564, 47.706375829013844, 47.63893795624372, 47.63846667319099, 47.63764022030142, 47.636236616426984, 47.56866213987346, 47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395])
        expected['lon_s'] = np.array([10.431394457070926, 10.388850062504563, 10.34621260661076, 10.303371952589146, 10.3727837500294, 10.330283751692642, 10.279889762222643, 10.229336117080793, 10.29091197999575, 10.240718627332619, 10.19039720862255, 10.133042402614661, 10.187670255596423, 10.130707331692163, 10.073583898342408, 10.016304224415387, 10.06012415680925, 9.992278180236221, 9.92441000554839, 9.856370222357661, 9.900196131167048, 9.834417140591594, 9.768539112273174, 9.702519357529479, 9.74821417685051, 9.682528247602496, 9.613475888867315, 9.544244237666419, 9.586760457702455, 9.517892514074864, 9.448996395916945, 9.37154461204994, 9.405763006018057, 9.328597236322546, 9.251353773225224, 9.174047130878082, 9.212407180804165, 9.139271222718962, 9.066014882549636, 8.992694509356841, 9.031144205515782, 8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431])
        expected['rho_s'] = np.array([327573.64975585934, 327586.94415283203, 327599.7767700195, 327612.51217041013, 327580.94101562497, 327594.11389160156, 327606.21738281247, 327618.0292236328, 327585.5102050781, 327597.8081298828, 327609.7414916992, 327621.0429443359, 327587.4302368164, 327598.82890625, 327609.93592529296, 327620.8728149414, 327584.87829589844, 327593.8951538086, 327602.32871093747, 327610.2518798828, 327573.4553222656, 327582.3992675781, 327590.71130371094, 327598.5858642578, 327561.9108276367, 327570.1742553711, 327577.3439941406, 327583.9790405273, 327546.30753173825, 327553.3314453125, 327559.9907958984, 327562.98021240236, 327521.7116821289, 327525.2357910156, 327527.98216552736, 327530.1452392578, 327488.8037963867, 327492.2549926758, 327495.12288818357, 327497.6262207031, 327455.6042602539, 327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_90_30000(self):
        """Test for latitude = -90, longitude = 90, rho = 30000"""
        lat = -90
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-6.744786989109073, -6.730306988068648, -6.716277779513447, -6.702590080416741, -6.756207065401341, -6.741665592658379, -6.732478988224713, -6.723668044195392, -6.782373520111662, -6.772742953381936, -6.763570009326614, -6.757060839047597, -6.818191032119202, -6.810944201409825, -6.804032049969763, -6.797632162717463, -6.864383601476007, -6.862641903237645, -6.861187072944432, -6.8602854879739965, -6.9274535682719005, -6.925110813386581, -6.923253001932338, -6.921886964098334, -6.9886657236135505, -6.986555195160022, -6.987593383913861, -6.989034553828731, -7.058839087146268, -7.05955625700912, -7.060744709924691, -7.066352295233273, -7.1403915458362235, -7.145329772606132, -7.150609508834549, -7.156449320574922, -7.22965528809911, -7.233432382710134, -7.2376875905630556, -7.242420911657874, -7.315954728262227, -7.328262729146587, -7.341035182894515, -7.35437454234355, -7.436835416193105, -7.449532737860167, -7.4656793050580745, -7.482440589281282, -7.568473652066771, -7.584729502291395])
        expected['lon_n'] = np.array([350.35105641894904, 350.3288719645248, 350.30668751010063, 350.28447573491974, 350.3770111377951, 350.35485400412756, 350.3423137768114, 350.3298555117653, 350.43236299082884, 350.4199047257827, 350.4075284230067, 350.3975563468185, 350.50271393928, 350.49287846687514, 350.4830156737137, 350.47315288055216, 350.59126051168005, 350.5942931156715, 350.5972710781496, 350.60035832365446, 350.7186572000791, 350.7172365207317, 350.71581584138437, 350.7144224827937, 350.8282953966361, 350.82703864182884, 350.8279402267993, 350.8287598494997, 350.9452009144701, 350.9461844617106, 350.9471133674377, 350.95541887746845, 351.07931850901247, 351.0877059813132, 351.0959022083172, 351.104207718348, 351.2237360288232, 351.227424330975, 351.2310853123701, 351.23477361452194, 351.3544385287806, 351.367552491987, 351.3807757382201, 351.3939443429399, 351.52339008809, 351.53661333432314, 351.5546177129753, 351.5726494123841, 351.70714949752005, 351.72534512146893])
        expected['rho_n'] = np.array([292487.1355957031, 292480.06307373045, 292473.2578979492, 292466.9631103515, 292520.0677856445, 292513.0438720703, 292511.31827392575, 292510.2974975586, 292569.08935546875, 292566.99919433595, 292565.39511718747, 292564.8604248047, 292624.673059082, 292623.40924072266, 292622.777331543, 292622.4856811523, 292690.14857177733, 292696.9537475586, 292703.8561401367, 292711.56057128904, 292779.7095458984, 292782.7961791992, 292786.49041748047, 292790.52491455077, 292855.12547607423, 292858.67388916016, 292862.78129882814, 292867.4963134765, 292933.06904296874, 292937.07923583983, 292941.5269042969, 292950.6409790039, 293021.0502441406, 293029.7025390625, 293038.8409179687, 293048.7813354492, 293116.1525756836, 293121.64532470703, 293127.52694091795, 293134.1619873047, 293201.970703125, 293212.2513793945, 293222.92092285154, 293234.46541748045, 293308.10714111326, 293319.1655517578, 293333.9425048828, 293349.740234375, 293427.4164550781, 293443.287097168])
        expected['lat_s'] = np.array([-78.96699553695964, -78.94655961096296, -78.92608270383127, -78.90552383442952, -78.88586654999821, -78.8652120579481, -78.84998073609898, -78.83470843311483, -78.82033771510112, -78.80495612909024, -78.78953356194435, -78.77495793825557, -78.76129755991553, -78.74663997395669, -78.73194140686283, -78.7171881982556, -78.70513974455969, -78.69208042286661, -78.67899378041687, -78.66585249645377, -78.65366743897448, -78.64257521176236, -78.6314283430369, -78.62028147431144, -78.6100908320698, -78.5989029822093, -78.58878064185936, -78.57864464113106, -78.56947852726489, -78.55930154540158, -78.54911090315991, -78.53949399680855, -78.53086063769763, -78.52118908983292, -78.51151754196817, -78.50181867334675, -78.49710584281945, -78.4913548235383, -78.48557648350047, -78.47979814346263, -78.47498969028695, -78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.4662060670143, -78.4636788970214, -78.46007255713963])
        expected['lon_s'] = np.array([136.90636078048658, 136.89591059105646, 136.88547406200468, 136.87501021219623, 136.86890402307824, 136.85842651289144, 136.88405338265733, 136.9095709693965, 136.93967844325792, 136.96534629415882, 136.990904862033, 136.99658757942245, 137.00672358015075, 137.01243361829688, 137.01802071303794, 137.02374441156243, 137.04441256399087, 137.06065475383718, 137.0769106040618, 137.0931527939081, 137.11382094633655, 137.1020593605858, 137.0904207182401, 137.0787820758944, 137.07140547159077, 137.05968486697503, 137.0382380729812, 137.0167093167173, 136.9996065230356, 136.97814606866342, 136.95675391618295, 136.9179721020756, 136.88354794865873, 136.84480711568642, 136.8060936034708, 136.7672161667151, 136.70971963428192, 136.64776981850991, 136.58576536122453, 136.52365162091243, 136.4660321450742, 136.39912361196477, 136.33206481469358, 136.26501967780072, 136.20220925819328, 136.13491823449033, 136.05295596445018, 135.9708161094916, 135.8930339152235, 135.81082575837323])
        expected['rho_s'] = np.array([6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.198860740661])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_90_30000(self):
        """Test for latitude = -60, longitude = 90, rho = 30000"""
        lat = -60
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([8.25467042826493, 8.279252279087814, 8.303437978938831, 8.327254848574654, 8.263344768510848, 8.2880700533063, 8.320670546214771, 8.352936359853913, 8.29720884641577, 8.330424056349543, 8.363208964365612, 8.399887080208572, 8.348469416136723, 8.386090098085148, 8.423478553601797, 8.4604913487141, 8.419298477829756, 8.467635726585925, 8.515856862126228, 8.563627205181291, 8.522523126756155, 8.569200639544022, 8.615625435332603, 8.66172238204102, 8.61865120913491, 8.6657112125163, 8.715940423672578, 8.765916917829557, 8.726226688562619, 8.777173069581735, 8.828105790222523, 8.885568171709835, 8.853015490125557, 8.911741456609334, 8.970180555147962, 9.028414748011485, 8.992918254894931, 9.049321957060897, 9.105322678065832, 9.161275587746573, 9.125881547467571, 9.196724269538947, 9.267321104800189, 9.33799307214231, 9.316805825336942, 9.388823339945546, 9.466270854944327, 9.543643237862227, 9.52881489717413, 9.607683091520272])
        expected['lon_n'] = np.array([27.638118004741063, 27.622265135677463, 27.606226143958978, 27.59020422771342, 27.646210071360233, 27.630488683438156, 27.619164229794276, 27.607831238413933, 27.66852771447325, 27.657471345754292, 27.646319354386954, 27.63786528774177, 27.70148166967127, 27.693280320025377, 27.685201913784546, 27.676805903747304, 27.74537758791965, 27.74227326694188, 27.739221879930174, 27.73597924762171, 27.804559469530517, 27.803097809048133, 27.80147222402567, 27.799824440888404, 27.86986290818501, 27.868454181668692, 27.87107697430998, 27.873597314113713, 27.947740725101493, 27.95054622530308, 27.953409782112605, 27.959755027851546, 28.037427939092932, 28.044065175418893, 28.050555562677697, 28.05699813861231, 28.134099021510707, 28.14028205025686, 28.14625675823333, 28.15234757942569, 28.2293767453378, 28.246515397512656, 28.26356013458642, 28.280560475430583, 28.36844110438655, 28.385740266006902, 28.406198390118384, 28.42648405195332, 28.51767903020404, 28.538290833571843])
        expected['rho_n'] = np.array([293947.6964477539, 293920.93752441404, 293894.5188598633, 293868.4890625, 293959.9943725586, 293933.6729248047, 293904.2405395508, 293874.8810668945, 293963.6400024414, 293933.4541870117, 293904.1433227539, 293872.35343017575, 293958.3416870117, 293925.72545166017, 293893.49808349606, 293861.1491943359, 293943.61334228516, 293907.1327392578, 293870.6278320312, 293834.46318359376, 293916.75720214844, 293877.4573120117, 293838.0359008789, 293799.0762695312, 293878.4051757812, 293838.61920166016, 293794.7258178711, 293750.85673828126, 293826.2240600586, 293781.4557250976, 293736.8089111328, 293688.5893798828, 293759.7277709961, 293710.31733398436, 293661.2714599609, 293612.2012817383, 293682.95080566406, 293632.78693847655, 293582.74459228513, 293532.7994628906, 293603.1358154297, 293541.91353759763, 293480.73986816406, 293419.1044189453, 293478.5767944336, 293415.7261352539, 293349.4728881836, 293282.58773193357, 293337.39370117185, 293269.29333496094])
        expected['lat_s'] = np.array([-67.13422555466781, -67.11291536445736, -67.0915641931119, -67.07019936138812, -67.04942192593293, -67.02797513193909, -67.01311264030514, -66.99819550715785, -66.98394773254918, -66.96894863713183, -66.95392222095782, -66.93691504992447, -66.92053625629478, -66.90343346261307, -66.88631700855302, -66.86911859222292, -66.85683791209524, -66.8438468922939, -66.83082855173583, -66.81776923004279, -66.80535194613168, -66.79338545470583, -66.78137798214496, -66.76935684920574, -66.7580187351835, -66.74597028148762, -66.73709103556658, -66.72823911040226, -66.7200292230199, -66.71114997709887, -66.70225707079953, -66.69367835320199, -66.68578265452146, -66.67721759730225, -66.66859789856969, -66.66000552059381, -66.6540222748809, -66.64739699138599, -66.64074438713439, -66.63405080174778, -66.62801291452149, -66.62855932965508, -66.62911940516702, -66.62963849954394, -66.63079963170284, -66.63129140532308, -66.62972046181397, -66.62813585792654, -66.62720695219943, -66.62558136717695])
        expected['lon_s'] = np.array([109.98509503480045, 110.00026488494706, 110.01527764074274, 110.03037235880846, 110.04385515223008, 110.05884058726909, 110.08866119318536, 110.11856376137168, 110.14681342377885, 110.17672282215433, 110.20674833374571, 110.23344071302212, 110.25856214878942, 110.28536381109255, 110.31209034131481, 110.33889200361794, 110.36147943920318, 110.38574710132423, 110.41000793325613, 110.43432340670137, 110.45699963474581, 110.46096114446442, 110.46503876739891, 110.46897295636084, 110.47134303200284, 110.47533186247811, 110.4720123905415, 110.46860412614566, 110.46362491824073, 110.46024397460157, 110.45686303096241, 110.43904989760702, 110.41946774525658, 110.40161363076618, 110.38375951627576, 110.36585076027198, 110.33519687127696, 110.30620954843944, 110.27709928219684, 110.2480231669001, 110.21721901374335, 110.18343006791929, 110.14947719755516, 110.11557896870438, 110.07986390953438, 110.045740284441, 110.00766197981818, 109.96945390160113, 109.92956559684825, 109.89126872617199])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.197721481323, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198860740661])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_90_30000(self):
        """Test for latitude = -30, longitude = 90, rho = 30000"""
        lat = -30
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([74.60422597867787, 74.60477239381147, 74.60533076177613, 74.60562275236315, 74.608137969525, 74.6084333752066, 74.60187980869797, 74.59546967616191, 74.59062024185121, 74.5841298545924, 74.57762068431339, 74.54721438967579, 74.51872054800577, 74.48838767790176, 74.45802919458833, 74.42758021126842, 74.38516985916452, 74.34059092199914, 74.2963910603327, 74.25159355711388, 74.20867947855868, 74.16834038132059, 74.1279756708731, 74.08727969625087, 74.04822296702945, 74.00742283202239, 73.96624874415825, 73.92506441101037, 73.88532978551382, 73.84401226367712, 73.80228151539563, 73.74697918123333, 73.69280724337867, 73.637199258251, 73.58128903725257, 73.52564348608448, 73.47563967117082, 73.42443545051053, 73.37294606945238, 73.32142936763756, 73.2715638640546, 73.20848365197494, 73.14552979839493, 73.08243763348423, 73.02056465734037, 72.95685777540437, 72.88176496812196, 72.80632723628646, 72.73206600453551, 72.65609381039745])
        expected['lon_n'] = np.array([61.36597120176296, 61.385806071112675, 61.40611222351512, 61.42725848918548, 61.48172924781634, 61.50237007948812, 61.556622272065525, 61.61078567218373, 61.699625942718086, 61.753297569216045, 61.80705798817321, 61.917345052701414, 62.06063559129911, 62.16858673113116, 62.27643541812566, 62.38414750133676, 62.56698483522884, 62.71528190248815, 62.86237002626437, 63.00975867836408, 63.188750615753435, 63.34747055168619, 63.50571237437705, 63.66387906498703, 63.85353292767078, 64.00935003320627, 64.1952063107115, 64.38022930513797, 64.596008641397, 64.77835420166885, 64.96072025250818, 65.16061256875777, 65.39033232111281, 65.58663195785898, 65.78218710398562, 65.97608251414394, 66.17839954754889, 66.34814340880204, 66.51718376057069, 66.68531569717972, 66.88080937160383, 67.12017335106692, 67.35732434923894, 67.59254240387584, 67.85375615849381, 68.08438432600845, 68.34354219368625, 68.60042560837043, 68.8817269493374, 69.13271591076786])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.198101234436, 6371.199620246887, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.198101234436, 6371.2, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.2, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775])
        expected['lat_s'] = np.array([-55.45073618687579, -55.430737392985975, -55.41069761796115, -55.39065784293635, -55.37035852072307, -55.35027776456323, -55.33536063141591, -55.320388856755244, -55.305184855662816, -55.290185760245464, -55.275159344071426, -55.258493682496606, -55.241595794489996, -55.22488915178013, -55.20819616944863, -55.19143488522542, -55.1815720920639, -55.17192786495585, -55.16228363784779, -55.15262575036141, -55.14269465530819, -55.132763560255, -55.12277782368844, -55.11283306825692, -55.102601444880236, -55.092629368692, -55.08462438698476, -55.076605744899155, -55.068313895246774, -55.06025427202616, -55.05218098842718, -55.048041893790156, -55.043602270829666, -55.03944951581428, -55.03525577966391, -55.03103472275683, -55.0290812886542, -55.02736008098336, -55.025652533690874, -55.02387668450666, -55.021841288134, -55.033452609723014, -55.04503661055537, -55.0566206113877, -55.067904083896565, -55.07943344321555, -55.092929897015495, -55.106385369680424, -55.11955397440019, -55.132968465930105])
        expected['lon_s'] = np.array([101.7597236135967, 101.76718218017037, 101.77469538825738, 101.7821607850202, 101.78899780437939, 101.7964632011422, 101.81417388166005, 101.83186407161038, 101.84890539358956, 101.86665705524243, 101.8844292074628, 101.90811630350441, 101.93115453157486, 101.95488260875149, 101.97861751611728, 102.00238657442893, 102.02074612291791, 102.03979552051308, 102.05888589924326, 102.0780104289193, 102.09649292081336, 102.09546156224869, 102.09449167538655, 102.09347397720022, 102.0917459393402, 102.09078971285639, 102.0789939761598, 102.06721189984151, 102.05470582347122, 102.04296472828797, 102.0311826519697, 102.01054182029792, 101.98919064895246, 101.96857713803736, 101.94799094787894, 101.92741158790969, 101.899824453852, 101.87303645192722, 101.84622795943493, 101.81943312732096, 101.79180501212826, 101.76375342520701, 101.73564036658324, 101.7075204777703, 101.67851949455442, 101.65033130384978, 101.62210896219929, 101.5938661299813, 101.5647763543062, 101.5365130315207])
        expected['rho_s'] = np.array([6371.196961975097, 6371.1965822219845, 6371.1965822219845, 6371.197341728211, 6371.196961975097, 6371.197341728211, 6371.1965822219845, 6371.197341728211, 6371.197721481323, 6371.197341728211, 6371.1965822219845, 6371.1965822219845, 6371.197341728211, 6371.197341728211, 6371.196961975097, 6371.197721481323, 6371.196961975097, 6371.197341728211, 6371.197721481323, 6371.197721481323, 6371.197341728211, 6371.197341728211, 6371.197721481323, 6371.197341728211, 6371.197341728211, 6371.196961975097, 6371.198101234436, 6371.1965822219845, 6371.197341728211, 6371.197341728211, 6371.197341728211, 6371.197341728211, 6371.197341728211, 6371.197341728211, 6371.197341728211, 6371.197721481323, 6371.196961975097, 6371.197721481323, 6371.197721481323, 6371.197341728211, 6371.197721481323, 6371.197721481323, 6371.196961975097, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_90_30000(self):
        """Test for latitude = 0, longitude = 90, rho = 30000"""
        lat = 0
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([67.66661671604103, 67.67011548044339, 67.67358009389987, 67.67706178282928, 67.67991509435505, 67.68340361347364, 67.68102329254789, 67.67871639615572, 67.67584088651515, 67.67355277314319, 67.67098632956255, 67.64523480884432, 67.61908372205963, 67.59361906928653, 67.5677736334672, 67.54216725426883, 67.50203647780042, 67.46214988059484, 67.42265772681381, 67.38269428998007, 67.34267962672756, 67.30323357672344, 67.26431345128542, 67.22502278808491, 67.18527279466274, 67.146243386198, 67.10558668516349, 67.06542346529652, 67.02401202835873, 66.9838180726405, 66.9431135602818, 66.89435796243896, 66.84447709093035, 66.79581370064132, 66.74672683862373, 66.69798660870651, 66.65499398297587, 66.61275267805392, 66.57054723162513, 66.52835886066924, 66.48560358401227, 66.42878153275963, 66.3723044060601, 66.31563432651649, 66.25819926578586, 66.20174092210652, 66.1335517285277, 66.0649476009568, 65.99583291674546, 65.92753102504531])
        expected['lon_n'] = np.array([86.79628178371618, 86.79516846288146, 86.79408246280343, 86.7929281608337, 86.79518895344897, 86.79404148166842, 86.79611785917609, 86.79818057630544, 86.80356276537141, 86.80557767117656, 86.80774967133263, 86.80637680330946, 86.80814582230448, 86.80649291652534, 86.80507906736715, 86.80350129366887, 86.81822035133025, 86.8296677483792, 86.84079412653715, 86.85215273112688, 86.86644831705972, 86.89754616835079, 86.92820688753497, 86.95902470107006, 86.99291609973167, 87.02335142267324, 87.06752225603572, 87.11131742899384, 87.15859599842867, 87.20221358646837, 87.24593362734562, 87.27305630853975, 87.30353944280552, 87.33034793529781, 87.35725888062767, 87.38393076933656, 87.39681933630038, 87.40637477094923, 87.41588239427388, 87.4252602440043, 87.43762288640204, 87.46719077531903, 87.49640349439916, 87.52544545875006, 87.55734927236318, 87.58583799139132, 87.62183308831727, 87.65778037391905, 87.69635045216212, 87.73158739809021])
        expected['rho_n'] = np.array([6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.197341728211, 6371.1984809875485, 6371.2, 6371.199620246887, 6371.199240493775, 6371.197721481323, 6371.1984809875485, 6371.199240493775, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.198101234436, 6371.2, 6371.198101234436, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.199240493775])
        expected['lat_s'] = np.array([-48.613484601261234, -48.59695554346979, -48.58042648567837, -48.56385644675191, -48.54649410588175, -48.529937727333646, -48.51808051893448, -48.50625063129203, -48.493587460570836, -48.48168927103666, -48.46977742112415, -48.457496740996476, -48.444437419303426, -48.43208843728402, -48.419753115642976, -48.4074041336236, -48.40497258627909, -48.40336066163496, -48.40174873699084, -48.400150472725045, -48.397773566893875, -48.395041491225896, -48.39237771744959, -48.38967296253824, -48.38624420757492, -48.38359409417694, -48.38138111288586, -48.37911349008141, -48.37613552760328, -48.373895225555515, -48.37169590464279, -48.37744692392394, -48.382337339369656, -48.38789711335406, -48.39345688733843, -48.39901666132283, -48.40625666184303, -48.41428896430699, -48.422307606392565, -48.43038088999154, -48.437634550890095, -48.46500994908348, -48.492371686898565, -48.519774405848665, -48.54639848323336, -48.57389682483185, -48.60709154419811, -48.64012233902429, -48.6724154734201, -48.705528230516336])
        expected['lon_s'] = np.array([100.85535875616283, 100.86004426593347, 100.86472977570409, 100.86940162509637, 100.87328117254495, 100.87794619174807, 100.89196173992494, 100.90598411829097, 100.91918004376743, 100.93322974289015, 100.9472931023912, 100.97189544378159, 100.99568499266074, 101.02028733405113, 101.04493065657654, 101.0695603187236, 101.08843896158952, 101.10817137810169, 101.12796526631638, 101.14770451301771, 101.16669243891036, 101.16246455181411, 101.15822983452871, 101.15400877762164, 101.14896809801417, 101.14476070148544, 101.12532198310758, 101.10586960435138, 101.08556345194894, 101.06608375243607, 101.04661088311236, 101.02202220210032, 100.99657974744201, 100.97197740605164, 100.94734774390456, 100.92275906289252, 100.89380589100084, 100.86568600218789, 100.83753879261828, 100.8093574321028, 100.78030180737356, 100.75301520163936, 100.72568761477014, 100.69828489582005, 100.66996693152117, 100.64244126916603, 100.6161860219965, 100.58982149180025, 100.56254171625521, 100.53611571435644])
        expected['rho_s'] = np.array([6371.196202468872, 6371.196961975097, 6371.195822715759, 6371.196202468872, 6371.197341728211, 6371.196961975097, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.196202468872, 6371.196202468872, 6371.1965822219845, 6371.196202468872, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.197721481323, 6371.199240493775, 6371.2, 6371.198860740661, 6371.198101234436, 6371.196961975097, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.196202468872, 6371.196202468872, 6371.196961975097, 6371.197721481323, 6371.199240493775, 6371.199240493775, 6371.197341728211, 6371.197721481323, 6371.196202468872, 6371.1965822219845, 6371.1965822219845, 6371.197341728211, 6371.1984809875485, 6371.199240493775, 6371.198101234436, 6371.196961975097, 6371.197341728211, 6371.197341728211, 6371.198860740661, 6371.198860740661, 6371.196961975097, 6371.196202468872, 6371.198101234436])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_90_30000(self):
        """Test for latitude = 30, longitude = 90, rho = 30000"""
        lat = 30
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([69.15238435109933, 69.15689569104612, 69.16144118193877, 69.1659456916964, 69.17005575802945, 69.17455685269249, 69.17534232444703, 69.17609706035032, 69.17642661697778, 69.17720867363775, 69.17795657935186, 69.15786045526639, 69.13732719907404, 69.11725839574524, 69.09718105468, 69.0771378645606, 69.04598878685076, 69.01526488841674, 68.98457855602317, 68.9539127141971, 68.9226406930822, 68.89064467191527, 68.85866914131586, 68.82671068618937, 68.79437657065853, 68.76245568157248, 68.72932414145603, 68.6961908937923, 68.66274687252134, 68.62968021920202, 68.59658112248412, 68.55767124232986, 68.51850352253442, 68.47976951975127, 68.44113796980568, 68.40260204250848, 68.369750540148, 68.33736690574568, 68.30502425247838, 68.27255353316414, 68.2396695874051, 68.19646864090478, 68.15335819441096, 68.11033141773446, 68.06702801839657, 68.02417541154392, 67.97124998321277, 67.9181042812809, 67.86472464536996, 67.81183678307927])
        expected['lon_n'] = np.array([89.09994749641056, 89.09591768480026, 89.09194934489248, 89.08787855214716, 89.085542627451, 89.08147866489485, 89.07952523079223, 89.07748983441957, 89.0772097966636, 89.07522904180429, 89.07317315486412, 89.06836470168842, 89.06523647504856, 89.06045534262955, 89.05564005926469, 89.05079062495399, 89.05895953120132, 89.06527745618358, 89.07165002267921, 89.07794745709396, 89.08596609917954, 89.10820519511711, 89.13038964954131, 89.15253995301967, 89.17622704906127, 89.1981792770537, 89.22950935477654, 89.2607711306077, 89.29355603862375, 89.32461973896896, 89.35571759026003, 89.37119479891928, 89.3882497812768, 89.40356989558515, 89.41880121743428, 89.43394374682418, 89.43495461482135, 89.43426476571518, 89.43345197320394, 89.43276212409778, 89.4336432185007, 89.44825982332453, 89.46269884322994, 89.4770354102978, 89.49281314728053, 89.50686284640325, 89.52664990442878, 89.54640964169762, 89.56765153001636, 89.58706975782671])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775])
        expected['lat_s'] = np.array([-49.736367700811314, -49.72048068080187, -49.7045800004141, -49.68870664078298, -49.67236882828831, -49.656522789413884, -49.64747961895279, -49.638299844708285, -49.62884686289698, -49.61989931508427, -49.61062391819138, -49.602195464755596, -49.59313863391617, -49.584614557832, -49.57606316099114, -49.56747078301524, -49.57242950035268, -49.57792097244538, -49.58348074642976, -49.58899953927914, -49.59387629434653, -49.59903991735905, -49.60444942718169, -49.60980429549099, -49.61446248450494, -49.619749050922536, -49.6278633156565, -49.63597758039049, -49.643381505450776, -49.65130452488799, -49.659309506595235, -49.67693139465388, -49.69408882984894, -49.71171071790758, -49.72933260596622, -49.746927173268176, -49.76504083494703, -49.78365993062451, -49.802251705545274, -49.82080249933102, -49.83888884025322, -49.87939186203138, -49.92008612910632, -49.96042522634443, -50.000327191475634, -50.04088485476717, -50.08798583928359, -50.135441993636846, -50.18185995923625, -50.22901558526601])
        expected['lon_s'] = np.array([103.36175814380154, 103.36567867238513, 103.36957188021204, 103.37345825784978, 103.37460572963033, 103.37849893745724, 103.39357999514463, 103.40861324150782, 103.42090075182468, 103.4360432812146, 103.45111750871281, 103.4781582276369, 103.50235075767708, 103.52937781622282, 103.55640487476856, 103.5834319333143, 103.60580763303527, 103.63106567258598, 103.65634420270419, 103.6816705441466, 103.70414186651594, 103.70188790408983, 103.6996407718529, 103.69743462075098, 103.69229148830597, 103.6900989975824, 103.67140476982406, 103.65273103263326, 103.63109982353183, 103.6124055957735, 103.59371819820434, 103.57301589483004, 103.54941076105847, 103.52874260863001, 103.5081086071474, 103.4874746056648, 103.46007871690388, 103.4356539604319, 103.41118139263575, 103.38667467389375, 103.35914901153859, 103.33584440609052, 103.31247149875072, 103.28900296876256, 103.26239938194537, 103.23882839911965, 103.2173406239908, 103.19585284886193, 103.17110024330981, 103.14943488326253])
        expected['rho_s'] = np.array([6371.1965822219845, 6371.1965822219845, 6371.196202468872, 6371.196961975097, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.1965822219845, 6371.197721481323, 6371.199240493775, 6371.197721481323, 6371.197721481323, 6371.197341728211, 6371.197721481323, 6371.197721481323, 6371.197721481323, 6371.195822715759, 6371.1965822219845, 6371.1984809875485, 6371.198860740661, 6371.197341728211, 6371.196961975097, 6371.198101234436, 6371.198860740661, 6371.196961975097, 6371.197341728211, 6371.198860740661, 6371.198860740661, 6371.197341728211, 6371.196961975097, 6371.196961975097, 6371.197341728211, 6371.197341728211, 6371.196961975097, 6371.197341728211, 6371.196961975097, 6371.196961975097, 6371.196961975097, 6371.197341728211, 6371.197721481323, 6371.197721481323, 6371.196961975097, 6371.198860740661, 6371.197721481323, 6371.197341728211, 6371.199620246887, 6371.196961975097, 6371.2, 6371.1965822219845, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_90_30000(self):
        """Test for latitude = 60, longitude = 90, rho = 30000"""
        lat = 60
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([78.86104234114536, 78.86351401585125, 78.86594641696942, 78.86842236054356, 78.87146179472421, 78.8739351769774, 78.87337424769181, 78.87282783255822, 78.8728978419972, 78.87236508724195, 78.87182635607117, 78.85209735465358, 78.83303088158549, 78.81333603111375, 78.79364971837849, 78.77401633960929, 78.7474904461939, 78.72031739235464, 78.69314519228904, 78.66597469977073, 78.63944112239253, 78.61110352129978, 78.5827889720955, 78.55439672948941, 78.526700312405, 78.49828928677869, 78.46841574689634, 78.4385089098418, 78.40931326623458, 78.37942606597389, 78.34952093646663, 78.31319372159228, 78.27761697375297, 78.24136232963855, 78.20516488835842, 78.16898366877757, 78.13804803447928, 78.10648060768278, 78.07492257239639, 78.04346784372119, 78.01276358208104, 77.97526584353767, 77.93784579839611, 77.90049747024084, 77.86388936383686, 77.8266375121036, 77.78063277294899, 77.73464852436187, 77.68942840318816, 77.64351672536098])
        expected['lon_n'] = np.array([87.97495386840694, 87.96924383026081, 87.96377284873563, 87.95819258418372, 87.95634160291866, 87.95078182893427, 87.94880107407496, 87.94692960224238, 87.94901280993923, 87.94716182867415, 87.94533816816576, 87.95450428203192, 87.96770703769755, 87.97676386853699, 87.98586168051145, 87.99496632267507, 88.02370775870249, 88.04828277933619, 88.0727894980782, 88.09718693379348, 88.1259966717126, 88.16697097654351, 88.20796577194191, 88.24886494469196, 88.29417641964582, 88.33487751690993, 88.38453982236508, 88.43416114668523, 88.48830405623592, 88.53749507863836, 88.58663145952742, 88.62307734893862, 88.6641472764179, 88.7002721469381, 88.73613747026984, 88.77181154830481, 88.79850392758122, 88.82010781592598, 88.8415136287848, 88.8628306491844, 88.88911321711062, 88.9332362391489, 88.97704507248535, 89.02062167939003, 89.06923213571302, 89.1124672331592, 89.1617811989667, 89.21064437228897, 89.26469848938045, 89.31309720983916])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.200759506225, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-58.271604314091036, -58.2574248413741, -58.243122425252096, -58.22875170723839, -58.210296536101, -58.196035101114006, -58.19088513847984, -58.185776156980666, -58.17661004311449, -58.17140543896696, -58.16628279708945, -58.16144702315708, -58.152226267777564, -58.14728121081848, -58.14234981423772, -58.13748671954866, -58.14378415396342, -58.15439826793363, -58.16512166493055, -58.175940684575835, -58.18286649639424, -58.197606044623114, -58.21238657398703, -58.22719442410761, -58.238054424887935, -58.2530261995486, -58.27581171061976, -58.2986518632043, -58.31770809098862, -58.34063020584318, -58.363415716914346, -58.3954766248784, -58.42337111744874, -58.455240780116014, -58.487151423918306, -58.51908938847728, -58.545576862078605, -58.57610780766856, -58.60648848909676, -58.63695113279502, -58.66354788942306, -58.711891968368406, -58.760386311475486, -58.80862110739412, -58.8529763558642, -58.90130677443119, -58.954022174445356, -59.00660097067612, -59.05497237037815, -59.107387242068825])
        expected['lon_s'] = np.array([119.45166209235725, 119.44677167691152, 119.44164903503402, 119.43622586483303, 119.38542291778647, 119.38021831363892, 119.38677529524213, 119.39303174852186, 119.3535532551192, 119.3599053310473, 119.3660115201653, 119.36918072794019, 119.32617785692578, 119.32938804583569, 119.33212012150369, 119.33463363111827, 119.29503219431052, 119.30316011942284, 119.31169785588536, 119.32044049802298, 119.28626223141623, 119.27036155102844, 119.25429694610057, 119.23813671852432, 119.17976592187739, 119.16435701510984, 119.13743240940165, 119.11033021877505, 119.04163217610305, 119.01495345720498, 118.98811081376684, 118.95890492487585, 118.88847201415469, 118.8595666535872, 118.83036076469622, 118.80078604559006, 118.72331804002378, 118.68683116947757, 118.64968860077104, 118.6124367490378, 118.5345589321213, 118.489711910031, 118.4443184728071, 118.39839228082793, 118.3119357463139, 118.26570902601127, 118.2164360413388, 118.16656200001935, 118.07588440859827, 118.02539565025354])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.200759506225, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_90_30000(self):
        """Test for latitude = 90, longitude = 90, rho = 30000"""
        lat = 90
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([85.85865951139581, 85.86528650243801, 85.87190196753599, 85.87851871329444, 85.88420015002342, 85.8908100655927, 85.90044148609607, 85.91007376037308, 85.91877072562062, 85.92836287253625, 85.9379588614333, 85.96213773109514, 85.98538257238799, 86.00949442081861, 86.03358236358713, 86.05763700918344, 86.08356654170693, 86.1103968054272, 86.13721853141102, 86.1640359885266, 86.18989380406381, 86.21774176597042, 86.2455671028754, 86.27336896100512, 86.30023209501078, 86.32797290832478, 86.3591271086765, 86.39022794817534, 86.4203657310012, 86.45135792780354, 86.48229420243206, 86.52514318074671, 86.56698299861151, 86.60962258893943, 86.65214841392898, 86.694575414619, 86.73572773125514, 86.77764396186127, 86.81943212642047, 86.86104868247676, 86.90162747666605, 86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361])
        expected['lon_n'] = np.array([264.32083499216316, 264.29329566942965, 264.2659749127496, 264.2384082692594, 264.21455724867775, 264.1870998882143, 264.14551769654724, 264.104126750177, 264.06604161536495, 264.0240769331044, 263.98205760933047, 263.83900612735374, 263.69863207953165, 263.5525753143201, 263.40509786976116, 263.2560904628282, 263.09041739432035, 262.91824198572266, 262.7439901996173, 262.5673068661674, 262.39215349509163, 262.2075744629612, 262.0200447891093, 261.829564473536, 261.6406960826068, 261.4440412760238, 261.226103599987, 261.0038765651515, 260.7826877190698, 260.55292015539055, 260.3186719876158, 260.0591247991553, 259.79821157286085, 259.5255777419506, 259.24644157095054, 258.96017468245697, 258.6797817566496, 258.3867119997427, 258.0864295630723, 257.77827874847793, 257.4688984998331, 257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([48.18526771248012, 48.19166076954325, 48.19770890205329, 48.20333014774022, 48.14152376594077, 48.1477802192205, 48.15184076668208, 48.155423200901744, 48.09146872460833, 48.09570002679916, 48.099562498774795, 48.10218870651067, 48.03728141882804, 48.040600890764665, 48.04341151360812, 48.04581915529055, 47.979409225990516, 47.9809630940267, 47.98201494315888, 47.98257501867082, 47.91573820254766, 47.91704276867913, 47.91777018382574, 47.918009240446686, 47.8512475564044, 47.852056933821046, 47.85141831113365, 47.8502230280289, 47.78217385332806, 47.78152498535691, 47.78046972113014, 47.77903196630985, 47.71076422555558, 47.709855810395965, 47.70833609330564, 47.706375829013844, 47.63893795624372, 47.63846667319099, 47.63764022030142, 47.636236616426984, 47.56866213987346, 47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395])
        expected['lon_s'] = np.array([10.431394457070926, 10.388850062504563, 10.34621260661076, 10.303371952589146, 10.3727837500294, 10.330283751692642, 10.279889762222643, 10.229336117080793, 10.29091197999575, 10.240718627332619, 10.19039720862255, 10.133042402614661, 10.187670255596423, 10.130707331692163, 10.073583898342408, 10.016304224415387, 10.06012415680925, 9.992278180236221, 9.92441000554839, 9.856370222357661, 9.900196131167048, 9.834417140591594, 9.768539112273174, 9.702519357529479, 9.74821417685051, 9.682528247602496, 9.613475888867315, 9.544244237666419, 9.586760457702455, 9.517892514074864, 9.448996395916945, 9.37154461204994, 9.405763006018057, 9.328597236322546, 9.251353773225224, 9.174047130878082, 9.212407180804165, 9.139271222718962, 9.066014882549636, 8.992694509356841, 9.031144205515782, 8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431])
        expected['rho_s'] = np.array([327573.64975585934, 327586.94415283203, 327599.7767700195, 327612.51217041013, 327580.94101562497, 327594.11389160156, 327606.21738281247, 327618.0292236328, 327585.5102050781, 327597.8081298828, 327609.7414916992, 327621.0429443359, 327587.4302368164, 327598.82890625, 327609.93592529296, 327620.8728149414, 327584.87829589844, 327593.8951538086, 327602.32871093747, 327610.2518798828, 327573.4553222656, 327582.3992675781, 327590.71130371094, 327598.5858642578, 327561.9108276367, 327570.1742553711, 327577.3439941406, 327583.9790405273, 327546.30753173825, 327553.3314453125, 327559.9907958984, 327562.98021240236, 327521.7116821289, 327525.2357910156, 327527.98216552736, 327530.1452392578, 327488.8037963867, 327492.2549926758, 327495.12288818357, 327497.6262207031, 327455.6042602539, 327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_180_30000(self):
        """Test for latitude = -90, longitude = 180, rho = 30000"""
        lat = -90
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-6.744793819298252, -6.730306988068648, -6.716284609702612, -6.702583250227576, -6.756200235212177, -6.741665592658379, -6.732478988224713, -6.723674874384571, -6.782373520111662, -6.772749783571115, -6.763570009326614, -6.757060839047597, -6.818184201930038, -6.810944201409825, -6.8040388801589415, -6.797638992906641, -6.864390431665171, -6.862641903237645, -6.861187072944432, -6.8602854879739965, -6.9274535682719005, -6.925110813386581, -6.923246171743173, -6.921886964098334, -6.9886657236135505, -6.986562025349187, -6.987593383913861, -6.989034553828731, -7.058832256957089, -7.05955625700912, -7.060744709924691, -7.066352295233273, -7.1403915458362235, -7.145329772606132, -7.150609508834549, -7.156456150764086, -7.22965528809911, -7.2334460430884775, -7.23769442075222, -7.242420911657874, -7.315954728262227, -7.328255898957423, -7.341035182894515, -7.35437454234355, -7.436835416193105, -7.449532737860167, -7.4656793050580745, -7.482447419470461, -7.568473652066771, -7.584729502291395])
        expected['lon_n'] = np.array([350.3510837397057, 350.3288719645248, 350.3067148308573, 350.28447573491974, 350.3770111377951, 350.35485400412756, 350.34234109756807, 350.329882832522, 350.43236299082884, 350.4199047257827, 350.4075284230067, 350.3975563468185, 350.50271393928, 350.49287846687514, 350.48304299447034, 350.47315288055216, 350.59126051168005, 350.5943204364282, 350.5972710781496, 350.60035832365446, 350.7186572000791, 350.7172365207317, 350.71581584138437, 350.7144224827937, 350.8282953966361, 350.82703864182884, 350.8279402267993, 350.8287598494997, 350.9452009144701, 350.9461844617106, 350.9471133674377, 350.95541887746845, 351.07931850901247, 351.0877059813132, 351.0959022083172, 351.10423503910465, 351.2237360288232, 351.2274516517317, 351.2310853123701, 351.2348009352786, 351.3544385287806, 351.367552491987, 351.3807757382201, 351.3939716636966, 351.52339008809, 351.53661333432314, 351.55464503373196, 351.5726494123841, 351.70714949752005, 351.72534512146893])
        expected['rho_n'] = np.array([292487.18420410156, 292480.06307373045, 292473.2822021484, 292466.9388061523, 292520.0677856445, 292513.0438720703, 292511.31827392575, 292510.2974975586, 292569.08935546875, 292566.99919433595, 292565.39511718747, 292564.8604248047, 292624.6973632812, 292623.43354492186, 292622.8259399414, 292622.4856811523, 292690.14857177733, 292697.002355957, 292703.8561401367, 292711.56057128904, 292779.73385009763, 292782.82048339845, 292786.49041748047, 292790.52491455077, 292855.12547607423, 292858.69819335936, 292862.7569946289, 292867.4963134765, 292933.04473876953, 292937.07923583983, 292941.5512084961, 292950.68958740233, 293021.0502441406, 293029.67823486327, 293038.8166137695, 293048.7327270508, 293116.1525756836, 293121.69393310545, 293127.55124511715, 293134.1619873047, 293201.970703125, 293212.2270751953, 293222.92092285154, 293234.46541748045, 293308.10714111326, 293319.1655517578, 293333.9911132812, 293349.740234375, 293427.4164550781, 293443.287097168])
        expected['lat_s'] = np.array([-78.96700919733797, -78.94655961096296, -78.92608270383127, -78.90552383442952, -78.88588021037657, -78.8652120579481, -78.84998073609898, -78.83470843311483, -78.82033771510112, -78.80494246871191, -78.78953356194435, -78.77495793825557, -78.76129755991553, -78.74663997395669, -78.73194140686283, -78.7171881982556, -78.70512608418136, -78.69208042286661, -78.67900744079523, -78.66586615683212, -78.65366743897448, -78.64256155138403, -78.6314283430369, -78.62028147431144, -78.6100908320698, -78.5989029822093, -78.58878064185936, -78.57864464113106, -78.56947852726489, -78.55930154540158, -78.54911090315991, -78.53949399680855, -78.53086063769763, -78.52118908983292, -78.51151754196817, -78.50181867334675, -78.49710584281945, -78.4913548235383, -78.48557648350047, -78.47979814346263, -78.47498969028695, -78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.4662060670143, -78.4636788970214, -78.46007255713963])
        expected['lon_s'] = np.array([136.9063334597299, 136.89589693067813, 136.88544674124802, 136.87499655181787, 136.8688903626999, 136.85842651289144, 136.88402606190064, 136.9095709693965, 136.93966478287956, 136.96533263378046, 136.990863880898, 136.99651927753075, 137.00672358015075, 137.01241995791855, 137.01802071303794, 137.02373075118408, 137.04442622436923, 137.0606274330805, 137.07688328330514, 137.09311181277306, 137.11379362557986, 137.1020593605858, 137.09040705786174, 137.07876841551604, 137.07139181121244, 137.05968486697503, 137.0382380729812, 137.01669565633898, 136.9996065230356, 136.9781324082851, 136.95672659542626, 136.91793112094058, 136.8835342882804, 136.84477979492974, 136.80607994309247, 136.76722982709342, 136.70970597390357, 136.64774249775323, 136.58575170084617, 136.52365162091243, 136.4660321450742, 136.3991372723431, 136.33206481469358, 136.2650469985574, 136.20218193743662, 136.1348772533553, 136.05294230407182, 135.9708161094916, 135.8930065944668, 135.81081209799487])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.198101234436, 6371.198860740661, 6371.2, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_180_30000(self):
        """Test for latitude = -60, longitude = 180, rho = 30000"""
        lat = -60
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-77.51865392345843, -77.35572659099694, -77.18010876705759, -76.99167750823526, -76.93147622089077, -76.71601107333356, -76.46953686694448, -76.19193065831848, -76.05748521469596, -75.69223401864033, -75.23413323100758, -74.64485183017555, -74.11351775426195, -71.85947336588768, 32.51854179433088, 36.67431233821719, 37.96499269510699, 39.66945957229451, 40.935011663227826, 41.96048601992979, 42.07366566957148, 42.86317065082784, 43.5988537414245, 44.281688243318186, 44.21548663478784, 44.86210747370212, 45.463468064081304, 46.01354417907745, 45.878419131632505, 46.41892273650688, 46.45859930539548, 46.95488085048861, 46.688527378520206, 46.974200040555985, 46.77351542236267, 46.46461670205426, 45.1011094531392, 42.77196030465149, -13.98581886788726, -17.27408367019568, -19.40905737076909, -19.84995974207176, -15.442957765039452, -20.41038359366047, -15.057229661852148, -18.920630543221478, -18.60549927529543, -17.955661417282073, -17.45035719229537, -17.152246755780993])
        expected['lon_n'] = np.array([4.074051599335342, 3.538525281375203, 2.9875817323544647, 2.4076062508231755, 2.472939571555286, 1.8181771471503536, 1.1217865340231918, 0.3547531953012923, 0.20020735872506001, 359.2492083035882, 358.11665365617336, 356.74670895396554, 355.7706476008141, 351.8432888280567, 230.22442137243274, 225.34975634292968, 224.76936784839702, 224.32972223190163, 224.1927769390429, 224.17366606974517, 224.13709723692892, 224.17128916391403, 224.23726879129634, 224.3233974767302, 224.25726758518613, 224.35791725279543, 224.45840299586467, 224.55617032364424, 224.45355356155397, 224.55914828612237, 224.48672096016355, 224.57958421211904, 224.42531755952513, 224.45453710879445, 224.31327513638024, 224.15256078520983, 223.69141373320727, 223.14543573171312, 277.24789690213936, 280.19356624586914, 280.9509249417968, 280.8162062906075, 279.02147846353824, 282.16907551989027, 278.38001441744734, 280.4186346393994, 282.2353829963527, 281.1595462398057, 280.70970998106867, 280.2280450407994])
        expected['rho_n'] = np.array([259288.085546875, 259142.33326416014, 259027.3014892578, 258892.3645751953, 258814.8341796875, 258627.0356323242, 258472.04775390626, 258272.85053710936, 258092.2217285156, 257858.41533203123, 257560.76180419922, 257248.72019042968, 256921.36693115233, 255919.01314697266, 77215.97208251953, 67548.02290344238, 64090.97537841797, 59281.781958007814, 55625.51898803711, 52668.196179199214, 52366.62967529297, 50121.298382568355, 48073.07110748291, 46217.952847290035, 46410.69426116943, 44692.23243713379, 43143.04936065674, 41761.92678375244, 42118.92812805175, 40784.266065979005, 40685.7550704956, 39511.03286743164, 40180.05455932617, 39504.8899810791, 39981.893309021, 40711.6785369873, 44108.381773376466, 50260.315364074704, 99333.6683227539, 100761.10255126953, 100429.96391296387, 100107.1251586914, 100451.6675628662, 101081.86329650879, 99949.54888305663, 100092.79783325196, 102052.20237426758, 101322.57816162109, 101055.38387145996, 100761.29090881348])
        expected['lat_s'] = np.array([-68.07565784872733, -68.05848675315394, -68.04132931795885, -68.02411724125042, -68.00646803243512, -67.98918765383499, -67.97310938852877, -67.95701746284422, -67.94050206543113, -67.92434183785488, -67.9081679499003, -67.89288198653782, -67.87713157031175, -67.86181828619257, -67.84642303980337, -67.83102779341417, -67.81732643393912, -67.80402122543595, -67.79070235655442, -67.77734250653788, -67.76357284517113, -67.756537750326, -67.74948899510255, -67.74246756063579, -67.73496801292711, -67.72789193694697, -67.72281027620448, -67.71775593621868, -67.71225080374765, -67.70718280338349, -67.70207382188434, -67.69983351983657, -67.69719706681693, -67.69495676476916, -67.69270280234306, -67.69047616067365, -67.6903258965119, -67.69057178332201, -67.69084499088882, -67.69111819845563, -67.69094061353721, -67.6949977459042, -67.69905487827117, -67.70312567101652, -67.70675933165495, -67.71083012440027, -67.7140129925535, -67.71719586070674, -67.71991427599639, -67.72309714414962])
        expected['lon_s'] = np.array([160.51139892940228, 160.48231598391635, 160.45317839691708, 160.42401348916113, 160.39572284561893, 160.3665306171063, 160.3548236728689, 160.34306208711814, 160.33210646368943, 160.3202902364253, 160.30846034878286, 160.2829837431787, 160.25825845838327, 160.2326725697524, 160.20710034149985, 160.18147347173397, 160.16797701793402, 160.15375656408207, 160.13948146871672, 160.12522003372976, 160.11155965538973, 160.09116471052806, 160.0707697656664, 160.0503201792914, 160.03049897032002, 160.0100220631883, 159.9897500617317, 159.96938243762673, 159.94960220979038, 159.92920726492872, 159.90883964082374, 159.8854803938623, 159.86258559976443, 159.83917171128962, 159.81570318130144, 159.79219367017828, 159.76249600766707, 159.73233389229227, 159.70207615426912, 159.6717637747326, 159.64182022541127, 159.614731695163, 159.58758852340137, 159.56036338936968, 159.53341146290484, 159.50604972508975, 159.46909840168, 159.43207877637852, 159.39529137750884, 159.3580805069106])
        expected['rho_s'] = np.array([6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.198860740661, 6371.2, 6371.198860740661, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.1984809875485, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_180_30000(self):
        """Test for latitude = -30, longitude = 180, rho = 30000"""
        lat = -30
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([67.06888124856384, 67.07005433355378, 67.07108227702388, 67.07238342806076, 67.07526235279593, 67.07634664532667, 67.07911116439323, 67.08143001361644, 67.08593793846865, 67.0885214575222, 67.09091031618442, 67.0882055612731, 67.08685318381744, 67.08399474964978, 67.08075382488862, 67.07777927750507, 67.08381374963678, 67.08775306124059, 67.09184092945884, 67.09579560898827, 67.10140831693873, 67.10424967563345, 67.10645241164079, 67.10897104389723, 67.11297182720357, 67.1151523650961, 67.12242993165674, 67.12934720573867, 67.13822132901781, 67.14476977288456, 67.15124137712314, 67.16945578409228, 67.18899012511851, 67.20666153204863, 67.22389068422999, 67.24105324206694, 67.25543249781711, 67.26813835722062, 67.2801458297815, 67.29224721744347, 67.30582392596617, 67.31325858687772, 67.32042174776977, 67.32725876712897, 67.33543962620735, 67.34163631533184, 67.35037383482758, 67.35864860900705, 67.36818184554109, 67.37564041211475])
        expected['lon_n'] = np.array([210.7167727254727, 210.70401393210312, 210.69124147835518, 210.6785509868773, 210.6580740797456, 210.6453426071327, 210.61406034073406, 210.58265513093033, 210.5436684111479, 210.51233150323588, 210.48096727456718, 210.41666787372068, 210.3445957175987, 210.28035095826556, 210.21602423666238, 210.15187509997762, 210.0411850542884, 209.93793991479447, 209.83465379416555, 209.73129937164492, 209.62010389195711, 209.49397761874366, 209.36756447758506, 209.24117865718316, 209.1068971381007, 208.98012882710526, 208.82524745748606, 208.67010654067838, 208.50715188746022, 208.351587498924, 208.19588650660438, 208.01013268193674, 207.8163055736701, 207.63000533386884, 207.4433772449874, 207.25659889194424, 207.077169822448, 206.90554082898393, 206.73376157135812, 206.56207793638066, 206.38255324423605, 206.17359043676868, 205.96470959157134, 205.7558560671307, 205.5392434477929, 205.33044456486562, 205.0919753401838, 204.85331487020522, 204.60671772041107, 204.367756722109])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.198101234436, 6371.1965822219845, 6371.199240493775, 6371.197721481323, 6371.1965822219845, 6371.1984809875485, 6371.196202468872, 6371.198101234436, 6371.197721481323, 6371.196202468872, 6371.199240493775, 6371.1965822219845, 6371.199620246887, 6371.196961975097, 6371.1984809875485, 6371.198101234436, 6371.196202468872, 6371.198101234436, 6371.198101234436, 6371.1965822219845, 6371.2, 6371.197341728211, 6371.198101234436, 6371.196961975097, 6371.198101234436, 6371.198860740661, 6371.1965822219845, 6371.1984809875485, 6371.196961975097, 6371.196961975097, 6371.199240493775, 6371.197341728211, 6371.197341728211, 6371.196961975097, 6371.198101234436, 6371.1965822219845, 6371.2, 6371.196961975097, 6371.197721481323, 6371.197721481323, 6371.197721481323, 6371.196202468872, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.196202468872, 6371.197341728211, 6371.197341728211, 6371.198101234436])
        expected['lat_s'] = np.array([-58.30727156193686, -58.29250469295127, -58.27771050320902, -58.262916313466775, -58.24783525577939, -58.2329864245238, -58.21903917823863, -58.205050950818446, -58.1908441573448, -58.176828609167956, -58.162813060991084, -58.14840136184236, -58.13378475701853, -58.119332076734764, -58.10485207569434, -58.090358414275585, -58.07966233803535, -58.069198488226874, -58.058679996905056, -58.04816150558324, -58.037438108586315, -58.035389051835324, -58.033339995084305, -58.03130459871164, -58.02905063628555, -58.027001579534556, -58.027001579534556, -58.027001579534556, -58.026769353102765, -58.026769353102765, -58.026769353102765, -58.02974731558089, -58.03256135351893, -58.03555297637541, -58.03854459923187, -58.04154988246668, -58.04590754315714, -58.050470109522706, -58.05505999664496, -58.05963622338888, -58.06400754445767, -58.072176450705, -58.080345356952336, -58.088555244334714, -58.09651924490694, -58.10472913228929, -58.11269313286152, -58.12067079381211, -58.128416228330906, -58.13640754965982])
        expected['lon_s'] = np.array([164.3179999576338, 164.29276923883975, 164.2675248596674, 164.24226682011667, 164.21774644099634, 164.1924474203106, 164.17802206078355, 164.16361036163482, 164.14990900215977, 164.135456321876, 164.12096266045725, 164.09477571517942, 164.06927178881858, 164.0430302020274, 164.01676129447952, 163.99045140579665, 163.97856687664083, 163.96591736629796, 163.9532678559551, 163.94059102485556, 163.92863819380804, 163.91645313632873, 163.9042680788494, 163.8920420402351, 163.88055366205114, 163.86831396305848, 163.8610876229166, 163.85386128277474, 163.84738626344156, 163.840132602543, 163.83287894164445, 163.82715524312, 163.8221418842692, 163.8163772046097, 163.81061252495022, 163.80482052453405, 163.79329116521507, 163.78099682470904, 163.76864784268966, 163.75628520029193, 163.74466021832455, 163.73643667056388, 163.7281858020465, 163.71986663163742, 163.71232610279372, 163.7039932720063, 163.68978647853268, 163.67553870392405, 163.6620422501241, 163.6477125132454])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.197721481323, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.196961975097, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.197721481323])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_180_30000(self):
        """Test for latitude = 0, longitude = 180, rho = 30000"""
        lat = 0
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([64.3821392536274, 64.38469545192429, 64.38727384833597, 64.38985224474764, 64.39250918833477, 64.39511319795585, 64.39689075468735, 64.39869221708094, 64.40059271721749, 64.40230367960457, 64.40397366085665, 64.39872295293219, 64.3935951884128, 64.38836155596128, 64.38319281030688, 64.37800015899037, 64.37524759275486, 64.37239428122909, 64.36943168667659, 64.36629150720569, 64.36319913905896, 64.35741226128467, 64.35159123256452, 64.34575995856062, 64.34000723173219, 64.33400691054632, 64.33069939144075, 64.3273389383691, 64.32399556077038, 64.32051045674638, 64.3169519281888, 64.32252023990966, 64.32789218369187, 64.33307117463004, 64.3381118542375, 64.34302276025075, 64.34502400567756, 64.34686815675346, 64.34825127006039, 64.34958144940124, 64.35091845893128, 64.34580947743211, 64.34064585441958, 64.33502460873265, 64.32928041964067, 64.3234235324274, 64.31748639049135, 64.31114455984701, 64.30449537068999, 64.29764981359435])
        expected['lon_n'] = np.array([198.8234736067562, 198.81921156871414, 198.81497685142872, 198.81075579452164, 198.8057560960492, 198.80154869952048, 198.78645398145474, 198.7713865841457, 198.75559518678463, 198.74048680834056, 198.72540575065318, 198.68646001200577, 198.64683125444134, 198.60796747806398, 198.56918566395663, 198.53041751022764, 198.4565421841648, 198.38339085815394, 198.31019855100809, 198.23695160234888, 198.16292601212427, 198.06772683547266, 197.97250033806432, 197.87730116141267, 197.78137798470902, 197.68613782692236, 197.56568061072002, 197.4452643756527, 197.324110480155, 197.20372156584438, 197.08331899115538, 196.94249415084806, 196.8009043293537, 196.66013413055973, 196.5194049129008, 196.37877131789023, 196.25524051656137, 196.13261130020297, 196.01000940460125, 195.8876124146746, 195.7646416888577, 195.61801118775588, 195.47169487535587, 195.32552882711758, 195.17889832601574, 195.03330601366775, 194.86175898247373, 194.69037587581977, 194.51846001441055, 194.34754136062014])
        expected['rho_n'] = np.array([6371.195063209533, 6371.1946834564205, 6371.194303703308, 6371.1946834564205, 6371.195063209533, 6371.195442962647, 6371.195442962647, 6371.196961975097, 6371.199620246887, 6371.1984809875485, 6371.196961975097, 6371.195063209533, 6371.195063209533, 6371.1946834564205, 6371.194303703308, 6371.195063209533, 6371.196202468872, 6371.1984809875485, 6371.199240493775, 6371.197721481323, 6371.195822715759, 6371.1946834564205, 6371.195442962647, 6371.195063209533, 6371.197341728211, 6371.199240493775, 6371.1965822219845, 6371.194303703308, 6371.1946834564205, 6371.1965822219845, 6371.199240493775, 6371.198860740661, 6371.196202468872, 6371.1946834564205, 6371.1946834564205, 6371.195442962647, 6371.1965822219845, 6371.2, 6371.196961975097, 6371.195822715759, 6371.1946834564205, 6371.195822715759, 6371.197721481323, 6371.197341728211, 6371.195442962647, 6371.195442962647, 6371.196961975097, 6371.197341728211, 6371.195063209533, 6371.195442962647])
        expected['lat_s'] = np.array([-53.08410296022305, -53.0705382045314, -53.05694612808307, -53.04332673087805, -53.02959805064634, -53.01593767230631, -53.00369797331365, -52.99144461394263, -52.97904099040991, -52.96674664990388, -52.954424988641165, -52.940559704626054, -52.92654415644918, -52.9126242309207, -52.89869064501386, -52.88470241759367, -52.87686136042652, -52.86914324666441, -52.86139781214561, -52.853652377626815, -52.845783999702945, -52.84895320747785, -52.852190717144424, -52.855387245675985, -52.858488151559186, -52.86172566122576, -52.86805041639718, -52.874402492325316, -52.88061796447002, -52.88701102153314, -52.893294795569574, -52.90226966413897, -52.911094268546606, -52.920069137116, -52.929071326442084, -52.93805985538984, -52.94766310136285, -52.95740295111929, -52.96719744238911, -52.976978273280565, -52.986677141901964, -53.00132106748248, -53.01582838927959, -53.03034937145503, -53.044761070603755, -53.05939133580594, -53.07400794062977, -53.08872016810196, -53.10335043330414, -53.118049000398])
        expected['lon_s'] = np.array([163.33524501909565, 163.3112983758656, 163.28732441187884, 163.26333678751377, 163.24026440849747, 163.2162494633757, 163.20237051898224, 163.1885052349671, 163.17555519630076, 163.16166259152897, 163.1477563263788, 163.12185624904612, 163.0968987378189, 163.07095767935118, 163.04501662088347, 163.0190482416591, 163.01173993924718, 163.00344808959477, 162.9951972210774, 162.98690537142502, 162.9795287671214, 162.9736548044352, 162.96773986061396, 162.96183857717108, 162.95685253907695, 162.95091027449905, 162.95346476524864, 162.95600559561987, 162.9594889920966, 162.9619888413328, 162.96457065283906, 162.97140084200907, 162.97920091804124, 162.98603110721126, 162.99284763600292, 162.9996641647946, 162.99853035339237, 162.99644031550636, 162.99428197572863, 162.99213729632925, 162.99090786227865, 162.99402242854018, 162.99721895707174, 163.0004154856033, 163.0045135991053, 163.00761450498848, 163.00648069358627, 163.005264919914, 163.00497805196886, 163.00374861791826])
        expected['rho_s'] = np.array([6371.197341728211, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.1984809875485, 6371.198101234436, 6371.197721481323, 6371.197341728211, 6371.196961975097, 6371.198101234436, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.196202468872, 6371.196202468872, 6371.1965822219845, 6371.196202468872, 6371.196961975097, 6371.196961975097, 6371.198101234436, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.197721481323, 6371.197341728211, 6371.196961975097, 6371.1965822219845, 6371.196202468872, 6371.1965822219845, 6371.1965822219845, 6371.197721481323, 6371.198101234436, 6371.2, 6371.197721481323, 6371.197341728211, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.197721481323, 6371.1984809875485, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_180_30000(self):
        """Test for latitude = 30, longitude = 180, rho = 30000"""
        lat = 30
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([67.73862910800779, 67.74308580644121, 67.74752030675984, 67.75196675990952, 67.75641492060649, 67.76087161903993, 67.7642269494697, 67.76760106291968, 67.77098712920072, 67.77436465774528, 67.7777319410061, 67.77435099736695, 67.77101103486281, 67.76763521386553, 67.76423719475345, 67.76083746809408, 67.75640638287003, 67.75193090141639, 67.74742297656417, 67.7428382120838, 67.73822612684674, 67.73082049423917, 67.72337558804385, 67.71587945542976, 67.70831843601856, 67.7006993599994, 67.69403992555864, 67.68729340620597, 67.680475169867, 67.67351862219734, 67.66649035754139, 67.66635716885258, 67.66612323487351, 67.66574928201645, 67.66522848009224, 67.66458131966837, 67.66377023470444, 67.6628430365246, 67.66182192324369, 67.6607290929765, 67.65953722496633, 67.65240138483095, 67.64507088430423, 67.63761914791975, 67.63006666624501, 67.62236221286123, 67.61422404246517, 67.60586901356295, 67.59735518276253, 67.58864839911806])
        expected['lon_n'] = np.array([194.72288757626902, 194.71843429293017, 194.71395368883464, 194.7094730847391, 194.70480123534682, 194.70034795200797, 194.68771210204346, 194.67511723321394, 194.66229013795265, 194.64972258987984, 194.63710040029366, 194.59517669916812, 194.55310273388085, 194.511247334647, 194.4694055957915, 194.42757751731435, 194.35700800280978, 194.28662973360196, 194.2162651247725, 194.14584587442965, 194.07526269954676, 193.98465341001736, 193.89407144124465, 193.80351679322862, 193.71277089991582, 193.62222991227816, 193.51039239480835, 193.39858219809526, 193.28670369949043, 193.17500278580405, 193.0634111551444, 192.93091914562447, 192.79841347572622, 192.66631761717818, 192.53435836241354, 192.40254937181064, 192.2855072501933, 192.1688612795478, 192.05246119571245, 191.93625235717386, 191.8200571790136, 191.68300260312813, 191.54624855556614, 191.40983601746262, 191.27351910200753, 191.13769396017264, 190.9738923634974, 190.8103503140106, 190.64691754755054, 190.48404485660242])
        expected['rho_n'] = np.array([6371.196202468872, 6371.196202468872, 6371.195822715759, 6371.195822715759, 6371.196961975097, 6371.195822715759, 6371.196202468872, 6371.195822715759, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.196202468872, 6371.197341728211, 6371.198101234436, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.2, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.196961975097, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.195822715759, 6371.195822715759, 6371.196202468872, 6371.1965822219845, 6371.1965822219845, 6371.196202468872, 6371.196202468872, 6371.1965822219845, 6371.196961975097, 6371.196961975097, 6371.197341728211, 6371.198101234436, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.197721481323, 6371.196202468872, 6371.196202468872, 6371.195822715759, 6371.196202468872])
        expected['lat_s'] = np.array([-54.8410598411821, -54.828396670460904, -54.81548761292956, -54.80257855539824, -54.78961485635355, -54.77684240260564, -54.76650149620224, -54.75629719358224, -54.746092890962245, -54.73569734304547, -54.72545205929046, -54.71389537921479, -54.70194254816727, -54.69029024544324, -54.6785013389358, -54.66669877205001, -54.663229035951645, -54.659540733799844, -54.65611197883649, -54.65257394084642, -54.64889929907295, -54.65799711104742, -54.66704028150852, -54.6760424708346, -54.68519492432242, -54.694074170243425, -54.708526850527164, -54.72308881383765, -54.737240965797895, -54.75169364608166, -54.76625560939212, -54.78401410123416, -54.80175893269785, -54.819790632106674, -54.83748082205702, -54.85540323843912, -54.873271013307885, -54.891097807041604, -54.90910218569377, -54.927106564345905, -54.944824075052935, -54.97110664297915, -54.99737555052701, -55.023671778831556, -55.04987238448771, -55.07625057506232, -55.10339374682394, -55.130591560098935, -55.15768009034721, -55.184905224378866])
        expected['lon_s'] = np.array([158.01907656164295, 157.99469278630602, 157.97041829399578, 157.94610282055055, 157.92395934726136, 157.8995345907894, 157.88893413719754, 157.87822440057894, 157.86965934335976, 157.85904522938955, 157.84832183239263, 157.82587783077997, 157.80583805575517, 157.78338039376416, 157.7610046940432, 157.7386153339439, 157.74022725858802, 157.73976280572447, 157.73914808869915, 157.73860167356557, 157.7403228812364, 157.7389568434024, 157.7376591074601, 157.73633405076112, 157.7371536734615, 157.73593789978926, 157.74433903246836, 157.7526445424991, 157.7634908829011, 157.77190567595855, 157.78023850674595, 157.7949643945965, 157.8119579052515, 157.82654718931863, 157.84132771868255, 157.85597164426306, 157.86215979565108, 157.86606666385634, 157.86986424903486, 157.87359353232168, 157.87986364597975, 157.8915842505955, 157.90330485521125, 157.91498447869196, 157.92904100800385, 157.94069331072788, 157.9502145944309, 157.95969489699885, 157.97157942615468, 157.9810050872093])
        expected['rho_s'] = np.array([6371.197341728211, 6371.199240493775, 6371.198101234436, 6371.196961975097, 6371.198101234436, 6371.199240493775, 6371.1965822219845, 6371.197341728211, 6371.199240493775, 6371.196961975097, 6371.197341728211, 6371.198860740661, 6371.197341728211, 6371.198860740661, 6371.198101234436, 6371.197721481323, 6371.2, 6371.197341728211, 6371.198860740661, 6371.198101234436, 6371.197341728211, 6371.1984809875485, 6371.1984809875485, 6371.197721481323, 6371.199620246887, 6371.197341728211, 6371.197721481323, 6371.199240493775, 6371.197341728211, 6371.197721481323, 6371.199240493775, 6371.198101234436, 6371.197721481323, 6371.199240493775, 6371.196961975097, 6371.198101234436, 6371.198860740661, 6371.198101234436, 6371.197721481323, 6371.199240493775, 6371.196961975097, 6371.197341728211, 6371.1965822219845, 6371.197341728211, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.197721481323])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_180_30000(self):
        """Test for latitude = 60, longitude = 180, rho = 30000"""
        lat = 60
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([76.69098537390018, 76.69528839307728, 76.69957177546053, 76.70385601161742, 76.70862689875267, 76.71289918207852, 76.71666944650036, 76.72043544205397, 76.72468211217043, 76.72843188602477, 76.73218080610546, 76.73113920225703, 76.73055607485664, 76.72946409836308, 76.72835077752838, 76.72720159820052, 76.72452416404587, 76.72130799872045, 76.71803463056071, 76.71470918220857, 76.71181318200048, 76.7066948089912, 76.70146629918156, 76.69614985068635, 76.69123296825758, 76.68575174144866, 76.6805710429632, 76.6752750850355, 76.6703539337385, 76.66483172579454, 76.65919596595563, 76.657940064922, 76.65701030542124, 76.65544277700671, 76.65371047027847, 76.65181423901015, 76.65186119656069, 76.65129343708594, 76.65058565873319, 76.6497421303707, 76.64922986618295, 76.64369655918159, 76.637977983299, 76.63207243098788, 76.62644435511179, 76.6201776565483, 76.61457263256065, 76.60870123119538, 76.60301680625864, 76.59661691900634])
        expected['lon_n'] = np.array([197.2279960581546, 197.21631643467387, 197.20456850930145, 197.19288888582074, 197.1788869980222, 197.16715273302813, 197.14425793893025, 197.1215134089941, 197.09631001095676, 197.07344253761553, 197.05067068692273, 196.96926849239452, 196.88555769392684, 196.80416915977696, 196.72288990865383, 196.64158333677398, 196.52869397017201, 196.41816784902286, 196.3076280674954, 196.19708828596788, 196.08404865520419, 195.95166592871098, 195.81925588146112, 195.68683217383293, 195.5519086169685, 195.41948490934027, 195.2656007473399, 195.1118122079879, 194.95555114015633, 194.8019538461011, 194.64837021242417, 194.4569473307454, 194.2631612036138, 194.07231205782531, 193.8816678177119, 193.69137874743535, 193.509695715513, 193.33097698569046, 193.15258610494806, 192.97456405442085, 192.79421973957585, 192.58984681923073, 192.38600665364086, 192.18269924280625, 191.9771105487889, 191.77481400595144, 191.53205542247085, 191.2898295937455, 191.04528150070237, 190.80408020035253])
        expected['rho_n'] = np.array([6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.200759506225, 6371.200759506225, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.1984809875485, 6371.199240493775])
        expected['lat_s'] = np.array([55.85952141483929, 55.86439133971751, 56.38346522607097, 56.43102041816718, 56.54406346402548, 56.59243144863292, 56.65493792482229, 56.71550804238197, 56.842761296808476, 56.905083357890256, 56.580891844030184, 57.039279499608085, 57.17476654707904, 57.229674437816776, 57.318927934795916, 57.393155015601025, 57.541540875319555, 57.62674748521546, 57.71160917055829, 57.800377724106355, 57.95560060318407, 58.05195749690002, 58.148321220805144, 58.24848936007797, 58.32903436586534, 58.520648492840884, 58.63294704807964, 58.747383452528624, 58.935807881161764, 59.05411017268098, 59.173761426561256, 59.285260849667125, 59.47420095758261, 59.58746598458893, 59.700949577648686, 59.81611339724428, 60.00970486398453, 60.12301770231504, 60.23772389923624, 60.35157973760576, 60.55191601615141, 60.71036957470653, 60.868396246438536, 61.02502955957985, 61.27223801130471, 61.42542720155705, 61.580477618358216, 61.72810191198428, 61.96771519335209, 62.10208208979917])
        expected['lon_s'] = np.array([189.60281384988738, 189.51829708909764, 190.2801090683642, 190.26668091645595, 190.30893246666164, 190.29580484307687, 190.27901623809697, 190.25647661383593, 190.29005382379572, 190.26883925623366, 189.5619283375157, 190.2204541961533, 190.24129993350016, 190.17969162718666, 190.1752656646045, 190.14101909610605, 190.15204302142644, 190.11740030195614, 190.0790966010907, 190.04613410815622, 190.05464452386207, 190.01567146445797, 189.97290081987535, 189.93491130771173, 189.74708110553638, 189.89563771998417, 189.84835915054933, 189.8021597510034, 189.79246088238196, 189.7465756715378, 189.70019868707342, 189.65526970271307, 189.64364472074573, 189.59881135903376, 189.5540462992135, 189.5128192773833, 189.4946236534344, 189.44951708415562, 189.40725187357157, 189.363921153477, 189.34218749153803, 189.27933609079557, 189.21850642604744, 189.15762211978594, 189.10519358771694, 189.04850301760584, 188.99403908916415, 188.94188376466192, 188.89187311955908, 188.84413009726072])
        expected['rho_s'] = np.array([247480.18200683594, 246251.24017333984, 258929.57430419922, 258844.16934814453, 258816.85142822264, 258746.6365966797, 258722.74556884766, 258617.4354736328, 258584.3817626953, 258519.51385498047, 247187.87540283203, 258426.57459716796, 258360.4914794922, 257786.7908569336, 258228.6898071289, 258161.4157836914, 258064.66076660156, 258027.47534179688, 257924.81440429686, 257904.83635253905, 257812.16444091796, 257765.01429443358, 257646.1424560547, 257601.9817260742, 254362.25627441405, 257433.26197509764, 257343.7739135742, 257262.1118041992, 257178.01927490233, 257106.97810058593, 257017.29560546874, 256943.02197265625, 256904.74285888672, 256825.6569946289, 256729.16932373046, 256675.14108886718, 256622.76553955078, 256535.68359375, 256480.51306152344, 256390.07713623045, 256353.76666259766, 256276.9896972656, 256207.16373291015, 256111.45379638672, 256012.82735595704, 255958.19151611326, 255916.6313354492, 255846.17346191406, 255784.85396728516, 255703.7508544922])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_180_30000(self):
        """Test for latitude = 90, longitude = 180, rho = 30000"""
        lat = 90
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([85.85865951139581, 85.86528650243801, 85.87190196753599, 85.87851871329444, 85.88420015002342, 85.8908100655927, 85.90044148609607, 85.91007376037308, 85.91877072562062, 85.92836287253625, 85.9379588614333, 85.96213773109514, 85.98538257238799, 86.00949442081861, 86.03358236358713, 86.05763700918344, 86.08356654170693, 86.1103968054272, 86.13721853141102, 86.1640359885266, 86.18989380406381, 86.21774176597042, 86.2455671028754, 86.27336896100512, 86.30023209501078, 86.32797290832478, 86.3591271086765, 86.39022794817534, 86.4203657310012, 86.45135792780354, 86.48229420243206, 86.52514318074671, 86.56698299861151, 86.60962258893943, 86.65214841392898, 86.694575414619, 86.73572773125514, 86.77764396186127, 86.81943212642047, 86.86104868247676, 86.90162747666605, 86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361])
        expected['lon_n'] = np.array([264.32083499216316, 264.29329566942965, 264.2659749127496, 264.2384082692594, 264.21455724867775, 264.1870998882143, 264.14551769654724, 264.104126750177, 264.06604161536495, 264.0240769331044, 263.98205760933047, 263.83900612735374, 263.69863207953165, 263.5525753143201, 263.40509786976116, 263.2560904628282, 263.09041739432035, 262.91824198572266, 262.7439901996173, 262.5673068661674, 262.39215349509163, 262.2075744629612, 262.0200447891093, 261.829564473536, 261.6406960826068, 261.4440412760238, 261.226103599987, 261.0038765651515, 260.7826877190698, 260.55292015539055, 260.3186719876158, 260.0591247991553, 259.79821157286085, 259.5255777419506, 259.24644157095054, 258.96017468245697, 258.6797817566496, 258.3867119997427, 258.0864295630723, 257.77827874847793, 257.4688984998331, 257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([48.18526771248012, 48.19166076954325, 48.19770890205329, 48.20333014774022, 48.14152376594077, 48.1477802192205, 48.15184076668208, 48.155423200901744, 48.09146872460833, 48.09570002679916, 48.099562498774795, 48.10218870651067, 48.03728141882804, 48.040600890764665, 48.04341151360812, 48.04581915529055, 47.979409225990516, 47.9809630940267, 47.98201494315888, 47.98257501867082, 47.91573820254766, 47.91704276867913, 47.91777018382574, 47.918009240446686, 47.8512475564044, 47.852056933821046, 47.85141831113365, 47.8502230280289, 47.78217385332806, 47.78152498535691, 47.78046972113014, 47.77903196630985, 47.71076422555558, 47.709855810395965, 47.70833609330564, 47.706375829013844, 47.63893795624372, 47.63846667319099, 47.63764022030142, 47.636236616426984, 47.56866213987346, 47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395])
        expected['lon_s'] = np.array([10.431394457070926, 10.388850062504563, 10.34621260661076, 10.303371952589146, 10.3727837500294, 10.330283751692642, 10.279889762222643, 10.229336117080793, 10.29091197999575, 10.240718627332619, 10.19039720862255, 10.133042402614661, 10.187670255596423, 10.130707331692163, 10.073583898342408, 10.016304224415387, 10.06012415680925, 9.992278180236221, 9.92441000554839, 9.856370222357661, 9.900196131167048, 9.834417140591594, 9.768539112273174, 9.702519357529479, 9.74821417685051, 9.682528247602496, 9.613475888867315, 9.544244237666419, 9.586760457702455, 9.517892514074864, 9.448996395916945, 9.37154461204994, 9.405763006018057, 9.328597236322546, 9.251353773225224, 9.174047130878082, 9.212407180804165, 9.139271222718962, 9.066014882549636, 8.992694509356841, 9.031144205515782, 8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431])
        expected['rho_s'] = np.array([327573.64975585934, 327586.94415283203, 327599.7767700195, 327612.51217041013, 327580.94101562497, 327594.11389160156, 327606.21738281247, 327618.0292236328, 327585.5102050781, 327597.8081298828, 327609.7414916992, 327621.0429443359, 327587.4302368164, 327598.82890625, 327609.93592529296, 327620.8728149414, 327584.87829589844, 327593.8951538086, 327602.32871093747, 327610.2518798828, 327573.4553222656, 327582.3992675781, 327590.71130371094, 327598.5858642578, 327561.9108276367, 327570.1742553711, 327577.3439941406, 327583.9790405273, 327546.30753173825, 327553.3314453125, 327559.9907958984, 327562.98021240236, 327521.7116821289, 327525.2357910156, 327527.98216552736, 327530.1452392578, 327488.8037963867, 327492.2549926758, 327495.12288818357, 327497.6262207031, 327455.6042602539, 327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_270_30000(self):
        """Test for latitude = -90, longitude = 270, rho = 30000"""
        lat = -90
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-6.744793819298252, -6.730306988068648, -6.716284609702612, -6.702590080416741, -6.756207065401341, -6.741672422847557, -6.732478988224713, -6.723674874384571, -6.782387180490005, -6.772749783571115, -6.763576839515778, -6.757067669236761, -6.818191032119202, -6.810957861788168, -6.804045710348106, -6.797638992906641, -6.864390431665171, -6.862655563615988, -6.861200733322775, -6.860292318163161, -6.9274535682719005, -6.925110813386581, -6.923253001932338, -6.921893794287513, -6.9886657236135505, -6.986562025349187, -6.987593383913861, -6.989034553828731, -7.058839087146268, -7.059563087198285, -7.060758370303034, -7.066352295233273, -7.1403915458362235, -7.145329772606132, -7.150609508834549, -7.156456150764086, -7.22965528809911, -7.233452873277642, -7.23769442075222, -7.242420911657874, -7.3159615584513915, -7.328296880092438, -7.3410488432728584, -7.354381372532714, -7.436842246382284, -7.449539568049332, -7.465699795625582, -7.482440589281282, -7.568480482255936, -7.584743162669739])
        expected['lon_n'] = np.array([350.3510837397057, 350.3288719645248, 350.3067148308573, 350.28447573491974, 350.37703845855174, 350.3548813248842, 350.34234109756807, 350.329882832522, 350.43236299082884, 350.4199047257827, 350.40755574376334, 350.39758366757513, 350.50271393928, 350.49290578763186, 350.48304299447034, 350.47315288055216, 350.5912878324367, 350.5943204364282, 350.59729839890633, 350.6003856444112, 350.7186572000791, 350.7172365207317, 350.71581584138437, 350.7144498035504, 350.82832271739284, 350.82703864182884, 350.8279402267993, 350.8287598494997, 350.94522823522675, 350.9461844617106, 350.9471133674377, 350.9554461982251, 351.07931850901247, 351.0877333020699, 351.0959022083172, 351.10423503910465, 351.2237633495799, 351.2274516517317, 351.2310853123701, 351.2348009352786, 351.3544385287806, 351.36760713350037, 351.3808030589768, 351.3939716636966, 351.52339008809, 351.53664065507985, 351.55464503373196, 351.5726494123841, 351.7071768182767, 351.72534512146893])
        expected['rho_n'] = np.array([292487.15989990236, 292480.06307373045, 292473.2822021484, 292466.9145019531, 292520.0434814453, 292513.0438720703, 292511.29396972654, 292510.2731933594, 292569.08935546875, 292566.9748901367, 292565.41942138673, 292564.8847290039, 292624.673059082, 292623.40924072266, 292622.8259399414, 292622.4856811523, 292690.14857177733, 292696.9537475586, 292703.8561401367, 292711.5119628906, 292779.73385009763, 292782.7961791992, 292786.49041748047, 292790.50061035156, 292855.12547607423, 292858.69819335936, 292862.73269042966, 292867.4720092773, 292933.06904296874, 292937.0306274414, 292941.5512084961, 292950.68958740233, 293021.0502441406, 293029.7268432617, 293038.8166137695, 293048.7327270508, 293116.1768798828, 293121.66962890624, 293127.55124511715, 293134.1619873047, 293201.970703125, 293212.27568359376, 293222.92092285154, 293234.46541748045, 293308.10714111326, 293319.189855957, 293334.01541748043, 293349.740234375, 293427.4407592773, 293443.287097168])
        expected['lat_s'] = np.array([-78.96699553695964, -78.94655961096296, -78.92606904345291, -78.90552383442952, -78.88586654999821, -78.8652120579481, -78.84998073609898, -78.83470843311483, -78.82033771510112, -78.80494246871191, -78.78951990156602, -78.77495793825557, -78.76129755991553, -78.74663997395669, -78.73194140686283, -78.7171881982556, -78.70512608418136, -78.69208042286661, -78.67899378041687, -78.66585249645377, -78.65366743897448, -78.64256155138403, -78.6314283430369, -78.62028147431144, -78.6100908320698, -78.5989029822093, -78.588766981481, -78.5786309807527, -78.56947852726489, -78.55930154540158, -78.54911090315991, -78.53948033643019, -78.5308469773193, -78.52118908983292, -78.51151754196817, -78.50181867334675, -78.49710584281945, -78.4913548235383, -78.48557648350047, -78.47978448308427, -78.47498969028695, -78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.46619240663597, -78.4636788970214, -78.46007255713963])
        expected['lon_s'] = np.array([136.90631979935156, 136.89591059105646, 136.88546040162635, 136.87499655181787, 136.86887670232156, 136.85839919213475, 136.88402606190064, 136.9095709693965, 136.93966478287956, 136.96531897340213, 136.99089120165465, 136.99653293790908, 137.00670991977242, 137.01241995791855, 137.0180070526596, 137.02373075118408, 137.04441256399087, 137.0606274330805, 137.07686962292678, 137.09311181277306, 137.11379362557986, 137.1020593605858, 137.09040705786174, 137.0787547551377, 137.07139181121244, 137.0596712065967, 137.03825173335954, 137.01668199596062, 136.99959286265727, 136.97811874790673, 136.9567402558046, 136.91794478131894, 136.88354794865873, 136.8447934553081, 136.8060662827141, 136.7672161667151, 136.70970597390357, 136.64772883737487, 136.58575170084617, 136.5236379605341, 136.46601848469587, 136.39912361196477, 136.33205115431522, 136.2650060174224, 136.20218193743662, 136.13490457411197, 136.0529286436935, 135.9708161094916, 135.8930065944668, 135.81079843761654])
        expected['rho_s'] = np.array([6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.198101234436, 6371.1984809875485, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198101234436, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_270_30000(self):
        """Test for latitude = -60, longitude = 270, rho = 30000"""
        lat = -60
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([59.90018961083255, 59.946252406595114, 59.99165267400819, 60.03737396031225, 60.07443798184333, 60.12002607945858, 60.17876912141527, 60.23736189921023, 60.28777894056868, 60.346180473066866, 60.40450004329502, 60.472525312333765, 60.53203333547751, 60.59998688752996, 60.66804972260914, 60.73571982181104, 60.80167212843668, 60.87594019037681, 60.950413157992045, 61.024633408608, 61.09069499826036, 61.16052343723998, 61.23010598940948, 61.29992759819994, 61.36123366864268, 61.43109796611545, 61.50729214140153, 61.58347094876197, 61.650953217761696, 61.72742743080374, 61.80366770986672, 61.89959600921226, 61.98699340228445, 62.08306855069714, 62.17939812365641, 62.27604188531751, 62.36234303302791, 62.45783761536116, 62.55374713168648, 62.64934587440456, 62.73621222026878, 62.84160374670937, 62.947288971284266, 63.05283417698118, 63.14973236318886, 63.25580178590457, 63.36890971855999, 63.482335255011805, 63.586949847434305, 63.70081422354029])
        expected['lon_n'] = np.array([289.81694761658025, 289.7720322925982, 289.7267891195361, 289.6813000596638, 289.60108631805116, 289.5556792204489, 289.50620133010136, 289.45636826991694, 289.3712367921019, 289.32132176964745, 289.27096961508613, 289.20979844087947, 289.11270047163856, 289.0511741275951, 288.9891560099314, 288.92650951486405, 288.84211569747936, 288.7937852789123, 288.74501772823845, 288.6956218001609, 288.6092062467819, 288.5672962060347, 288.5249763539373, 288.48194616216625, 288.4017050997969, 288.3584563419724, 288.3159725653349, 288.27277844902375, 288.1916904431974, 288.1481411570494, 288.1039908142544, 288.0731730007193, 288.00394220329207, 287.97252333311, 287.94074929309113, 287.90842883793863, 287.83712166300364, 287.8045826417977, 287.7712239978914, 287.73731893885144, 287.66390806565215, 287.6067530426775, 287.54872375548905, 287.4896562795468, 287.3904272912848, 287.33029430583207, 287.27152735821323, 287.211858825624, 287.1114550448248, 287.0504477951583])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.2, 6371.197721481323, 6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199240493775, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.197721481323, 6371.198860740661, 6371.198101234436, 6371.199240493775, 6371.2, 6371.198101234436, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198860740661])
        expected['lat_s'] = np.array([-80.59676063519473, -80.59717044654491, -80.59756659751679, -80.59793542773195, -80.59916486178255, -80.59946539010605, -80.59486184260547, -80.59020365359152, -80.58646070992634, -80.58178886053406, -80.57710335076342, -80.57654327525148, -80.5768574639533, -80.57628372806303, -80.5756963317944, -80.57508161476909, -80.5705053880252, -80.5650139159325, -80.5595224438398, -80.55398999061211, -80.54935912235484, -80.54817066943926, -80.546995876902, -80.54579376360809, -80.5454932352846, -80.54427746161235, -80.54420915972065, -80.54411353707226, -80.54494682015101, -80.5448375371243, -80.54478289561092, -80.54496048052934, -80.54610795230991, -80.54631285798501, -80.54651776366009, -80.5467226693352, -80.55422221704387, -80.5608884816738, -80.5675410859254, -80.57423467131201, -80.58180252091239, -80.5906134649417, -80.59947905048438, -80.60834463602706, -80.61812546691851, -80.62704569397457, -80.63655331729922, -80.64612924251557, -80.656552111189, -80.66612803640538])
        expected['lon_s'] = np.array([214.7157938430018, 214.62085421353862, 214.52555941423861, 214.42990944510174, 214.34846626943852, 214.25198301722293, 214.19885780585855, 214.1456233114675, 214.1068961388735, 214.05330647464558, 213.9995938670126, 213.95507469400246, 213.92514480505946, 213.88028412259084, 213.83545076087887, 213.7902758897084, 213.7616847178427, 213.71842229963985, 213.67485935311353, 213.6311734631821, 213.60225444223627, 213.55026304227414, 213.4981350385286, 213.4459114121347, 213.40824974905124, 213.35572559433385, 213.3082421192239, 213.26047177616886, 213.2275229436127, 213.1795340345042, 213.1313538800989, 213.07262791361515, 213.02872345763032, 212.96961500055303, 212.9106021661241, 212.85086533164318, 212.8189000463275, 212.7716214768927, 212.72413800178276, 212.67640863986273, 212.643637392225, 212.62475874935907, 212.6057298423314, 212.5866053126554, 212.5826847840718, 212.563328027964, 212.52248349672732, 212.4815570032206, 212.45569790702294, 212.41411571535588])
        expected['rho_s'] = np.array([6371.196961975097, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.197341728211, 6371.197341728211, 6371.1965822219845, 6371.196961975097, 6371.197341728211, 6371.196961975097, 6371.197341728211, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.196961975097, 6371.197721481323, 6371.1965822219845, 6371.1965822219845, 6371.197341728211, 6371.1965822219845, 6371.196961975097, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.196961975097, 6371.196961975097, 6371.195822715759, 6371.196202468872, 6371.196202468872, 6371.1965822219845, 6371.195822715759, 6371.196202468872, 6371.195442962647, 6371.195063209533, 6371.195442962647, 6371.195822715759, 6371.195063209533, 6371.195063209533, 6371.195822715759, 6371.195063209533, 6371.1946834564205, 6371.195442962647, 6371.195063209533, 6371.195063209533, 6371.195063209533, 6371.195822715759, 6371.1946834564205, 6371.1946834564205])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_270_30000(self):
        """Test for latitude = -30, longitude = 270, rho = 30000"""
        lat = -30
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([50.89242002020384, 50.93156383433718, 50.97049591260626, 51.00919576444355, 51.047574597389854, 51.08660912849648, 51.13598115091191, 51.18597130544723, 51.234854969337015, 51.284746086129374, 51.33397125947766, 51.382994942245425, 51.43111362494816, 51.48010998695925, 51.5287955753631, 51.577696314725806, 51.63101618648151, 51.684520473344804, 51.738171609275255, 51.7919593489891, 51.845405579244456, 51.89650222442532, 51.947393963931084, 51.99828570343685, 52.04886325424079, 52.10016480509676, 52.15624065818257, 52.212459945240944, 52.26806451527401, 52.3243725947916, 52.38075922148464, 52.456161094826996, 52.53117023229208, 52.60667455847198, 52.68242477146201, 52.7584789278701, 52.83517170696559, 52.91269776913982, 52.99048679359709, 53.0684977992024, 53.14607167270081, 53.231551490163525, 53.31734208123347, 53.40374397423413, 53.48904962187301, 53.57550274129244, 53.667857144154766, 53.76033790551674, 53.85240544043393, 53.9454326169295])
        expected['lon_n'] = np.array([275.25501162686976, 275.20577962333226, 275.15649297828145, 275.10709705020395, 275.05483244267504, 275.0055457976242, 274.9517785484779, 274.89801129933153, 274.8411294839236, 274.7873349140206, 274.7333217780642, 274.66335332020657, 274.59005173003396, 274.5197281023395, 274.44913126707826, 274.37837050727694, 274.3169261255035, 274.25840506469484, 274.1997200793461, 274.14089849021394, 274.07879841028017, 274.0276812745318, 273.97640021424337, 273.9249825501715, 273.87023175378465, 273.81862284441604, 273.7704563503891, 273.7221805733355, 273.67054434321017, 273.6220773208598, 273.57350101548263, 273.52896818209416, 273.48093829185063, 273.43613225089535, 273.39121692691333, 273.34613767839124, 273.29600408988335, 273.2491489921771, 273.20210264917404, 273.15483774011756, 273.1039664911793, 273.0465929021512, 272.9890007470696, 272.9311353844213, 272.86947243659444, 272.8111426210825, 272.75368706978435, 272.6959856316761, 272.63432268384923, 272.5761567928774])
        expected['rho_n'] = np.array([6371.196961975097, 6371.198101234436, 6371.197721481323, 6371.197721481323, 6371.197721481323, 6371.199620246887, 6371.1965822219845, 6371.2, 6371.196961975097, 6371.199620246887, 6371.196961975097, 6371.1984809875485, 6371.196961975097, 6371.198860740661, 6371.197341728211, 6371.1984809875485, 6371.2, 6371.1984809875485, 6371.197341728211, 6371.197341728211, 6371.197341728211, 6371.199620246887, 6371.198860740661, 6371.196961975097, 6371.196961975097, 6371.198101234436, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.197721481323, 6371.198860740661, 6371.200759506225, 6371.198101234436, 6371.197341728211, 6371.1965822219845, 6371.196202468872, 6371.196961975097, 6371.197341728211, 6371.198101234436, 6371.1984809875485, 6371.197341728211, 6371.1965822219845, 6371.2, 6371.197721481323, 6371.197341728211, 6371.197341728211, 6371.197721481323, 6371.197341728211, 6371.198101234436])
        expected['lat_s'] = np.array([-73.33450735866529, -73.34420622728672, -73.35391875628648, -73.36369958717793, -73.37348041806939, -73.3832749093392, -73.38573377744041, -73.38824728705495, -73.39080177780454, -73.39336992893246, -73.39589709892539, -73.40110170307292, -73.40637460911216, -73.41162019439474, -73.41689310043401, -73.42213868571656, -73.42148298755623, -73.42079996863924, -73.42011694972223, -73.41946125156193, -73.4188055534016, -73.42200208193316, -73.4252259312214, -73.42844978050965, -73.43171461093291, -73.43500676211286, -73.43611325275842, -73.4372334037823, -73.43836721518451, -73.43952834734341, -73.440716800259, -73.4431073664685, -73.44552525343468, -73.44795680077922, -73.45044298963711, -73.45291551811664, -73.46011453750182, -73.46736819840038, -73.47467650081231, -73.48198480322421, -73.48929310563614, -73.49665604956141, -73.50403265386501, -73.51132729589858, -73.51855363604045, -73.5258346176957, -73.53379861826792, -73.54178993959684, -73.54978126092576, -73.55784088414637])
        expected['lon_s'] = np.array([236.97087242322115, 236.94652962901924, 236.9221595140606, 236.89762547456192, 236.87483996349076, 236.85008735793863, 236.84470516887268, 236.83926833829335, 236.8357712814383, 236.83030713010228, 236.82492494103633, 236.8427653951484, 236.8626002644981, 236.8806866054203, 236.8987729463425, 236.91707785331812, 236.933142458246, 236.94745853474635, 236.96171996973334, 236.97603604623367, 236.992319217215, 237.00297431232022, 237.01357476591207, 237.0242025402606, 237.03687937136016, 237.0476164287354, 237.07021069450983, 237.09294156406762, 237.1176122073497, 237.14039771842087, 237.16329251251875, 237.18394700456886, 237.20684179866674, 237.22763289450026, 237.24864255638724, 237.26973418054422, 237.30585222087527, 237.34011244975204, 237.37459124468225, 237.40923396415258, 237.44592574037387, 237.50966506570845, 237.57370491936646, 237.63837315042815, 237.70539096656432, 237.77068757502965, 237.8345635041476, 237.898657999319, 237.96504743805153, 238.02971566911322])
        expected['rho_s'] = np.array([6371.193544197083, 6371.193923950195, 6371.193544197083, 6371.193544197083, 6371.193923950195, 6371.193544197083, 6371.193923950195, 6371.194303703308, 6371.1946834564205, 6371.195063209533, 6371.195822715759, 6371.195822715759, 6371.197341728211, 6371.198101234436, 6371.199240493775, 6371.2, 6371.197721481323, 6371.196202468872, 6371.195822715759, 6371.193923950195, 6371.193164443969, 6371.193164443969, 6371.193164443969, 6371.192784690857, 6371.192784690857, 6371.193164443969, 6371.1946834564205, 6371.1946834564205, 6371.196202468872, 6371.197721481323, 6371.199620246887, 6371.198860740661, 6371.1965822219845, 6371.194303703308, 6371.193544197083, 6371.192404937744, 6371.192025184631, 6371.192784690857, 6371.192404937744, 6371.193923950195, 6371.195063209533, 6371.1965822219845, 6371.1984809875485, 6371.1984809875485, 6371.195822715759, 6371.194303703308, 6371.192784690857, 6371.191645431519, 6371.192025184631, 6371.193544197083])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_270_30000(self):
        """Test for latitude = 0, longitude = 270, rho = 30000"""
        lat = 0
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([49.91934004462505, 49.95241865077542, 49.98547676635828, 50.01850073099529, 50.05155543148357, 50.08451450932347, 50.125867889653314, 50.1669378171326, 50.20809312197651, 50.249163049455795, 50.29024322221884, 50.329089923123284, 50.36801858629777, 50.40695407966143, 50.44592030887635, 50.48490019846962, 50.525857427827596, 50.566602921321305, 50.60748843369301, 50.64845932342933, 50.68961462827324, 50.728502310312706, 50.76748219990597, 50.80644159893172, 50.845360016822454, 50.884268189429434, 50.92669390945897, 50.96929038421775, 51.01209517974622, 51.05504340924726, 51.09772867646525, 51.15686103920464, 51.216420288767154, 51.27618444400476, 51.336273033227954, 51.396040603560145, 51.458212400480186, 51.5206642351562, 51.583423428344865, 51.64627482908732, 51.70909207888393, 51.77671778185622, 51.844722560327455, 51.912898093527936, 51.98092677766127, 52.049266235401824, 52.12186090099531, 52.19447264206171, 52.267190251060256, 52.34029035065232])
        expected['lon_n'] = np.array([272.8881871549202, 272.84687817082, 272.8055145452064, 272.7641782403495, 272.7222135580889, 272.6808499324753, 272.63440464611926, 272.5878773974931, 272.5406398091933, 272.49405791905383, 272.447421387401, 272.38638681697773, 272.3245872653675, 272.26325216662076, 272.20178046409063, 272.1401721577771, 272.0871152483045, 272.0345774332087, 271.9819030143296, 271.9291193124237, 271.8755159878175, 271.8295624750816, 271.7835543208324, 271.7374095627998, 271.69041786131015, 271.64405453722406, 271.6009150624263, 271.55766630460175, 271.5136798863469, 271.4702672039823, 271.42671791783425, 271.3842068204401, 271.3408214588322, 271.29803715387123, 271.25508892437017, 271.21197677032904, 271.1674166161839, 271.1233755564157, 271.07917057210733, 271.03477434250226, 270.9895038486834, 270.93786761855813, 270.88601282237937, 270.83393946014723, 270.7809098714312, 270.7283720563355, 270.6763533356167, 270.6241160488444, 270.57092253558835, 270.5182207959525])
        expected['rho_n'] = np.array([6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.197721481323, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.197341728211, 6371.197721481323, 6371.199240493775, 6371.198860740661, 6371.197721481323, 6371.197341728211, 6371.196202468872, 6371.1965822219845, 6371.197341728211, 6371.199240493775, 6371.198860740661, 6371.197721481323, 6371.196961975097, 6371.196961975097, 6371.1965822219845, 6371.197341728211, 6371.2, 6371.1984809875485, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.2, 6371.1984809875485, 6371.1965822219845, 6371.1965822219845, 6371.198101234436, 6371.199240493775, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.199620246887, 6371.196961975097, 6371.196202468872, 6371.197341728211, 6371.197721481323, 6371.196202468872, 6371.197341728211])
        expected['lat_s'] = np.array([-71.59306866750052, -71.60457070606282, -71.61611372576013, -71.62772504734914, -71.64025161428697, -71.6519722189027, -71.65462233230068, -71.65734074759033, -71.66113833276887, -71.66407531411195, -71.66691667280668, -71.67337803176153, -71.68089123984853, -71.68758482523515, -71.69423742948675, -71.70069878844157, -71.69888195812234, -71.69617720321102, -71.69325388224627, -71.69068573111832, -71.68865033474566, -71.6895792404728, -71.69061742922662, -71.69136875003534, -71.69337682565131, -71.69425108986508, -71.69169659911549, -71.6892787121493, -71.68751652334345, -71.68515327789063, -71.68236656070926, -71.6794432397445, -71.67701169239996, -71.67400640916517, -71.67074157874188, -71.66762701248038, -71.67102844668705, -71.67358293743663, -71.67609644705118, -71.67862361704408, -71.68206603238579, -71.68388286270499, -71.68593191945601, -71.68751652334345, -71.69019395749808, -71.6919698066823, -71.6931036180845, -71.69431939175678, -71.69649139191284, -71.69737931650494])
        expected['lon_s'] = np.array([235.36162521325278, 235.3558605335933, 235.35017781620385, 235.34441313654435, 235.34138053255288, 235.33542460759662, 235.35329238246538, 235.37110551582077, 235.3915687625741, 235.40902672609266, 235.42700378398814, 235.47120876829646, 235.51820046978617, 235.56235081258112, 235.60671972142953, 235.65207217751842, 235.69821693555102, 235.74146569337555, 235.78567067768387, 235.82911068080514, 235.8765668351584, 235.91563551721086, 235.95467687850666, 235.99456518325954, 236.0366664693035, 236.07669137783975, 236.13007613639257, 236.1834608949454, 236.24047931413668, 236.29386407268947, 236.34858754831964, 236.40246408049268, 236.46062997146453, 236.51508023952786, 236.57040477180496, 236.62551073802862, 236.69616221480322, 236.76410893666653, 236.83238350760996, 236.90087664460685, 236.97251166862196, 237.07788782713692, 237.1830727403551, 237.28973297443403, 237.3990706426676, 237.50581283901656, 237.61665314886753, 237.72752077947516, 237.84172154239778, 237.95406449386616])
        expected['rho_s'] = np.array([6371.193544197083, 6371.192784690857, 6371.193923950195, 6371.195063209533, 6371.196961975097, 6371.198101234436, 6371.195442962647, 6371.192784690857, 6371.194303703308, 6371.196961975097, 6371.1965822219845, 6371.192784690857, 6371.192784690857, 6371.196961975097, 6371.197721481323, 6371.193923950195, 6371.193544197083, 6371.196961975097, 6371.193164443969, 6371.197721481323, 6371.193164443969, 6371.194303703308, 6371.197341728211, 6371.192404937744, 6371.1965822219845, 6371.194303703308, 6371.193923950195, 6371.195822715759, 6371.192784690857, 6371.198860740661, 6371.192404937744, 6371.199620246887, 6371.192784690857, 6371.1965822219845, 6371.192784690857, 6371.193923950195, 6371.193923950195, 6371.193923950195, 6371.193923950195, 6371.193544197083, 6371.194303703308, 6371.192025184631, 6371.199620246887, 6371.192025184631, 6371.193164443969, 6371.196961975097, 6371.191645431519, 6371.193164443969, 6371.198101234436, 6371.192784690857])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_270_30000(self):
        """Test for latitude = 30, longitude = 270, rho = 30000"""
        lat = 30
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([57.0215312530498, 57.048175821002026, 57.07480331348132, 57.10140007010935, 57.127768015400186, 57.15430330032568, 57.187504849881115, 57.22070298434196, 57.25369962822229, 57.28680897022393, 57.319911482036396, 57.35587925820568, 57.39170018530781, 57.427630395436665, 57.46350254895757, 57.499436174181014, 57.535356139026106, 57.571457103884214, 57.60760246497192, 57.64380588266757, 57.67986586639066, 57.71490473683282, 57.75006313558546, 57.78525910037853, 57.82033895195572, 57.8556476148701, 57.895399315839576, 57.9352534696466, 57.97499834042691, 58.015033494246936, 58.05511304429657, 58.10933108592813, 58.163351052073764, 58.2176203201241, 58.2720569278091, 58.32659598833165, 58.383570011293315, 58.44089237390264, 58.49835475538996, 58.55598789160653, 58.61356297121515, 58.67394867366723, 58.734494885564814, 58.7951981918133, 58.855761479183805, 58.91656723826984, 58.98045682776614, 59.0445376625592, 59.10862191244685, 59.173112558590105])
        expected['lon_n'] = np.array([272.28857850806315, 272.26052009095275, 272.2323797115723, 272.2043486152186, 272.1752246885976, 272.14713895073055, 272.1133158539606, 272.0794654364341, 272.0444948678836, 272.010644450357, 271.97673939131704, 271.92649651978246, 271.8750515349539, 271.8245627766092, 271.7739647352377, 271.7232300900829, 271.67847869064093, 271.63473815919616, 271.59086102396805, 271.54690192646984, 271.501713394921, 271.46463912810617, 271.4274828990213, 271.3902447076664, 271.3517497615042, 271.31426568333916, 271.2801147374891, 271.24582718785564, 271.210364845685, 271.1759133715114, 271.14135261431113, 271.10725630997445, 271.0718486093171, 271.03750641817027, 271.0030003024834, 270.96841222452645, 270.9325947125189, 270.8978153892652, 270.8628994622281, 270.8277922898942, 270.7914010419964, 270.7505018692463, 270.70943877195623, 270.6681844293693, 270.6255367281918, 270.5838725742547, 270.54433943933867, 270.50461505912585, 270.46352464107906, 270.42341777027275])
        expected['rho_n'] = np.array([6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.200759506225, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.1984809875485])
        expected['lat_s'] = np.array([-73.66996526956132, -73.67299787355279, -73.67616708132769, -73.67924066645418, -73.69410315808813, -73.69725870548467, -73.6963161393792, -73.6953872336521, -73.70627455518908, -73.70541395135368, -73.70455334751824, -73.71104202722975, -73.72933327382705, -73.73587659505193, -73.74248821816849, -73.7491818035551, -73.75914021936501, -73.75725508715408, -73.75546557759154, -73.75370338878565, -73.76383938951398, -73.7672681444773, -73.77077886171071, -73.77433056007911, -73.78958920268491, -73.7930725991616, -73.79797667498568, -73.80283976967473, -73.81939614822284, -73.8241636202635, -73.82899939419588, -73.83259207369932, -73.84763215025168, -73.85103358445835, -73.85451698093505, -73.85806867930347, -73.88056732242947, -73.89144098358815, -73.90242392777353, -73.9134068719589, -73.93583721319322, -73.95274876157816, -73.96974227223316, -73.98664016023977, -74.0148351811336, -74.03167842762684, -74.04945057984722, -74.0671954113109, -74.09607345112173, -74.11369533918037])
        expected['lon_s'] = np.array([207.05028619787484, 207.01711879926526, 206.9837191742239, 206.95064739826267, 206.9550597004665, 206.92094973575146, 206.94051139753438, 206.96019600272237, 207.01777449742556, 207.03734981958684, 207.0571027266665, 207.08878114403703, 207.15847639432783, 207.19012749094168, 207.22205179512233, 207.25429028800477, 207.3633820694282, 207.43414282922956, 207.505149475841, 207.57666155645103, 207.68619046998137, 207.75331756914426, 207.82086814003569, 207.8887328996289, 207.9955160771129, 208.063736006543, 208.15064333354223, 208.23871179270037, 208.36516591499398, 208.4537671289074, 208.54268253152262, 208.65222510543128, 208.80097296517582, 208.91118489762314, 209.02186128293403, 209.132701592785, 209.2803702826407, 209.39033632827787, 209.50079414753532, 209.6118393630614, 209.76151612853306, 209.9287328197933, 210.09648226580882, 210.2658163157118, 210.47405512312713, 210.64450932405396, 210.82519514835747, 211.00685085952313, 211.22776649803802, 211.41088386968607])
        expected['rho_s'] = np.array([6371.197341728211, 6371.1965822219845, 6371.198860740661, 6371.198101234436, 6371.197341728211, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.197341728211, 6371.196961975097, 6371.197721481323, 6371.197341728211, 6371.199240493775, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.196202468872, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.197721481323, 6371.196961975097, 6371.1965822219845, 6371.198101234436, 6371.197341728211, 6371.1984809875485, 6371.196961975097, 6371.196961975097, 6371.197721481323, 6371.1965822219845, 6371.196202468872, 6371.196202468872, 6371.197721481323, 6371.199240493775, 6371.197341728211, 6371.196961975097, 6371.1965822219845, 6371.197341728211, 6371.1965822219845, 6371.197341728211, 6371.195822715759, 6371.1965822219845, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.2, 6371.199620246887])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_270_30000(self):
        """Test for latitude = 60, longitude = 270, rho = 30000"""
        lat = 60
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([70.11249208989697, 70.1292499590256, 70.14596855456651, 70.16268202746552, 70.17847513237388, 70.19514933168517, 70.2164629369902, 70.23772360832916, 70.25806391167745, 70.27929213961787, 70.30047938642323, 70.33132110562043, 70.36125270211072, 70.39206710055123, 70.42285247068777, 70.45363442572972, 70.48399803168502, 70.51523419430677, 70.54650450787439, 70.577774821442, 70.60817770098501, 70.6406142693534, 70.67310206414058, 70.70561376458983, 70.7372358328997, 70.76979192957856, 70.80753213983748, 70.84529967085305, 70.88217586218195, 70.91999291206903, 70.9578321600709, 71.00929592791965, 71.0598956768384, 71.11144311450448, 71.16304519368393, 71.21470020682945, 71.26786469178153, 71.32199052585929, 71.37614026559916, 71.43031220345382, 71.48360816954744, 71.54002894718633, 71.59650265879128, 71.65299173832187, 71.70856386495639, 71.76508880298012, 71.82615240170733, 71.88728088723164, 71.94760341043391, 72.00890777332935])
        expected['lon_n'] = np.array([269.5050032137162, 269.48669830674055, 269.46844804127824, 269.4501431343026, 269.4296525667926, 269.41134765981695, 269.3864857712381, 269.3616238826593, 269.33452169203264, 269.3096324826971, 269.28479791487496, 269.2333529300464, 269.1796130016568, 269.12797677153145, 269.0762039376228, 269.02432182068736, 268.9747892888264, 268.9273877759865, 268.8797676970932, 268.83203833517314, 268.7818227843952, 268.7410602154286, 268.70005175965184, 268.6589067000917, 268.61538473470034, 268.57391182606005, 268.53418744584724, 268.4942718203377, 268.45184268521353, 268.41157188986716, 268.3711644907374, 268.3304019217707, 268.28709852243287, 268.24589882135933, 268.2045625165024, 268.16278907953864, 268.11716341588294, 268.0735321674649, 268.0297096737501, 267.98564129322517, 267.9387588747622, 267.88613909739644, 267.83313682943714, 267.7797520708843, 267.7234713121234, 267.66940353465355, 267.61926994614566, 267.5688358293143, 267.5154237500048, 267.46419733122974])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.198860740661, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887])
        expected['lat_s'] = np.array([32.93367727698967, 32.929780654068175, 32.925675710377, 32.921229257227324, 32.86978427239879, 32.866167687233265, 32.85057778045271, 32.83462587364615, 32.77128269928345, 32.75613333970436, 32.74036584800538, 32.7171534501111, 32.645829199703236, 32.6231529716588, 32.60021036623672, 32.57663255322184, 32.489233452602356, 32.450847789466884, 32.411977182900344, 32.37253967063269, 32.28407506050268, 32.24699396349868, 32.209216187199345, 32.171083241063165, 32.083240178147626, 32.045530703739985, 32.000779304298064, 31.95524926329076, 31.85940121866797, 31.814492724875137, 31.76888072159779, 31.708023736092976, 31.59583104878635, 31.534871610443986, 31.47326330413047, 31.410828544927384, 31.30202363144908, 31.245025702825323, 31.187474528878795, 31.129308637906966, 31.018441007299323, 30.94342503964507, 30.867637260614607, 30.791145972099635, 30.66079181178994, 30.58405463646485, 30.495596856524017, 30.406141868964355, 30.261854122747835, 30.17176392759537])
        expected['lon_s'] = np.array([329.53433371566433, 329.5098816384357, 329.48540224045036, 329.4611960500318, 329.5619003591545, 329.53753024419586, 329.510619298866, 329.4838449573196, 329.58209039834105, 329.55515213225453, 329.5283777907081, 329.5001281283009, 329.59684360694826, 329.5684846615144, 329.5403442821339, 329.5122858650235, 329.60370111687496, 329.570479076752, 329.53739364041246, 329.5044448078563, 329.5958600597078, 329.56307515169175, 329.5303722059457, 329.49788782625313, 329.5893850403746, 329.55687333992535, 329.52283167710203, 329.48879001427866, 329.57867530375603, 329.54471560320275, 329.5108378649195, 329.4720697111905, 329.5567094153853, 329.5179139408996, 329.4793097117107, 329.44084208630517, 329.5269571113607, 329.4900194483293, 329.4532457098379, 329.41660857513, 329.50280556245553, 329.4653214842905, 329.4280832929356, 329.39122759217423, 329.4767142398261, 329.43996782209143, 329.4016368004693, 329.36368826944073, 329.4476722754752, 329.4100242727701])
        expected['rho_s'] = np.array([319819.51651611325, 319841.3902954101, 319862.7536865234, 319884.1656860351, 319819.12764892576, 319840.9771240234, 319853.2264404297, 319865.2813232422, 319790.6188232422, 319802.89244384767, 319814.94732666016, 319821.55806884763, 319741.40281982423, 319748.4753417969, 319755.11038818356, 319761.7697387695, 319665.35498046875, 319656.28951416013, 319647.1025268555, 319636.84615478513, 319539.19188232423, 319531.0256713867, 319522.2275512695, 319512.724609375, 319415.3133789062, 319406.3208251953, 319390.69322509767, 319374.36080322263, 319269.4881835937, 319253.30158691405, 319236.2886474609, 319203.3078491211, 319080.76607666013, 319046.66728515626, 319012.15532226564, 318976.5982788086, 318856.8514892578, 318826.3983276367, 318794.87578125, 318762.81854248047, 318640.54411621095, 318595.45982666017, 318549.11171875, 318501.9858764648, 318364.1324584961, 318315.76710205077, 318256.07598876953, 318195.55853271484, 318042.8552490234, 317979.98028564453])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_270_30000(self):
        """Test for latitude = 90, longitude = 270, rho = 30000"""
        lat = 90
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([85.85865951139581, 85.86528650243801, 85.87190196753599, 85.87851871329444, 85.88420015002342, 85.8908100655927, 85.90044148609607, 85.91007376037308, 85.91877072562062, 85.92836287253625, 85.9379588614333, 85.96213773109514, 85.98538257238799, 86.00949442081861, 86.03358236358713, 86.05763700918344, 86.08356654170693, 86.1103968054272, 86.13721853141102, 86.1640359885266, 86.18989380406381, 86.21774176597042, 86.2455671028754, 86.27336896100512, 86.30023209501078, 86.32797290832478, 86.3591271086765, 86.39022794817534, 86.4203657310012, 86.45135792780354, 86.48229420243206, 86.52514318074671, 86.56698299861151, 86.60962258893943, 86.65214841392898, 86.694575414619, 86.73572773125514, 86.77764396186127, 86.81943212642047, 86.86104868247676, 86.90162747666605, 86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361])
        expected['lon_n'] = np.array([264.32083499216316, 264.29329566942965, 264.2659749127496, 264.2384082692594, 264.21455724867775, 264.1870998882143, 264.14551769654724, 264.104126750177, 264.06604161536495, 264.0240769331044, 263.98205760933047, 263.83900612735374, 263.69863207953165, 263.5525753143201, 263.40509786976116, 263.2560904628282, 263.09041739432035, 262.91824198572266, 262.7439901996173, 262.5673068661674, 262.39215349509163, 262.2075744629612, 262.0200447891093, 261.829564473536, 261.6406960826068, 261.4440412760238, 261.226103599987, 261.0038765651515, 260.7826877190698, 260.55292015539055, 260.3186719876158, 260.0591247991553, 259.79821157286085, 259.5255777419506, 259.24644157095054, 258.96017468245697, 258.6797817566496, 258.3867119997427, 258.0864295630723, 257.77827874847793, 257.4688984998331, 257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([48.18526771248012, 48.19166076954325, 48.19770890205329, 48.20333014774022, 48.14152376594077, 48.1477802192205, 48.15184076668208, 48.155423200901744, 48.09146872460833, 48.09570002679916, 48.099562498774795, 48.10218870651067, 48.03728141882804, 48.040600890764665, 48.04341151360812, 48.04581915529055, 47.979409225990516, 47.9809630940267, 47.98201494315888, 47.98257501867082, 47.91573820254766, 47.91704276867913, 47.91777018382574, 47.918009240446686, 47.8512475564044, 47.852056933821046, 47.85141831113365, 47.8502230280289, 47.78217385332806, 47.78152498535691, 47.78046972113014, 47.77903196630985, 47.71076422555558, 47.709855810395965, 47.70833609330564, 47.706375829013844, 47.63893795624372, 47.63846667319099, 47.63764022030142, 47.636236616426984, 47.56866213987346, 47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395])
        expected['lon_s'] = np.array([10.431394457070926, 10.388850062504563, 10.34621260661076, 10.303371952589146, 10.3727837500294, 10.330283751692642, 10.279889762222643, 10.229336117080793, 10.29091197999575, 10.240718627332619, 10.19039720862255, 10.133042402614661, 10.187670255596423, 10.130707331692163, 10.073583898342408, 10.016304224415387, 10.06012415680925, 9.992278180236221, 9.92441000554839, 9.856370222357661, 9.900196131167048, 9.834417140591594, 9.768539112273174, 9.702519357529479, 9.74821417685051, 9.682528247602496, 9.613475888867315, 9.544244237666419, 9.586760457702455, 9.517892514074864, 9.448996395916945, 9.37154461204994, 9.405763006018057, 9.328597236322546, 9.251353773225224, 9.174047130878082, 9.212407180804165, 9.139271222718962, 9.066014882549636, 8.992694509356841, 9.031144205515782, 8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431])
        expected['rho_s'] = np.array([327573.64975585934, 327586.94415283203, 327599.7767700195, 327612.51217041013, 327580.94101562497, 327594.11389160156, 327606.21738281247, 327618.0292236328, 327585.5102050781, 327597.8081298828, 327609.7414916992, 327621.0429443359, 327587.4302368164, 327598.82890625, 327609.93592529296, 327620.8728149414, 327584.87829589844, 327593.8951538086, 327602.32871093747, 327610.2518798828, 327573.4553222656, 327582.3992675781, 327590.71130371094, 327598.5858642578, 327561.9108276367, 327570.1742553711, 327577.3439941406, 327583.9790405273, 327546.30753173825, 327553.3314453125, 327559.9907958984, 327562.98021240236, 327521.7116821289, 327525.2357910156, 327527.98216552736, 327530.1452392578, 327488.8037963867, 327492.2549926758, 327495.12288818357, 327497.6262207031, 327455.6042602539, 327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_0_60000(self):
        """Test for latitude = -90, longitude = 0, rho = 60000"""
        lat = -90
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-12.808027877599443, -12.979076304984062, -13.164181261680582, -13.315866102768226, -12.416473623050123, -12.579223370593184, -13.213433755785545, -13.839345461136347, -13.452483546546816, -14.051245249946817, -14.629290989594168, -15.416087800844622, -15.348646512979911, -16.122042493078794, -16.846800696098015, -17.581127994144424, -17.88578858207285, -18.93506273293771, -19.853873440466174, -20.77461026134057, -21.030571600486795, -21.85780996181289, -22.639682206671765, -23.337713879468723, -23.54028362987296, -24.203331073741126, -24.965375279439442, -25.693582727989522, -25.94321248177515, -26.604374793432385, -27.240101480620496, -27.98644991160613, -28.32595129449078, -29.00697579625441, -29.66934022120556, -30.26980947189807, -30.52061401822094, -31.027072545177376, -31.534514619374306, -32.009103483663466, -32.2034770070637, -32.80199282365358, -33.378474449980985, -33.885055920342495, -34.155941222825206, -34.57930366833928, -34.911032295948445, -35.23009775283643, -35.431274144649976, -35.724740052528745])
        expected['lon_n'] = np.array([355.49011807122326, 355.4970029019067, 355.5121112803507, 355.51664652595963, 355.44370010562386, 355.44809874744936, 355.5440765656664, 355.64040955372025, 355.6633863100882, 355.75354480713236, 355.8394139453777, 355.96459765248574, 356.04688777160607, 356.171306497527, 356.288293977631, 356.4136962507924, 356.56565429944686, 356.7546046526461, 356.9154692679782, 357.08761735581925, 357.2337560833008, 357.3851130753083, 357.52977648192916, 357.6531296983396, 357.78782102877227, 357.90784111286774, 358.05132972695134, 358.1929058880674, 358.3341542001032, 358.46073126580194, 358.5843030482658, 358.7336109835223, 358.8964700140921, 359.0332650427891, 359.17115290175326, 359.2920472500625, 359.43769420392385, 359.5337539844109, 359.635359878504, 359.72912471543, 359.8586251020934, 359.98451914887505, 0.10806184359147995, 0.20914364739781677, 0.3554047846759409, 0.4288793707335895, 0.45909527384807963, 0.48788185315703886, 0.6056764161609894, 0.6296286089197548])
        expected['rho_n'] = np.array([312165.0061889648, 311824.7717041015, 311692.6297729492, 311433.15814208984, 312780.4128173828, 312417.08934326173, 311539.70775146486, 310755.4841552734, 311410.6767578125, 310614.34967041016, 309826.0672729492, 308792.7013305664, 309079.7096191406, 308128.07869873045, 307290.410168457, 306703.4151489258, 306440.5895385742, 305490.87864990236, 304513.6311035156, 303960.7591796875, 303928.26446533203, 303400.2557373047, 302979.9389160156, 302423.25123291014, 302508.2187133789, 302104.16140136716, 301739.938671875, 301588.3047729492, 301536.58543701173, 301349.32158203126, 301264.9860107422, 301117.0949584961, 301077.308984375, 300993.5810180664, 301119.2580322265, 301120.81350097654, 301296.5328613281, 301194.8927001953, 301314.71240234375, 301428.1887084961, 301517.7496826172, 301707.56547851564, 302019.8744384766, 302077.815649414, 302133.20491943357, 301862.86931152345, 300665.0125488281, 299566.8273071289, 299156.49951171875, 298128.33466796874])
        expected['lat_s'] = np.array([-75.09847201371281, -75.07509910637302, -75.05167155751985, -75.02817570677502, -75.01167396974029, -74.98808249634706, -74.96914921196776, -74.95016128607514, -74.93819479464926, -74.91911124610826, -74.89997305605388, -74.88188671513169, -74.8708627898113, -74.85269448661904, -74.83447154191347, -74.81620761607286, -74.80798406831215, -74.79260248230128, -74.77715259439873, -74.76167538573947, -74.75336987570873, -74.74229130887497, -74.73119908166288, -74.72012051482912, -74.71622730700221, -74.70508043827675, -74.69661100370593, -74.68816888989178, -74.68693945584118, -74.6784427005137, -74.66993228480786, -74.66355288812306, -74.66444081271516, -74.65802043489535, -74.65158639669721, -74.64516601887738, -74.64975590599963, -74.64702383033162, -74.64427809428528, -74.64153235823895, -74.64613590573953, -74.65020669848485, -74.65431847236522, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67629802111432, -74.67860662505376, -74.68829183329686, -74.69058677685797])
        expected['lon_s'] = np.array([142.56377914925147, 142.54589771400438, 142.52800261837893, 142.51013484351017, 142.51064027750877, 142.49274518188332, 142.50681537157357, 142.52081725937208, 142.5532469975513, 142.56735816837656, 142.58144201844513, 142.57862798050706, 142.59409152878797, 142.59127749084993, 142.58849077366858, 142.58564941497386, 142.61641258699558, 142.628843531285, 142.6414384001145, 142.6538966651606, 142.68463251642567, 142.67400474207713, 142.6632130431885, 142.65239402354322, 142.65993455238691, 142.64926579690334, 142.63264111646353, 142.61582519072695, 142.61739613423606, 142.60083975568796, 142.58417409411314, 142.55827401678044, 142.55061054453168, 142.5247787690907, 142.49889235213635, 142.47296495404697, 142.4479254805497, 142.4047450246169, 142.36135966300895, 142.31798796177938, 142.29262063920197, 142.25000025878109, 142.20713399155008, 142.164103799779, 142.13929655271352, 142.09623904018576, 142.0419526966625, 141.98754340973417, 141.95116582221468, 141.89661993150295])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_0_60000(self):
        """Test for latitude = -60, longitude = 0, rho = 60000"""
        lat = -60
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-0.7024942908031448, -0.6976926678166251, -0.6933145165586581, -0.6893393464617077, -0.7245489716331264, -0.7196995373224127, -0.7178075749223183, -0.716284442737404, -0.7541646718742925, -0.7518833886915104, -0.749868482886356, -0.749438180968653, -0.7887049385070526, -0.7875438063481539, -0.7867036930802414, -0.7861914288924794, -0.8284429790981847, -0.8300549037423082, -0.8318512434940146, -0.8340300738392585, -0.8764933599092188, -0.8771968693937282, -0.8782418883367455, -0.8794918129548535, -0.9213198914320202, -0.9218458159981111, -0.9237924199115639, -0.926025891770152, -0.9691107250545912, -0.9705177440236241, -0.9723140837753306, -0.9761458198997133, -1.0210748042600528, -1.0240322761706722, -1.027303936783099, -1.0308078238273168, -1.0749786571897886, -1.0769252611032414, -1.0791314122051574, -1.081624431252223, -1.1258430759388744, -1.131204774437336, -1.1366689257733498, -1.142481416757036, -1.1904020239738458, -1.195285609230396, -1.2017879493202486, -1.2084268931935043, -1.2576998778659743, -1.2634235763904513])
        expected['lon_n'] = np.array([357.02349553989114, 357.00934338793087, 356.99516391521394, 356.98101176325366, 357.04048905054617, 357.0263368985859, 357.01559984121064, 357.004808142322, 357.0677551657128, 357.05701810833756, 357.046308371719, 357.03704663520443, 357.1016055832394, 357.09239848823825, 357.0831640724804, 357.07390233596584, 357.14305117112303, 357.13840664248744, 357.1336801515818, 357.1288990191628, 357.19812981659004, 357.19195532558035, 357.18575351381395, 357.17944241902086, 357.2472252163441, 357.240996083821, 357.23564121551175, 357.2302317056891, 357.2989980502528, 357.2935885404301, 357.2881517098508, 357.2856655209929, 357.3573278657647, 357.35478703539343, 357.3521369219955, 357.34948680859753, 357.41964651175186, 357.4154117944665, 357.41120439793775, 357.406915039139, 357.47702010078, 357.4764736856464, 357.4758179874861, 357.4751076478124, 357.5489556531186, 357.54819067193154, 357.54939278522545, 357.5504309739793, 357.6262460737664, 357.62731158327693])
        expected['rho_n'] = np.array([339794.8705078125, 339791.58944091795, 339788.84306640626, 339786.704296875, 339825.0563232422, 339821.5808227539, 339822.72312011715, 339824.27858886716, 339866.5678955078, 339867.22410888673, 339868.0504516601, 339870.94265136716, 339915.0061645508, 339917.144934082, 339919.5753540039, 339922.5647705078, 339971.5377319336, 339978.4158203125, 339985.48834228516, 339993.14416503906, 340042.38447265624, 340047.73139648436, 340053.27275390626, 340059.1786743164, 340106.8635131836, 340111.96739501954, 340118.48092041013, 340125.50483398436, 340174.52640380856, 340180.40802001953, 340186.8729370117, 340196.5460083008, 340248.48408203124, 340256.9176391601, 340266.03171386715, 340275.41313476564, 340325.35826416017, 340332.09052734374, 340339.33317871095, 340346.8188720703, 340396.8612182617, 340407.72519531247, 340418.7349975586, 340430.1822753906, 340484.7208984375, 340495.05018310546, 340507.9071044922, 340520.83693847654, 340577.24698486325, 340589.107434082])
        expected['lat_s'] = np.array([-80.41959918850293, -80.4022231872544, -80.38479254449254, -80.36726627908229, -80.35598280657342, -80.33831993737977, -80.33164001237151, -80.32491910622821, -80.32449563449967, -80.31769276608634, -80.31083525615963, -80.30395042547627, -80.30345865185603, -80.29650551928097, -80.28953872632755, -80.28253095223911, -80.28463465050348, -80.28024966905633, -80.27585102723083, -80.27139774389198, -80.27346046102133, -80.26511396985558, -80.25674015793314, -80.24833902525404, -80.2464812137998, -80.23803909998566, -80.22947404276644, -80.22090898554725, -80.21888724955292, -80.21026755082039, -80.20164785208783, -80.19066490790243, -80.18626626607696, -80.17524234075654, -80.16420475505782, -80.1531398486024, -80.14947886720725, -80.13920626269555, -80.1289063374272, -80.1185654310238, -80.11482248735862, -80.11049214742482, -80.10613448673436, -80.10174950528722, -80.10397614695663, -80.09955018437446, -80.09113539131701, -80.08270693788123, -80.08084912642698, -80.07235237109947])
        expected['lon_s'] = np.array([98.17487150695884, 98.23977879464147, 98.30455630872987, 98.3691903888457, 98.40939971248956, 98.47430016998302, 98.55294979827572, 98.63153112467673, 98.68594724179421, 98.76532087013894, 98.8446808381053, 98.90475235185558, 98.94060401480897, 99.00129024558453, 99.06194915560341, 99.12234168824466, 99.16773512546857, 99.23799045127133, 99.30818430537154, 99.37827570663421, 99.42436582315347, 99.46292224101818, 99.50153330039626, 99.53987115220755, 99.55401647397865, 99.59283926922099, 99.61466855380836, 99.6364773478282, 99.6340867816187, 99.65616195301618, 99.67812101119777, 99.67656372806701, 99.65064999035599, 99.6493454242245, 99.64793157506632, 99.64633331080053, 99.5763716831321, 99.53130609498835, 99.48583752568358, 99.44036212618963, 99.37038683814285, 99.28698339818783, 99.20334773180102, 99.1196915748467, 99.01081152928752, 98.92700510817147, 98.84362898897312, 98.75976109615453, 98.65088105059537, 98.56740930874864])
        expected['rho_s'] = np.array([6371.196202468872, 6371.195822715759, 6371.195063209533, 6371.195063209533, 6371.196202468872, 6371.195442962647, 6371.195442962647, 6371.195822715759, 6371.196202468872, 6371.195063209533, 6371.195063209533, 6371.1946834564205, 6371.195063209533, 6371.196202468872, 6371.195063209533, 6371.195442962647, 6371.195063209533, 6371.195063209533, 6371.195822715759, 6371.195063209533, 6371.1946834564205, 6371.195442962647, 6371.195822715759, 6371.195063209533, 6371.195442962647, 6371.1946834564205, 6371.195822715759, 6371.195063209533, 6371.195442962647, 6371.195063209533, 6371.195442962647, 6371.196202468872, 6371.195442962647, 6371.195442962647, 6371.196202468872, 6371.195822715759, 6371.196961975097, 6371.197341728211, 6371.196961975097, 6371.197341728211, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.197341728211, 6371.197341728211, 6371.196202468872, 6371.195822715759, 6371.194303703308, 6371.195442962647])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_0_60000(self):
        """Test for latitude = -30, longitude = 0, rho = 60000"""
        lat = -30
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([10.680794602588477, 10.684018451876724, 10.686941772841493, 10.689591886239455, 10.649129845596306, 10.652483468478778, 10.654498374283932, 10.656185431008922, 10.614487126125994, 10.61689135271385, 10.619008711356543, 10.620934824702488, 10.579134066982007, 10.581695387920774, 10.583935689968527, 10.585909614638666, 10.5431731210019, 10.54510606453701, 10.54673847974864, 10.548152328906838, 10.505142627703265, 10.508202552451436, 10.510975609254459, 10.513482288679853, 10.471285379987506, 10.474475078329903, 10.477514512510567, 10.480273908935246, 10.437947226648674, 10.441451113692892, 10.444661302602796, 10.447891982080222, 10.405708733766218, 10.409649752917318, 10.413303904123268, 10.416794130789157, 10.374781637204407, 10.379337373380807, 10.383606241612057, 10.38773167587074, 10.345493786043392, 10.34895669195258, 10.352221522375842, 10.35519948485397, 10.311636538327633, 10.315365821614463, 10.318733104875278, 10.32196378435269, 10.277881743449427, 10.281877404113885])
        expected['lon_n'] = np.array([358.17749698129984, 358.16471086717354, 358.1519247530473, 358.13905667665097, 358.1903650576961, 358.1775516228132, 358.1666779616545, 358.15574965898253, 358.20907977602195, 358.1982061148633, 358.18733245370464, 358.1773876982731, 358.23170136255305, 358.2218112486349, 358.21189381396, 358.2019217377718, 358.258858194693, 358.2515362319028, 358.2441596275992, 358.23675570253886, 358.29371948021674, 358.2853046871593, 358.2768079318318, 358.26828385574765, 358.324182123915, 358.31568536858754, 358.3075437830969, 358.29926559382284, 358.3555736733404, 358.3473774463364, 358.339126577819, 358.3325969169725, 358.3906535249176, 358.38412386407106, 358.37753956171116, 358.3709279385946, 358.4281102823259, 358.4206243949956, 358.41311118690857, 358.40551601655153, 358.4626983602829, 358.45715224667686, 358.45160613307075, 358.445950736438, 358.505127495407, 358.4995267402876, 358.49504613619206, 358.49053821133987, 358.5509444043594, 358.5464638002639])
        expected['rho_n'] = np.array([360267.14815673826, 360263.13796386716, 360259.2735961914, 360255.50644531247, 360287.344946289, 360283.3833618164, 360280.32103271486, 360277.4045288086, 360310.5311523437, 360307.2500854492, 360304.06623535155, 360301.0525146484, 360334.6895263672, 360331.26263427734, 360327.68991699215, 360324.554675293, 360359.2610717773, 360356.63621826173, 360353.5981933594, 360351.11916503904, 360386.11721191404, 360382.447277832, 360378.77734375, 360375.5205810547, 360409.9839355469, 360406.0466552734, 360402.5225463867, 360398.7553955078, 360433.437487793, 360429.4029907226, 360425.5872314453, 360422.0631225586, 360456.9639526367, 360452.5162841797, 360448.7734375, 360444.8604614258, 360479.5182495117, 360474.87614746095, 360470.4284790039, 360466.0051147461, 360500.881640625, 360496.7499267578, 360492.93416748045, 360489.3614501953, 360525.0400146484, 360520.7867797851, 360517.3112792969, 360517.1654541016, 360552.64958496095, 360549.87890625])
        expected['lat_s'] = np.array([-75.7407556825041, -75.7348134179262, -75.72883017221326, -75.72279228498698, -75.71641288830219, -75.7103750010759, -75.71389937868761, -75.71742375629935, -75.72067492634429, -75.72430858698272, -75.72792858724284, -75.73190375733978, -75.73559205949158, -75.7396628522369, -75.74373364498223, -75.74779077734922, -75.75478489105933, -75.76218881611962, -75.7695927411799, -75.7769966662402, -75.78410006297702, -75.78466013848896, -75.78523387437923, -75.78575296875616, -75.78599885556628, -75.78657259145655, -75.78453719508389, -75.78243349681952, -75.78007025136671, -75.77804851537238, -75.77599945862138, -75.7687321373445, -75.76121892925747, -75.75397892873727, -75.7466842867037, -75.73933500315675, -75.7260707757886, -75.71303877485221, -75.69993847202412, -75.686797188061, -75.67342367766614, -75.65831529922207, -75.64316593964298, -75.62794827817217, -75.61247106951294, -75.59722608728546, -75.58080631252076, -75.56434555662102, -75.54759793277614, -75.53108253536305])
        expected['lon_s'] = np.array([49.84138960010586, 49.88885941483745, 49.93651364467663, 49.98444108208261, 50.00603472514361, 50.05448467202109, 50.0691729938312, 50.0838647307359, 50.07197678648549, 50.08705784417288, 50.10218671318446, 50.09974492055618, 50.07034437127386, 50.068069918280244, 50.06544371054437, 50.06285506884894, 50.03859765201164, 50.04194102961036, 50.045205860033626, 50.048590218767366, 50.024025443417415, 50.04824529421428, 50.07276567333463, 50.0972587316983, 50.09389827862665, 50.11863039361126, 50.13620788544029, 50.15356681121588, 50.14312686706951, 50.16084779287111, 50.17865068094275, 50.1924203423095, 50.17796766202575, 50.191949059256764, 50.20593728667695, 50.21989477824587, 50.18345913411844, 50.175583926005416, 50.167609680149425, 50.15969007580679, 50.12335346942233, 50.062803842430164, 50.002291781478434, 49.941673852594576, 49.85246475184504, 49.79214052109548, 49.7480858009489, 49.70414377892362, 49.63142958501967, 49.58774369508827])
        expected['rho_s'] = np.array([6371.197721481323, 6371.195442962647, 6371.193923950195, 6371.193164443969, 6371.191265678405, 6371.19050617218, 6371.190126419067, 6371.189746665955, 6371.190885925293, 6371.191265678405, 6371.192404937744, 6371.193164443969, 6371.197721481323, 6371.199620246887, 6371.198101234436, 6371.1965822219845, 6371.192404937744, 6371.191265678405, 6371.191265678405, 6371.189746665955, 6371.189366912841, 6371.189366912841, 6371.189746665955, 6371.188987159729, 6371.191645431519, 6371.192025184631, 6371.194303703308, 6371.195063209533, 6371.199620246887, 6371.197341728211, 6371.196202468872, 6371.195063209533, 6371.191265678405, 6371.190126419067, 6371.189366912841, 6371.188607406616, 6371.188987159729, 6371.188987159729, 6371.189366912841, 6371.19050617218, 6371.193923950195, 6371.1965822219845, 6371.198101234436, 6371.195063209533, 6371.19050617218, 6371.189366912841, 6371.187847900391, 6371.188607406616, 6371.190126419067, 6371.192404937744])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_0_60000(self):
        """Test for latitude = 0, longitude = 0, rho = 60000"""
        lat = 0
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([64.59375046694545, 64.63297965844342, 64.67236252919771, 64.71158489050651, 64.7498954215611, 64.78896751870816, 64.83363354078547, 64.87768484583746, 64.92112143386416, 64.96540838044253, 65.00932308171112, 65.06073220804652, 65.11130463620859, 65.16243030969342, 65.21339888882734, 65.264406741549, 65.30801579185224, 65.3523266440927, 65.39675190200175, 65.44066148062848, 65.48360971012951, 65.52102548640283, 65.55828929096714, 65.59521158607294, 65.63117423960034, 65.66807262904405, 65.70747257527127, 65.7470757196263, 65.785200128026, 65.82441565914564, 65.86339042609703, 65.90616619332152, 65.94770740385354, 65.98987016160002, 66.03200559858983, 66.07374829970237, 66.11229447228334, 66.15214350344849, 66.19117632700781, 66.23054041474188, 66.26792033252207, 66.31638906241977, 66.36427551869073, 66.41203732400935, 66.45838698771705, 66.50558871752372, 66.55438529650158, 66.6029411113112, 66.64962033164636, 66.69745726904584])
        expected['lon_n'] = np.array([339.2212631664132, 339.2648670940746, 339.3084163802226, 339.35204762864066, 339.4044215191963, 339.4482713336678, 339.50856824366065, 339.56911104046367, 339.63839647940426, 339.6991851630174, 339.7601104504139, 339.8701584583211, 339.98952284425627, 340.1005817201607, 340.2121323696853, 340.32401086829015, 340.4613796328774, 340.5903062836506, 340.719724708044, 340.84974418908433, 340.98957182177287, 341.11216005699623, 341.23515810356986, 341.3585386407369, 341.49170000879553, 341.615900168663, 341.7391714228034, 341.86271588451064, 341.9962870639194, 342.12073311059703, 342.2454796855981, 342.41309252783026, 342.59103261608743, 342.75982025085676, 342.92909965924633, 343.09889816201286, 343.27713877859355, 343.4457351680661, 343.6148506519156, 343.7842120225753, 343.96428312985347, 344.1436712182147, 344.32371500473624, 344.5043052063914, 344.6959056729886, 344.877834591721, 345.0817293988243, 345.28633454560116, 345.50244173094035, 345.70874076463144])
        expected['rho_n'] = np.array([6371.198101234436, 6371.195822715759, 6371.197341728211, 6371.1984809875485, 6371.1965822219845, 6371.195063209533, 6371.199240493775, 6371.195442962647, 6371.195822715759, 6371.197721481323, 6371.196202468872, 6371.195442962647, 6371.197341728211, 6371.196961975097, 6371.195822715759, 6371.198101234436, 6371.197341728211, 6371.195063209533, 6371.198860740661, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.197721481323, 6371.195822715759, 6371.196202468872, 6371.1984809875485, 6371.195822715759, 6371.199240493775, 6371.195822715759, 6371.1965822219845, 6371.1965822219845, 6371.196961975097, 6371.1984809875485, 6371.196202468872, 6371.196961975097, 6371.196202468872, 6371.195442962647, 6371.199620246887, 6371.195442962647, 6371.199620246887, 6371.196961975097, 6371.199240493775, 6371.196202468872, 6371.1965822219845, 6371.197341728211, 6371.195822715759, 6371.195822715759, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845])
        expected['lat_s'] = np.array([-69.12013037660569, -69.1072349794527, -69.09435324267807, -69.08136222287669, -69.07148576933687, -69.05850840991383, -69.05345406992802, -69.04844071107723, -69.04643263546126, -69.04141927661047, -69.03643323851637, -69.03372848360505, -69.03408365344188, -69.03137889853056, -69.02868780399757, -69.02596938870789, -69.03196629479916, -69.03508086106069, -69.0381544461872, -69.04125535207038, -69.04730689967502, -69.04404206925176, -69.04081821996351, -69.03756704991858, -69.03722554046007, -69.03404267230687, -69.02547761508765, -69.01684425597676, -69.01114787820896, -69.00254183985476, -68.99396312225721, -68.98489263103943, -68.97863617775971, -68.96956568654193, -68.96052251608083, -68.95142470410639, -68.93649391058074, -68.9187763998737, -68.90097692689667, -68.88321843505463, -68.86815103774558, -68.84308424349163, -68.81796280772434, -68.79269110779529, -68.77011050239923, -68.74472951944347, -68.72156151777878, -68.69828423308738, -68.67758875990222, -68.65428415445413])
        expected['lon_s'] = np.array([39.29865793692185, 39.30270482400508, 39.306710729953295, 39.31082591892823, 39.30287899382892, 39.30694637147966, 39.27812638827679, 39.249057103169214, 39.207669571893526, 39.17835098488125, 39.14894019031517, 39.09774109229676, 39.033769540530415, 38.981849857554565, 38.9296364764444, 38.8771362273891, 38.818359034486555, 38.771667861320346, 38.72477178247904, 38.677640062111365, 38.61751732194233, 38.60250456614664, 38.58738594241882, 38.572294639447676, 38.544406977066515, 38.52912442879861, 38.50864752166691, 38.48809206735976, 38.454651461183374, 38.43401745970077, 38.41340736388025, 38.39415989079916, 38.36187700168709, 38.34239730217421, 38.323016640404305, 38.30360865787771, 38.25990569247339, 38.229296199707974, 38.19869353713173, 38.16808745946091, 38.124254720462346, 38.04237782778682, 37.960483859638366, 37.87849768393612, 37.783325828041164, 37.70128159573097, 37.63044911894335, 37.55955858554779, 37.4753047870411, 37.40431863099715])
        expected['rho_s'] = np.array([6371.187847900391, 6371.198101234436, 6371.186328887939, 6371.193923950195, 6371.192784690857, 6371.186328887939, 6371.199240493775, 6371.186708641052, 6371.19050617218, 6371.19050617218, 6371.187468147277, 6371.1946834564205, 6371.186328887939, 6371.193164443969, 6371.188607406616, 6371.188607406616, 6371.197721481323, 6371.185949134827, 6371.191645431519, 6371.190126419067, 6371.185189628601, 6371.195063209533, 6371.188607406616, 6371.186708641052, 6371.197341728211, 6371.187088394165, 6371.188987159729, 6371.192784690857, 6371.185189628601, 6371.193164443969, 6371.188227653503, 6371.185949134827, 6371.1946834564205, 6371.188607406616, 6371.185189628601, 6371.196202468872, 6371.188227653503, 6371.1844301223755, 6371.197721481323, 6371.186328887939, 6371.185189628601, 6371.193544197083, 6371.184050369263, 6371.195063209533, 6371.1844301223755, 6371.198860740661, 6371.184050369263, 6371.195442962647, 6371.185569381713, 6371.19050617218])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_0_60000(self):
        """Test for latitude = 30, longitude = 0, rho = 60000"""
        lat = 30
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([70.17984629284976, 70.20715851179335, 70.23443999488568, 70.26168391193757, 70.28902345163783, 70.31617686868321, 70.34721666336634, 70.37821035427257, 70.40928429990154, 70.44012089645685, 70.47090797414069, 70.5095361089917, 70.54826328158566, 70.58661820886988, 70.62486043803277, 70.66302070492563, 70.69651936770997, 70.72962358706972, 70.76263730642297, 70.79551783708742, 70.82853497153526, 70.8560623414377, 70.88349067359718, 70.91078069442597, 70.93819195111253, 70.96520534927993, 70.99490130424385, 71.0244538252352, 71.05421125190165, 71.08354349929228, 71.11272719006844, 71.14649223023041, 71.18035972322991, 71.21362103694058, 71.24668256761802, 71.27951699450557, 71.31078901562047, 71.34143095178445, 71.37182700113829, 71.40203180519538, 71.4323902885088, 71.46872518734597, 71.504773218238, 71.54059243779284, 71.5766148554755, 71.61192693348445, 71.64843770969276, 71.68464795757758, 71.7210904318942, 71.75669791558477])
        expected['lon_n'] = np.array([339.06375900415276, 339.1053138750631, 339.14686874597345, 339.1884782583972, 339.22374935527114, 339.26544082996486, 339.3237979662335, 339.3822097440154, 339.4342831062476, 339.4928861293263, 339.551571114675, 339.6683400287256, 339.7790164140365, 339.89668691305747, 340.014821864942, 340.13333930742004, 340.2605447505224, 340.3949355526315, 340.52981812836094, 340.66527443998064, 340.79458358134735, 340.92512215676464, 341.05604322277543, 341.1874014208931, 341.31242120346104, 341.44454438276574, 341.5792083924417, 341.71433685498124, 341.843044939701, 341.9789110626709, 342.11521431774764, 342.2993835385279, 342.4773782682984, 342.66277692312923, 342.8487493138503, 343.03526811970505, 343.2153938684966, 343.4028415800784, 343.59072642376714, 343.77902107880607, 343.9609226767818, 344.1609106156798, 344.3615815734948, 344.5628809087134, 344.7579237906523, 344.9604798806782, 345.1883623121465, 345.4170370455585, 345.6395372879609, 345.869823946017])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.1984809875485, 6371.197721481323, 6371.196961975097, 6371.196961975097, 6371.197341728211, 6371.1965822219845, 6371.1965822219845, 6371.197341728211, 6371.197341728211, 6371.196961975097, 6371.196961975097, 6371.196961975097, 6371.1984809875485, 6371.197341728211, 6371.198101234436, 6371.197721481323, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.197341728211, 6371.197341728211, 6371.197721481323, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.1965822219845, 6371.197341728211, 6371.198101234436, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198860740661])
        expected['lat_s'] = np.array([22.227051809576622, 22.24623781095518, 22.265095963253586, 22.283619436282663, 22.237454187682545, 22.25668800038531, 22.275805699872166, 22.294588720089706, 22.24852592432714, 22.268019284218354, 22.287171134651075, 22.30546238124836, 22.25874388732548, 22.277745473596454, 22.296398720219756, 22.314724117762907, 22.268893548432118, 22.28890600270026, 22.30857694750989, 22.327913213050195, 22.281987021071032, 22.300954456396155, 22.319594042641114, 22.33789211942758, 22.290804795289517, 22.309813211749656, 22.327510231889164, 22.344872572759343, 22.296740229678264, 22.314799249843773, 22.33254408130746, 22.34974249763755, 22.301343777178843, 22.319273023750128, 22.336847100484576, 22.35408649794968, 22.30503207933066, 22.322387590011658, 22.339415251612493, 22.356087743376506, 22.30697868324411, 22.323289174982094, 22.339258157261582, 22.354885630082578, 22.304635928358792, 22.32098057104264, 22.336874421241248, 22.3524199317922, 22.302026796095845, 22.318275816131305])
        expected['lon_s'] = np.array([1.5314334260753477, 1.5154011578243296, 1.4993786012485375, 1.4833887015147145, 1.5196519900873104, 1.503604674075777, 1.4865755183710543, 1.4695555407330287, 1.5048378433860816, 1.4877767778913302, 1.4707611758432417, 1.4527956438919927, 1.4871481870442775, 1.4691378319766002, 1.4511670706617676, 1.4332067680741019, 1.4662860145547083, 1.447015276141752, 1.4277938431568666, 1.4085664337564576, 1.441646320567299, 1.423259237878213, 1.4048882701667, 1.3865430223862805, 1.4204549116153937, 1.4020700700821294, 1.3837547043793286, 1.3654270656803629, 1.3993548564436378, 1.3810074742291025, 1.362681329634018, 1.3433592580805807, 1.3761882421611296, 1.356795093951642, 1.3374223295879586, 1.3180965227748191, 1.351348871862203, 1.3324160143697508, 1.3135283001588443, 1.2946591555247435, 1.3278519539003013, 1.3078755715687183, 1.287955324854376, 1.2680517267261358, 1.3001213858700496, 1.2801794746494337, 1.2595116423860926, 1.2388618460910283, 1.2700979019910834, 1.249434552039385])
        expected['rho_s'] = np.array([377071.87353515625, 377074.2796508789, 377076.734375, 377079.1890991211, 377073.98800048826, 377076.6128540039, 377079.2134033203, 377081.81395263673, 377076.5399414062, 377079.1890991211, 377081.9354736328, 377084.5603271484, 377079.2377075195, 377081.91116943356, 377084.6575439453, 377087.30670166016, 377082.0812988281, 377085.1679321289, 377088.23026123043, 377091.17106933595, 377085.9456665039, 377088.7406494141, 377091.46271972655, 377094.2333984375, 377088.7892578125, 377091.58424072264, 377094.0389648437, 377096.4936889648, 377090.73359375, 377093.38275146484, 377095.9346923828, 377098.4137207031, 377092.7022338867, 377095.2784790039, 377097.80611572263, 377100.3094482422, 377094.4278320312, 377096.8582519531, 377099.21575927734, 377101.57326660154, 377095.52152099606, 377097.80611572263, 377099.7990600586, 377101.8892211914, 377095.61873779295, 377097.83041992184, 377100.0177978515, 377101.9378295898, 377095.6916503906, 377097.8790283203])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_0_60000(self):
        """Test for latitude = 60, longitude = 0, rho = 60000"""
        lat = 60
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([79.0975367873834, 79.1150861046821, 79.13261407763964, 79.15011558361417, 79.16756159430167, 79.1849990672527, 79.20623241783498, 79.22743161747141, 79.2485744680472, 79.2696848752244, 79.29076027768212, 79.32463886973903, 79.35843293820497, 79.39212113873876, 79.42572310813429, 79.45923030865508, 79.49167882985451, 79.52400953029077, 79.55626339109887, 79.58846517171453, 79.62063621647894, 79.64983356763345, 79.67896261689626, 79.70798579822694, 79.73710545597964, 79.76594422220273, 79.79774985184653, 79.82948461827772, 79.86128256395871, 79.89276546716424, 79.92413737979575, 79.96224044385431, 80.00034265413923, 80.03793516155733, 80.07534752273608, 80.11248155370615, 80.14927236767042, 80.18549883725451, 80.22142563228878, 80.2571483754216, 80.29291380723672, 80.33396665924309, 80.37470959141585, 80.41518529243736, 80.45583260196175, 80.49567565671137, 80.53807149466328, 80.58004983730218, 80.62217588278187, 80.66342083385865])
        expected['lon_n'] = np.array([327.0792905203949, 327.11182954160086, 327.14455980810357, 327.17731739536293, 327.1929995096973, 327.2259210214968, 327.2786500818893, 327.3315703875785, 327.36738789958605, 327.4205814128421, 327.47385688836823, 327.60485991664905, 327.7191699625984, 327.8517575947667, 327.9851375288787, 328.1195010102312, 328.25886419005616, 328.4169694089636, 328.57619477889494, 328.73651297909345, 328.87989231015035, 329.0428606237469, 329.2067578430705, 329.37180253417466, 329.5191980164636, 329.6862371228054, 329.8651061167897, 330.04501329952785, 330.2073532357207, 330.38963732429, 330.5730415638832, 330.8323428655335, 331.0745140527455, 331.337558298061, 331.6022964302907, 331.8689743362447, 332.1197515618109, 332.3916477322908, 332.6653197519548, 332.94052173399297, 333.1980471864592, 333.4914994339596, 333.7873012665345, 334.08526143888713, 334.3653811571277, 334.66795853735925, 335.0177735058906, 335.3707030406835, 335.70622925347124, 336.0649234679236])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775])
        expected['lat_s'] = np.array([34.24096182375177, 34.253549862392106, 34.26583737270896, 34.27775263771604, 34.229019237988005, 34.24163118229043, 34.25304101329894, 34.264129825416454, 34.214518746380065, 34.22627008684707, 34.237731144274356, 34.248317937487876, 34.19807506595326, 34.209286821475835, 34.220146822256154, 34.230767766415525, 34.18006044201735, 34.190879461662654, 34.201377462416964, 34.211598840509886, 34.16111691235432, 34.171864215013336, 34.18221536670049, 34.1923035561046, 34.14176698643568, 34.15256210041888, 34.162230233189035, 34.17160808291946, 34.1204875320765, 34.13059962714271, 34.140377042939576, 34.14955681718408, 34.098367964449416, 34.10831613497554, 34.117970607367354, 34.12725624954398, 34.07637134022739, 34.086507340955684, 34.09633939826592, 34.1058640970635, 34.055153357570745, 34.06346569779065, 34.07144677383581, 34.079192208354605, 34.026893449879815, 34.03533897878854, 34.043026356699386, 34.05049516855679, 33.998029070447345, 34.00615699555966])
        expected['lon_s'] = np.array([3.705020320460961, 3.6828964839656435, 3.6607389234112993, 3.6385642873840296, 3.6874774064646063, 3.6653377751568335, 3.640792636600746, 3.616186026342129, 3.6627090061000245, 3.6381540491470057, 3.6136165945537346, 3.5872576138861865, 3.631999194931178, 3.6057080892685063, 3.579412501294192, 3.553049251758412, 3.594482246480767, 3.5649333540274113, 3.535391718650048, 3.5057465632180773, 3.5471229954363683, 3.518494470973671, 3.4898300880178317, 3.4611232298230914, 3.5033502340364606, 3.474688839288383, 3.445240905568696, 3.415699697078156, 3.4570865880236137, 3.427624140151941, 3.3981277547778292, 3.366171007198632, 3.405032649141889, 3.373088921610797, 3.341072623319774, 3.3090642224349787, 3.3490221097400226, 3.3181453858233336, 3.2872791206338112, 3.2563223554377863, 3.2961908099533854, 3.2620834065592805, 3.2280131423187877, 3.1938643309028394, 3.23045941725871, 3.196343689571554, 3.160354995947934, 3.124282419063569, 3.158981701037938, 3.1229959956220794])
        expected['rho_s'] = np.array([369627.7646606445, 369637.3162109375, 369646.60041503905, 369655.6415771484, 369625.35854492185, 369635.00731201173, 369643.7811279297, 369652.48203124997, 369621.54278564453, 369630.5839477539, 369639.5035888672, 369647.7913208008, 369616.6333374023, 369625.5286743164, 369634.1080566406, 369642.44439697266, 369614.9077392578, 369623.8516845703, 369632.3581542969, 369640.47575683595, 369610.94615478517, 369619.5741455078, 369627.934790039, 369635.9794799805, 369605.91518554685, 369614.4216552734, 369622.61217041017, 369630.486730957, 369600.22800292965, 369608.4185180664, 369616.4389038086, 369624.046118164, 369593.3985229492, 369601.37030029297, 369609.12333984376, 369616.5361206055, 369585.8642211914, 369594.05473632814, 369601.75916748046, 369609.48790283204, 369578.7430908203, 369586.47182617185, 369593.83599853516, 369601.1272583008, 369569.8477539062, 369577.600793457, 369584.8677490234, 369591.91596679686, 369560.63646240236, 369568.29228515626])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_0_60000(self):
        """Test for latitude = 90, longitude = 0, rho = 60000"""
        lat = 90
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([84.14229828665555, 84.15131029437855, 84.16028900492935, 84.1692719843484, 84.17346273229101, 84.18240942633008, 84.19517419611519, 84.20790609561492, 84.21584064099636, 84.22851960653003, 84.2411878998931, 84.26737954092599, 84.28869613443877, 84.31474562215955, 84.34069607213738, 84.3666115173957, 84.39167148146046, 84.4214873916217, 84.45127256593165, 84.48096638646146, 84.50577875616882, 84.53655601545572, 84.56726369219045, 84.59789879816525, 84.62357817501038, 84.65407795786226, 84.68851961363886, 84.722848998181, 84.7521534979281, 84.78626218198268, 84.82023212781974, 84.86767291424736, 84.91000685360991, 84.9570519159525, 85.0038814004494, 85.05049402644015, 85.09144570626944, 85.13712473077801, 85.18255359960823, 85.22772334813682, 85.26770001846208, 85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089])
        expected['lon_n'] = np.array([255.73632539421095, 255.67977142788322, 255.622834970962, 255.56548870269057, 255.50158545281593, 255.4443757883279, 255.36555540530597, 255.2863525316905, 255.20026482739166, 255.12073410469603, 255.04106677821702, 254.8571434442469, 254.66532441159626, 254.47896953028163, 254.290565592216, 254.10104150312648, 253.88124601563547, 253.66573988694324, 253.4481573807433, 253.2274056667685, 252.99916806546335, 252.76855355832703, 252.5352342962794, 252.29890975099696, 252.054689507034, 251.81317401798233, 251.54605897992147, 251.27498243214202, 250.99609214795206, 250.71848593932606, 250.43675429644136, 250.1120197825423, 249.77794156985863, 249.44195090420735, 249.0993759361962, 248.75046255263527, 248.40165845210106, 248.04957586076523, 247.69159198598652, 247.3269145258212, 246.95466921605552, 246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.198860740661, 6371.2, 6371.200759506225, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.200759506225, 6371.200759506225, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([47.544486685306204, 47.55122466692242, 47.55758015794512, 47.56364878102267, 47.510786531941356, 47.517585985260105, 47.52181045726176, 47.52572757075076, 47.47063184981086, 47.47521149164935, 47.47951109573187, 47.48260517142589, 47.42664543155597, 47.43043960163991, 47.43391275283287, 47.43705463985107, 47.37875897528501, 47.380323088604946, 47.381460315101755, 47.38230042836966, 47.32409355626282, 47.32634410359434, 47.32829753769696, 47.32985140573314, 47.27244366575918, 47.27466689233402, 47.27610123205972, 47.27720772270526, 47.21921941665185, 47.22109088848444, 47.222644756520616, 47.2230579829654, 47.16460180895385, 47.165950771314925, 47.166900167609555, 47.167610507283236, 47.11030863524142, 47.112682125978, 47.11483363556655, 47.116544597953634, 47.0595876504649, 47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269])
        expected['lon_s'] = np.array([5.791035651950549, 5.756712243823942, 5.722305165880003, 5.687765753020894, 5.768920780078517, 5.734572185629346, 5.696477232420422, 5.65820213297214, 5.735513471074338, 5.697460779660904, 5.659333809940247, 5.618563130123971, 5.6932947911540195, 5.652742677391185, 5.612097075414085, 5.571295232859722, 5.6405584736855285, 5.59470698690041, 5.548805981243808, 5.502688543967882, 5.57183182959639, 5.527048413642732, 5.482137785415783, 5.437168246807242, 5.507260074956736, 5.462479647210841, 5.416120165106202, 5.369575414120327, 5.438118923762343, 5.3917888968485, 5.345322693038081, 5.294993163478373, 5.35955168462651, 5.3092379498792575, 5.258862316542651, 5.208312940269034, 5.27469597569921, 5.2261760193827325, 5.177641548914268, 5.128896623242002, 5.195070911015671, 5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653])
        expected['rho_s'] = np.array([353340.98607177736, 353350.90218505857, 353360.57525634766, 353370.0781982422, 353328.6881469726, 353338.7014770508, 353346.1871704101, 353353.08956298826, 353309.4635253906, 353317.3137817383, 353324.9939086914, 353330.8755249023, 353285.9370605469, 353292.5964111328, 353298.98841552733, 353304.8943359375, 353256.6018920898, 353260.24752197263, 353263.26124267577, 353266.177746582, 353218.0554321289, 353222.3086669922, 353225.954296875, 353229.405493164, 353182.13382568356, 353186.21693115233, 353189.0362182617, 353191.66107177734, 353143.4172363281, 353146.69830322266, 353149.8821533203, 353150.31962890626, 353099.91271972656, 353101.2251464844, 353102.1730102539, 353103.0479614258, 353053.66182861326, 353056.1651611328, 353058.4254516601, 353060.3211791992, 353011.39682617184, 353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_90_60000(self):
        """Test for latitude = -90, longitude = 90, rho = 60000"""
        lat = -90
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-12.798697839193196, -12.986719286665306, -13.145022581058697, -13.299494139327706, -12.425680718051296, -12.59521284344018, -13.222688662110912, -13.86089470796773, -13.452770414491965, -14.044394570209292, -14.613540573368113, -15.431619651017229, -15.365858589688344, -16.11273294524007, -16.85565262126235, -17.550562897608614, -17.877373789015394, -18.91091801422172, -19.86222676182109, -20.7710039214588, -21.03585816690439, -21.861307018667944, -22.60960205356703, -23.343000445886318, -23.54028362987296, -24.204464885143338, -25.005960263487665, -25.675810575769148, -25.94310319874843, -26.63597124853287, -27.23284781972194, -27.973158363481275, -28.325937634112435, -29.012139419266944, -29.66957244763735, -30.257419508743666, -30.52786767911948, -31.03877948941478, -31.532615826785047, -32.01389827646082, -32.18509013781802, -32.78376987894798, -33.37848811035933, -33.88487833542408, -34.17342650710043, -34.57963151741943, -34.91092301292173, -35.234182205960096, -35.42688916320283, -35.724740052528745])
        expected['lon_n'] = np.array([355.48719475025854, 355.49989890211475, 355.50615535539447, 355.51178343127054, 355.44656878507527, 355.4530438044085, 355.5469452451178, 355.6471851013769, 355.66341363084484, 355.7514957503813, 355.834441567662, 355.96951538868814, 356.05232460218537, 356.16835585580554, 356.2911626570824, 356.403997382171, 356.5628949030222, 356.74670895396554, 356.91831062667296, 357.0863879217686, 357.2356138947551, 357.3863425093589, 357.5194492359041, 357.6550967928206, 357.78782102877227, 357.90830556573127, 358.0658916902618, 358.1864581894909, 358.33412687934657, 358.47236990814764, 358.58159829335443, 358.7286932473199, 358.8964700140921, 359.0352594580267, 359.17126218478, 359.2872387968868, 359.4405355626186, 359.5384258338032, 359.6345675765603, 359.7310644891542, 359.85119385627644, 359.97714254457145, 0.10806322430354849, 0.20905967075558, 0.36276919602466917, 0.4290091976886726, 0.4590617098716114, 0.4896805941471323, 0.6038198320063625, 0.6296228459476426])
        expected['rho_n'] = np.array([312129.32762451173, 311874.3522705078, 311618.9880493164, 311378.91116943356, 312815.84833984374, 312477.898449707, 311575.7022705078, 310841.44810791017, 311408.7810302734, 310591.13916015625, 309762.4145751953, 308857.05885009764, 309151.7229614258, 308090.0669311523, 307328.73789062497, 306577.64091796876, 306401.9701660156, 305380.87784423825, 304555.3128051758, 303942.94420166017, 303957.38089599606, 303419.3588378906, 302826.7495483398, 302455.94038085936, 302508.2187133789, 302112.7893920898, 301969.151574707, 301486.00839843747, 301536.60974121094, 301538.09229736327, 301220.8981933594, 301038.47087402345, 301077.38189697266, 301028.3603271484, 301122.32036132814, 301039.2, 301346.01621093747, 301277.74571533204, 301300.51875, 301462.60345458984, 301384.85432128905, 301575.0832885742, 302019.8987426758, 302076.1872680664, 302272.5651977539, 301865.37264404295, 300664.91533203126, 299602.5787841797, 299121.40424804686, 298128.33466796874])
        expected['lat_s'] = np.array([-75.09847201371281, -75.07509910637302, -75.05167155751985, -75.02817570677502, -75.01167396974029, -74.98808249634706, -74.96914921196776, -74.95017494645347, -74.93819479464926, -74.91911124610826, -74.89997305605388, -74.88190037551004, -74.8708627898113, -74.85269448661904, -74.83447154191347, -74.81620761607286, -74.80798406831215, -74.79260248230128, -74.77715259439873, -74.76167538573947, -74.75336987570873, -74.74229130887497, -74.73121274204121, -74.72012051482912, -74.71622730700221, -74.70508043827675, -74.69661100370593, -74.68816888989178, -74.68693945584118, -74.6784427005137, -74.66994594518619, -74.66355288812306, -74.66444081271516, -74.65802043489535, -74.65158639669721, -74.64516601887738, -74.64975590599963, -74.64702383033162, -74.64427809428528, -74.64153235823895, -74.6461222453612, -74.65022035886321, -74.65433213274355, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67629802111432, -74.67860662505376, -74.68829183329686, -74.69058677685797])
        expected['lon_s'] = np.array([142.5637928096298, 142.5459113743827, 142.5280162787573, 142.51013484351017, 142.5106266171304, 142.49274518188332, 142.50681537157357, 142.52083091975044, 142.5532469975513, 142.5673445079982, 142.58145567882346, 142.57862798050706, 142.59409152878797, 142.5912638304716, 142.58849077366858, 142.58564941497386, 142.61642624737394, 142.62885719166334, 142.6414384001145, 142.6538966651606, 142.68463251642567, 142.67401840245546, 142.6632130431885, 142.65240768392155, 142.65993455238691, 142.64926579690334, 142.63264111646353, 142.6157978699703, 142.61740979461442, 142.60083975568796, 142.58416043373478, 142.55827401678044, 142.55061054453168, 142.52476510871236, 142.49890601251468, 142.47296495404697, 142.4479254805497, 142.4047450246169, 142.36135966300895, 142.31798796177938, 142.2926342995803, 142.24998659840273, 142.20713399155008, 142.16411746015734, 142.13929655271352, 142.09623904018576, 142.0419526966625, 141.98752974935584, 141.95116582221468, 141.89661993150295])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_90_60000(self):
        """Test for latitude = -60, longitude = 90, rho = 60000"""
        lat = -60
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([2.5221882805216183, 2.5478561314225203, 2.573189303054093, 2.59803070106544, 2.5393798666625287, 2.5651023590768034, 2.5956742858017776, 2.625624665312287, 2.571925718057642, 2.602934776889498, 2.6334862130469645, 2.6663325927655563, 2.615058362666275, 2.648799497166138, 2.682076178802447, 2.715086483061114, 2.670287272294999, 2.710920067667402, 2.7510405988520574, 2.790785469632368, 2.7458701456503576, 2.7851300729995927, 2.823891396539409, 2.8623931728907763, 2.815920565778015, 2.85530343653231, 2.8963938545791024, 2.9369788386273115, 2.8923981939146444, 2.9338642723657955, 2.9752210677902156, 3.0204232597173615, 2.9801183134251232, 3.0263928450519586, 3.072298546463614, 3.117835417660089, 3.0753584712117714, 3.11963858760096, 3.1636591568016996, 3.2072562542738865, 3.164656364420523, 3.2164838398425815, 3.2679766359953106, 3.3191962245812334, 3.2838226748697394, 3.3361009427770085, 3.3914664561891357, 3.4464221582510532, 3.414600306907971, 3.470628348669578])
        expected['lon_n'] = np.array([25.04666739401006, 25.030285185285784, 25.013660504845973, 24.996933371568613, 25.054706526663168, 25.038286751898454, 25.025848977419862, 25.013168731225733, 25.074898273397014, 25.062653451762475, 25.05026007351349, 25.04041094073033, 25.104840115171058, 25.095259067312824, 25.085660943981665, 25.076018424420898, 25.145201410523956, 25.140763495110743, 25.13619751365059, 25.131513711427253, 25.200759876780133, 25.19666517837271, 25.19249022524254, 25.18826746078818, 25.257657060113633, 25.253676767374806, 25.252531003141538, 25.251192286064214, 25.32333274407789, 25.322424328918277, 25.321413460921118, 25.324154074325584, 25.400182617524287, 25.403283523407474, 25.406312712304373, 25.40931116535001, 25.48403343486995, 25.48615420860724, 25.488148623844882, 25.49007132209624, 25.56476456331221, 25.574671752703313, 25.584645536438824, 25.594438320161327, 25.676786495889587, 25.687069345685043, 25.700424073059708, 25.713701960806215, 25.79927569337001, 25.812983883034228])
        expected['rho_n'] = np.array([313257.04246826173, 313225.88448486326, 313194.96954345703, 313164.5163818359, 313265.5732421875, 313234.5610839844, 313200.87546386715, 313168.307836914, 313266.73984374997, 313233.0056152344, 313199.8303833008, 313164.9052490234, 313261.4415283203, 313225.6657470703, 313190.3760498047, 313155.305090332, 313248.7547363281, 313209.5763671875, 313170.7868652344, 313132.43483886716, 313225.86018066405, 313186.4873779297, 313147.6006591797, 313108.859765625, 313202.3337158203, 313162.742175293, 313122.29998779297, 313081.83349609375, 313173.48463134764, 313132.24040527345, 313091.1177001953, 313047.6617919922, 313136.71237792965, 313092.06556396483, 313047.9291381836, 313003.98714599607, 313093.71824951173, 313049.8248657226, 313006.24743652344, 312963.13178710936, 313052.64415283204, 313004.25449218747, 312956.205090332, 312907.985559082, 312992.92873535154, 312943.93146972655, 312892.8926513672, 312842.1454833984, 312924.63393554685, 312873.0361206055])
        expected['lat_s'] = np.array([-68.65302739964687, -68.63061071879088, -68.60813939642154, -68.58561343253885, -68.56721290291483, -68.54463229751875, -68.52811690010566, -68.51153320080087, -68.49910225651146, -68.48245025531497, -68.46574361260511, -68.44749334714282, -68.4333821763176, -68.4150636089636, -68.39669040009628, -68.37824888933724, -68.36941062455125, -68.3563513028582, -68.34323733965175, -68.33006873493198, -68.3211348474976, -68.31140865811952, -68.30166880836308, -68.29192895860663, -68.28636918462223, -68.27661567448746, -68.27113786277312, -68.2656054095454, -68.26429401322477, -68.25880254113207, -68.25328374828268, -68.24958178575255, -68.25008721975112, -68.24643989873434, -68.24272427582585, -68.23902231329572, -68.24148118139692, -68.23978729448274, -68.23807974719026, -68.23633121876273, -68.23876276610724, -68.24503287976532, -68.25127567266671, -68.25749114481141, -68.26788669272818, -68.27411582525124, -68.2787057123735, -68.28329559949574, -68.29202458125502, -68.29660080799891])
        expected['lon_s'] = np.array([122.78863297363702, 122.80088633300804, 122.81296210746062, 122.82491493850813, 122.82726452358263, 122.83938127917023, 122.8720705645379, 122.90470520839223, 122.92760000249011, 122.96041223126285, 122.9931971392789, 123.01707548061728, 123.03139155711762, 123.05550212488777, 123.07954439076622, 123.10343639248292, 123.122642884429, 123.15173949029325, 123.1807677942658, 123.2098780605084, 123.22911187321115, 123.2352044019508, 123.2413105910688, 123.24728017640338, 123.24344161008983, 123.24956145958618, 123.24920628974932, 123.24875549726411, 123.23851021350909, 123.23814138329391, 123.23778621345707, 123.22519134462756, 123.20270636187988, 123.19024809683378, 123.17774885065265, 123.16511300068814, 123.1300877906243, 123.10511661901874, 123.07994054173807, 123.05472348332239, 123.0193704241784, 122.99217261090341, 122.96485185422335, 122.93740815413824, 122.89962354764974, 122.87216618718628, 122.83882120365828, 122.80525765407684, 122.76146248111871, 122.7279399126723])
        expected['rho_s'] = np.array([6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_90_60000(self):
        """Test for latitude = -30, longitude = 90, rho = 60000"""
        lat = -30
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([82.34395558196809, 82.34094005344951, 82.33790915700533, 82.3348646001828, 82.33510109548281, 82.33200872733609, 82.32079355671893, 82.3094648342068, 82.30109272983167, 82.28965216297189, 82.27819195931826, 82.24392831534689, 82.21271264329265, 82.17827568327108, 82.14387372796902, 82.10915160754962, 82.06352679766758, 82.01473961019983, 81.96556054062845, 81.91640964558738, 81.87008047244719, 81.82330136059541, 81.77619525343712, 81.72883045286403, 81.68461180817737, 81.6370685689122, 81.58628440488586, 81.53547377387649, 81.48741229400805, 81.43629601203332, 81.38502519702863, 81.31825497524987, 81.25451821123625, 81.18766773473475, 81.12084116389534, 81.0537516307729, 80.99524081524798, 80.9338399759305, 80.8721266554585, 80.8104799293309, 80.7520783968327, 80.67822270756372, 80.60428505602468, 80.53035081958024, 80.45930319183375, 80.38532285161241, 80.29802620383047, 80.21083627775432, 80.12619742733317, 80.03874368520033])
        expected['lon_n'] = np.array([68.67164399084615, 68.76435014845073, 68.85819011745754, 68.95168174681667, 69.1171020983252, 69.20870176528425, 69.3278134342201, 69.44649480123825, 69.63721417343251, 69.75407187994226, 69.87061539775019, 70.04646544812134, 70.29154629591973, 70.46189804400902, 70.63019390515814, 70.79821655874045, 71.05858336990134, 71.24294383597832, 71.42788486813475, 71.60859118300579, 71.85843950284485, 72.05982080033351, 72.25942624863796, 72.4566889420571, 72.72216473471715, 72.91438674852883, 73.13388853788553, 73.35075387422259, 73.63611234755656, 73.8463797211554, 74.05544498146033, 74.24643073103222, 74.50524708925151, 74.68842593260209, 74.8692961720132, 75.0460204865981, 75.2753372577921, 75.43032108024887, 75.58372712900736, 75.73380004545088, 75.95153281581254, 76.16100788746766, 76.36654193997168, 76.56831255824304, 76.83639065297686, 77.02903613851707, 77.23407158721169, 77.43370435627281, 77.6991528281762, 77.88929846448018])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([-61.25029079226863, -61.230141734217085, -61.2099380346522, -61.18970701433062, -61.166388748504176, -61.14614406780427, -61.131841651682265, -61.11753923556026, -61.10014957393341, -61.085806176676385, -61.07139447752766, -61.05442828762935, -61.03434753146951, -61.01736768119284, -61.000319529024495, -60.983244056099466, -60.97064918726997, -60.96119620545866, -60.951674921755654, -60.94212631729599, -60.92944948619643, -60.92272858005316, -60.91600767390986, -60.90920480549653, -60.89928737082167, -60.89253914392168, -60.890776955115825, -60.888973785174954, -60.88398774708085, -60.882239218653325, -60.88040872795574, -60.88161084124968, -60.87973936941708, -60.881023444981054, -60.88218457713995, -60.88337303005554, -60.88286759605697, -60.88555869058993, -60.88818148323122, -60.89074963435914, -60.890216879603884, -60.903726993782186, -60.91722344758213, -60.930665259868704, -60.94089688324539, -60.954352355910316, -60.96787613046695, -60.98137258426689, -60.99160420764355, -61.00504601993015])
        expected['lon_s'] = np.array([113.19831654869793, 113.2020390017956, 113.20577511527159, 113.20934047401833, 113.20338454906208, 113.20707968140306, 113.22334236181686, 113.2394616082581, 113.24608689175301, 113.26238372311265, 113.27867372428314, 113.29189697051628, 113.295421348128, 113.30884950003625, 113.32213421797192, 113.33537795477258, 113.33703086055172, 113.34853289911402, 113.35993931502794, 113.37127742905017, 113.37296448577516, 113.36293776807358, 113.35282225791279, 113.34258380434694, 113.32272844442971, 113.31257878332308, 113.29268244227083, 113.27258119554348, 113.24287670284309, 113.222946210845, 113.20281764336097, 113.17110507504461, 113.12965948716096, 113.09795374903376, 113.06607725617732, 113.0341051406725, 112.98550834472785, 112.94656260608043, 112.90751441459547, 112.86830229857043, 112.81951425732903, 112.77599229193771, 112.73223126992544, 112.68837462526479, 112.63489424406359, 112.59083269372783, 112.5425978978092, 112.49412404526961, 112.43613573921621, 112.38753894327157])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_90_60000(self):
        """Test for latitude = 0, longitude = 90, rho = 60000"""
        lat = 0
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([75.22989008967126, 75.23051334443302, 75.2307677689796, 75.23110415579623, 75.2278529857513, 75.22811594803434, 75.2221395325106, 75.21600089999404, 75.2060322389004, 75.20018218187629, 75.19402647388682, 75.1647608208406, 75.13174539394005, 75.10249681636677, 75.07320042746929, 75.04393818951766, 74.99787197866051, 74.95587656054869, 74.91386065186936, 74.87166545072431, 74.82553606061734, 74.78577923700598, 74.74611803604301, 74.70612727845258, 74.66212549227207, 74.62223889506649, 74.58119458079659, 74.54042518164079, 74.49514615008547, 74.45375178862062, 74.41264600264819, 74.35997670641093, 74.3029941457128, 74.25038632117807, 74.19772897777186, 74.14513139852089, 74.09336368725407, 74.04578458949575, 73.99825671815623, 73.95081593172861, 73.89910286197515, 73.84107186723942, 73.78273009889647, 73.72480497209288, 73.6618699015331, 73.60373303886524, 73.53431270368853, 73.46462769868147, 73.39059869336228, 73.32104346194946])
        expected['lon_n'] = np.array([71.92424837549792, 71.93674079148988, 71.9504011698299, 71.96411618968328, 72.03237027005922, 72.04561400685988, 72.07755197141886, 72.11021393602987, 72.19736714983922, 72.22875186907542, 72.260874248742, 72.32570640434376, 72.44400528076838, 72.50736211550942, 72.57063698798042, 72.63395284158644, 72.77631447445701, 72.86338572599634, 72.95024524167138, 73.03717305923814, 73.17603763525366, 73.27907786907248, 73.38168780097358, 73.48473486498156, 73.63878295152203, 73.73975363802234, 73.86226674116485, 73.98419244803875, 74.15658642268987, 74.2772280539998, 74.3970568927985, 74.51641444854448, 74.68469664931526, 74.80138360109575, 74.91731923206756, 75.03282456112163, 75.17758359039088, 75.27220220096308, 75.36649979264426, 75.4601280257868, 75.60067282833816, 75.74088978180934, 75.88058764090361, 76.0187555376238, 76.20261739989138, 76.33782099451179, 76.4913090055403, 76.64390226178756, 76.83951887961673, 76.98792522990277])
        expected['rho_n'] = np.array([6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.198860740661, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([-55.34582448122438, -55.32697315911514, -55.308080855870884, -55.2891748922483, -55.26465451312794, -55.245707568370335, -55.232730208947316, -55.219698208010925, -55.20101081044177, -55.187937828370366, -55.17485118592063, -55.1600979773134, -55.13967571169505, -55.124854201196115, -55.11004635107554, -55.095224840576606, -55.084720009633145, -55.07993887721412, -55.07514408441676, -55.07015804632266, -55.05959857386583, -55.05647034722597, -55.05339676209945, -55.05030951659461, -55.04152589332196, -55.03847962895213, -55.03987298754282, -55.04102045932339, -55.036553515606215, -55.03774196852177, -55.03895774219404, -55.04596551628248, -55.04716762957639, -55.05425736593489, -55.06114219661825, -55.06804068767997, -55.070690801077916, -55.07920121678376, -55.08769797211124, -55.096126425547055, -55.09872189743166, -55.122395333094914, -55.14616439140656, -55.17004273274492, -55.18782854534365, -55.21144733949356, -55.23812605839163, -55.26469549426298, -55.28536364669142, -55.31178281840104])
        expected['lon_s'] = np.array([107.30780431412803, 107.31479159764895, 107.32175839060237, 107.32867054204242, 107.32369816432664, 107.33064446671256, 107.3499260907395, 107.36918039400976, 107.37651601717836, 107.39588643366451, 107.41527051052901, 107.43990700286524, 107.45256334339727, 107.47726130743604, 107.50197293185315, 107.52665723551358, 107.53856908542608, 107.56259086073702, 107.58661946623712, 107.6106070906022, 107.62255992164972, 107.62317463867502, 107.62368690286277, 107.62424014818555, 107.61264931716404, 107.61322305305431, 107.60208301451803, 107.59086784390087, 107.5675359176961, 107.5563275772681, 107.54518070854265, 107.52797546202339, 107.49854417688981, 107.48140723226224, 107.4642224763105, 107.44701039960206, 107.41148658572882, 107.38823662179409, 107.36499348804854, 107.34171620335714, 107.3059669932413, 107.28261457646902, 107.25916653704837, 107.23566385611436, 107.19961411767503, 107.1759748329576, 107.15277268034708, 107.12943392395314, 107.09356860062141, 107.07020935365996])
        expected['rho_s'] = np.array([6371.198101234436, 6371.197721481323, 6371.197721481323, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.2, 6371.198101234436, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.2, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.1984809875485, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_90_60000(self):
        """Test for latitude = 30, longitude = 90, rho = 60000"""
        lat = 30
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([75.94646657983846, 75.94807850448258, 75.94962468855594, 75.95124344338923, 75.95105219809247, 75.95266753783118, 75.94954187251226, 75.94643669776083, 75.94163080590609, 75.93839841888138, 75.93519762148158, 75.91045184611862, 75.88397803289566, 75.85931678112368, 75.8346811425611, 75.8101138058902, 75.77310357209883, 75.73769587144149, 75.70210631699749, 75.66660555501271, 75.6294356655495, 75.593881115825, 75.55836071704634, 75.5227259125991, 75.48518377782612, 75.44938846393337, 75.41225784805789, 75.37513064727699, 75.3363642010953, 75.29918748144291, 75.26182976177753, 75.21664464532331, 75.16999103819754, 75.12508766704659, 75.08029187137505, 75.03523140587319, 74.99439370482568, 74.95537966428657, 74.91648173696335, 74.87773236625458, 74.83720031617243, 74.7888494070379, 74.74065900734888, 74.69263765484182, 74.64313073619027, 74.59493179876478, 74.53649953041533, 74.47822947905865, 74.41863607855029, 74.36024649888314])
        expected['lon_n'] = np.array([85.53947916510964, 85.54125501429384, 85.54314014650477, 85.54509358060739, 85.56615105381854, 85.56811131811034, 85.57546743184643, 85.58297380974427, 85.60960471731816, 85.61698815181094, 85.62452185046547, 85.64107139882441, 85.67626053342832, 85.69214755343776, 85.70799359231219, 85.7238464613758, 85.7727711064006, 85.80278295761363, 85.83288360128587, 85.86291594306643, 85.9110960974717, 85.95998659155065, 86.00880195354873, 86.05756950422261, 86.12468977319634, 86.17290407854745, 86.23369276216057, 86.29437899293613, 86.37247537590605, 86.43241711606208, 86.49224274300222, 86.53694633111996, 86.5988688261353, 86.64242494247247, 86.68573517199953, 86.72922981663416, 86.77383778210351, 86.80044136892072, 86.82688786138701, 86.85310895761069, 86.89625526259765, 86.94894334185513, 87.00125576070826, 87.05303542480613, 87.1209616561019, 87.17185339560767, 87.2304973998214, 87.28851985682066, 87.36219027720841, 87.41909941337296])
        expected['rho_n'] = np.array([6371.199240493775, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([-55.52002162581638, -55.50324668121485, -55.486376113964894, -55.46936894293157, -55.44916524336668, -55.432144411954994, -55.42274607165706, -55.41327942946742, -55.40043867382781, -55.39109497504322, -55.381601012096894, -55.37078199245161, -55.35671180276137, -55.3458654623594, -55.33495082006573, -55.324077158907045, -55.32329851734167, -55.32598961187466, -55.32854410262425, -55.331276178292256, -55.330552178240225, -55.33691791454669, -55.343502216906586, -55.34988161359138, -55.35300984023124, -55.35947119918606, -55.37309059639108, -55.386682672839385, -55.39695527735108, -55.41042441039437, -55.42415309062608, -55.44570916764664, -55.46375452743382, -55.48532426483271, -55.50690766260996, -55.528532041522226, -55.54655008055272, -55.56803785568158, -55.58940268740537, -55.6108085002642, -55.62882653929469, -55.66747174961861, -55.70623990334761, -55.74489877404989, -55.780197191680514, -55.81881508124775, -55.86155840507371, -55.904247087386295, -55.943315769438755, -55.98585418758958])
        expected['lon_s'] = np.array([114.01606777726688, 114.02211249468235, 114.02807524982776, 114.03394238232481, 114.02222177770906, 114.02819819323283, 114.0505465721971, 114.07290861153973, 114.07752581941865, 114.10007910405804, 114.12251627548153, 114.14904473021787, 114.15785567424717, 114.18456854409109, 114.21106967807074, 114.2375913026179, 114.25424330381439, 114.28883138177135, 114.32341945972829, 114.35790508484767, 114.37455708604418, 114.3835251244244, 114.39248633261545, 114.40127678607726, 114.39202870994106, 114.4010377294563, 114.40027274826927, 114.39935750292048, 114.38028078456864, 114.37952946375994, 114.3786961806812, 114.37728916171217, 114.3574679527408, 114.35623851869019, 114.35491346199122, 114.3535064430222, 114.32587832782949, 114.31694444039512, 114.30782613785314, 114.29862587304113, 114.27060160687658, 114.26511696497306, 114.25952304004282, 114.25373103962664, 114.2287803585886, 114.22280394306483, 114.21908148996717, 114.21500386703268, 114.19155582761202, 114.18738258202914])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.2, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_90_60000(self):
        """Test for latitude = 60, longitude = 90, rho = 60000"""
        lat = 60
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([84.31637718359754, 84.31379366454398, 84.31121569501913, 84.30863132219193, 84.30847508161466, 84.30589070878746, 84.29925219180103, 84.29262562764565, 84.28844213677901, 84.28178995941424, 84.27516979856121, 84.24898584149113, 84.2252129411981, 84.19905288979012, 84.17287320158829, 84.14674815489981, 84.11473377759948, 84.08026352040572, 84.04579070189102, 84.01127946356225, 83.9792036145594, 83.94305996101897, 83.90693082163055, 83.87075002893653, 83.83695937556517, 83.80071753805544, 83.76149645740712, 83.72227537675879, 83.68538979391978, 83.64605003873461, 83.60667442505633, 83.55875082963175, 83.51333135231164, 83.46549654934627, 83.41771980298886, 83.37001050474949, 83.32910578247076, 83.28582458124768, 83.24263601446522, 83.19953367882103, 83.15890942118509, 83.10793358497517, 83.05703928414846, 83.00626536540588, 82.95781413786793, 82.90702357053924, 82.84628824777901, 82.78566733068739, 82.72754668971882, 82.66705768065553])
        expected['lon_n'] = np.array([101.87718920694259, 101.89099984944436, 101.90491294478367, 101.91879871936631, 101.95982766571058, 101.9737612516174, 101.99042691319224, 102.00716770684794, 102.05096970999523, 102.06731435267906, 102.08397318406473, 102.08571488230308, 102.11398503527776, 102.11505737497745, 102.11620484675801, 102.1173932996736, 102.15456318913681, 102.16479481251349, 102.17554553026709, 102.18579081402211, 102.22134194865203, 102.25369655475038, 102.28572331176856, 102.31787301219181, 102.37414011057437, 102.40496475429865, 102.44510577605081, 102.48512385439791, 102.54876072689493, 102.58748789948889, 102.62604431735362, 102.62146809060971, 102.64017597874637, 102.6349986953555, 102.63018341199064, 102.62562767581424, 102.63572269540752, 102.62290926052458, 102.61045782566765, 102.59817714553996, 102.60748669337869, 102.62878322321079, 102.64954699828763, 102.67031760355364, 102.71114847441197, 102.73089455130248, 102.73835994806531, 102.74598926936821, 102.77322806377822, 102.779928479354])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.200759506225, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-39.91846254253824, -39.82292185642811, -39.72897943458375, -39.63283769182664, -39.40970907202066, -39.313020914129964, -39.137990486459216, -38.96022798312046, -38.65329294219845, -38.47253881600321, -38.29144318034949, -38.06331486207108, -37.703336572054724, -37.47089157422084, -37.23695759514791, -37.00056474797377, -36.51671414717006, -36.1607929895207, -35.80164798258309, -35.438541465926875, -34.94004693954267, -34.596119594075844, -34.249774361642835, -33.899986713868145, -33.416245396091156, -33.06201812535596, -32.67231485207171, -32.280207352199625, -31.7535178049216, -31.35684773868394, -30.957855408128466, -30.477693109476576, -29.865025140926434, -29.380887672177593, -28.892856995601832, -28.40339197930038, -27.828508617238754, -27.379915452930646, -26.930652930083895, -26.478671991947465, -25.90209474297167, -25.31661092731818, -24.73112711166469, -24.143887937394496, -23.436526226191305, -22.84904116511099, -22.194872967163846, -21.5403427691907, -20.7723016574011, -20.11984100674647])
        expected['lon_s'] = np.array([103.62412620038924, 103.44641833856384, 103.27075953348947, 103.09141925645244, 102.76507964809839, 102.58434601247069, 102.27727436776526, 101.96473857172381, 101.50304510458763, 101.18534568553366, 100.86621192675398, 100.47057321908099, 99.92374827412978, 99.52188726412291, 99.11715757466463, 98.70852784719027, 97.97118843571988, 97.3775493741974, 96.780338123739, 96.17837306161826, 95.43205195138931, 94.86186775947664, 94.29047462408089, 93.71736028101428, 93.0086052112204, 92.43744430225642, 91.81098618177202, 91.18572334439239, 90.43285891293856, 89.81320366086749, 89.19524912589975, 88.46730805472731, 87.6247632394712, 86.90841299932028, 86.19446015556801, 85.4859851235301, 84.73327778642718, 84.09509906132705, 83.46212494037445, 82.83190338565737, 82.11034854135889, 81.31414655962134, 80.52826499371966, 79.74961659814905, 78.89609566889673, 78.13868233145568, 77.30942906432443, 76.49036643943485, 75.61134158382171, 74.81804243248142])
        expected['rho_s'] = np.array([56833.790100097656, 57031.94831237793, 57226.284689331056, 57423.46465759277, 57788.86006469726, 57982.93517150879, 58336.45190124511, 58690.07192382812, 59208.510873413084, 59557.77436828613, 59903.36792907715, 60337.30725402832, 60932.29835510254, 61360.550497436525, 61784.54332885742, 62206.74372558593, 62989.557678222656, 63604.660504150386, 64213.99108276367, 64818.0658782959, 65574.19989624023, 66126.45813903808, 66674.63935241698, 67221.42307434081, 67919.43967590331, 68459.60050354003, 69048.36365356445, 69635.13385925292, 70378.48994445801, 70962.37402648925, 71545.62619934081, 72249.88898010254, 73115.21568908691, 73819.8308807373, 74528.4623413086, 75238.57028198242, 76054.91795349121, 76708.0750793457, 77363.78414611817, 78025.41736145019, 78865.27326965332, 79735.57626342773, 80613.67482910155, 81501.89001770019, 82586.74440612792, 83498.65618896484, 84530.41197814941, 85577.91688842773, 86842.54336242676, 87927.1486328125])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_90_60000(self):
        """Test for latitude = 90, longitude = 90, rho = 60000"""
        lat = 90
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([84.14229828665555, 84.15131029437855, 84.16028900492935, 84.1692719843484, 84.17346273229101, 84.18240942633008, 84.19517419611519, 84.20790609561492, 84.21584064099636, 84.22851960653003, 84.2411878998931, 84.26737954092599, 84.28869613443877, 84.31474562215955, 84.34069607213738, 84.3666115173957, 84.39167148146046, 84.4214873916217, 84.45127256593165, 84.48096638646146, 84.50577875616882, 84.53655601545572, 84.56726369219045, 84.59789879816525, 84.62357817501038, 84.65407795786226, 84.68851961363886, 84.722848998181, 84.7521534979281, 84.78626218198268, 84.82023212781974, 84.86767291424736, 84.91000685360991, 84.9570519159525, 85.0038814004494, 85.05049402644015, 85.09144570626944, 85.13712473077801, 85.18255359960823, 85.22772334813682, 85.26770001846208, 85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089])
        expected['lon_n'] = np.array([255.73632539421095, 255.67977142788322, 255.622834970962, 255.56548870269057, 255.50158545281593, 255.4443757883279, 255.36555540530597, 255.2863525316905, 255.20026482739166, 255.12073410469603, 255.04106677821702, 254.8571434442469, 254.66532441159626, 254.47896953028163, 254.290565592216, 254.10104150312648, 253.88124601563547, 253.66573988694324, 253.4481573807433, 253.2274056667685, 252.99916806546335, 252.76855355832703, 252.5352342962794, 252.29890975099696, 252.054689507034, 251.81317401798233, 251.54605897992147, 251.27498243214202, 250.99609214795206, 250.71848593932606, 250.43675429644136, 250.1120197825423, 249.77794156985863, 249.44195090420735, 249.0993759361962, 248.75046255263527, 248.40165845210106, 248.04957586076523, 247.69159198598652, 247.3269145258212, 246.95466921605552, 246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.198860740661, 6371.2, 6371.200759506225, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.200759506225, 6371.200759506225, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([47.544486685306204, 47.55122466692242, 47.55758015794512, 47.56364878102267, 47.510786531941356, 47.517585985260105, 47.52181045726176, 47.52572757075076, 47.47063184981086, 47.47521149164935, 47.47951109573187, 47.48260517142589, 47.42664543155597, 47.43043960163991, 47.43391275283287, 47.43705463985107, 47.37875897528501, 47.380323088604946, 47.381460315101755, 47.38230042836966, 47.32409355626282, 47.32634410359434, 47.32829753769696, 47.32985140573314, 47.27244366575918, 47.27466689233402, 47.27610123205972, 47.27720772270526, 47.21921941665185, 47.22109088848444, 47.222644756520616, 47.2230579829654, 47.16460180895385, 47.165950771314925, 47.166900167609555, 47.167610507283236, 47.11030863524142, 47.112682125978, 47.11483363556655, 47.116544597953634, 47.0595876504649, 47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269])
        expected['lon_s'] = np.array([5.791035651950549, 5.756712243823942, 5.722305165880003, 5.687765753020894, 5.768920780078517, 5.734572185629346, 5.696477232420422, 5.65820213297214, 5.735513471074338, 5.697460779660904, 5.659333809940247, 5.618563130123971, 5.6932947911540195, 5.652742677391185, 5.612097075414085, 5.571295232859722, 5.6405584736855285, 5.59470698690041, 5.548805981243808, 5.502688543967882, 5.57183182959639, 5.527048413642732, 5.482137785415783, 5.437168246807242, 5.507260074956736, 5.462479647210841, 5.416120165106202, 5.369575414120327, 5.438118923762343, 5.3917888968485, 5.345322693038081, 5.294993163478373, 5.35955168462651, 5.3092379498792575, 5.258862316542651, 5.208312940269034, 5.27469597569921, 5.2261760193827325, 5.177641548914268, 5.128896623242002, 5.195070911015671, 5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653])
        expected['rho_s'] = np.array([353340.98607177736, 353350.90218505857, 353360.57525634766, 353370.0781982422, 353328.6881469726, 353338.7014770508, 353346.1871704101, 353353.08956298826, 353309.4635253906, 353317.3137817383, 353324.9939086914, 353330.8755249023, 353285.9370605469, 353292.5964111328, 353298.98841552733, 353304.8943359375, 353256.6018920898, 353260.24752197263, 353263.26124267577, 353266.177746582, 353218.0554321289, 353222.3086669922, 353225.954296875, 353229.405493164, 353182.13382568356, 353186.21693115233, 353189.0362182617, 353191.66107177734, 353143.4172363281, 353146.69830322266, 353149.8821533203, 353150.31962890626, 353099.91271972656, 353101.2251464844, 353102.1730102539, 353103.0479614258, 353053.66182861326, 353056.1651611328, 353058.4254516601, 353060.3211791992, 353011.39682617184, 353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_180_60000(self):
        """Test for latitude = -90, longitude = 180, rho = 60000"""
        lat = -90
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-12.808034707788607, -12.97745071996161, -13.1326531084718, -13.299528290273557, -12.40710260350886, -12.58209205004458, -13.232114323165533, -13.820924440944822, -13.452483546546816, -14.010694416844458, -14.629284159405003, -15.425451990196706, -15.324467643318059, -16.1064833221495, -16.846568469666238, -17.581127994144424, -17.87736012863705, -18.93476220461423, -19.886125593726973, -20.783195809127278, -21.028303977682356, -21.84793350827306, -22.617955374921948, -23.329135161871193, -23.54028362987296, -24.19739563935238, -25.005960263487665, -25.68560506703895, -25.94308953837009, -26.63593026739784, -27.240101480620496, -27.98644991160613, -28.333150313875976, -29.0145163250981, -29.654914861678492, -30.257419508743666, -30.514220961157804, -31.039107338494944, -31.53956895936011, -32.00232793600682, -32.18511745857471, -32.783797199704665, -33.37848811035933, -33.88485101466739, -34.166828544362204, -34.57930366833928, -34.91100497519177, -35.232351715262524, -35.42892455957549, -35.724726392150416])
        expected['lon_n'] = np.array([355.49011807122326, 355.4970029019067, 355.5022758079459, 355.5117561105139, 355.4408041054158, 355.4490003324198, 355.5499232075959, 355.63464487406077, 355.6633863100882, 355.74089529678946, 355.8394139453777, 355.96757561496383, 356.0392652804923, 356.1663887613246, 356.2882393361176, 356.4136962507924, 356.5628949030222, 356.7545500111327, 356.9261790045968, 357.0905133560273, 357.2330730643838, 357.38180726375003, 357.5223452361122, 357.6502610188882, 357.78782102877227, 357.90571009384666, 358.0658916902618, 358.1900098878593, 358.33412687934657, 358.4723425873909, 358.5843030482658, 358.7336109835223, 358.89920208976, 359.0361337222405, 359.16557946739056, 359.2872387968868, 359.4351806943093, 359.5385897583433, 359.63738161449834, 359.72639263976197, 359.8512211770331, 359.97714254457145, 0.1080659857276856, 0.20905609557843632, 0.3600146821180235, 0.42887651592795983, 0.4590881768546452, 0.48887826036306736, 0.6046723249921447, 0.6296240732472591])
        expected['rho_n'] = np.array([312165.0061889648, 311838.4063598633, 311570.6226928711, 311377.3557006836, 312745.2689453125, 312428.0748413086, 311612.9606079101, 310682.45003662107, 311410.6767578125, 310455.5217285156, 309826.1158813476, 308832.2199584961, 308981.44774169923, 308064.6204345703, 307289.6081298828, 306703.4151489258, 306402.018774414, 305491.12169189454, 304665.994128418, 304002.8783569336, 303920.4628173828, 303352.74102783203, 302870.27836914064, 302382.32296142576, 302508.2187133789, 302070.93756103516, 301969.1272705078, 301542.41844482423, 301536.63404541014, 301538.06799316406, 301264.9860107422, 301117.0949584961, 301123.02518310543, 301041.9706787109, 301024.5931762695, 301039.2243041992, 301253.2713867187, 301280.8323486328, 301350.7798339844, 301379.19144287106, 301384.90292968747, 301575.01037597656, 302019.8987426758, 302076.2115722656, 302220.52990722656, 301862.89361572266, 300664.9882446289, 299586.6595336914, 299137.15336914064, 298128.33466796874])
        expected['lat_s'] = np.array([-75.09847201371281, -75.07511276675135, -75.05167155751985, -75.02818936715337, -75.01167396974029, -74.98808249634706, -74.96914921196776, -74.95017494645347, -74.93819479464926, -74.91911124610826, -74.89997305605388, -74.88190037551004, -74.8708627898113, -74.85269448661904, -74.83447154191347, -74.81620761607286, -74.80798406831215, -74.79260248230128, -74.77715259439873, -74.76167538573947, -74.75338353608706, -74.74229130887497, -74.73121274204121, -74.72012051482912, -74.71622730700221, -74.70508043827675, -74.69661100370593, -74.68816888989178, -74.68693945584118, -74.6784427005137, -74.66994594518619, -74.66355288812306, -74.66444081271516, -74.65802043489535, -74.65158639669721, -74.64516601887738, -74.64975590599963, -74.64702383033162, -74.64429175466364, -74.64153235823895, -74.6461222453612, -74.65022035886321, -74.65431847236522, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67629802111432, -74.67862028543212, -74.68830549367519, -74.69060043723633])
        expected['lon_s'] = np.array([142.56377914925147, 142.54589771400438, 142.5280162787573, 142.51010752275351, 142.51061295675208, 142.49274518188332, 142.50681537157357, 142.52081725937208, 142.5532469975513, 142.5673445079982, 142.58144201844513, 142.57862798050706, 142.59409152878797, 142.5912638304716, 142.58847711329022, 142.58564941497386, 142.61641258699558, 142.62882987090666, 142.64142473973618, 142.65388300478227, 142.68460519566898, 142.67401840245546, 142.6632130431885, 142.65240768392155, 142.65993455238691, 142.64926579690334, 142.63264111646353, 142.61581153034862, 142.61738247385773, 142.60083975568796, 142.5841331129781, 142.55827401678044, 142.55061054453168, 142.52476510871236, 142.49889235213635, 142.47295129366864, 142.44791182017136, 142.40473136423856, 142.36134600263063, 142.31798796177938, 142.2926342995803, 142.24998659840273, 142.20713399155008, 142.16409013940066, 142.13928289233516, 142.09623904018576, 142.0419526966625, 141.98750242859916, 141.95112484107966, 141.89660627112463])
        expected['rho_s'] = np.array([6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_180_60000(self):
        """Test for latitude = -60, longitude = 180, rho = 60000"""
        lat = -60
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-74.77667448115682, -74.70887802345527, -74.64058979213348, -74.56982903233214, -74.55429718215953, -74.48210208263251, -74.40766668105769, -74.33335422288798, -74.31265874970282, -74.23240402695518, -74.15351534204154, -74.07928484614183, -74.0664031093672, -73.99116174547032, -73.91396694747084, -73.83510558331389, -73.81337192137491, -73.72718859442767, -73.6384644371092, -73.54878405330695, -73.52347137224288, -73.44615363083832, -73.36674585154776, -73.28654577031347, -73.27365037316048, -73.19145587668856, -73.11562711652309, -73.04180643197358, -73.03800884679504, -72.95958461474495, -72.88073691096633, -72.80776316987391, -72.81235305699616, -72.735813957157, -72.66034036682836, -72.58261281407363, -72.59104126750941, -72.51876420571233, -72.44464299283936, -72.3701529497512, -72.37787106351331, -72.3120963418061, -72.24647188426061, -72.17845686050563, -72.19769067320837, -72.12927949848154, -72.06250756915549, -71.99543511150597, -72.02063850954332, -71.95195412724965])
        expected['lon_n'] = np.array([24.58576452127031, 24.33370492768745, 24.080072683048197, 23.8264506836927, 23.95628574962547, 23.69849562732897, 23.416758861802407, 23.132143171540683, 23.233848103376758, 22.94793296962544, 22.65741941101541, 22.371760409357968, 22.467191812441385, 22.175407838645732, 21.883456525215415, 21.589635447499806, 21.643250724937115, 21.308419483897456, 20.972495412590597, 20.632970124043847, 20.67873068393564, 20.37127142599481, 20.062137064160034, 19.751414683343228, 19.829068811564394, 19.513223788870082, 19.21525337123369, 18.911345811661263, 18.99966015762953, 18.69658758868314, 18.389527896808755, 18.07793295932548, 18.1557800403907, 17.84373772551679, 17.527074787669367, 17.211431255555567, 17.310574866452885, 17.011240118529784, 16.712398851774218, 16.411650254692923, 16.51108585617726, 16.21003160813061, 15.906436529712712, 15.606386319474053, 15.705716053026256, 15.403082323734038, 15.097190594207724, 14.79015310044814, 14.882568975013, 14.576848000215936])
        expected['rho_n'] = np.array([258045.36323242186, 258001.66428222656, 257949.92064208983, 257937.16093749998, 257869.98413085938, 257836.3228149414, 257776.51018066405, 257685.51525878906, 257635.1326538086, 257622.25142822266, 257540.68653564452, 257534.5618774414, 257441.01501464844, 257387.13260498046, 257356.48500976563, 257335.68061523436, 257268.86837158204, 257212.72567138672, 257179.74487304688, 257125.6194213867, 257063.03610839843, 257002.85891113282, 256961.2501220703, 256907.48923339843, 256876.8416381836, 256790.3916015625, 256805.0713378906, 256726.20421142576, 256700.2959350586, 256683.453125, 256638.12579345703, 256607.6240234375, 256521.90311279296, 256527.97916259765, 256466.17358398438, 256440.87291259764, 256423.42249755858, 256371.1927734375, 256347.90935058592, 256304.01596679687, 256304.67218017578, 256269.16374511717, 256198.87600097657, 256192.14373779297, 256178.89794921875, 256129.05003662108, 256127.15430908202, 256109.02337646484, 256042.7701293945, 256033.72896728516])
        expected['lat_s'] = np.array([-68.23450072806517, -68.21517129271402, -68.19575989509283, -68.17630751633666, -68.15964185476182, -68.14017581562729, -68.12290909740548, -68.10560139804869, -68.09108041587322, -68.07371807500306, -68.05638305488958, -68.03974471407142, -68.02589309043464, -68.00921376848146, -67.99247980501494, -67.97574584154839, -67.96491316152475, -67.95132108507644, -67.93766070673641, -67.92395934726136, -67.91307202572435, -67.90656968563451, -67.90004002478798, -67.8934967035631, -67.88975375989793, -67.8832240990514, -67.87997292900647, -67.87669443820488, -67.87621632496297, -67.87297881529639, -67.86971398487313, -67.86955006033304, -67.87215919259597, -67.8720225888126, -67.87185866427251, -67.87168107935409, -67.87683104198828, -67.87923526857611, -67.88163949516397, -67.88401640099514, -67.88912538249429, -67.8976221378218, -67.9061598742843, -67.91467028999014, -67.92587180022898, -67.93442319706983, -67.94229157499367, -67.95014629253919, -67.96062380272599, -67.96851950140652])
        expected['lon_s'] = np.array([155.09051001159494, 155.0595146131414, 155.02854653544458, 154.99757845774775, 154.9884943061516, 154.95744426618475, 154.9474995107532, 154.93754109494333, 154.94941196372082, 154.9394398875326, 154.9293448679393, 154.90508403600742, 154.90257052639285, 154.87822773219094, 154.85388493798902, 154.82941920038203, 154.84198674845484, 154.8327250119403, 154.82349059618247, 154.81424252004626, 154.82660516244397, 154.80757625541634, 154.78850636725366, 154.7694091583343, 154.7719636490839, 154.7528254590295, 154.7346025143239, 154.71636590923998, 154.7197536830683, 154.70146243647102, 154.68314386911703, 154.6623800940402, 154.66321337711892, 154.6423813001504, 154.62152190242517, 154.60062152356494, 154.59216574937247, 154.56197631324102, 154.53171857521784, 154.50140619568134, 154.4927318554354, 154.4672962309663, 154.44171034233543, 154.41606981219118, 154.41208098171592, 154.38627652703158, 154.3520436189115, 154.31768776738633, 154.30503825704346, 154.2705184809782])
        expected['rho_s'] = np.array([6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_180_60000(self):
        """Test for latitude = -30, longitude = 180, rho = 60000"""
        lat = -30
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([71.22917167013341, 71.23160321747793, 71.23408940633581, 71.23656022726807, 71.24676282234077, 71.2494009829077, 71.25427773797509, 71.25949429495368, 71.27215746567488, 71.27733816416034, 71.2821978437548, 71.28533460813114, 71.2957967503923, 71.29882081664732, 71.30143677909943, 71.3044266944086, 71.32144069563111, 71.33092441329367, 71.34025103660532, 71.34948715991047, 71.36636797244417, 71.37510036929802, 71.38385667181399, 71.39247466299923, 71.40843681508956, 71.41682770248492, 71.4299963072047, 71.4432093081541, 71.46334982846918, 71.47603348975788, 71.48854639631735, 71.51297798297848, 71.54439002297137, 71.56828031714079, 71.59184959241921, 71.61508931107018, 71.64291037910692, 71.66336850321841, 71.68364050467501, 71.70342414760594, 71.73010969669319, 71.74795356589985, 71.76550544451949, 71.78270386084958, 71.806273136128, 71.82247946748114, 71.84325007274717, 71.86358696100088, 71.89004540629821, 71.90915286050132])
        expected['lon_n'] = np.array([217.348039466098, 217.3244206719481, 217.3009111608249, 217.27744263083676, 217.22659870265517, 217.20311651228866, 217.15682149009433, 217.1108133358451, 217.03744344378083, 216.99123038385653, 216.94490804090552, 216.8500913548474, 216.7279812328659, 216.6330279430244, 216.53785608712943, 216.443066721828, 216.28270754049444, 216.14916368184234, 216.0154012571368, 215.88148856826953, 215.7201048585605, 215.56631631920848, 215.4123365345597, 215.25813818385748, 215.07625024626006, 214.92127325399247, 214.73864765596466, 214.55578983150508, 214.34487358993508, 214.16096391634332, 213.97683567669813, 213.75223173603143, 213.49915956690413, 213.2732578902951, 213.04686444006586, 212.81997921621638, 212.57808123657122, 212.3638865041996, 212.14950052653126, 211.93482768091775, 211.6916046445736, 211.43152470135786, 211.171198871332, 210.910627154496, 210.6212456997409, 210.35982703944782, 210.0625908671472, 209.764808279713, 209.43760123733438, 209.13817940449934])
        expected['rho_n'] = np.array([6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.197721481323, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198101234436, 6371.199620246887, 6371.198101234436, 6371.199240493775, 6371.198101234436, 6371.2, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.197721481323, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.197721481323, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.199620246887, 6371.199620246887])
        expected['lat_s'] = np.array([-63.655992700973826, -63.64011934134271, -63.62420500057658, -63.60822235791878, -63.59244462093602, -63.57646197827822, -63.562460090479675, -63.54840356116779, -63.53452461677432, -63.52048174784079, -63.50639789777222, -63.49236868921702, -63.47850340520188, -63.464405894754975, -63.45028106355139, -63.43612891159111, -63.42651200523974, -63.416731174348286, -63.4069366830785, -63.397128531430354, -63.3874843043223, -63.38600898346158, -63.3845200022225, -63.38303102098345, -63.38171962466279, -63.38025796418043, -63.38223871903972, -63.38417849276399, -63.386364153298416, -63.38838588929272, -63.390393964908725, -63.39585811624474, -63.401472531742485, -63.4069366830785, -63.41244181554953, -63.41794694802056, -63.42530989194583, -63.43254989246603, -63.43973525147288, -63.446934270858094, -63.454310875161696, -63.46767072517824, -63.481071556329795, -63.49445872710302, -63.50803714317303, -63.521506276216286, -63.53467488093608, -63.54782982527752, -63.56120333567239, -63.57446756304057])
        expected['lon_s'] = np.array([157.65909827162662, 157.62879955246842, 157.59848717293193, 157.5681747933954, 157.54816233912726, 157.5177953180774, 157.5043944869258, 157.49099365577425, 157.48796105178278, 157.47451923949617, 157.46105010645292, 157.43468557625667, 157.41871659397717, 157.39239304491596, 157.3660285147197, 157.33963666376678, 157.34014209776535, 157.33023832346882, 157.3202935680373, 157.31032149184907, 157.31079960509098, 157.2954863209718, 157.2801593764743, 157.26477779046343, 157.2598873750177, 157.24449212862848, 157.23322231649797, 157.2219661647458, 157.22116020242373, 157.2098220884015, 157.1984566536226, 157.18773325662568, 157.18756933208562, 157.176818614332, 157.16598593430837, 157.15512593352804, 157.14660185744387, 157.1274363466328, 157.1082571754434, 157.08902336274068, 157.08041732438645, 157.06690721020817, 157.05327415262482, 157.03964109504147, 157.036649472185, 157.0228934711966, 157.00312690373858, 156.9833056947672, 156.9741532412794, 156.95415444738958])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_180_60000(self):
        """Test for latitude = 0, longitude = 180, rho = 60000"""
        lat = 0
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([71.2301996136035, 71.2329777930484, 71.23582085929041, 71.23860928401908, 71.24256737864309, 71.24523627506127, 71.24857111492354, 71.25199986988689, 71.25676221928568, 71.26001168178331, 71.26323382352426, 71.26216489891915, 71.26247738007368, 71.26138796490106, 71.26008852141148, 71.25883518169877, 71.26227588949317, 71.26441715379796, 71.26628008789409, 71.26804910688912, 71.27114830522501, 71.27152567317665, 71.27169642790591, 71.27164178639255, 71.27288658836878, 71.27289341855796, 71.27528056967287, 71.27765918305133, 71.2813321172775, 71.28321895703571, 71.28506140056433, 71.29637219382987, 71.3084821192283, 71.3190518369689, 71.32950373394631, 71.33977292336343, 71.34964596180868, 71.35820248129141, 71.36673680265935, 71.37469226549511, 71.38378495482769, 71.38739812489862, 71.39018996472187, 71.39305864417328, 71.3966069274471, 71.39878063515046, 71.40323904113119, 71.40727056028878, 71.41219171158578, 71.41536262690795])
        expected['lon_n'] = np.array([207.5714569523035, 207.5530974038145, 207.53488811948725, 207.5165832125116, 207.49516373927443, 207.4768178511638, 207.44142381088477, 207.40619369514584, 207.36784901314542, 207.33242765210971, 207.29697897031735, 207.21731164383831, 207.13472099639452, 207.05502634915882, 206.97534536230145, 206.89575999809247, 206.7761087442122, 206.65938081129667, 206.54242065194936, 206.42543317184538, 206.3053447858582, 206.16660998343693, 206.02771125647556, 205.8887169068658, 205.7465396891028, 205.60738141495295, 205.44147612001336, 205.27555716469539, 205.1064826619809, 204.94019487644778, 204.7738524494013, 204.57155590656384, 204.3659125710331, 204.1632198772238, 203.9605271834145, 203.75783448960522, 203.56672579662825, 203.37885461331788, 203.19127029795266, 203.00353571842567, 202.8127275537722, 202.58951697169618, 202.36615612545842, 202.14323241132755, 201.9171121686651, 201.6944889828577, 201.43641711525794, 201.1783452476582, 200.91703587039186, 200.6589366820354])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.199620246887, 6371.198860740661, 6371.197721481323, 6371.197341728211, 6371.1984809875485, 6371.2, 6371.1984809875485, 6371.197721481323, 6371.197721481323, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.197721481323, 6371.197721481323, 6371.199240493775, 6371.199620246887, 6371.198101234436, 6371.197721481323, 6371.198860740661, 6371.199240493775, 6371.197721481323, 6371.198101234436, 6371.2, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.198101234436, 6371.197721481323, 6371.199620246887, 6371.1984809875485, 6371.197721481323, 6371.198860740661, 6371.2, 6371.198101234436, 6371.198101234436, 6371.2, 6371.198101234436, 6371.198101234436, 6371.2, 6371.197721481323, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198860740661])
        expected['lat_s'] = np.array([-60.879206614661825, -60.86463099097301, -60.850069027662556, -60.835479743595414, -60.82121830860842, -60.80649242075788, -60.794116117981815, -60.78168517369238, -60.76969136150984, -60.75734237949047, -60.744884114444346, -60.7319613965347, -60.71950313148858, -60.70660773433562, -60.69373965793929, -60.68083060040797, -60.6739730904813, -60.66671942958274, -60.65952041019753, -60.65229407005566, -60.64550486202069, -60.647663201798395, -60.64982154157613, -60.6520618436239, -60.65476659853522, -60.65689761755627, -60.663195051971, -60.669560788277465, -60.67632267555578, -60.682606449592186, -60.68894486514196, -60.69897158284354, -60.709353470381956, -60.719284565435146, -60.72928396238004, -60.7393380008383, -60.7509083412923, -60.762109851531136, -60.773366003283314, -60.78463581541382, -60.796260797381194, -60.814470081708436, -60.832693026414034, -60.85086132960629, -60.86958970831046, -60.88777167188101, -60.90626782415342, -60.924750316047465, -60.94360163815671, -60.962248054590845])
        expected['lon_s'] = np.array([157.2090980883495, 157.18120359577915, 157.15322714093878, 157.1252097049634, 157.10776540182317, 157.07987090925283, 157.06810932350209, 157.0563067566163, 157.05496803953898, 157.04309717076148, 157.03128094349736, 157.0074709040507, 156.99407007289915, 156.97017807118243, 156.94620410719568, 156.92220282245228, 156.92717520016805, 156.92164274694034, 156.91601467106423, 156.91035927443147, 156.91522236912053, 156.90530493444567, 156.8953191978791, 156.8852514990425, 156.88567497077105, 156.87571655496117, 156.87161844145916, 156.86739738455208, 156.87383142275024, 156.86976063000492, 156.86558055423288, 156.86398228996708, 156.87312108307657, 156.8716730829725, 156.87011579984176, 156.86843557330593, 156.86873610162942, 156.85827225182095, 156.84764447747241, 156.83700304274552, 156.8371942880423, 156.83364258967387, 156.83003624979213, 156.82647089104537, 156.8335469670255, 156.82996794790043, 156.8218673435448, 156.8136847769191, 156.81647149410048, 156.80811134255637])
        expected['rho_s'] = np.array([6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.2, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.2, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.200759506225])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_180_60000(self):
        """Test for latitude = 30, longitude = 180, rho = 60000"""
        lat = 30
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([73.29961203588125, 73.30246705495432, 73.30531524383821, 73.30815489498565, 73.3113975272941, 73.31424571617801, 73.31705121637958, 73.31986696186493, 73.32304641492357, 73.32575287738219, 73.32841323606391, 73.32653322649486, 73.32504766035038, 73.32310447153151, 73.32116299025994, 73.31919248068439, 73.31858971649012, 73.31749176358105, 73.31633746161131, 73.31507387661486, 73.31412448032023, 73.31104235745727, 73.30788510251342, 73.30463051737391, 73.30176013037521, 73.29835869616855, 73.29653332811287, 73.294520129855, 73.29273232783976, 73.29027516728584, 73.28769677087416, 73.29260596934012, 73.29783447914976, 73.30232020588716, 73.30663517789532, 73.31057790459371, 73.31506533887841, 73.31882023537462, 73.32236510355386, 73.3257733679497, 73.32957949086469, 73.32781388696424, 73.32569140567965, 73.32316252813945, 73.32106907515885, 73.31810306551176, 73.31684631070448, 73.3152941502156, 73.3137898010509, 73.31127629143634])
        expected['lon_n'] = np.array([203.7194761472264, 203.70031063641537, 203.68125440863102, 203.66208889781996, 203.64851048174998, 203.6293996124523, 203.59539893076396, 203.56142556983232, 203.53297100175004, 203.49890201817, 203.46483303459, 203.3815593682292, 203.3038181550961, 203.22053082835697, 203.1374620676713, 203.05440696736395, 202.94174982719375, 202.82345095076911, 202.70516573472284, 202.58677123564985, 202.4737589256428, 202.3329340853355, 202.1921502261632, 202.0512707443425, 201.91580077234445, 201.77483932825373, 201.60856520309895, 201.4422364364308, 201.28119423618026, 201.1147288657287, 200.9482498348988, 200.74490144292918, 200.54700354191723, 200.3437234518393, 200.14063460705816, 199.93762772454704, 199.75420982457553, 199.56568294310483, 199.3773882880659, 199.18938050097213, 199.00679588407937, 198.786262736158, 198.56596181466838, 198.34601606301564, 198.13158910421225, 197.91235369223318, 197.6556068813324, 197.39917425913345, 197.14787793919035, 196.89177316607154])
        expected['rho_n'] = np.array([6371.199240493775, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.199240493775, 6371.197721481323, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199240493775, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.197721481323, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.198101234436, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199240493775])
        expected['lat_s'] = np.array([-62.01245794137199, -61.99874292151861, -61.98506888280025, -61.97117627802845, -61.95758420158012, -61.943828200591724, -61.933309709269906, -61.922914161353134, -61.91253227381472, -61.9020820843846, -61.89157725344111, -61.880799214930846, -61.87032170474404, -61.85951634547709, -61.84877928810181, -61.83785098542981, -61.83423098516968, -61.83014653204603, -61.82626698459745, -61.82212788996043, -61.81842592743027, -61.824204267468104, -61.83007823015433, -61.83596585321888, -61.84205838195854, -61.84787770313139, -61.85901091147849, -61.870144119825625, -61.88152321498285, -61.89258812143828, -61.90373499016374, -61.91877506671611, -61.934238614997014, -61.949319672684396, -61.96437340961512, -61.97942714654582, -61.99579227979717, -62.01193884699509, -62.028003451922956, -62.044109037985834, -62.06039220896716, -62.085609267382836, -62.110880967311886, -62.136166327619264, -62.161588291710075, -62.1869965954225, -62.21289667275519, -62.23886505197959, -62.2650110161224, -62.29112965950853])
        expected['lon_s'] = np.array([153.56601178884807, 153.53773480568424, 153.50930755835864, 153.48101691481645, 153.46917336679564, 153.44074611947005, 153.43108823198367, 153.42118445768713, 153.42808294874885, 153.4183021178574, 153.40850762658758, 153.3872110967555, 153.3822933605531, 153.36109245336937, 153.33965931975385, 153.3183764503001, 153.33370339479762, 153.33287011171888, 153.33170897955998, 153.3307937342112, 153.34627094287043, 153.33761026300286, 153.3287856585952, 153.3198517711608, 153.32772014908468, 153.3189501861904, 153.31546678971367, 153.31191509134527, 153.3252066394701, 153.3218188656418, 153.31829448803006, 153.31822618613836, 153.33493282884822, 153.33493282884822, 153.33491916846987, 153.33474158355145, 153.34150347082976, 153.3311762248047, 153.32084897877965, 153.31038512897118, 153.31720165776287, 153.31437395944647, 153.31142331772503, 153.3083360722202, 153.32273411099058, 153.31957856359404, 153.31299426123414, 153.30616407206412, 153.31696943133107, 153.310016298756])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.200759506225, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_180_60000(self):
        """Test for latitude = 60, longitude = 180, rho = 60000"""
        lat = 60
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([78.05341516047366, 78.05619846256045, 78.05897920332629, 78.06174969880837, 78.06122377424228, 78.0639763404778, 78.06700979824292, 78.07002703430878, 78.06972821353258, 78.0727206901627, 78.07569865264082, 78.07690418102933, 78.07477657710288, 78.07588562906936, 78.07696309141093, 78.07798761978643, 78.07629714796686, 78.07786126128678, 78.0792810868605, 78.08061126620136, 78.07858867643338, 78.07905825193883, 78.07939207743452, 78.07960808216701, 78.07647473288527, 78.0764568036387, 78.07782113392541, 78.07903093118215, 78.07687942159359, 78.07776136977017, 78.07847341699114, 78.08631532793197, 78.09078397919644, 78.09813923915891, 78.10521616891269, 78.1120352590253, 78.11620935838182, 78.12322652398036, 78.13000463295795, 78.13654539286189, 78.13985888838299, 78.14202320457623, 78.14387247829401, 78.14542036991467, 78.1437281905478, 78.14468014816337, 78.1481379314307, 78.15117309674312, 78.15096306842614, 78.15320507802119])
        expected['lon_n'] = np.array([208.27085466293445, 208.2422361703121, 208.21359035693305, 208.1849308831757, 208.1767756373067, 208.14817080506268, 208.1011108016813, 208.05390053413817, 208.02744038129353, 207.98040769866884, 207.93337501604412, 207.81140149784602, 207.71027371699483, 207.58850510447184, 207.4667228315705, 207.34480395488578, 207.2054680958175, 207.04490400880886, 206.8842716199085, 206.72340700457636, 206.58363401340122, 206.4007625285633, 206.21775443994198, 206.03447314375384, 205.87250203777617, 205.68901583591295, 205.47997106617555, 205.27044818319624, 205.0826179810209, 204.87316339993328, 204.6635175735489, 204.4019213283374, 204.16211338657826, 203.90028491493499, 203.6384427829134, 203.376409405595, 203.15078093655282, 202.9031729187615, 202.6554556179435, 202.40799786431393, 202.1830797349454, 201.89869797866277, 201.6146167507036, 201.3306311453928, 201.06982720212503, 200.78659291762293, 200.4556292712008, 200.12465196440033, 199.81699292342626, 199.48634346570594])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([78.35908149375737, 78.4298883573356, 78.50169413607993, 78.57559251403124, 78.72009028856475, 78.79297182210337, 78.86579786035497, 78.94208765828944, 79.08744176779012, 79.16556547151673, 79.24401617054986, 79.31065064231885, 79.44919505321704, 79.52005741208227, 79.58882204909867, 79.65892881828697, 79.79580239415944, 79.8653379887784, 79.93488297490747, 80.00425122989176, 80.14139886710468, 80.20195617805966, 80.26358839003527, 80.32433438496608, 80.45145103560918, 80.51513486565673, 80.56785453453912, 80.61650938709171, 80.73421489083219, 80.78655548421564, 80.83829416717849, 80.86844518349587, 80.9629630487778, 80.99728048048887, 81.02744430341095, 81.05850629620886, 81.15317613319982, 81.18309406931178, 81.2133714441288, 81.24222387073023, 81.33408308487772, 81.36583065791359, 81.39706852808264, 81.42676533682021, 81.51654646591274, 81.54890363333203, 81.5697288801114, 81.59715550472359, 81.67991178425476, 81.70342300292522])
        expected['lon_s'] = np.array([193.71882244752047, 193.808393548296, 193.8978007245315, 193.9839840514787, 194.0134358271798, 194.1078427018877, 194.20533682210046, 194.29539969649625, 194.3285944158625, 194.42085661117105, 194.51518152360893, 194.61249805890327, 194.63764681542727, 194.73108380327304, 194.8298756594281, 194.92705559093903, 194.95169891346444, 195.04794993924827, 195.14570360664948, 195.24536972701833, 195.27158399305281, 195.367206641433, 195.46039774246864, 195.55891639105693, 195.57856001510987, 195.67336304078967, 195.75462863153447, 195.84650833624949, 195.84605754376426, 195.9332380783303, 196.02041861289635, 196.08635725914365, 196.06499242741984, 196.1259996770864, 196.18971168166428, 196.2545301768877, 196.22365772183923, 196.28676866977017, 196.34892339121728, 196.41199335801318, 196.38168097847665, 196.4284541139129, 196.47672989096657, 196.52622144169246, 196.48101924976532, 196.5241177434281, 196.56777631260283, 196.5988263525697, 196.53671261225762, 196.57160121853804])
        expected['rho_s'] = np.array([260249.46245117186, 260356.3037109375, 260449.3887939453, 260506.9168334961, 260577.27749023438, 260681.68833007812, 260818.0591918945, 260887.18033447265, 260991.51826171874, 261063.5316040039, 261140.84326171875, 261266.78762207032, 261336.07889404296, 261421.33802490233, 261533.25886230468, 261621.60462646483, 261697.40942382812, 261786.36279296875, 261875.70502929686, 261967.47768554688, 262046.39342041014, 262156.22409667965, 262240.2680175781, 262352.6263305664, 262433.1704467773, 262506.3989990234, 262570.19752197264, 262703.2873168945, 262733.3273071289, 262823.884753418, 262908.998059082, 262989.3477416992, 263029.23093261715, 263066.7809204101, 263136.5582763672, 263208.4500976562, 263205.2905517578, 263278.34897460934, 263344.28626708983, 263417.0530395508, 263436.4477905273, 263486.36861572263, 263546.01112060546, 263617.95155029296, 263655.8661010742, 263684.9339233398, 263794.4243408203, 263799.06644287106, 263814.0135253906, 263869.2326660156])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_180_60000(self):
        """Test for latitude = 90, longitude = 180, rho = 60000"""
        lat = 90
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([84.14229828665555, 84.15131029437855, 84.16028900492935, 84.1692719843484, 84.17346273229101, 84.18240942633008, 84.19517419611519, 84.20790609561492, 84.21584064099636, 84.22851960653003, 84.2411878998931, 84.26737954092599, 84.28869613443877, 84.31474562215955, 84.34069607213738, 84.3666115173957, 84.39167148146046, 84.4214873916217, 84.45127256593165, 84.48096638646146, 84.50577875616882, 84.53655601545572, 84.56726369219045, 84.59789879816525, 84.62357817501038, 84.65407795786226, 84.68851961363886, 84.722848998181, 84.7521534979281, 84.78626218198268, 84.82023212781974, 84.86767291424736, 84.91000685360991, 84.9570519159525, 85.0038814004494, 85.05049402644015, 85.09144570626944, 85.13712473077801, 85.18255359960823, 85.22772334813682, 85.26770001846208, 85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089])
        expected['lon_n'] = np.array([255.73632539421095, 255.67977142788322, 255.622834970962, 255.56548870269057, 255.50158545281593, 255.4443757883279, 255.36555540530597, 255.2863525316905, 255.20026482739166, 255.12073410469603, 255.04106677821702, 254.8571434442469, 254.66532441159626, 254.47896953028163, 254.290565592216, 254.10104150312648, 253.88124601563547, 253.66573988694324, 253.4481573807433, 253.2274056667685, 252.99916806546335, 252.76855355832703, 252.5352342962794, 252.29890975099696, 252.054689507034, 251.81317401798233, 251.54605897992147, 251.27498243214202, 250.99609214795206, 250.71848593932606, 250.43675429644136, 250.1120197825423, 249.77794156985863, 249.44195090420735, 249.0993759361962, 248.75046255263527, 248.40165845210106, 248.04957586076523, 247.69159198598652, 247.3269145258212, 246.95466921605552, 246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.198860740661, 6371.2, 6371.200759506225, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.200759506225, 6371.200759506225, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([47.544486685306204, 47.55122466692242, 47.55758015794512, 47.56364878102267, 47.510786531941356, 47.517585985260105, 47.52181045726176, 47.52572757075076, 47.47063184981086, 47.47521149164935, 47.47951109573187, 47.48260517142589, 47.42664543155597, 47.43043960163991, 47.43391275283287, 47.43705463985107, 47.37875897528501, 47.380323088604946, 47.381460315101755, 47.38230042836966, 47.32409355626282, 47.32634410359434, 47.32829753769696, 47.32985140573314, 47.27244366575918, 47.27466689233402, 47.27610123205972, 47.27720772270526, 47.21921941665185, 47.22109088848444, 47.222644756520616, 47.2230579829654, 47.16460180895385, 47.165950771314925, 47.166900167609555, 47.167610507283236, 47.11030863524142, 47.112682125978, 47.11483363556655, 47.116544597953634, 47.0595876504649, 47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269])
        expected['lon_s'] = np.array([5.791035651950549, 5.756712243823942, 5.722305165880003, 5.687765753020894, 5.768920780078517, 5.734572185629346, 5.696477232420422, 5.65820213297214, 5.735513471074338, 5.697460779660904, 5.659333809940247, 5.618563130123971, 5.6932947911540195, 5.652742677391185, 5.612097075414085, 5.571295232859722, 5.6405584736855285, 5.59470698690041, 5.548805981243808, 5.502688543967882, 5.57183182959639, 5.527048413642732, 5.482137785415783, 5.437168246807242, 5.507260074956736, 5.462479647210841, 5.416120165106202, 5.369575414120327, 5.438118923762343, 5.3917888968485, 5.345322693038081, 5.294993163478373, 5.35955168462651, 5.3092379498792575, 5.258862316542651, 5.208312940269034, 5.27469597569921, 5.2261760193827325, 5.177641548914268, 5.128896623242002, 5.195070911015671, 5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653])
        expected['rho_s'] = np.array([353340.98607177736, 353350.90218505857, 353360.57525634766, 353370.0781982422, 353328.6881469726, 353338.7014770508, 353346.1871704101, 353353.08956298826, 353309.4635253906, 353317.3137817383, 353324.9939086914, 353330.8755249023, 353285.9370605469, 353292.5964111328, 353298.98841552733, 353304.8943359375, 353256.6018920898, 353260.24752197263, 353263.26124267577, 353266.177746582, 353218.0554321289, 353222.3086669922, 353225.954296875, 353229.405493164, 353182.13382568356, 353186.21693115233, 353189.0362182617, 353191.66107177734, 353143.4172363281, 353146.69830322266, 353149.8821533203, 353150.31962890626, 353099.91271972656, 353101.2251464844, 353102.1730102539, 353103.0479614258, 353053.66182861326, 353056.1651611328, 353058.4254516601, 353060.3211791992, 353011.39682617184, 353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_270_60000(self):
        """Test for latitude = -90, longitude = 270, rho = 60000"""
        lat = -90
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-12.77351493172337, -12.991206720950004, -13.139701863695251, -13.299849309164543, -12.40977320747433, -12.595185522683494, -13.245474173182075, -13.86089470796773, -13.454648716513717, -14.035679248828359, -14.61375913942156, -15.431640141584737, -15.340300021814159, -16.11246656786244, -16.846773375341343, -17.54304968952161, -17.885959336802102, -18.932194053486313, -19.886098272970287, -20.77461026134057, -21.005197447720207, -21.874755661143695, -22.612477563207605, -23.329770369464, -23.52932800644426, -24.19538756373639, -24.967902449432344, -25.68560506703895, -25.937447802115656, -26.63593026739784, -27.240101480620496, -27.978800099735707, -28.325937634112435, -29.007016777389424, -29.662414409387168, -30.276612340311402, -30.52055937670758, -31.043751867130553, -31.53956895936011, -32.02082408827921, -32.1852540623581, -32.795791011887204, -33.37848811035933, -33.88485101466739, -34.1580585814679, -34.57930366833928, -34.911032295948445, -35.23022069624149, -35.43126048427165, -35.724740052528745])
        expected['lon_n'] = np.array([355.479381013848, 355.5007731663285, 355.50429754394025, 355.5117561105139, 355.44165104887287, 355.45307112516514, 355.5540759626113, 355.6471851013769, 355.6640693290052, 355.7487090332, 355.834523529932, 355.9695427094448, 356.0442376582081, 356.16824657277886, 356.288293977631, 356.401347268773, 356.56570894096023, 356.7534845016222, 356.9261790045968, 357.08761735581925, 357.2252320072167, 357.3909050757245, 357.52032350011785, 357.6503156604016, 357.78402344359375, 357.90502707492965, 358.0522313119218, 358.1900098878593, 358.3320778225956, 358.4723425873909, 358.5843030482658, 358.7307423040709, 358.8964700140921, 359.0332650427891, 359.1684754675986, 359.29469736346044, 359.43769420392385, 359.5403929282842, 359.63738161449834, 359.733905847849, 359.8512484977898, 359.98195099774716, 0.10806736643975413, 0.2090603244260279, 0.3563294482152578, 0.4288829192303067, 0.4590981019732828, 0.4879725132461002, 0.6056791375644869, 0.6296300496627828])
        expected['rho_n'] = np.array([312033.56907958986, 311869.7344726562, 311590.74656982423, 311375.4113647461, 312755.1364501953, 312477.8498413086, 311664.2181640625, 310841.4967163086, 311419.1346191406, 310553.5162597656, 309762.90065917966, 308857.1317626953, 309044.3227050781, 308087.7823364258, 307290.45877685543, 306534.5495727539, 306441.8776611328, 305470.3659057617, 304666.0427368164, 303960.7591796875, 303804.385961914, 303486.0738647461, 302835.83931884763, 302378.92037353513, 302451.128149414, 302060.8027099609, 301753.1115478515, 301542.39414062497, 301503.21577148436, 301538.09229736327, 301264.9860107422, 301069.48303222656, 301077.35759277345, 300993.6053222656, 301073.4446166992, 301167.04008789064, 301296.5328613281, 301311.6986816406, 301350.7555297851, 301513.18049316405, 301384.90292968747, 301659.7591186523, 302019.8987426758, 302076.1629638672, 302150.77685546875, 301862.8450073242, 300665.0125488281, 299569.28203125, 299156.52381591796, 298128.33466796874])
        expected['lat_s'] = np.array([-75.09847201371281, -75.07509910637302, -75.05167155751985, -75.02817570677502, -75.01167396974029, -74.98808249634706, -74.96914921196776, -74.95017494645347, -74.93819479464926, -74.91911124610826, -74.89997305605388, -74.88190037551004, -74.8708627898113, -74.85269448661904, -74.83447154191347, -74.81620761607286, -74.80798406831215, -74.79260248230128, -74.77715259439873, -74.76167538573947, -74.75336987570873, -74.74229130887497, -74.73119908166288, -74.72012051482912, -74.71622730700221, -74.70508043827675, -74.69661100370593, -74.68816888989178, -74.68693945584118, -74.6784427005137, -74.66993228480786, -74.66355288812306, -74.66444081271516, -74.65802043489535, -74.65158639669721, -74.64516601887738, -74.64975590599963, -74.64702383033162, -74.64427809428528, -74.64153235823895, -74.6461222453612, -74.65022035886321, -74.65431847236522, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67628436073596, -74.67860662505376, -74.68829183329686, -74.69060043723633])
        expected['lon_s'] = np.array([142.56376548887312, 142.54589771400438, 142.52800261837893, 142.51012118313184, 142.51064027750877, 142.492731521505, 142.5068017111952, 142.52081725937208, 142.55321967679464, 142.5673445079982, 142.58144201844513, 142.57862798050706, 142.59407786840964, 142.59127749084993, 142.5884634529119, 142.5856357545955, 142.61641258699558, 142.628843531285, 142.64142473973618, 142.65388300478227, 142.68463251642567, 142.6739910816988, 142.6632130431885, 142.65240768392155, 142.65993455238691, 142.64926579690334, 142.63265477684186, 142.61581153034862, 142.61739613423606, 142.60083975568796, 142.58416043373478, 142.5582603564021, 142.55059688415335, 142.524751448334, 142.498878691758, 142.47295129366864, 142.44791182017136, 142.40473136423856, 142.36134600263063, 142.31798796177938, 142.2926342995803, 142.2499729380244, 142.20713399155008, 142.164103799779, 142.13928289233516, 142.09623904018576, 142.0419526966625, 141.9875160889775, 141.95116582221468, 141.89660627112463])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_270_60000(self):
        """Test for latitude = -60, longitude = 270, rho = 60000"""
        lat = -60
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([36.98854885136197, 36.933815130448075, 36.87923167369592, 36.82472676411921, 36.905719147297226, 36.851835784935, 36.7711131942292, 36.69145611303392, 36.74359094696863, 36.664777394135854, 36.5870976527053, 36.50156319372923, 36.54357227221939, 36.459554115239065, 36.37687126024148, 36.295356367591964, 36.29596425442809, 36.17790443462442, 36.063044558446904, 35.95075283339731, 35.95210521085297, 35.85638010963524, 35.76321632935627, 35.67217673790917, 35.68642451251782, 35.598332147697576, 35.51000414135097, 35.42416573895684, 35.4356541171408, 35.35298492252155, 35.27269263373346, 35.17638013624711, 35.169806079170975, 35.078274714103635, 34.99033261344513, 34.90552215452109, 34.91048770204769, 34.83911564031564, 34.76998388063135, 34.703208536210724, 34.70843704602037, 34.63732794657137, 34.569138752992544, 34.503183031272314, 34.50214142742389, 34.439532498396964, 34.37195802184345, 34.30645992279761, 34.29881011092719, 34.23583576677968])
        expected['lon_n'] = np.array([282.52828882871955, 282.5432879241369, 282.5580957742575, 282.5735866432951, 282.4513808986652, 282.46624339029916, 282.51307116724877, 282.5601721517652, 282.46763674888984, 282.51561199762, 282.56473471813075, 282.6258785715807, 282.5457467922381, 282.60814740049534, 282.6717774428032, 282.7372652965653, 282.7213099746641, 282.85228568218827, 282.9852284841934, 283.1211765694334, 283.11289838015927, 283.23291846425474, 283.35534277493804, 283.4802805952359, 283.4581234615684, 283.58573871602096, 283.72277963152806, 283.8625799434599, 283.8528264333251, 283.9968614625423, 284.1427543032138, 284.3420045816814, 284.39235673624273, 284.5978634679901, 284.80610227540546, 285.01868508313294, 285.05146999114896, 285.23968268391786, 285.4306547731114, 285.6244135794863, 285.66659682780033, 285.8944246177553, 286.12547625699847, 286.36106314185054, 286.4460580158822, 286.6873822596371, 286.96613594004367, 287.2492336207623, 287.3826955171444, 287.6735796135169])
        expected['rho_n'] = np.array([63359.418981933595, 63642.64189147949, 63928.49573059082, 64215.87465820312, 63915.59627685547, 64200.514404296875, 64645.33593444824, 65091.92559509277, 64952.000244140625, 65399.20358581543, 65846.82009887695, 66360.48934936523, 66282.78274841308, 66796.03275146484, 67311.13594970702, 67828.64526367188, 68001.53318481446, 68768.05117187499, 69538.11757202148, 70315.16535339355, 70493.25437316894, 71193.51911315917, 71896.12920837403, 72603.19912414551, 72706.18209228515, 73415.69458007812, 74168.21334838867, 74925.04003601075, 75073.14375, 75832.15173950195, 76596.3425994873, 77540.96175842285, 77868.98338317871, 78819.18035583496, 79773.94651794434, 80734.81303405762, 80974.38560180664, 81841.14018249512, 82713.96473693848, 83589.77263183593, 83835.56707458496, 84873.69056396485, 85917.67744140625, 86967.49732666016, 87378.04993591308, 88432.4086303711, 89607.55919494628, 90790.96103515624, 91327.05698547362, 92515.59308776855])
        expected['lat_s'] = np.array([-77.6142355907036, -77.59998781609494, -77.5856717395946, -77.5713010215809, -77.5572718130257, -77.54276449122858, -77.52909045251022, -77.51536177227848, -77.50202924301863, -77.48820494013853, -77.47432599574506, -77.46487301393375, -77.45577520195928, -77.4461856163646, -77.43656871001323, -77.42692448290515, -77.41650161423172, -77.40547768891133, -77.39449474472593, -77.38349814016223, -77.37293866770537, -77.36726961069428, -77.36158689330483, -77.35593149667204, -77.35075421328119, -77.34501685437837, -77.34156077865833, -77.33818666520835, -77.33524968386524, -77.33179360814523, -77.32832387204687, -77.32634311718755, -77.32493609821853, -77.32292802260255, -77.32097458849992, -77.31903481477565, -77.32350175849282, -77.32743594745475, -77.3313701364167, -77.3353316461353, -77.3398395709875, -77.35003021322916, -77.36026183660584, -77.37057542225256, -77.38138078151951, -77.39169436716622, -77.40163912259777, -77.411611198786, -77.42214335048615, -77.43214274743104])
        expected['lon_s'] = np.array([175.56199397309217, 175.48391125050057, 175.4058968298007, 175.3281556166676, 175.2855898777601, 175.2074661740335, 175.17403922823544, 175.14063960319407, 175.1426886599451, 175.1090431480936, 175.07554790040385, 175.02878842534597, 175.01775083964722, 174.9709230626976, 174.9239586819646, 174.87710358425832, 174.8915699249204, 174.87021875357493, 174.8487719595811, 174.8275437316407, 174.84203739305946, 174.80630184332196, 174.77049799169276, 174.73494002687366, 174.73500832876536, 174.69923179789285, 174.6611330027025, 174.6230751886472, 174.62099881113951, 174.58272243103076, 174.54441873016532, 174.50738544448552, 174.5062516330833, 174.46886317756665, 174.43157034469837, 174.39422287031675, 174.3817509448923, 174.3324642998415, 174.28304105100727, 174.2337270851998, 174.22106391447858, 174.18679002522347, 174.15250247558998, 174.11801002028145, 174.12060549216605, 174.08573054626396, 174.03668978802327, 173.9875534071342, 173.97543665154657, 173.9258494781723])
        expected['rho_s'] = np.array([6371.199240493775, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.197341728211, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.198860740661, 6371.197721481323, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.197721481323, 6371.197341728211, 6371.197721481323, 6371.1984809875485, 6371.198860740661])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_270_60000(self):
        """Test for latitude = -30, longitude = 270, rho = 60000"""
        lat = -30
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([58.39521548382818, 58.43501841121643, 58.47464716878085, 58.51403003953514, 58.54821172123647, 58.587529705193646, 58.63858536923949, 58.68910486343549, 58.73434803649766, 58.785120247692944, 58.835475817348865, 58.893214821497565, 58.94540429694564, 59.00324575393189, 59.06113843733692, 59.11905161130946, 59.17533578516495, 59.23787641230017, 59.30064585077258, 59.3633708930154, 59.419798500843456, 59.47963778816194, 59.53910483017066, 59.59891338163787, 59.65233570623113, 59.712427710548894, 59.77856784737672, 59.84452356909694, 59.90415795074033, 59.970666917783326, 60.03705635651585, 60.12191462676409, 60.20029104748998, 60.28584258193898, 60.371206286185796, 60.45707200933661, 60.53601533576362, 60.622389908007605, 60.708784970819096, 60.7952210147656, 60.874895171433806, 60.96954793295184, 61.0640811661594, 61.158873946555424, 61.24675799060598, 61.341854714420066, 61.44362453305325, 61.54515017242362, 61.639686820725764, 61.74166496012864])
        expected['lon_n'] = np.array([282.85408885212917, 282.8253747368584, 282.79649669704764, 282.7674820534534, 282.6873502741108, 282.6584722343, 282.62956687373253, 282.6002790225715, 282.5195461865819, 282.49058618450107, 282.4613529748534, 282.4195248963763, 282.32545953112685, 282.2836587734064, 282.24153016660574, 282.1991283522383, 282.1226029127775, 282.0990524205193, 282.07539264523433, 282.0513777001126, 281.97386871341126, 281.95556380643563, 281.93709497491994, 281.9182982943241, 281.84546115701505, 281.826937683986, 281.8116380602451, 281.7960652289375, 281.72563231821636, 281.71033269447554, 281.69475986316786, 281.6939129197108, 281.63744091565314, 281.6370584250596, 281.63645736841266, 281.6358836325224, 281.5767615150667, 281.5746304960457, 281.57225359021453, 281.56979472211333, 281.5097163781739, 281.49581011302377, 281.4816579610635, 281.46717796002304, 281.3940129736339, 281.37953297259344, 281.3707083681858, 281.36161055621136, 281.2927349286209, 281.2838283619432])
        expected['rho_n'] = np.array([6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.200759506225, 6371.1984809875485, 6371.198101234436, 6371.2, 6371.199240493775, 6371.198860740661, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.2, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.199620246887, 6371.198101234436, 6371.198101234436, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.198860740661])
        expected['lat_s'] = np.array([-77.27189284912421, -77.27340915111995, -77.27495277387237, -77.27645541548978, -77.27689254759667, -77.2783678684574, -77.27556749089769, -77.27279443409466, -77.26892854702442, -77.26614182984306, -77.26330047114834, -77.26652432043659, -77.26866899983597, -77.2718518679892, -77.27510303803413, -77.27834054770071, -77.27469322668392, -77.27213873593433, -77.26957058480642, -77.26702975443516, -77.26343707493174, -77.26570469773617, -77.26806794318901, -77.2704175282635, -77.27159232080072, -77.27388726436186, -77.27750726462196, -77.28118190639543, -77.28379103865836, -77.2874793408102, -77.29122228447534, -77.29523843570732, -77.29798417175365, -77.30200032298563, -77.30601647421759, -77.31007360658458, -77.31975881482765, -77.33061881560798, -77.34145149563162, -77.35232515679027, -77.36206500654671, -77.37718704536914, -77.39236372570491, -77.40748576452731, -77.42136470892078, -77.43659603076989, -77.4528382206162, -77.46914871235418, -77.48406584550148, -77.50033535610447])
        expected['lon_s'] = np.array([209.5779889455348, 209.5149189787389, 209.45190365345638, 209.38865610174207, 209.3329763996281, 209.26914145164517, 209.24803616710983, 209.2270674863579, 209.21381691936807, 209.19241110650924, 209.17096431251542, 209.15798695309238, 209.15301457537663, 209.1397640083868, 209.12649978101862, 209.11330385554217, 209.12619925269516, 209.13084378133075, 209.13542000807468, 209.14007819708863, 209.15275502818815, 209.1509245374906, 209.14901208452298, 209.1472089145821, 209.15431231131893, 209.15294627348493, 209.16077367027376, 209.16861472744094, 209.1848159361522, 209.19216521969912, 209.1994598617327, 209.21018325872961, 209.22977224126922, 209.2402360910777, 209.25063163899443, 209.2609588850195, 209.28468696219613, 209.29978168026184, 209.31549111535287, 209.33117322968724, 209.35567994842924, 209.40651021623248, 209.45750440857577, 209.50948214815958, 209.57110411485144, 209.62325943935366, 209.6678469142555, 209.71233876650896, 209.76763597802938, 209.8132070001717])
        expected['rho_s'] = np.array([6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.197721481323, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.1965822219845, 6371.1965822219845, 6371.196202468872, 6371.197341728211, 6371.197721481323, 6371.198860740661, 6371.2, 6371.198101234436, 6371.197341728211, 6371.196961975097, 6371.1965822219845, 6371.1965822219845, 6371.196961975097, 6371.198101234436, 6371.199620246887, 6371.1984809875485, 6371.197341728211, 6371.196961975097, 6371.196202468872, 6371.196202468872, 6371.197721481323, 6371.199620246887, 6371.199240493775, 6371.196961975097, 6371.195822715759, 6371.196202468872, 6371.1965822219845, 6371.1984809875485, 6371.198860740661, 6371.196961975097, 6371.195822715759, 6371.195442962647, 6371.196961975097, 6371.1984809875485, 6371.197721481323, 6371.196202468872, 6371.195822715759, 6371.1965822219845, 6371.199620246887, 6371.1965822219845, 6371.195822715759])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_270_60000(self):
        """Test for latitude = 0, longitude = 270, rho = 60000"""
        lat = 0
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([56.84624469328518, 56.87583307276968, 56.90533948998413, 56.93483566191483, 56.967562513322946, 56.99688451542981, 57.03482621626923, 57.072767917108656, 57.114032504979285, 57.151950300156614, 57.18982711419892, 57.23133758887967, 57.27627340342919, 57.31753457620523, 57.35887771125132, 57.40021743120282, 57.44719205721958, 57.49108285282608, 57.534953157865075, 57.57861514213438, 57.625651239853674, 57.66806329950487, 57.710598302561124, 57.75279521125346, 57.79837647867954, 57.84099002891126, 57.889149692749015, 57.937042979209146, 57.988218171565464, 58.03655200522706, 58.08506683890166, 58.149540409572, 58.21759299936741, 58.28266079649555, 58.34767053701572, 58.41314814549405, 58.48337273544554, 58.55063643839182, 58.61833044325582, 58.68595273113353, 58.75670666074569, 58.829724798067716, 58.90233995422871, 58.97548103495579, 59.05158300268807, 59.124946064563176, 59.20297756073599, 59.280961245584606, 59.36234294954531, 59.44073986083872])
        expected['lon_n'] = np.array([274.55786787866487, 274.52221429119743, 274.48650606221656, 274.4507158709657, 274.40000854656756, 274.36427299683004, 274.32364703164683, 274.28288446268016, 274.22715011905285, 274.18646951235627, 274.1457342641463, 274.0881147883081, 274.0152230094857, 273.9574122883507, 273.8993283596489, 273.8411351479204, 273.77892578495994, 273.73198872498364, 273.6848604197105, 273.63748622762733, 273.57464848726323, 273.5343776919168, 273.493970292787, 273.45334432760376, 273.39695428581615, 273.35624635836285, 273.3189262047379, 273.2813328435462, 273.22794808499333, 273.1903000822883, 273.1525427965564, 273.1189929073533, 273.06937841322235, 273.0358285240192, 273.00200542724934, 272.9681003682094, 272.9167646664076, 272.88157553180366, 272.8463317556864, 272.810760130489, 272.7585774852301, 272.71396268957153, 272.66904736558956, 272.62394079631076, 272.56184071637705, 272.5163243357481, 272.473294143977, 272.42996342388244, 272.3694206270794, 272.32576205790474])
        expected['rho_n'] = np.array([6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.198101234436, 6371.2, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.197721481323, 6371.198860740661, 6371.1984809875485])
        expected['lat_s'] = np.array([-74.48199279960579, -74.48352276197986, -74.48502540359726, -74.48643242256628, -74.49438276276018, -74.4958580836209, -74.49271619660271, -74.4895060076928, -74.49286646076445, -74.48954698882781, -74.48643242256628, -74.48982019639462, -74.49979227258282, -74.50313906527614, -74.50658148061783, -74.51017416012127, -74.51237348103399, -74.50815242412693, -74.50398600873322, -74.49983325371787, -74.50219649917068, -74.50318004641116, -74.50420457478666, -74.50524276354051, -74.51260570746578, -74.51363023584128, -74.51573393410564, -74.51786495312669, -74.52645733110256, -74.52853370861024, -74.53066472763129, -74.53198978433028, -74.5396805773357, -74.54099197365636, -74.5423716718687, -74.54381967197273, -74.55853189944494, -74.5670149943941, -74.57544344782991, -74.58395386353573, -74.59880269479135, -74.61169809194433, -74.62475741363738, -74.6378030749521, -74.65703688765487, -74.67004156783457, -74.68386587071467, -74.69763553208142, -74.71766164672789, -74.73136300620294])
        expected['lon_s'] = np.array([209.84513130435235, 209.8045326599258, 209.76382473247253, 209.72396374847634, 209.68721733074165, 209.64635913912664, 209.64891362987623, 209.65182329046266, 209.65810706449906, 209.66160412135412, 209.66398102718526, 209.67794193384879, 209.69485348223373, 209.7090739360857, 209.7228709182091, 209.7364766550358, 209.78044941291233, 209.82014647236844, 209.8597888903112, 209.89993674225252, 209.9436772736973, 209.9787707856528, 210.01430142971523, 210.05026920588452, 210.09058098236594, 210.1262618905901, 210.17572612055932, 210.22555918074372, 210.2787936751348, 210.3285857541842, 210.37886960685384, 210.43894795079325, 210.5026462949928, 210.5626153559055, 210.62252977530486, 210.6825807984876, 210.7487106900317, 210.81027801521017, 210.8728835291425, 210.93585787329002, 211.0019194629424, 211.10791033848264, 211.21336845926766, 211.31978280653644, 211.43045919184732, 211.53766584105986, 211.64603362243128, 211.75546691331323, 211.86861582710364, 211.97892338219935])
        expected['rho_s'] = np.array([6371.196961975097, 6371.197721481323, 6371.1984809875485, 6371.196202468872, 6371.1965822219845, 6371.1965822219845, 6371.196961975097, 6371.197341728211, 6371.200759506225, 6371.1965822219845, 6371.198101234436, 6371.1965822219845, 6371.199240493775, 6371.196961975097, 6371.196961975097, 6371.198860740661, 6371.195822715759, 6371.196202468872, 6371.196961975097, 6371.196202468872, 6371.198101234436, 6371.198860740661, 6371.2, 6371.198860740661, 6371.195822715759, 6371.1965822219845, 6371.196202468872, 6371.196202468872, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.1965822219845, 6371.196961975097, 6371.195822715759, 6371.197721481323, 6371.198860740661, 6371.1965822219845, 6371.199240493775, 6371.1984809875485, 6371.196961975097, 6371.199620246887, 6371.196202468872, 6371.1965822219845, 6371.196961975097, 6371.196961975097, 6371.196202468872, 6371.196961975097, 6371.196202468872, 6371.198101234436, 6371.195822715759])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_270_60000(self):
        """Test for latitude = 30, longitude = 270, rho = 60000"""
        lat = 30
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([62.750024570317834, 62.773450411623685, 62.7968625925512, 62.82024745272203, 62.84369719968998, 62.8669932674016, 62.89672678840596, 62.92644664903197, 62.9562570096645, 62.98588124764213, 63.01547987241038, 63.0511898089385, 63.0869509718854, 63.12260797444745, 63.15825302417845, 63.19386050786902, 63.23079987844774, 63.26762825845245, 63.30447883657196, 63.341401131677756, 63.37843953999945, 63.41562479738829, 63.45291763025656, 63.490213878219414, 63.52771161676279, 63.56507958171193, 63.60846152822526, 63.65181786152921, 63.69533470427865, 63.73875763192701, 63.782209587879336, 63.84050866753998, 63.89911510571328, 63.95767885520425, 64.01633993489091, 64.07512737307721, 64.1367390944853, 64.19830471211651, 64.25997449013255, 64.32156572097315, 64.38347114051555, 64.44818547535414, 64.51303641397611, 64.57803420166525, 64.64333764031974, 64.70861375821755, 64.77805287641449, 64.84758932480709, 64.91733409396939, 64.98708227822627])
        expected['lon_n'] = np.array([270.5059810969599, 270.47330547197055, 270.4404659224411, 270.4076263729117, 270.37139904955393, 270.3386141415379, 270.3009388180761, 270.26320885310093, 270.2220091520274, 270.18430650780897, 270.1465765428338, 270.0887931424555, 270.02734876068206, 269.9693467942503, 269.9111809032785, 269.85285108776657, 269.80244429169187, 269.75547991095885, 269.7084335679558, 269.6611686588993, 269.61024276844773, 269.56841468997055, 269.52639536619665, 269.48423943863935, 269.4383952089302, 269.3961573191028, 269.35640561813335, 269.3165446341372, 269.2728314234491, 269.23272455264276, 269.1924537572964, 269.1546691508079, 269.1131416006542, 269.0750837865989, 269.03691668951683, 268.998476384868, 268.95323321180587, 268.9114870955987, 268.8693858095548, 268.82706595745736, 268.78083923715474, 268.73166187513067, 268.68210202251305, 268.6322689623286, 268.57841975091225, 268.5280675963509, 268.4816769515082, 268.43490381607194, 268.3840598878904, 268.33682229959055])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([-74.59704050598549, -74.59186322259461, -74.58668593920373, -74.58138571240781, -74.58197310867644, -74.57670020263717, -74.57088088146432, -74.56521182445323, -74.56533476785827, -74.55954276744211, -74.55387371043099, -74.55213884238182, -74.55648284269395, -74.55482993691481, -74.55324533302735, -74.55163340838322, -74.5512509177897, -74.5448305399699, -74.53825989798835, -74.53177121827684, -74.53151167108837, -74.53157997298007, -74.53171657676347, -74.53181219941186, -74.53836918101507, -74.53850578479847, -74.5412105397098, -74.54388797386443, -74.55313605000063, -74.55586812566864, -74.55861386171497, -74.55996623917065, -74.56801220201294, -74.56922797568518, -74.57068963616757, -74.57202835324489, -74.58710941093227, -74.59540126058468, -74.60363846872372, -74.61202594102448, -74.62743484779205, -74.64313062250471, -74.65893568024413, -74.67479537949691, -74.69797704153993, -74.71380942003603, -74.72934127020864, -74.7449004411379, -74.76795915977587, -74.78354565146185])
        expected['lon_s'] = np.array([182.23734275462073, 182.19655286489743, 182.15643233371276, 182.1156287836111, 182.08629995131506, 182.04512757099823, 182.05565972269838, 182.06601428948014, 182.0883626684444, 182.09858063144276, 182.10868931141437, 182.11281474567306, 182.12898863362764, 182.1334419169665, 182.13782689841364, 182.14212991759075, 182.20045973310266, 182.24604441562332, 182.29217551327758, 182.33745966747477, 182.3969915962806, 182.4319075233177, 182.46782065797365, 182.50339228317108, 182.5520368904399, 182.5881549307709, 182.63113048102863, 182.6751032389052, 182.7326544128517, 182.77619003862137, 182.81983494741775, 182.87946249887196, 182.95346076833988, 183.01296537638905, 183.07357647508374, 183.13357285675312, 183.19630131409053, 183.24408531752394, 183.2923064530642, 183.3400767961193, 183.4042805743174, 183.4888792973772, 183.57477575637927, 183.66063123424632, 183.76289282649975, 183.8501416629575, 183.93647525406647, 184.02279518479708, 184.1272697583416, 184.21476448160945])
        expected['rho_s'] = np.array([6371.198860740661, 6371.2, 6371.1984809875485, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.197721481323, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.2, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.1984809875485, 6371.2, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.197721481323, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.2, 6371.198101234436, 6371.1984809875485, 6371.197721481323, 6371.2, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.2, 6371.198101234436, 6371.197721481323, 6371.199620246887, 6371.196961975097, 6371.1984809875485])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_270_60000(self):
        """Test for latitude = 60, longitude = 270, rho = 60000"""
        lat = 60
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([72.69767941715907, 72.71418115419382, 72.73068118368128, 72.74718121316874, 72.76026956316578, 72.776709828498, 72.79826590551856, 72.81976904857305, 72.83788441779922, 72.85934145707681, 72.88074897748292, 72.91282866846718, 72.94150521769748, 72.97350636150628, 73.00548359965299, 73.03744546987406, 73.06855698154348, 73.10304260666287, 73.13750944876205, 73.17196775312476, 73.20299730252412, 73.23898215416634, 73.27498237373419, 73.31099625368037, 73.34356942583216, 73.37960379634586, 73.42146773331615, 73.46334191557021, 73.50175831455695, 73.54362395907454, 73.58550667906506, 73.6427573246881, 73.69654164930736, 73.75380595530876, 73.811104412256, 73.86841994467616, 73.92368471279802, 73.98243629249117, 74.04116225897495, 74.0998933481006, 74.155079569047, 74.21708744142697, 74.2791021439961, 74.34111343147065, 74.39956106774574, 74.46150917597046, 74.52975471860994, 74.59804294993172, 74.66279314326344, 74.73112064817296])
        expected['lon_n'] = np.array([265.5183310782065, 265.4906824724462, 265.46303386668603, 265.43516669487235, 265.3952510693628, 265.36754782208925, 265.3307467628412, 265.29397302434984, 265.24479566232577, 265.2079672823211, 265.1710569400463, 265.0993672745179, 265.01500077788984, 264.94287398025455, 264.87052861656576, 264.7978554037968, 264.71594777527, 264.6461432419525, 264.57603818031146, 264.50557794883366, 264.42214035793273, 264.35725356081764, 264.29195695235234, 264.22625053253677, 264.1474847910282, 264.08114999380905, 264.0139955738895, 263.9462947388363, 263.8653160160366, 263.79693216206647, 263.72816581750277, 263.6560936613808, 263.5703338061621, 263.497250782043, 263.42351205976354, 263.34925424310717, 263.2595055574132, 263.1824883443321, 263.1049520368741, 263.0267873520125, 262.93447051519064, 262.83964016875416, 262.7439628788606, 262.64743864551, 262.53613388279547, 262.43810700782745, 262.3405719064797, 262.2420532578914, 262.12834426858905, 262.02799512930324])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.200759506225, 6371.200759506225, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([40.18340740167587, 40.18683615663922, 40.189971213468255, 40.192641817433724, 40.15182460695373, 40.15529775814668, 40.154973324161105, 40.15443715431126, 40.109911151111945, 40.1100580001791, 40.10983601903108, 40.107237132051885, 40.06028299660264, 40.058367128540446, 40.05614048687102, 40.053401581013844, 40.00219906790085, 39.99634218068756, 39.989973029286524, 39.98328627408908, 39.9315339307479, 39.92605270393896, 39.92007287332061, 39.9137890992842, 39.86208115217262, 39.85643258572902, 39.84850615119722, 39.8401289241802, 39.786102127845396, 39.77839767446162, 39.77038244747062, 39.75875405040867, 39.70101163116538, 39.689984290750395, 39.678530063512284, 39.66656698718101, 39.609528077422226, 39.59966186916615, 39.58937218918152, 39.57853267896871, 39.52087905218463, 39.50468467366253, 39.48803267246604, 39.47080352028468, 39.40632311942518, 39.38963696728284, 39.3696006073526, 39.348990511532094, 39.280579336805246, 39.26038929761869])
        expected['lon_s'] = np.array([335.23814272927194, 335.2094012932445, 335.1807691402438, 335.1521916287565, 335.2543985794965, 335.2256844642258, 335.19464808463727, 335.1635570635354, 335.26327782541756, 335.23207752128894, 335.20109578321376, 335.16831087519773, 335.26614650486897, 335.23327963458286, 335.20041276429674, 335.1676551770374, 335.26142001396335, 335.2245369924453, 335.1877632539539, 335.1510441569759, 335.24472703163184, 335.2080625761672, 335.17150740372927, 335.13497955204804, 335.22877170973067, 335.19218921653606, 335.1540494402107, 335.1159369846421, 335.2080352554105, 335.16992279984186, 335.1317557027598, 335.0898729827693, 335.17784581927907, 335.1357445332351, 335.0936705679478, 335.05167856493057, 335.14085351473426, 335.09992702122753, 335.0591371315042, 335.0183745625376, 335.10733094628785, 335.0638089808965, 335.0205055815586, 334.97725682373414, 335.06348113181633, 335.02009577020846, 334.97471599336285, 334.92941817878733, 335.0134568263352, 334.96813169100295])
        expected['rho_s'] = np.array([350859.2356811523, 350880.7448974609, 350902.1325927734, 350923.2529418945, 350851.53125, 350873.2835083008, 350892.3866088867, 350911.00362548826, 350836.87581787107, 350856.29487304686, 350875.3493652344, 350892.5567382812, 350816.751940918, 350834.5669189453, 350852.11455078126, 350869.419140625, 350790.06593017577, 350804.7456665039, 350818.7691894531, 350832.7197998047, 350752.97772216797, 350767.51163330075, 350781.8511108398, 350795.534375, 350715.6950805664, 350730.20468749997, 350742.429699707, 350754.533190918, 350672.871081543, 350685.38774414064, 350697.6856689453, 350706.43518066406, 350621.03022460936, 350630.2415161133, 350639.0882446289, 350647.5704101562, 350562.55432128906, 350572.78638916014, 350582.21641845704, 350591.3791015625, 350505.9498413086, 350511.5884155273, 350516.69229736325, 350521.4802246094, 350431.3116455078, 350436.41552734375, 350438.79733886715, 350440.6444580078, 350347.21911621094, 350349.45510253904])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_270_60000(self):
        """Test for latitude = 90, longitude = 270, rho = 60000"""
        lat = 90
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([84.14229828665555, 84.15131029437855, 84.16028900492935, 84.1692719843484, 84.17346273229101, 84.18240942633008, 84.19517419611519, 84.20790609561492, 84.21584064099636, 84.22851960653003, 84.2411878998931, 84.26737954092599, 84.28869613443877, 84.31474562215955, 84.34069607213738, 84.3666115173957, 84.39167148146046, 84.4214873916217, 84.45127256593165, 84.48096638646146, 84.50577875616882, 84.53655601545572, 84.56726369219045, 84.59789879816525, 84.62357817501038, 84.65407795786226, 84.68851961363886, 84.722848998181, 84.7521534979281, 84.78626218198268, 84.82023212781974, 84.86767291424736, 84.91000685360991, 84.9570519159525, 85.0038814004494, 85.05049402644015, 85.09144570626944, 85.13712473077801, 85.18255359960823, 85.22772334813682, 85.26770001846208, 85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089])
        expected['lon_n'] = np.array([255.73632539421095, 255.67977142788322, 255.622834970962, 255.56548870269057, 255.50158545281593, 255.4443757883279, 255.36555540530597, 255.2863525316905, 255.20026482739166, 255.12073410469603, 255.04106677821702, 254.8571434442469, 254.66532441159626, 254.47896953028163, 254.290565592216, 254.10104150312648, 253.88124601563547, 253.66573988694324, 253.4481573807433, 253.2274056667685, 252.99916806546335, 252.76855355832703, 252.5352342962794, 252.29890975099696, 252.054689507034, 251.81317401798233, 251.54605897992147, 251.27498243214202, 250.99609214795206, 250.71848593932606, 250.43675429644136, 250.1120197825423, 249.77794156985863, 249.44195090420735, 249.0993759361962, 248.75046255263527, 248.40165845210106, 248.04957586076523, 247.69159198598652, 247.3269145258212, 246.95466921605552, 246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.198860740661, 6371.2, 6371.200759506225, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.200759506225, 6371.200759506225, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([47.544486685306204, 47.55122466692242, 47.55758015794512, 47.56364878102267, 47.510786531941356, 47.517585985260105, 47.52181045726176, 47.52572757075076, 47.47063184981086, 47.47521149164935, 47.47951109573187, 47.48260517142589, 47.42664543155597, 47.43043960163991, 47.43391275283287, 47.43705463985107, 47.37875897528501, 47.380323088604946, 47.381460315101755, 47.38230042836966, 47.32409355626282, 47.32634410359434, 47.32829753769696, 47.32985140573314, 47.27244366575918, 47.27466689233402, 47.27610123205972, 47.27720772270526, 47.21921941665185, 47.22109088848444, 47.222644756520616, 47.2230579829654, 47.16460180895385, 47.165950771314925, 47.166900167609555, 47.167610507283236, 47.11030863524142, 47.112682125978, 47.11483363556655, 47.116544597953634, 47.0595876504649, 47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269])
        expected['lon_s'] = np.array([5.791035651950549, 5.756712243823942, 5.722305165880003, 5.687765753020894, 5.768920780078517, 5.734572185629346, 5.696477232420422, 5.65820213297214, 5.735513471074338, 5.697460779660904, 5.659333809940247, 5.618563130123971, 5.6932947911540195, 5.652742677391185, 5.612097075414085, 5.571295232859722, 5.6405584736855285, 5.59470698690041, 5.548805981243808, 5.502688543967882, 5.57183182959639, 5.527048413642732, 5.482137785415783, 5.437168246807242, 5.507260074956736, 5.462479647210841, 5.416120165106202, 5.369575414120327, 5.438118923762343, 5.3917888968485, 5.345322693038081, 5.294993163478373, 5.35955168462651, 5.3092379498792575, 5.258862316542651, 5.208312940269034, 5.27469597569921, 5.2261760193827325, 5.177641548914268, 5.128896623242002, 5.195070911015671, 5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653])
        expected['rho_s'] = np.array([353340.98607177736, 353350.90218505857, 353360.57525634766, 353370.0781982422, 353328.6881469726, 353338.7014770508, 353346.1871704101, 353353.08956298826, 353309.4635253906, 353317.3137817383, 353324.9939086914, 353330.8755249023, 353285.9370605469, 353292.5964111328, 353298.98841552733, 353304.8943359375, 353256.6018920898, 353260.24752197263, 353263.26124267577, 353266.177746582, 353218.0554321289, 353222.3086669922, 353225.954296875, 353229.405493164, 353182.13382568356, 353186.21693115233, 353189.0362182617, 353191.66107177734, 353143.4172363281, 353146.69830322266, 353149.8821533203, 353150.31962890626, 353099.91271972656, 353101.2251464844, 353102.1730102539, 353103.0479614258, 353053.66182861326, 353056.1651611328, 353058.4254516601, 353060.3211791992, 353011.39682617184, 353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_0_90000(self):
        """Test for latitude = -90, longitude = 0, rho = 90000"""
        lat = -90
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.90688287510915, -72.88874189267358, -72.8686747968921, -72.85101192769844, -72.88099645815478, -72.86271887193584, -72.84416807815009, -72.82800785057384, -72.85974090945771, -72.84296596485618, -72.82430588804368, -72.81035864175851, -72.84452324798693, -72.82994762429811, -72.81564520817611, -72.79945765984317, -72.83621773795619, -72.82401902009855, -72.81206618905102, -72.8003865655703, -72.83520686995902, -72.82410098236858, -72.81326830234494, -72.80268150913142, -72.84099887037519, -72.82978369975805, -72.81889637822104, -72.81034498138018, -72.85094362580674, -72.8417501911839, -72.83282996412785, -72.8274477750619, -72.86945343845747, -72.86344287198787, -72.85766453195004, -72.85213207872232, -72.89609117622052, -72.88967079840072, -72.88348264701267, -72.87751306167809, -72.91960068734372, -72.9163904984338, -72.91345351709069, -72.91069412066602, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99974612706464, -72.99780635334037])
        expected['lon_n'] = np.array([28.70190289249762, 28.64750726594764, 28.59361024320707, 28.53976103179069, 28.673280984780686, 28.61871289595416, 28.547513296498654, 28.476248810246034, 28.59222542235285, 28.5201703417038, 28.448601912033116, 28.36504820791635, 28.468491422896193, 28.384020765883353, 28.29983697681565, 28.216030555699597, 28.2942003632031, 28.184700477976747, 28.07530475313524, 27.96591756603019, 28.04308162817841, 27.93465408265175, 27.82636314090849, 27.718179774644657, 27.795961968912763, 27.68673529125321, 27.568448367659638, 27.449942878012624, 27.517005090378394, 27.39750410065985, 27.27808507321135, 27.13911633681098, 27.18515010426957, 27.04481703758249, 26.904538612408768, 26.764360932525303, 26.817279530667268, 26.68440332300655, 26.55180203045992, 26.419303190750842, 26.47082330766025, 26.311299116952725, 26.151631492272628, 25.992225122328286, 26.015910510822597, 25.85506126341609, 25.679313665882493, 25.503724870247098, 25.510640436781735, 25.33370780142714])
        expected['rho_n'] = np.array([257214.93735351562, 257202.37208251952, 257252.87620849608, 257240.5782836914, 257196.92794189454, 257184.0710205078, 257235.5230102539, 257224.14864501954, 257181.5919921875, 257169.6829345703, 257220.867578125, 257210.87855224608, 257169.9259765625, 257159.45086669922, 257149.097277832, 257201.91030273438, 257162.5375, 257153.52064208983, 257144.67391357422, 257135.87579345703, 257159.49947509766, 257151.30895996094, 257143.19135742186, 257135.21958007812, 257096.86755371094, 257088.45830078123, 257144.67391357422, 257137.96595458983, 257101.09648437498, 257093.92674560545, 257087.02435302734, 257082.57668457032, 257111.15842285156, 257106.24897460936, 257101.4853515625, 257096.8432495117, 257062.3555908203, 257057.1544921875, 257052.19643554688, 257047.38420410155, 257075.91733398437, 257072.95222167968, 257070.06002197266, 257067.4108642578, 257035.20780029296, 257032.02395019532, 257030.27404785156, 257028.6213623047, 257060.654296875, 257058.53983154296])
        expected['lat_s'] = np.array([-68.33946707522992, -68.30527514824485, -68.27094661747634, -68.23646782254613, -68.21161959434562, -68.17694955411864, -68.14398706118416, -68.11077868143954, -68.08737845334309, -68.05406079057175, -68.02046992023364, -67.98901006891657, -67.96737202962595, -67.93565263112043, -67.90385127034483, -67.87180402275914, -67.85184621000437, -67.82172507576462, -67.79142635660642, -67.7609363921515, -67.74043216426313, -67.71996891750976, -67.69928710470296, -67.67850966924777, -67.66795019679094, -67.64703615755238, -67.63134038283968, -67.61552166472194, -67.61003019262924, -67.59411585186311, -67.57811954882695, -67.56609841588772, -67.56439086859521, -67.55228777338596, -67.54008905552831, -67.52780837540061, -67.5294066396664, -67.5205956956371, -67.5116891289594, -67.50270060001165, -67.50417592087237, -67.50285086417338, -67.50144384520436, -67.50003682623534, -67.50916195896647, -67.50771395886244, -67.50597909081327, -67.50410761898067, -67.51287758187496, -67.5110470911774])
        expected['lon_s'] = np.array([152.8373262270561, 152.8151690933886, 152.79301195972107, 152.77086848643188, 152.80079837537488, 152.77868222284238, 152.78021218521647, 152.7817967891039, 152.83523618917008, 152.83653392511238, 152.83813218937817, 152.82361120720273, 152.86101332309772, 152.84649234092225, 152.83169815118, 152.81708154635618, 152.87289785225354, 152.87672275818875, 152.88045204147556, 152.88393543795226, 152.93976540422796, 152.92906932798772, 152.91849619515253, 152.9078001189123, 152.94915008414756, 152.93848132866398, 152.92726615804682, 152.9159963459163, 152.95689551866636, 152.94569400842752, 152.93435589440531, 152.92036766698513, 152.95880797163395, 152.94481974421376, 152.93064027149683, 152.91644713840154, 152.9439181592433, 152.91863279893593, 152.89316985371013, 152.86762494621428, 152.8948364198676, 152.87409996554746, 152.85318592630887, 152.8320669813952, 152.8639776251975, 152.84269475574374, 152.8135025272311, 152.7841873553134, 152.80809301740845, 152.77857293981566])
        expected['rho_s'] = np.array([6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_0_90000(self):
        """Test for latitude = -60, longitude = 0, rho = 90000"""
        lat = -60
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-6.292915333999375, -6.303386013996999, -6.31445092045243, -6.326574506229193, -6.321035222812313, -6.331826921700937, -6.349913262623133, -6.368921679083272, -6.3674600186008945, -6.385320963280478, -6.404889455252558, -6.429437155129591, -6.427982324836378, -6.451375722743677, -6.47683183778031, -6.504794632242337, -6.507383273937776, -6.543528635025481, -6.583334977508315, -6.627983924112684, -6.6226495463709085, -6.6649489079007935, -6.713245175521962, -6.768501405907358, -6.747348310047826, -6.802263030974743, -6.8743078663400325, -6.969814401504323, -6.917652246812935, -7.015569838754232, -7.1992746066708975, -9.138399462983372, -7.524835573459555, -9.692204861266347, -11.5460548057912, -13.271162514650499, -11.917043360749616, -13.474374302836708, -14.88247927193737, -16.17211460988416, -15.053363774781914, -16.47693912235266, -17.77632114024506, -18.983837113800774, -18.18404928274896, -19.355044234812624, -20.43765653901633, -21.480578613953085, -20.85218071974441, -21.848076942245626])
        expected['lon_n'] = np.array([358.0126162147357, 358.00212504417055, 357.99163387360545, 357.98133394833707, 358.0312216500348, 358.02075780022636, 358.0143374224066, 358.0079716861001, 358.0611378785995, 358.05466285926633, 358.04859765128333, 358.0446634623214, 358.09889516433134, 358.094824371586, 358.0912180317042, 358.08804882392934, 358.14624203565785, 358.14823645089547, 358.1509958473202, 358.15482075325536, 358.21099222898954, 358.21361502163086, 358.2175218898361, 358.22293139965876, 358.274485667514, 358.27992249809336, 358.29008581957834, 358.30557668861593, 358.35046469184124, 358.36650197601244, 358.4019096766698, 358.7047602644681, 358.509307571179, 358.848658689902, 359.1392695787077, 359.42031820267533, 359.25327909633353, 359.4996576800742, 359.72532713025146, 359.93460412642065, 359.79991279598795, 0.028635141208455332, 0.24032732962075956, 0.4409943387326203, 0.3520257086666358, 0.5454451933780253, 0.7204271160334945, 0.8945325866802334, 0.8304130119079751, 0.9936463154999347])
        expected['rho_n'] = np.array([360389.68992919923, 360418.441796875, 360448.45748291013, 360481.21954345703, 360451.42259521486, 360481.04941406247, 360526.5225708008, 360573.7213256836, 360551.7989379883, 360596.9318359375, 360645.7832763672, 360705.10982666013, 360682.09375, 360739.33013916016, 360801.7433227539, 360869.50343017577, 360853.3897460937, 360939.2321777344, 361033.9213378906, 361140.25220947264, 361104.403515625, 361206.28671874997, 361322.24205322267, 361454.94298095704, 361380.49921875, 361513.1029296875, 361686.12452392577, 361915.6047729492, 361765.1131713867, 361999.57578124997, 362414.6185913086, 358358.4907836914, 361842.27900390624, 357537.9081054687, 354082.23984374997, 351651.0178833008, 353886.3723022461, 351446.08487548825, 349508.3110717773, 347971.50794677733, 349560.273449707, 347685.6662597656, 346238.3025878906, 345217.0158325195, 346000.6075195312, 345072.9162353515, 344028.6134033203, 343446.38200683595, 343929.7925292969, 343331.54466552736])
        expected['lat_s'] = np.array([-76.24000152969703, -76.20893782935181, -76.17792877051994, -76.14690605130974, -76.14458378699194, -76.11327419983661, -76.08802982066425, -76.06286740376191, -76.06652838515703, -76.0410517795529, -76.01552053243537, -75.99123237974683, -75.99612279519255, -75.9714931330455, -75.94695909354681, -75.92242505404812, -75.93033441310698, -75.90854610965465, -75.88677146658065, -75.86507878577669, -75.87294716370053, -75.85483350202168, -75.83677448185614, -75.81875644282567, -75.83058633046812, -75.81237704614088, -75.79720036580511, -75.7821876100094, -75.79713206391341, -75.78183244017256, -75.76669674097181, -75.75348715511703, -75.77031674123194, -75.75691591008035, -75.74354239968548, -75.73030549307398, -75.75112390966419, -75.74158896558285, -75.73205402150151, -75.72265568120358, -75.74354239968548, -75.7420534184464, -75.74066005985574, -75.73936232391344, -75.76847259015602, -75.76698360891697, -75.76352753319694, -75.76024904239534, -75.78737855377861, -75.78388149692358])
        expected['lon_s'] = np.array([135.10354600944007, 135.11696050096995, 135.13014276606808, 135.1426420122492, 135.13653582313123, 135.14988201276944, 135.19817145020141, 135.2460510762832, 135.2752023236608, 135.32398353471305, 135.3722593117667, 135.4018886723862, 135.41307652224668, 135.44363478859333, 135.47382422472478, 135.50339894383094, 135.5322906440201, 135.58097623242395, 135.62892416039745, 135.67663986193915, 135.7057637885601, 135.72297586526852, 135.73981911176176, 135.7559110374463, 135.7538209995603, 135.77095111399868, 135.7779179069521, 135.78458417158203, 135.77230349145435, 135.77972107689297, 135.7868654547648, 135.7820433412108, 135.75817866025076, 135.75446303734228, 135.75014635778683, 135.7452969234761, 135.69816861820303, 135.6710254464414, 135.64349978408626, 135.61578287643434, 135.56773932581245, 135.53718105946584, 135.50611735912062, 135.47482143234362, 135.42228361724787, 135.39145214333445, 135.3500748573425, 135.30827409962203, 135.2445484346658, 135.20326677132223])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.199240493775, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.198101234436, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.2, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.2, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.198101234436, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.197721481323, 6371.198860740661, 6371.2, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.198101234436, 6371.198860740661])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_0_90000(self):
        """Test for latitude = -30, longitude = 0, rho = 90000"""
        lat = -30
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([5.965430075097615, 5.969022754601042, 5.972349056726827, 5.975497773934208, 5.9476784134447485, 5.951394036353236, 5.953647998779331, 5.955697055530337, 5.923615656998791, 5.926245279829246, 5.928615355471237, 5.930432185790465, 5.900795994981777, 5.903213881947963, 5.905392712293192, 5.907318825639138, 5.873208860924095, 5.874445125163874, 5.875435502593518, 5.87624146491558, 5.844836255111858, 5.846837500538683, 5.848674821425405, 5.850238934745349, 5.816313385137889, 5.81851270605064, 5.820247574099824, 5.821818517608918, 5.7903859870485235, 5.792598968339604, 5.794566062820564, 5.795904779897896, 5.760790777374851, 5.762682739774945, 5.764472249337501, 5.7661046645491325, 5.731222888457879, 5.736844134144789, 5.739316662624333, 5.741618436374637, 5.706538584797443, 5.708218811333268, 5.709728283139839, 5.71106700021717, 5.67455963910345, 5.676567714719425, 5.6779405827426075, 5.679204167739059, 5.644916618105597, 5.646890542775722])
        expected['lon_n'] = np.array([358.5262464403207, 358.51411602435473, 358.50195828763214, 358.4897459093961, 358.535781384402, 358.5236236476794, 358.5131871186276, 358.5027232688191, 358.5504526307392, 358.5400161016874, 358.52957957263567, 358.5199353455276, 358.56859361317476, 358.5589767068234, 358.5493871212287, 358.5397428941206, 358.5907234260856, 358.58342878405205, 358.57616146277513, 358.56878485847153, 358.61984735270653, 358.61186969175594, 358.6038647100487, 358.5958324075848, 358.6461845621461, 358.6381795804389, 358.6306936931085, 358.62309852275143, 358.67407905471646, 358.6665112051161, 358.65897067627236, 358.6529327890461, 358.70544328338514, 358.69940539615885, 358.6932582259058, 358.6871383764095, 358.7389112103182, 358.73199905887816, 358.72511422819474, 358.7181201144847, 358.7699202691501, 358.7649205706776, 358.7598935514485, 358.754811890706, 358.8085791398523, 358.8034974791098, 358.7994540071212, 358.7953285728625, 358.8502432937894, 358.8461451802874])
        expected['rho_n'] = np.array([382312.39357910154, 382307.3383056641, 382302.50177001953, 382298.0784057617, 382331.9341552734, 382326.660144043, 382323.37907714845, 382320.2681396484, 382295.3077270508, 382291.443359375, 382287.9921630859, 382285.14857177733, 382321.67778320314, 382318.22658691404, 382315.0670410156, 382312.17484130856, 382290.0580200195, 382288.38103027345, 382286.97138671874, 382285.6589599609, 382324.3026367187, 382321.75069580076, 382319.3202758789, 382317.2301147461, 382294.9431640625, 382292.002355957, 382289.8635864258, 382287.773425293, 382326.5629272461, 382323.93807373045, 382321.48334960936, 382320.14661865233, 382299.2693115234, 382297.227758789, 382295.2105102539, 382293.4606079101, 382272.2430419922, 382329.86829833983, 382326.90318603517, 382324.2540283203, 382303.2308959961, 382301.5052978516, 382299.9012207031, 382298.56448974606, 382279.36417236325, 382277.29831542965, 382275.9858886719, 382274.7463745117, 382317.0599853516, 382315.1156494141])
        expected['lat_s'] = np.array([-78.73579363355469, -78.72479702899099, -78.71369114140055, -78.70250329154007, -78.69713476285244, -78.68590593185692, -78.68645234699054, -78.68693046023245, -78.69336449843058, -78.69395189469921, -78.69448464945447, -78.69524963064151, -78.70207981981153, -78.7029540840253, -78.70378736710401, -78.70456600866942, -78.71464736788434, -78.71878646252136, -78.72285725526669, -78.726887066877, -78.73702306760532, -78.73385385983042, -78.73061635016384, -78.72731053860554, -78.73023385957032, -78.72701001028207, -78.72213325521469, -78.71721551901226, -78.71856789646793, -78.71370480177887, -78.70878706557647, -78.69949800830526, -78.69646540431378, -78.68723098855591, -78.67792827090636, -78.66854359098676, -78.6623827603554, -78.64999279720101, -78.63749355101987, -78.62498064446041, -78.61871053080236, -78.6087930961275, -78.59883468031762, -78.58875332110267, -78.58477815100574, -78.5747377729258, -78.56177407388111, -78.54871475218806, -78.54172063847795, -78.52867497716323])
        expected['lon_s'] = np.array([69.88347664395732, 69.96303468740963, 70.04221024026842, 70.12122869877629, 70.11741062303025, 70.1973033457519, 70.25665085945013, 70.31607350522926, 70.29194927708076, 70.35245109274874, 70.4127275121741, 70.45689834553657, 70.41625188978583, 70.46128332698372, 70.50612351888485, 70.55054023905745, 70.5152691421835, 70.56614722131093, 70.61675892306071, 70.66730232291881, 70.63205171661238, 70.68047775782777, 70.7286237612872, 70.77655802888233, 70.738097233666, 70.78687844471823, 70.82264814540159, 70.85811048757229, 70.80676795558131, 70.84302942988491, 70.87925675324266, 70.90053962269641, 70.83515422177187, 70.85729086487189, 70.87920211172928, 70.90058743402061, 70.79844878517223, 70.78423516150943, 70.769393160443, 70.75473557448414, 70.65230322750146, 70.58165175072685, 70.51074072676379, 70.43926962728877, 70.28041991776179, 70.20981625231136, 70.15161621039368, 70.09299952693664, 69.94665589377993, 69.88886566321246])
        expected['rho_s'] = np.array([6371.193164443969, 6371.192784690857, 6371.192404937744, 6371.193544197083, 6371.195442962647, 6371.1946834564205, 6371.194303703308, 6371.195063209533, 6371.1984809875485, 6371.197721481323, 6371.197721481323, 6371.197721481323, 6371.193164443969, 6371.193544197083, 6371.193164443969, 6371.192784690857, 6371.193164443969, 6371.193164443969, 6371.193923950195, 6371.193164443969, 6371.198860740661, 6371.197721481323, 6371.197721481323, 6371.196961975097, 6371.194303703308, 6371.195063209533, 6371.195442962647, 6371.1946834564205, 6371.192025184631, 6371.192404937744, 6371.192404937744, 6371.192404937744, 6371.196202468872, 6371.195442962647, 6371.1946834564205, 6371.1946834564205, 6371.195822715759, 6371.195822715759, 6371.195442962647, 6371.1946834564205, 6371.192025184631, 6371.191645431519, 6371.191645431519, 6371.192404937744, 6371.199620246887, 6371.197721481323, 6371.1965822219845, 6371.195442962647, 6371.191265678405, 6371.192025184631])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_0_90000(self):
        """Test for latitude = 0, longitude = 0, rho = 90000"""
        lat = 0
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([18.13162940210195, 18.214875747706074, 18.296394055450165, 18.376211646090937, 18.271989789545714, 18.35315292745298, 18.43762870710769, 18.52019886398398, 18.422520328663623, 18.50637456110387, 18.58832317076569, 18.66919261053863, 18.57434177353467, 18.656488458682418, 18.736743181430057, 18.815263036128528, 18.724804010760877, 18.8067526204227, 18.8868365884411, 18.965049084626912, 18.876516172605207, 18.95447595179172, 19.030543768578156, 19.1048562267479, 19.01705414496739, 19.092432112647643, 19.167325136896835, 19.240326198745933, 19.155898230415403, 19.22997846215337, 19.30220771262624, 19.371192623243374, 19.28744084364068, 19.357600546795055, 19.425984400765216, 19.492619726307865, 19.409933456215683, 19.476862479892645, 19.54212493741211, 19.605707168395767, 19.52494701164953, 19.597442639500045, 19.6679711728696, 19.736594083460716, 19.665566946281757, 19.73494800787074, 19.80391925810953, 19.870875602543165, 19.803290880705887, 19.870943904434867])
        expected['lon_n'] = np.array([358.42742726340896, 358.39827601603133, 358.369698504544, 358.3417220497036, 358.39395933647586, 358.3658189570954, 358.3384708796587, 358.31180582113893, 358.36330544748085, 358.33661306820443, 358.31054906633165, 358.2859330645629, 358.3372687663648, 358.31262544383935, 358.2886378194743, 358.2652512517561, 358.3173792555017, 358.2957685369678, 358.27486815810755, 358.25456883589425, 358.3058772169394, 358.2851680833759, 358.2651146479727, 358.2455803069465, 358.29552265015764, 358.2760975921581, 358.2573555530756, 358.2391872498834, 358.2884465741775, 358.2703602332553, 358.2528749489801, 358.2381763818862, 358.2888563855277, 358.2742397807039, 358.2601149495003, 358.24650921267363, 358.2958778199945, 358.2816710265209, 358.2679560066675, 358.254678118921, 358.3033637073248, 358.29055027244186, 358.2782832526925, 358.26661728959016, 358.3150569911839, 358.30360959413497, 358.29371948021674, 358.2844031021889, 358.33330725664615, 358.3242914069417])
        expected['rho_n'] = np.array([175750.15377197266, 177148.24067993162, 178530.93087768555, 179897.94486694335, 178428.50083007812, 179815.55363159179, 181283.32067871094, 182734.2813720703, 181365.88204345704, 182838.02384643554, 184293.80892333985, 185737.32037963867, 184386.4322265625, 185851.294921875, 187301.06470947264, 188739.08341674804, 187467.0745422363, 188989.27084350586, 190498.1362915039, 191993.41569213866, 190730.25354614257, 192189.1860168457, 193632.9283630371, 195065.08975830078, 193758.36233520508, 195208.41162109375, 196650.97521362305, 198078.53096313475, 196788.13596191406, 198235.9249572754, 199669.45953979492, 201056.31790771484, 199746.3094177246, 201154.0207885742, 202547.5871154785, 203926.69244384766, 202592.24608154295, 203958.16638183594, 205310.88950195312, 206649.63770751952, 205325.61784667967, 206840.88745117188, 208341.6960571289, 209828.52974853516, 208668.56323242188, 210174.59724121093, 211699.9530883789, 213207.83421630858, 212092.66033935547, 213620.37369384765])
        expected['lat_s'] = np.array([-70.5732531225259, -70.56082217823649, -70.54829561129867, -70.5357144028475, -70.52660293049473, -70.5140353824219, -70.50980066513648, -70.50555228747274, -70.50470534401566, -70.50053892862195, -70.49637251322824, -70.49429613572056, -70.49555289052785, -70.49357213566853, -70.49160504118757, -70.4895969655716, -70.49563485279788, -70.4986811171677, -70.5017137211592, -70.50471900439399, -70.51072957086362, -70.50764232535877, -70.50451409871889, -70.50138587207903, -70.50120828716061, -70.4982030039258, -70.4907990788655, -70.48338149342689, -70.47887356857467, -70.4715789265411, -70.46422964299416, -70.45511817064136, -70.44879341546994, -70.43980488652221, -70.43078903681777, -70.42169122484333, -70.40744345023467, -70.39066850563313, -70.37381159876153, -70.35687272961991, -70.34247469084951, -70.32011265150689, -70.29769597065092, -70.2751700067682, -70.25488434493326, -70.23235838105057, -70.21129407765025, -70.1901478119799, -70.17111890495224, -70.15005460155191])
        expected['lon_s'] = np.array([41.32556779954191, 41.34173144221274, 41.357816537708125, 41.373942614338524, 41.349999386203045, 41.36636451945439, 41.348882650273744, 41.33127442259345, 41.27301290897324, 41.25528515298248, 41.2374242083029, 41.19917173385624, 41.11985274702488, 41.081275838592646, 41.04245645844488, 41.003343380162804, 40.92734386526807, 40.89276603259488, 40.85789791688197, 40.82272927284557, 40.745274927657626, 40.741760795329654, 40.73821251205583, 40.73458226651197, 40.68857411226276, 40.685070225218546, 40.676665677444845, 40.66822356363071, 40.617174729774035, 40.60886238955413, 40.60057395499632, 40.59225136949266, 40.540758573339936, 40.532442818025444, 40.524130477805535, 40.5156883639914, 40.45196611412977, 40.43163605606522, 40.41124794139274, 40.39078810973396, 40.32692242589976, 40.256236798179295, 40.18549652894548, 40.114725523860386, 40.00030619488434, 39.92962056716387, 39.87097997804473, 39.81228816250681, 39.70988996646998, 39.65125620754])
        expected['rho_s'] = np.array([6371.189746665955, 6371.195822715759, 6371.194303703308, 6371.188227653503, 6371.188987159729, 6371.193923950195, 6371.195442962647, 6371.188227653503, 6371.188987159729, 6371.195442962647, 6371.193544197083, 6371.187468147277, 6371.188987159729, 6371.196961975097, 6371.192784690857, 6371.187468147277, 6371.188607406616, 6371.1965822219845, 6371.193923950195, 6371.187468147277, 6371.187468147277, 6371.1946834564205, 6371.196961975097, 6371.189366912841, 6371.185949134827, 6371.189366912841, 6371.196202468872, 6371.193544197083, 6371.185949134827, 6371.186328887939, 6371.191645431519, 6371.2, 6371.188607406616, 6371.185189628601, 6371.186328887939, 6371.189746665955, 6371.195442962647, 6371.188987159729, 6371.185189628601, 6371.185569381713, 6371.1946834564205, 6371.193164443969, 6371.185569381713, 6371.185569381713, 6371.198860740661, 6371.189746665955, 6371.184809875488, 6371.186708641052, 6371.198101234436, 6371.188607406616])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_0_90000(self):
        """Test for latitude = 30, longitude = 0, rho = 90000"""
        lat = 30
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([71.4000920314711, 71.42573597670992, 71.45135260119204, 71.47694702755938, 71.50546989753335, 71.53091576728623, 71.56047170337217, 71.58999519605955, 71.62244371725899, 71.6517759646496, 71.68103991014853, 71.7193094600681, 71.7605398969929, 71.7984815978323, 71.83637890244212, 71.87414643345771, 71.91057524739597, 71.9436658063774, 71.97667269554148, 72.0095583488478, 72.04542196463224, 72.07327889116215, 72.10102311957073, 72.1287041687295, 72.15942123697434, 72.18674882384356, 72.21704754300174, 72.24723356403861, 72.28052902869513, 72.3104196440504, 72.34018902354791, 72.3752210638009, 72.41333010427498, 72.44782597467812, 72.48205717525094, 72.51611591354721, 72.55186170856847, 72.58387651275561, 72.61569494900412, 72.64725896070604, 72.68208780283123, 72.71998681498833, 72.75759042146385, 72.79496009396027, 72.83570729500127, 72.87250323160742, 72.91078131926346, 72.948746925765, 72.99016690043923, 73.02746656349667])
        expected['lon_n'] = np.array([336.8606131854734, 336.905309943402, 336.9500613428439, 336.9948673837992, 337.01997515918816, 337.0649997661969, 337.12775554429095, 337.1905659638984, 337.2335961556695, 337.2967071036004, 337.3599546553147, 337.48328055096846, 337.58688086029923, 337.71129958622015, 337.8361008027347, 337.96142111362605, 338.0838181035527, 338.2272793968796, 338.37131442609683, 338.51597783271774, 338.64072440771884, 338.78098917251424, 338.92171839017317, 339.0628300984256, 339.1837517674915, 339.3259289852545, 339.4713846938191, 339.61730485524726, 339.74292569446214, 339.88988404464413, 340.0373068476897, 340.2357921449703, 340.4141420445776, 340.61415730423226, 340.81496486583063, 341.0163188425626, 341.1975374216214, 341.4004213607275, 341.6037697526971, 341.807691880557, 341.9911507616635, 342.207831682893, 342.4253322268229, 342.6435977519398, 342.84148199257345, 343.06138676309115, 343.3089401393691, 343.55747706288753, 343.78574198494937, 344.03630064446213])
        expected['rho_n'] = np.array([6371.197721481323, 6371.196961975097, 6371.197341728211, 6371.197341728211, 6371.198101234436, 6371.196961975097, 6371.197341728211, 6371.196961975097, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.198101234436, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.199240493775, 6371.197721481323, 6371.197341728211, 6371.197721481323, 6371.197721481323, 6371.197341728211, 6371.197341728211, 6371.197721481323, 6371.198101234436, 6371.197721481323, 6371.197721481323, 6371.197341728211, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.2])
        expected['lat_s'] = np.array([24.06170892253374, 24.077438848192287, 24.092840924770655, 24.107969793782246, 24.06526062090215, 24.081031527695714, 24.096556547679143, 24.111781039339107, 24.069065036269848, 24.084911075144277, 24.100456585695227, 24.115387379220877, 24.072234244044736, 24.087793414974016, 24.103031567012323, 24.117989681294645, 24.075335149927923, 24.091488547314995, 24.10732775600026, 24.122859606172867, 24.0801231125361, 24.095613981573692, 24.110783831720283, 24.125659983732575, 24.082151678719597, 24.097649377946354, 24.11225232239184, 24.12653424794634, 24.082342924016345, 24.097260057163666, 24.11186300160915, 24.12615858754198, 24.081864810774448, 24.096795604300098, 24.111398548745584, 24.12570779505677, 24.080997376749863, 24.09559349100617, 24.109868586371505, 24.123836323224182, 24.07908492378226, 24.092854585149, 24.10631005781393, 24.1194581719662, 24.07380518755383, 24.087588509298925, 24.10096201969381, 24.114028171576038, 24.06823858338028, 24.081939942855314])
        expected['lon_s'] = np.array([1.128565816939547, 1.1142530488120794, 1.099949458751309, 1.0856619836681118, 1.1233018755235826, 1.1089786486689486, 1.0940309931137873, 1.0790815232896276, 1.116082472292585, 1.1011136858396788, 1.0861856670783814, 1.0707903139674668, 1.1073341739045621, 1.0919176898959029, 1.0765345030594475, 1.0611472607981725, 1.0967747014477225, 1.0804752020454276, 1.0642162568913296, 1.0479431177503626, 1.0835459056858772, 1.067705096175031, 1.0518758126084091, 1.0360629641844774, 1.07205742078021, 1.0562217339112412, 1.0403552044693016, 1.024470639059085, 1.0603814258374846, 1.0444931251675658, 1.0286179512674578, 1.0121377720172748, 1.04734665013675, 1.0308111890429723, 1.0142902421011808, 0.9978028057750048, 1.0333002593869132, 1.0170958490244695, 1.0009325265187516, 0.9847731527161476, 1.0202029447665906, 1.0033409152531216, 0.9865257365684906, 0.9697136528133273, 1.004407598702407, 0.9875750243797338, 0.9703104405920572, 0.9530483114036136, 0.9872190007692468, 0.9699592194583304])
        expected['rho_s'] = np.array([382301.0921264648, 382304.13015136716, 382307.2896972656, 382310.32772216795, 382300.31439208984, 382303.3767211914, 382306.53626708983, 382309.64720458986, 382299.6581787109, 382302.8420288086, 382306.00157470704, 382309.0395996094, 382299.0748779297, 382302.18581542966, 382305.442578125, 382308.50490722654, 382298.44296874997, 382301.89416503906, 382305.1752319336, 382308.35908203124, 382298.46727294923, 382301.6997314453, 382304.8349731445, 382307.99451904296, 382297.98118896486, 382301.1650390625, 382304.27597656247, 382307.3383056641, 382297.1791503906, 382300.31439208984, 382303.4739379883, 382306.46335449215, 382296.47432861326, 382299.6338745117, 382302.81772460934, 382305.75853271486, 382295.7452026367, 382298.92905273434, 382301.89416503906, 382304.9078857422, 382294.77303466795, 382297.73814697267, 382300.60604248045, 382303.3767211914, 382293.2418701172, 382296.20698242186, 382299.0991821289, 382301.6997314453, 382291.6377929687, 382294.50568847655])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_0_90000(self):
        """Test for latitude = 60, longitude = 0, rho = 90000"""
        lat = 60
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([80.52434281443155, 80.54199714588874, 80.5596822131972, 80.57737240314754, 80.59574561201487, 80.61333164158036, 80.63556390732876, 80.65782349383383, 80.68075414642485, 80.70290103480862, 80.72497791375339, 80.76256615230326, 80.80083911731742, 80.83822928038137, 80.8755656557056, 80.91282604517531, 80.95157712343138, 80.98935660727801, 81.0271002326315, 81.0648327589276, 81.1034608937786, 81.13883615103738, 81.17418921018137, 81.20950470328492, 81.24587716816254, 81.28099885464839, 81.3196346734622, 81.35818340736411, 81.397931693239, 81.43628918184415, 81.47459971289875, 81.5218620606343, 81.5702735876977, 81.6170791665325, 81.66366447176658, 81.71004657887285, 81.75747199737484, 81.80307290159477, 81.84843816428835, 81.89354302601983, 81.94020346333477, 81.99164076420048, 82.04278778202648, 82.09373074795101, 82.14635308663773, 82.19667365157416, 82.25103000453642, 82.30501667350984, 82.36091664922452, 82.41410162474412])
        expected['lon_n'] = np.array([317.3189775172034, 317.34903034955147, 317.3788099743327, 317.4085349576006, 317.3792471064396, 317.40940922181437, 317.45885979140525, 317.5082283987261, 317.49803775648445, 317.54776153364213, 317.59797708442005, 317.71687701749164, 317.77599913494726, 317.89697544552655, 318.01888066183295, 318.14185138764987, 318.22987886567296, 318.38161834827395, 318.5346145856823, 318.6889768609245, 318.7812117354764, 318.9427320489689, 319.1053998342419, 319.2692697328088, 319.369591551338, 319.53663065767984, 319.71722085933493, 319.899614230931, 320.01687491860173, 320.2024648187293, 320.38939343593427, 320.6610163988473, 320.86805309296875, 321.14549537705466, 321.4259429443754, 321.70920454963414, 321.92700562188753, 322.2165783219394, 322.5088557769026, 322.8042477981273, 323.03177505975873, 323.3445157614753, 323.6611360106404, 323.98108939212045, 324.23254963660366, 324.56039871676427, 324.94510229157606, 325.3350241309138, 325.6557424935809, 326.05645603180716])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([36.22117026792187, 36.2327337781867, 36.24401725069556, 36.255030930732204, 36.21303551262038, 36.2246468342094, 36.23526777836877, 36.24559160929925, 36.20288585151374, 36.21384489003703, 36.2244658341964, 36.23445498585754, 36.19131551105974, 36.20191937974619, 36.21221588991998, 36.22223919252698, 36.178587453541425, 36.188655152378026, 36.198439398364066, 36.20799141791833, 36.16201741461497, 36.17454398155277, 36.18434188791716, 36.19389390747142, 36.147991621154354, 36.15818226339601, 36.17004630198432, 36.17915777433712, 36.13295154460198, 36.14272554530427, 36.15221609315601, 36.16351664113779, 36.117091845349215, 36.126660940376404, 36.13593975236386, 36.1449521869737, 36.10124922156937, 36.1110129769879, 36.12052060031256, 36.12975160097583, 36.0861647487874, 36.0948869003575, 36.103305108509545, 36.11143986381103, 36.06449597364553, 36.07568382350601, 36.08393810711797, 36.091854296366016, 36.04480453826838, 36.05342423700094])
        expected['lon_s'] = np.array([2.8552956957267153, 2.834491152958268, 2.8136018731554304, 2.792700213634722, 2.8450292810739515, 2.824208089719402, 2.801594827481371, 2.7789655569874725, 2.8295375582627162, 2.806959514187593, 2.7843795491217653, 2.76068135402276, 2.81016842243703, 2.7865310587103194, 2.762900738616191, 2.739183760496968, 2.7862478193031754, 2.7602510521050494, 2.734232727122355, 2.7081382028417327, 2.755095966828982, 2.729713489769334, 2.70426740657304, 2.678755582805984, 2.726225397537573, 2.700788492407976, 2.6747469020934216, 2.648580233939691, 2.6953924295202545, 2.669322878118785, 2.6432085036008877, 2.615420732736337, 2.6604777831436186, 2.6326684544945, 2.60480427088861, 2.5769262134609683, 2.6227725776042097, 2.595783511985727, 2.568732974664715, 2.54164657885056, 2.587412901714465, 2.5580776661160836, 2.5287518220278105, 2.4993591701517186, 2.542740689778231, 2.5134280791814754, 2.4827793128283355, 2.4520639521307883, 2.494123189822669, 2.463498329131624])
        expected['rho_s'] = np.array([382299.6095703125, 382310.9353271484, 382321.77499999997, 382332.66328125, 382295.550769043, 382306.9251342773, 382317.2787231445, 382327.55939941405, 382289.9121948242, 382300.6789550781, 382311.2755859375, 382321.1430908203, 382283.3257568359, 382293.84947509767, 382304.00863037107, 382313.97335205076, 382275.5241088867, 382285.44022216793, 382295.113293457, 382304.61623535154, 382327.82674560545, 382276.2046264648, 382285.7075683594, 382295.33203125, 382318.6154541015, 382328.6044799805, 382276.49627685547, 382285.6589599609, 382308.9180786133, 382318.6397583008, 382328.21561279293, 382275.3782836914, 382298.3214477539, 382307.6785644531, 382316.9384643555, 382325.9553222656, 382287.5789916992, 382297.2034545898, 382306.68209228513, 382315.96629638673, 382277.39553222654, 382286.63112792966, 382295.2834228515, 382303.86280517577, 382326.7330566406, 382274.1630737305, 382282.81536865234, 382291.12740478513, 382313.8761352539, 382322.8200805664])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_0_90000(self):
        """Test for latitude = 90, longitude = 0, rho = 90000"""
        lat = 90
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([82.34980649276585, 82.35883258775402, 82.36779123462414, 82.37671829186934, 82.3705207489712, 82.37948793357778, 82.3916328636957, 82.40375218060426, 82.4006760341568, 82.41277400672419, 82.42483270570385, 82.44572881569586, 82.45134237741996, 82.47210871381773, 82.49274869171587, 82.51323328281039, 82.52234731648412, 82.54664912955103, 82.57078872562515, 82.5948292839563, 82.60333969966213, 82.62841930052078, 82.65333070797111, 82.6780534314456, 82.68715295096736, 82.71162466498987, 82.73989311041726, 82.76792506054464, 82.78009987274018, 82.80777665303073, 82.83521352292667, 82.87523373570788, 82.89924099686684, 82.93864563884911, 82.97768273127667, 83.01635825056505, 83.03864558471362, 83.0763614624236, 83.11364874576317, 83.15056378379298, 83.17138348104365, 83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544])
        expected['lon_n'] = np.array([239.34851591481595, 239.28043258916927, 239.21185748990234, 239.14295454155524, 239.06115619605518, 238.99288162511172, 238.89602954268094, 238.79876764889997, 238.68896552780285, 238.59211344537206, 238.49471494780767, 238.287869498983, 238.0690302379758, 237.8613924872074, 237.65249798163174, 237.44212815519535, 237.18687032553365, 236.94062834557633, 236.6923919503814, 236.44207917767875, 236.18225878165148, 235.91757529093513, 235.65048757363095, 235.38099562973895, 235.10336210035626, 234.83108343928288, 234.52913443645494, 234.2242347919056, 233.91245031667285, 233.6039443322417, 233.29270627214257, 232.91871243394934, 232.5390358783667, 232.15840309630022, 231.77345363467828, 231.3842421350143, 231.00396452278466, 230.62024449521334, 230.2324536748967, 229.84072866561812, 229.44985059979663, 228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([48.517652038249615, 48.525165246336634, 48.53232328458681, 48.53916030394599, 48.49639990464713, 48.50390969763955, 48.5095480187994, 48.51500192485165, 48.47052714807111, 48.47649331831112, 48.482288733821875, 48.487216715308044, 48.44232871208272, 48.44792263701296, 48.45331848645727, 48.45830452455138, 48.411500653263865, 48.41536654033409, 48.41881578586495, 48.422087446477384, 48.37538944312201, 48.380078367987224, 48.38436431169141, 48.38848974595009, 48.342594289822195, 48.347385667524954, 48.3516818565129, 48.355674102082766, 48.30976498557653, 48.31448806138759, 48.31890036359142, 48.322499873284016, 48.27616728504923, 48.28038151176713, 48.2843976629991, 48.288202078366794, 48.24280181095372, 48.24802690566878, 48.252961717344114, 48.25768479315518, 48.21248601632262, 48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855])
        expected['lon_s'] = np.array([3.5521261090034026, 3.5211221728134223, 3.489967545574878, 3.458780048050954, 3.5427674691800157, 3.511755422140396, 3.4786102216456167, 3.445244320663281, 3.5271468265481967, 3.494132039977882, 3.4609479927821982, 3.4263814726098216, 3.5069271187274316, 3.472532847388186, 3.438059388543251, 3.4035411065818875, 3.4810074046008794, 3.4435687898825122, 3.4061756386108084, 3.368535746755337, 3.4458618124529328, 3.409107497312036, 3.372313908583412, 3.3353341971999053, 3.413228449259081, 3.3763868357628555, 3.3387919804672244, 3.300950171144415, 3.37792661653387, 3.3403176739730758, 3.302593898856861, 3.2626253393812386, 3.3372528400264443, 3.297408931503175, 3.2573996043359443, 3.217312797210316, 3.292921710661888, 3.2540583611713387, 3.215103444457229, 3.1759666739564674, 3.251479324429424, 3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065])
        expected['rho_s'] = np.array([377304.9994140625, 377315.9606079101, 377326.75167236326, 377336.95943603513, 377284.7540161133, 377295.9339477539, 377304.2216796875, 377312.72814941406, 377258.0923095703, 377266.86612548825, 377275.5427246094, 377282.6395507812, 377226.9586303711, 377234.93040771486, 377242.73205566406, 377249.9260986328, 377191.3529785156, 377196.60268554685, 377201.4635253906, 377206.25145263673, 377147.7755493164, 377154.1432495117, 377159.87904052733, 377165.37178955076, 377107.7222290039, 377114.3572753906, 377119.5583740234, 377124.61364746094, 377066.623828125, 377072.55405273434, 377078.1683227539, 377082.00838623045, 377022.75474853517, 377027.34824218747, 377031.45565185545, 377035.73319091793, 376977.743371582, 376983.6006835937, 376989.2878662109, 376994.8292236328, 376936.8394042969, 376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_90_90000(self):
        """Test for latitude = -90, longitude = 90, rho = 90000"""
        lat = -90
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.90688287510915, -72.88874189267358, -72.8686747968921, -72.85101192769844, -72.88099645815478, -72.86271887193584, -72.84415441777173, -72.82800785057384, -72.85974090945771, -72.84296596485618, -72.82429222766535, -72.81035864175851, -72.84452324798693, -72.82994762429811, -72.81564520817611, -72.79945765984317, -72.83621773795619, -72.82401902009855, -72.81206618905102, -72.8003865655703, -72.83520686995902, -72.82410098236858, -72.81326830234494, -72.80268150913142, -72.84099887037519, -72.82978369975805, -72.81889637822104, -72.81034498138018, -72.85094362580674, -72.8417501911839, -72.83282996412785, -72.82743411468354, -72.86945343845747, -72.86344287198787, -72.85766453195004, -72.85213207872232, -72.89609117622052, -72.88967079840072, -72.88348264701267, -72.87751306167809, -72.91960068734372, -72.9163904984338, -72.91345351709069, -72.91068046028767, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99974612706464, -72.99780635334037])
        expected['lon_n'] = np.array([28.70189947740304, 28.647503850853056, 28.593606828112485, 28.539757616696104, 28.6732775696861, 28.61870606576499, 28.547506466309486, 28.476245395151448, 28.59221176197451, 28.52016351151463, 28.44859166674936, 28.365037962632595, 28.46848288515973, 28.384010520599595, 28.29983526926836, 28.216023725510425, 28.29418841037205, 28.18469535533487, 28.07529963049336, 27.965905613199144, 28.043076505536533, 27.934650667557165, 27.826358018266614, 27.718171236908194, 27.795950016081715, 27.686730168611334, 27.568438122375884, 27.449937755370748, 27.516998260189226, 27.39749556292339, 27.278083365664056, 27.13910609152722, 27.185141566533108, 27.04481191494061, 26.90453348976689, 26.76435068724155, 26.817269285383514, 26.684396492817378, 26.551798615365335, 26.419296360561674, 26.47081647747108, 26.311299116952725, 26.151628077178042, 25.992218292139118, 26.01590538818072, 25.855054433226922, 25.67930512814603, 25.503711209868758, 25.510626776403395, 25.33370097123797])
        expected['rho_n'] = np.array([257214.93735351562, 257202.37208251952, 257252.87620849608, 257240.5539794922, 257196.92794189454, 257184.0710205078, 257235.5230102539, 257224.14864501954, 257181.5919921875, 257169.6829345703, 257220.8918823242, 257210.87855224608, 257169.9259765625, 257159.45086669922, 257149.097277832, 257201.91030273438, 257162.5375, 257153.52064208983, 257144.67391357422, 257135.87579345703, 257159.49947509766, 257151.30895996094, 257143.19135742186, 257135.24388427733, 257096.86755371094, 257088.45830078123, 257144.67391357422, 257137.96595458983, 257101.07218017578, 257093.92674560545, 257087.02435302734, 257082.57668457032, 257111.13411865235, 257106.24897460936, 257101.4853515625, 257096.8432495117, 257062.3555908203, 257057.17879638672, 257052.19643554688, 257047.38420410155, 257075.91733398437, 257072.95222167968, 257070.08432617187, 257067.4108642578, 257035.20780029296, 257032.02395019532, 257030.27404785156, 257028.6456665039, 257060.654296875, 257058.56413574217])
        expected['lat_s'] = np.array([-68.33948073560825, -68.30527514824485, -68.2709602778547, -68.23646782254613, -68.2116469151023, -68.17694955411864, -68.14394608004912, -68.1107923418179, -68.08739211372142, -68.05406079057175, -68.02046992023364, -67.98901006891657, -67.96737202962595, -67.93569361225545, -67.90381028920982, -67.8717903623808, -67.85184621000437, -67.82172507576462, -67.7914126962281, -67.7609363921515, -67.74050046615483, -67.71996891750976, -67.69928710470296, -67.67850966924777, -67.66796385716927, -67.64703615755238, -67.63134038283968, -67.61556264585695, -67.61001653225088, -67.59411585186311, -67.57810588844859, -67.56609841588772, -67.5644181893519, -67.55228777338596, -67.54008905552831, -67.5277673942656, -67.52944762080142, -67.52062301639376, -67.51167546858105, -67.50270060001165, -67.50417592087237, -67.50285086417338, -67.50145750558272, -67.5000504866137, -67.50916195896647, -67.50771395886244, -67.50597909081327, -67.50412127935903, -67.51287758187496, -67.5110470911774])
        expected['lon_s'] = np.array([152.83727158554274, 152.8151690933886, 152.79298463896438, 152.7708821468102, 152.8006754319698, 152.77868222284238, 152.7802941474865, 152.7817967891039, 152.83522252879175, 152.83653392511238, 152.83813218937817, 152.82361120720273, 152.86101332309772, 152.84641037865222, 152.83184841534177, 152.81708154635618, 152.87289785225354, 152.8767090978104, 152.8804657018539, 152.88393543795226, 152.93958781930954, 152.92906932798772, 152.91850985553089, 152.9078001189123, 152.94912276339087, 152.93849498904234, 152.9272524976685, 152.91583242137622, 152.95696382055806, 152.94569400842752, 152.93436955478364, 152.92036766698513, 152.9587533301206, 152.94481974421376, 152.93064027149683, 152.91652910067157, 152.9438771781083, 152.9184825347742, 152.89316985371013, 152.86762494621428, 152.89482275948927, 152.87409996554746, 152.85318592630887, 152.83205332101684, 152.8639776251975, 152.84277671801377, 152.81351618760945, 152.7841873553134, 152.80807935703012, 152.77857293981566])
        expected['rho_s'] = np.array([6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_90_90000(self):
        """Test for latitude = -60, longitude = 90, rho = 90000"""
        lat = -60
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-65.6771139782723, -65.68237322393321, -65.68778273375585, -65.69346545114533, -65.72284892495472, -65.72803986872393, -65.7381348883172, -65.74844847396392, -65.7829136085158, -65.79274908092063, -65.8028031193789, -65.81461934664301, -65.85101059454084, -65.8623623689414, -65.8739327093954, -65.88570795552451, -65.92929822280752, -65.94745286562141, -65.96582607448875, -65.98445883054455, -66.02851355069112, -66.04478306129408, -66.06121649643714, -66.07788215801199, -66.12046155729783, -66.1366627660091, -66.15463982390457, -66.1728081270968, -66.21725899821524, -66.23497650892227, -66.25289892530438, -66.27511070048527, -66.32409681721259, -66.34585779990826, -66.367823688279, -66.39009010497327, -66.43699984419291, -66.4563429399224, -66.47586362057027, -66.49558920689327, -66.54290875746312, -66.56767502339358, -66.59259155348579, -66.6177403100098, -66.67090650250915, -66.69552250427787, -66.72336235533487, -66.75142077244527, -66.80792009725963, -66.83543209923644])
        expected['lon_n'] = np.array([51.260412626594785, 51.175871960142956, 51.086027651800606, 51.00189338160439, 51.08624621785405, 50.99537055094702, 50.89699533633133, 50.79308766848793, 50.863742560357124, 50.75852349619308, 50.65899056451307, 50.55135702847742, 50.61861048613995, 50.509556270756946, 50.4061574520067, 50.29727057625836, 50.34546439104197, 50.21680070255186, 50.08806871217004, 49.964688175002934, 50.005901536454786, 49.89035522626568, 49.76925938737594, 49.648112322067426, 49.70322511848026, 49.58047637381137, 49.462457535142725, 49.338851601733005, 49.38638971835629, 49.26667016258431, 49.14146938000339, 49.008133842120984, 49.05185729809282, 48.91666736385076, 48.78141937300075, 48.651758476891814, 48.697093857507774, 48.57421533924466, 48.44574289605131, 48.31727728304712, 48.36703179605608, 48.22212933281426, 48.07720637900493, 47.932088764804256, 47.96552254079146, 47.81861541702825, 47.66339936813971, 47.50795109281938, 47.531566471874704, 47.37415793226259])
        expected['rho_n'] = np.array([256597.22182617188, 256571.67811279296, 256608.86353759764, 256583.05247802733, 256543.92271728514, 256580.6463623047, 256550.87371826172, 256584.02464599608, 256540.98190917968, 256573.2821899414, 256542.5616821289, 256573.35510253906, 256528.61107177733, 256559.015625, 256526.6181274414, 256557.04698486326, 256506.44564208985, 256530.6526245117, 256554.83530273437, 256515.92427978516, 256527.42016601562, 256490.74512939452, 256516.8964477539, 256542.92624511718, 256494.14771728514, 256519.69143066407, 256482.11713867186, 256507.1747680664, 256520.54207763672, 256482.1900512695, 256506.56716308594, 256528.5624633789, 256476.13830566406, 256497.33156738282, 256518.4276123047, 256476.50286865234, 256489.1653564453, 256449.5009033203, 256472.5898925781, 256495.5087524414, 256444.73728027343, 256462.52795410156, 256480.245715332, 256497.8662597656, 256441.77216796874, 256458.63928222656, 256472.8086303711, 256486.85645751952, 256427.50560302733, 256440.75139160155])
        expected['lat_s'] = np.array([-66.53366068132692, -66.50393569805902, -66.47401946949438, -66.444034939038, -66.4233531262312, -66.39325931274814, -66.36548776358288, -66.33767523328257, -66.31912443949682, -66.2911616450348, -66.26307590716769, -66.23534533913744, -66.21712239443187, -66.18936450564493, -66.16145635269626, -66.13335695445082, -66.11897257605878, -66.09490298942364, -66.07066947824845, -66.04627204253316, -66.03164177733098, -66.01641045548186, -66.00108351098436, -65.98570192497348, -65.98015581136744, -65.96476056497823, -65.95467920576328, -65.94455686541332, -65.94433829935988, -65.93422961938828, -65.92401165638992, -65.9178918068936, -65.92171671282881, -65.91561052371082, -65.90936773080941, -65.90312493790802, -65.90947701383615, -65.9058433531977, -65.9020867491542, -65.89833014511069, -65.90465490028211, -65.90827490054221, -65.9118539196673, -65.91535097652235, -65.9290523359974, -65.93265867587917, -65.93575958176234, -65.93894244991557, -65.95228863955379, -65.95549882846367])
        expected['lon_s'] = np.array([145.33638515979143, 145.3421498394509, 145.34810576440717, 145.3539114252017, 145.39802078686162, 145.404222598628, 145.43422078946267, 145.46410969727066, 145.53250721161916, 145.5628195911557, 145.59305000842215, 145.6062732546553, 145.65780020175387, 145.6711190706354, 145.68450624140863, 145.69789341218186, 145.76925522863016, 145.80242262723974, 145.83574029001105, 145.86918089618743, 145.94115742966102, 145.95443531740753, 145.96775418628906, 145.9810457344139, 146.0331873985378, 146.0469843806612, 146.0580356267383, 146.06908687281538, 146.1190975179182, 146.13055857534547, 146.14225185920455, 146.14864491626767, 146.1942842403017, 146.2011007690934, 146.20806756204678, 146.2151709587836, 146.24896673479682, 146.24374847027093, 146.23870779066345, 146.23365345067765, 146.26815956636457, 146.26688915117893, 146.26568703788502, 146.2645532264828, 146.30313013491502, 146.30233783297132, 146.29436017202073, 146.28617760539507, 146.31797896617064, 146.3103564750569])
        expected['rho_s'] = np.array([6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.199240493775, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_90_90000(self):
        """Test for latitude = -30, longitude = 90, rho = 90000"""
        lat = -30
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-55.84571236619928, -55.93393108951915, -56.02139849203033, -56.108237517137894, -56.206455637402655, -56.29116364348917, -56.3893681033756, -56.48432139321713, -56.594150835070934, -56.68691846437804, -56.78177613157118, -56.86703055279128, -56.968376899695926, -57.05204671702859, -57.137738270355555, -57.22062944612284, -57.33607330347439, -57.43545255589808, -57.534367355458215, -57.630536418971985, -57.746799899023955, -57.82765567841855, -57.9105741749425, -57.99102014298691, -58.09024913124887, -58.17173328804711, -58.24668778399882, -58.32161495919388, -58.418234815192875, -58.49211014125572, -58.56590350504854, -58.642319661482645, -58.73689046073065, -58.81026035279493, -58.88573394312357, -58.95913115594453, -59.046257048997205, -59.111567317840866, -59.1789539641922, -59.24434619530592, -59.33136280533188, -59.40422726339759, -59.47509730622562, -59.54797542466966, -59.64123482759703, -59.71132622885969, -59.78584359270454, -59.860456579197745, -59.95617485022632, -60.029886251749076])
        expected['lon_n'] = np.array([59.66259032465348, 59.52550842801133, 59.38865875780095, 59.25199350269816, 59.26092055994337, 59.12355179535607, 58.96556951985368, 58.813201659849035, 58.802273357177015, 58.64892878012106, 58.48967608943304, 58.3386742672624, 58.330382417610004, 58.17806236892955, 58.020148395318856, 57.867794195692554, 57.84056223147171, 57.661918633730025, 57.4829881680432, 57.30924864612559, 57.275528002193234, 57.12054417973648, 56.95977518705271, 56.80442253438078, 56.79600091113415, 56.6332033522669, 56.48317483205298, 56.33281504766432, 56.325226707496434, 56.173039847504796, 56.020631006365136, 55.861224636420786, 55.85702748517582, 55.70127868153202, 55.53952614160777, 55.38313188508741, 55.388684828882624, 55.24145668622842, 55.088381901644674, 54.9407371174511, 54.9451084385199, 54.777437539679845, 54.61512150914908, 54.446730025351584, 54.4369321189872, 54.27210599393646, 54.09547388690534, 53.91833976097023, 53.90098083519464, 53.72152444494173])
        expected['rho_n'] = np.array([255311.0192993164, 255316.4634399414, 255321.73745117188, 255326.91424560547, 255269.70216064452, 255274.29565429688, 255323.29291992186, 255311.0192993164, 255297.09299316406, 255283.50694580076, 255330.29252929686, 255320.95971679688, 255310.36308593748, 255299.15885009765, 255347.86446533204, 255334.64298095703, 255298.59985351562, 255323.3172241211, 255346.5520385742, 255307.15493164063, 255328.54262695313, 255309.77978515625, 255351.14553222657, 255330.29252929686, 255309.41522216797, 255348.30194091797, 255333.1604248047, 255317.28978271483, 255362.42268066405, 255344.7292236328, 255326.40385742186, 255368.54733886718, 255349.15258789062, 255328.22667236326, 255368.13416748046, 255345.38543701172, 255336.07692871094, 255324.96990966797, 255374.86643066406, 255362.5928100586, 255350.34349365233, 255386.36231689452, 255359.74921875, 255394.16396484376, 255367.1619995117, 255338.1184814453, 255365.5822265625, 255392.14671630858, 255357.3187988281, 255382.42503662108])
        expected['lat_s'] = np.array([-64.76846659222883, -64.74839949644732, -64.72826409877413, -64.70814236147928, -64.69501473789452, -64.67500228362638, -64.65867813151004, -64.64238130015039, -64.63291465796075, -64.6167544303845, -64.60063518394327, -64.58345042799152, -64.5733007668849, -64.55632091660823, -64.53934106633159, -64.52236121605492, -64.5189324610916, -64.50860521506652, -64.4982506482848, -64.48799170415143, -64.4848771378899, -64.48236362827532, -64.4798637790391, -64.47735026942453, -64.48210408108687, -64.4797954771474, -64.48243193016702, -64.4850137416733, -64.49499947823986, -64.49785449731291, -64.50057291260259, -64.50703427155742, -64.52099517822091, -64.52759314095917, -64.53415012256238, -64.54066612303058, -64.5560340486631, -64.56395706810031, -64.57168884224077, -64.57942061638121, -64.5950071080672, -64.6106755620232, -64.62618009143912, -64.64164363972003, -64.66528975462663, -64.68080794442088, -64.6958616813516, -64.7107788144989, -64.73423368410872, -64.74920545876938])
        expected['lon_s'] = np.array([143.77774965157286, 143.80355410625717, 143.82948150434652, 143.85565478924602, 143.9124683027622, 143.9394748707404, 143.99190340280944, 144.04444121790516, 144.1282749597779, 144.18194658627587, 144.23590508071896, 144.27050681905425, 144.33628154076146, 144.37166192066215, 144.40708328169782, 144.44253196349018, 144.533359819073, 144.59364306868756, 144.65411756359885, 144.7146057188885, 144.8077148576541, 144.84429735084868, 144.8808935044216, 144.91763992215627, 144.98755373850054, 145.02510611855726, 145.0580549511134, 145.09089450064283, 145.15787133564396, 145.19140756446873, 145.2249437932935, 145.25181375748832, 145.3136133090986, 145.3408521035086, 145.36796795451357, 145.3947696168167, 145.44390599770577, 145.456692111832, 145.46932796179655, 145.4815949815459, 145.53178321156716, 145.54858547692538, 145.56496427055507, 145.5806737056461, 145.63616216246328, 145.65195355982436, 145.65912525845286, 145.66547733438097, 145.71443613035163, 145.72046035719958])
        expected['rho_s'] = np.array([6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.200759506225, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.200759506225, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_90_90000(self):
        """Test for latitude = 0, longitude = 90, rho = 90000"""
        lat = 0
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([82.24806826375756, 82.24585869756106, 82.24379854175267, 82.24187669727495, 82.22865771991003, 82.22668806410813, 82.21750828986363, 82.20835669014947, 82.1886464717521, 82.17949487203794, 82.17033900345552, 82.14180844951873, 82.10225055516696, 82.07366706726411, 82.04497429633453, 82.01655217165082, 81.9632673046146, 81.9214383723638, 81.87969481747763, 81.83765158804161, 81.78398166909093, 81.73904329322049, 81.69371986543558, 81.64838619236691, 81.59127727316914, 81.54607678878929, 81.49561620497488, 81.4453186919269, 81.38275159780865, 81.33243444796682, 81.28221462832065, 81.22397019017336, 81.15305489734206, 81.09499231298143, 81.03701681353272, 80.97898496502334, 80.91481533777107, 80.86411569733572, 80.81347752860289, 80.76284789760652, 80.69832395429106, 80.63331165244995, 80.56841461005106, 80.50370710540166, 80.42437616573925, 80.35977623656927, 80.28303991501782, 80.20652728216169, 80.1150411670976, 80.03837570875879])
        expected['lon_n'] = np.array([119.96227337432907, 120.13641587740771, 120.31236155042724, 120.48890828009374, 120.22254456284158, 120.39675536781192, 120.54766156733419, 120.69923712539511, 120.40794321767241, 120.55801613411593, 120.70729674861572, 120.78354898050975, 120.41586623710963, 120.49062948776458, 120.56658119133512, 120.64354376290282, 120.23785784696075, 120.27174924562235, 120.30719792741472, 120.34434049612125, 119.94555307124088, 120.00191579227183, 120.06061443799892, 120.11715474394828, 119.74675358525849, 119.80517902341877, 119.85449298922627, 119.90592431367646, 119.53477183417797, 119.58557478122452, 119.63699244529639, 119.6087018017542, 119.16631044921246, 119.14009618317796, 119.11504304930234, 119.09138327401743, 118.67374452702782, 118.66237909224893, 118.65205184622387, 118.64303599651944, 118.23677634468709, 118.22437272115435, 118.2125838146469, 118.20218826673015, 117.80346914374148, 117.79471284122553, 117.74963359270345, 117.70616626882548, 117.28550857822273, 117.24475966963443])
        expected['rho_n'] = np.array([6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.2, 6371.200759506225, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.200759506225, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([-61.42179684232764, -61.4097347282534, -61.39765895380083, -61.38552853783489, -61.348140082318224, -61.3360506474873, -61.32900189226385, -61.321871174770365, -61.289373134699446, -61.28229705871931, -61.27520732236084, -61.265262566929295, -61.22980022475858, -61.21989645046207, -61.20999267616554, -61.20000693959898, -61.17321893767419, -61.17198950362359, -61.17074640919466, -61.169503314765706, -61.14257870905752, -61.14741448298989, -61.15225025692226, -61.1570040685846, -61.13599440669762, -61.140871161765006, -61.15205901162551, -61.163274182242674, -61.148507313257085, -61.159831766900965, -61.1711698809232, -61.18638754239399, -61.175500220856975, -61.19077252384113, -61.206099468338635, -61.22133079018778, -61.21098988378438, -61.22698618682054, -61.24290052758667, -61.25880120797447, -61.248432980814385, -61.27459260533553, -61.30076589023503, -61.3269391751345, -61.32676159021608, -61.35300317700728, -61.379490650608574, -61.40585518080482, -61.405896161939864, -61.43241095629784])
        expected['lon_s'] = np.array([127.2410147275149, 127.26897752197694, 127.29702227870901, 127.32514899771111, 127.24548167123208, 127.27358106947752, 127.31782703492087, 127.36222326452595, 127.29799216557115, 127.34234741404121, 127.38683926629467, 127.4220010801419, 127.34781156537721, 127.38287775657606, 127.41802591004495, 127.45324236540554, 127.38663436061957, 127.42939134482386, 127.4724078762166, 127.51546538874435, 127.44835194995981, 127.47034515908724, 127.4926798776732, 127.51494629436743, 127.42654998612913, 127.44887104433673, 127.46550938515487, 127.48236629202647, 127.38748130407666, 127.40436553170493, 127.42135904235991, 127.4298831184441, 127.32590031851981, 127.33445171536067, 127.34305775371489, 127.35163647131243, 127.23888370849386, 127.23948476514082, 127.23970333119426, 127.24031804821956, 127.12710083253742, 127.12764724767104, 127.1283849081014, 127.12920453080179, 127.01503108863587, 127.01549554149942, 127.0114247487541, 127.00734029563043, 126.88744315494002, 126.88301719235785])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2, 6371.2, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_90_90000(self):
        """Test for latitude = 30, longitude = 90, rho = 90000"""
        lat = 30
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([79.80784145634523, 79.80767240916327, 79.8075750789676, 79.80744445159972, 79.80357173434032, 79.80343171546234, 79.79756031409707, 79.79174099292422, 79.7825569498115, 79.77668127957799, 79.77082353859106, 79.74371281022798, 79.71328687879651, 79.68639727780783, 79.65928654944474, 79.63223302391597, 79.59015478976046, 79.55091279165781, 79.51186886904107, 79.47305802662976, 79.43056656602947, 79.39065008674626, 79.35074214519953, 79.31057465646433, 79.26750348355823, 79.22720365990787, 79.18423408606567, 79.14143441317907, 79.09525038155874, 79.05230556715229, 79.00942393199566, 78.9576169471411, 78.9029993394431, 78.85134432629758, 78.80005814336725, 78.7484355736203, 78.70083171642625, 78.65585492074172, 78.61123244112038, 78.56691475869074, 78.51937408074652, 78.46452936925861, 78.40950878039956, 78.35497142742432, 78.29733146101857, 78.24271812218879, 78.17672824952272, 78.11111830612923, 78.04279080121971, 77.97696741441464])
        expected['lon_n'] = np.array([94.46967829733451, 94.49057184600558, 94.51238064002543, 94.53452411331462, 94.5923143438821, 94.61348110011996, 94.64391642306154, 94.67517819889268, 94.74110318476164, 94.77085548878622, 94.80163915137547, 94.8333380593135, 94.89962504520848, 94.9295276133948, 94.96007221936308, 94.99085588195233, 95.07036611408046, 95.11407249457936, 95.15780619583495, 95.20156721784723, 95.27967043100632, 95.34586179425291, 95.41235368582299, 95.47925538874327, 95.57929033932727, 95.64438887230666, 95.7253744252955, 95.8061414122309, 95.9204377978019, 95.99923086006717, 96.07840641292596, 96.13322551120447, 96.22033091368965, 96.27303948351464, 96.32546801558365, 96.3786888495964, 96.45138255293284, 96.49064248028208, 96.5298136151721, 96.56923063687223, 96.64057196275303, 96.71586796816324, 96.79125276603267, 96.86606382801182, 96.97208202430876, 97.04450252007841, 97.11995561983954, 97.19501939881798, 97.2999720856044, 97.37236526061736])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.200759506225, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([-59.100010637765195, -59.08389139132399, -59.06771750336938, -59.05148897390143, -59.02800678353492, -59.011791914445325, -59.00248919679578, -58.99337772444298, -58.97687598740822, -58.96753228862363, -58.95836617475749, -58.947383230572115, -58.92907832359646, -58.91794511524935, -58.90690752955061, -58.895951906121894, -58.88942224527537, -58.89024186797579, -58.89099318878448, -58.89173084921484, -58.885433414800076, -58.891416660513016, -58.897577491144375, -58.90367001988403, -58.90275477453525, -58.90883364289655, -58.92254866274993, -58.93644126752173, -58.94299824912497, -58.956904514275095, -58.97057855299346, -58.99085055445008, -59.003923536521455, -59.024099915329685, -59.04422165262454, -59.0644116918111, -59.077430032369136, -59.09746980739396, -59.1174139597704, -59.13754935744359, -59.15064966027168, -59.18497819104016, -59.21937502370034, -59.253853818630574, -59.28143412249909, -59.315789974024256, -59.35257737289393, -59.38925548873692, -59.41911707578819, -59.455767870874496])
        expected['lon_s'] = np.array([121.5996336429212, 121.60071281281007, 121.60173734118557, 121.60303507712787, 121.58106918875711, 121.582189339781, 121.60087673735015, 121.61960511605433, 121.6151245119588, 121.63385289066298, 121.65269055239386, 121.66972504418388, 121.6635232324175, 121.68053040345085, 121.69745561221413, 121.71453108513916, 121.71862919864117, 121.74567674775442, 121.77299750443447, 121.80016799695278, 121.80436173310316, 121.80895162022541, 121.81376007340111, 121.81841826241505, 121.80008603468275, 121.80477154445337, 121.80355577078112, 121.80247660089225, 121.77832505198708, 121.77717758020653, 121.77601644804761, 121.77262867421929, 121.74615486099633, 121.74258950224957, 121.73902414350283, 121.73548610551276, 121.7003106312872, 121.68796164926782, 121.67542142195167, 121.66281289274383, 121.62752813549154, 121.61852594616546, 121.60951009646105, 121.60042594486492, 121.56850164068429, 121.55907597962967, 121.54936345062991, 121.53989680844028, 121.50737144761267, 121.49731740915442])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_90_90000(self):
        """Test for latitude = 60, longitude = 90, rho = 90000"""
        lat = 60
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([85.31715699135532, 85.31166381171533, 85.30616508254666, 85.30061598073284, 85.29678339083482, 85.29130728666776, 85.28190040863336, 85.27247047871052, 85.26485140269136, 85.25545989258259, 85.24608417728628, 85.22163167317082, 85.19905704419027, 85.17460752828256, 85.15010635906927, 85.12555865919224, 85.09503240935733, 85.06244600371403, 85.0297225674005, 84.99685270890663, 84.96618174443869, 84.93012304137608, 84.89396231236273, 84.85762784041239, 84.8236591752358, 84.78721542025872, 84.7472063065349, 84.7070499168571, 84.66941045188847, 84.62918405277168, 84.58881507345596, 84.54330765745027, 84.50047383361783, 84.45483322892333, 84.40907394969202, 84.36318703130058, 84.32433520775425, 84.28246444059478, 84.24056635267863, 84.19860551239948, 84.15979253555406, 84.1078801096542, 84.05589041723935, 84.00385376727397, 83.95519678028727, 83.90316354541646, 83.84263014012357, 83.78196610746282, 83.72482090288509, 83.66411375465519])
        expected['lon_n'] = np.array([140.1649210900946, 140.1379281824947, 140.11157731267681, 140.08592312215424, 140.21381158417356, 140.18649082749351, 140.13165806883666, 140.07840991406724, 140.17722909097898, 140.12183625681016, 140.06817829069055, 139.87865420160102, 139.84148431213782, 139.65219244948008, 139.46679379464928, 139.28441408343159, 139.20235619074305, 138.972274438362, 138.74592196926778, 138.52343538724378, 138.44843991015705, 138.23524238540426, 138.0264161817203, 137.82077284618956, 137.75961533236125, 137.5567313932552, 137.34111598153623, 137.12935279650915, 137.05831882914103, 136.85013466323903, 136.6460895919741, 136.34246036261032, 136.17697853939927, 135.8811903672027, 135.59072974255872, 135.30671681649125, 135.17167031622176, 134.90945935398497, 134.65226175059897, 134.40024143060384, 134.27661500662663, 134.02831030953996, 133.78556538643772, 133.54824363353646, 133.4354635499612, 133.2037971936927, 132.91612328623012, 132.63512930377578, 132.47456521676713, 132.20266904628727])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.200759506225, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([5.186583604041061, 5.279897648481764, 5.373245843868332, 5.466689661903274, 5.593621897438794, 5.687024734338721, 5.850498481933812, 6.014641588067548, 6.210326507788423, 6.373984670491097, 6.537424267140324, 6.736797489013, 6.964837014832213, 7.163581859301246, 7.361479760313202, 7.559459623595188, 7.877896703079514, 8.16671125213351, 8.455553121944178, 8.74320653883926, 9.054376297046701, 9.321511825675074, 9.586939807010936, 9.85294835442626, 10.138190714544336, 10.401494507048326, 10.689912905130456, 10.977129189918656, 11.279679249393538, 11.563828779244417, 11.846455176910368, 12.18403727663825, 12.532773075280758, 12.865246193509478, 13.195936632364806, 13.524181863497276, 13.830126526989659, 14.126126435050509, 14.420254871278757, 14.71126874124549, 15.00456389439502, 15.369842411207301, 15.731487267381127, 16.08887008551288, 16.43903339369193, 16.786751494148106, 17.167220351674487, 17.541425925731986, 17.90029772510279, 18.26177865673654])
        expected['lon_s'] = np.array([54.458730667723295, 54.35879816997684, 54.25847293635311, 54.15770032533874, 54.08094948963531, 53.9810067466051, 53.816983168781825, 53.65180870408341, 53.51173859967937, 53.34753743693768, 53.183698274221996, 52.98666780723464, 52.815438379837005, 52.61933681857677, 52.42345723846456, 52.22682975263823, 51.97122358332884, 51.69173907268109, 51.41135639215748, 51.1306936738779, 50.876088127281925, 50.61264773099453, 50.349265391315086, 50.08419941000523, 49.84617756271404, 49.58157261917316, 49.29073291903026, 48.998906256552296, 48.73389833185039, 48.44162770707637, 48.14840427091314, 47.7979438495052, 47.47291905256639, 47.120559838569186, 46.76578956779497, 46.40893267422931, 46.10723980349527, 45.77683281749049, 45.44431188793758, 45.11069129792831, 44.800812445474, 44.37171947124629, 43.938729874097085, 43.50225346537659, 43.088436209227616, 42.64571359251114, 42.1506888022253, 41.651709332410014, 41.173821486751734, 40.666785808810424])
        expected['rho_s'] = np.array([157939.8421508789, 158294.3796569824, 158651.61492919922, 159011.65733642576, 159742.31447753907, 160102.91588134764, 160802.19630126952, 161510.4206665039, 162597.81484375, 163314.18111572266, 164036.3682434082, 164947.69064941406, 166245.59564819335, 167171.08740234375, 168102.82533569334, 169044.9776184082, 170878.3648864746, 172329.70229492188, 173804.3352783203, 175296.12702636718, 177212.03920288084, 178638.03948364258, 180076.20401611328, 181539.1466796875, 183431.15567626953, 184921.7929748535, 186594.91835327147, 188289.54079589844, 190430.04807739257, 192163.59369506835, 193917.12951660156, 196076.98294067383, 198714.46248779297, 200934.26221923827, 203189.48532104492, 205476.57122802734, 208048.66032714842, 210191.05118408203, 212365.4506713867, 214562.18571777342, 217271.2532836914, 220195.51022949218, 223172.75032958985, 226203.1437133789, 229782.5689575195, 232920.60563964842, 236487.68435058594, 240124.6862426758, 244362.90111083983, 248157.12686767578])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_90_90000(self):
        """Test for latitude = 90, longitude = 90, rho = 90000"""
        lat = 90
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([82.34980649276585, 82.35883258775402, 82.36779123462414, 82.37671829186934, 82.3705207489712, 82.37948793357778, 82.3916328636957, 82.40375218060426, 82.4006760341568, 82.41277400672419, 82.42483270570385, 82.44572881569586, 82.45134237741996, 82.47210871381773, 82.49274869171587, 82.51323328281039, 82.52234731648412, 82.54664912955103, 82.57078872562515, 82.5948292839563, 82.60333969966213, 82.62841930052078, 82.65333070797111, 82.6780534314456, 82.68715295096736, 82.71162466498987, 82.73989311041726, 82.76792506054464, 82.78009987274018, 82.80777665303073, 82.83521352292667, 82.87523373570788, 82.89924099686684, 82.93864563884911, 82.97768273127667, 83.01635825056505, 83.03864558471362, 83.0763614624236, 83.11364874576317, 83.15056378379298, 83.17138348104365, 83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544])
        expected['lon_n'] = np.array([239.34851591481595, 239.28043258916927, 239.21185748990234, 239.14295454155524, 239.06115619605518, 238.99288162511172, 238.89602954268094, 238.79876764889997, 238.68896552780285, 238.59211344537206, 238.49471494780767, 238.287869498983, 238.0690302379758, 237.8613924872074, 237.65249798163174, 237.44212815519535, 237.18687032553365, 236.94062834557633, 236.6923919503814, 236.44207917767875, 236.18225878165148, 235.91757529093513, 235.65048757363095, 235.38099562973895, 235.10336210035626, 234.83108343928288, 234.52913443645494, 234.2242347919056, 233.91245031667285, 233.6039443322417, 233.29270627214257, 232.91871243394934, 232.5390358783667, 232.15840309630022, 231.77345363467828, 231.3842421350143, 231.00396452278466, 230.62024449521334, 230.2324536748967, 229.84072866561812, 229.44985059979663, 228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([48.517652038249615, 48.525165246336634, 48.53232328458681, 48.53916030394599, 48.49639990464713, 48.50390969763955, 48.5095480187994, 48.51500192485165, 48.47052714807111, 48.47649331831112, 48.482288733821875, 48.487216715308044, 48.44232871208272, 48.44792263701296, 48.45331848645727, 48.45830452455138, 48.411500653263865, 48.41536654033409, 48.41881578586495, 48.422087446477384, 48.37538944312201, 48.380078367987224, 48.38436431169141, 48.38848974595009, 48.342594289822195, 48.347385667524954, 48.3516818565129, 48.355674102082766, 48.30976498557653, 48.31448806138759, 48.31890036359142, 48.322499873284016, 48.27616728504923, 48.28038151176713, 48.2843976629991, 48.288202078366794, 48.24280181095372, 48.24802690566878, 48.252961717344114, 48.25768479315518, 48.21248601632262, 48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855])
        expected['lon_s'] = np.array([3.5521261090034026, 3.5211221728134223, 3.489967545574878, 3.458780048050954, 3.5427674691800157, 3.511755422140396, 3.4786102216456167, 3.445244320663281, 3.5271468265481967, 3.494132039977882, 3.4609479927821982, 3.4263814726098216, 3.5069271187274316, 3.472532847388186, 3.438059388543251, 3.4035411065818875, 3.4810074046008794, 3.4435687898825122, 3.4061756386108084, 3.368535746755337, 3.4458618124529328, 3.409107497312036, 3.372313908583412, 3.3353341971999053, 3.413228449259081, 3.3763868357628555, 3.3387919804672244, 3.300950171144415, 3.37792661653387, 3.3403176739730758, 3.302593898856861, 3.2626253393812386, 3.3372528400264443, 3.297408931503175, 3.2573996043359443, 3.217312797210316, 3.292921710661888, 3.2540583611713387, 3.215103444457229, 3.1759666739564674, 3.251479324429424, 3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065])
        expected['rho_s'] = np.array([377304.9994140625, 377315.9606079101, 377326.75167236326, 377336.95943603513, 377284.7540161133, 377295.9339477539, 377304.2216796875, 377312.72814941406, 377258.0923095703, 377266.86612548825, 377275.5427246094, 377282.6395507812, 377226.9586303711, 377234.93040771486, 377242.73205566406, 377249.9260986328, 377191.3529785156, 377196.60268554685, 377201.4635253906, 377206.25145263673, 377147.7755493164, 377154.1432495117, 377159.87904052733, 377165.37178955076, 377107.7222290039, 377114.3572753906, 377119.5583740234, 377124.61364746094, 377066.623828125, 377072.55405273434, 377078.1683227539, 377082.00838623045, 377022.75474853517, 377027.34824218747, 377031.45565185545, 377035.73319091793, 376977.743371582, 376983.6006835937, 376989.2878662109, 376994.8292236328, 376936.8394042969, 376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_180_90000(self):
        """Test for latitude = -90, longitude = 180, rho = 90000"""
        lat = -90
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.90688287510915, -72.88874189267358, -72.8686747968921, -72.85101192769844, -72.88099645815478, -72.86270521155751, -72.84415441777173, -72.82800785057384, -72.85974090945771, -72.84296596485618, -72.82429222766535, -72.81035864175851, -72.84452324798693, -72.82994762429811, -72.81564520817611, -72.79945765984317, -72.83621773795619, -72.82401902009855, -72.81206618905102, -72.8003865655703, -72.83520686995902, -72.82410098236858, -72.81326830234494, -72.80268150913142, -72.84099887037519, -72.8297700393797, -72.81889637822104, -72.81034498138018, -72.85094362580674, -72.8417501911839, -72.83282996412785, -72.82743411468354, -72.86945343845747, -72.86344287198787, -72.85766453195004, -72.85213207872232, -72.89609117622052, -72.88967079840072, -72.88348264701267, -72.87751306167809, -72.91960068734372, -72.9163904984338, -72.91345351709069, -72.91069412066602, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99973246668631, -72.99780635334037])
        expected['lon_n'] = np.array([28.70189947740304, 28.64750043575847, 28.59360853565978, 28.539755909148813, 28.673280984780686, 28.6187043582177, 28.547506466309486, 28.476247102698743, 28.592215177069097, 28.520161803967337, 28.448595081843944, 28.365041377727177, 28.468491422896193, 28.384019058336058, 28.29983526926836, 28.216022017963134, 28.294190117919342, 28.18469535533487, 28.07530475313524, 27.96591073584102, 28.043073090441947, 27.93465408265175, 27.826358018266614, 27.718176359550075, 27.795950016081715, 27.68673187615863, 27.568439829923175, 27.449941170465333, 27.517005090378394, 27.39750239311256, 27.27808507321135, 27.139114629263684, 27.185146689174985, 27.044815330035195, 26.904535197314182, 26.76435410233613, 26.81727440802539, 26.684401615459254, 26.55180032291263, 26.419303190750842, 26.47081647747108, 26.31129740940543, 26.151633199819923, 25.992225122328286, 26.015907095728014, 25.855057848321504, 25.679308543240616, 25.50372145515251, 25.510635314139858, 25.33370097123797])
        expected['rho_n'] = np.array([257214.93735351562, 257202.37208251952, 257252.87620849608, 257240.5539794922, 257196.92794189454, 257184.0710205078, 257235.5230102539, 257224.14864501954, 257181.61629638672, 257169.6829345703, 257220.8918823242, 257210.87855224608, 257169.9259765625, 257159.45086669922, 257149.097277832, 257201.86169433594, 257162.5375, 257153.52064208983, 257144.649609375, 257135.87579345703, 257159.49947509766, 257151.30895996094, 257143.19135742186, 257135.1952758789, 257096.86755371094, 257088.43399658203, 257144.67391357422, 257137.96595458983, 257101.07218017578, 257093.92674560545, 257087.02435302734, 257082.57668457032, 257111.13411865235, 257106.24897460936, 257101.4853515625, 257096.89185791014, 257062.3555908203, 257057.17879638672, 257052.19643554688, 257047.38420410155, 257075.89302978516, 257072.95222167968, 257070.06002197266, 257067.4108642578, 257035.20780029296, 257032.04825439453, 257030.27404785156, 257028.6456665039, 257060.654296875, 257058.56413574217])
        expected['lat_s'] = np.array([-68.3394943959866, -68.30527514824485, -68.2709602778547, -68.23645416216777, -68.21161959434562, -68.17697687487532, -68.14398706118416, -68.1107923418179, -68.08737845334309, -68.05406079057175, -68.020483580612, -67.98901006891657, -67.96737202962595, -67.93569361225545, -67.90382394958817, -67.87180402275914, -67.85184621000437, -67.82172507576462, -67.7914126962281, -67.7609363921515, -67.74045948501981, -67.71996891750976, -67.69928710470296, -67.67850966924777, -67.66795019679094, -67.64704981793071, -67.63134038283968, -67.61552166472194, -67.61000287187255, -67.59411585186311, -67.57810588844859, -67.56609841588772, -67.56437720821688, -67.55228777338596, -67.54008905552831, -67.52782203577897, -67.5294066396664, -67.5205956956371, -67.51170278933773, -67.50270060001165, -67.50417592087237, -67.50283720379505, -67.50145750558272, -67.5000504866137, -67.50916195896647, -67.50768663810575, -67.50597909081327, -67.50412127935903, -67.51286392149663, -67.51103343079907])
        expected['lon_s'] = np.array([152.83729890629942, 152.81515543301023, 152.7929982993427, 152.77086848643188, 152.80078471499652, 152.778627581329, 152.78021218521647, 152.7817967891039, 152.83523618917008, 152.83653392511238, 152.83813218937817, 152.82361120720273, 152.86101332309772, 152.84641037865222, 152.83182109458508, 152.81706788597785, 152.87289785225354, 152.87669543743206, 152.8804657018539, 152.88393543795226, 152.93973808347127, 152.92906932798772, 152.91850985553089, 152.9078001189123, 152.94915008414756, 152.93846766828565, 152.92726615804682, 152.91601000629464, 152.95696382055806, 152.94569400842752, 152.93435589440531, 152.92036766698513, 152.9588216320123, 152.94481974421376, 152.93064027149683, 152.91644713840154, 152.94390449886498, 152.91863279893593, 152.89308789144008, 152.86762494621428, 152.89482275948927, 152.8741136259258, 152.85318592630887, 152.83205332101684, 152.8639776251975, 152.84281769914878, 152.8134342253394, 152.7841873553134, 152.8081476589218, 152.77857293981566])
        expected['rho_s'] = np.array([6371.200759506225, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_180_90000(self):
        """Test for latitude = -60, longitude = 180, rho = 90000"""
        lat = -60
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-78.5693829046165, -78.55538101681799, -78.53867437410813, -78.52513693917317, -78.53396154358083, -78.52006893880903, -78.50680471144085, -78.49661406919921, -78.51004222110745, -78.49923686184047, -78.48847248370853, -78.47933369059908, -78.50050727702612, -78.49075376689132, -78.4840601815047, -78.47742123763146, -78.5012585978348, -78.49918222032713, -78.49425082374637, -78.49235203115711, -78.51658554233231, -78.51561565547019, -78.51179074953495, -78.51105308910459, -78.53725369476078, -78.5359013173051, -78.53826456275794, -78.53782743065105, -78.57102215001731, -78.5699566405068, -78.57192373498776, -78.57988773555999, -78.6164702287546, -78.6237785311665, -78.62825913526203, -78.63573136221402, -78.67161717611327, -78.67738185577275, -78.68325581845897, -78.68626110169376, -78.72550736866467, -78.73452321836908, -78.7465443513083, -78.7556558236611, -78.8019371854771, -78.8133709221477, -78.82552865887033, -78.84067801844941, -78.88805221053263, -78.90246390968136])
        expected['lon_n'] = np.array([54.00218716322131, 53.82215703707811, 53.631328381857124, 53.455184633351664, 53.67256564897107, 53.49506610791537, 53.2771967337703, 53.072926266262726, 53.2645335630491, 53.05753101987352, 52.85143689185755, 52.62641972474607, 52.821230380253176, 52.59333428840653, 52.378852688089786, 52.16491067271748, 52.30811241885597, 52.05315511751773, 51.78709875877822, 51.53363726886822, 51.674526995972656, 51.434462922119636, 51.18332711162202, 50.94477250957557, 51.100176388666284, 50.85855844677709, 50.612941429128846, 50.356457580511105, 50.516751875047554, 50.25703393185782, 50.00948055557988, 49.741764460872055, 49.867617526518714, 49.596001393794815, 49.31356282633103, 49.04272533517251, 49.18204753386244, 48.92363074170959, 48.665770609974096, 48.39729977936299, 48.5455080541631, 48.240471805830325, 47.9468385583169, 47.642399951536504, 47.754336506749254, 47.45703886274611, 47.13029968832563, 46.814287325996645, 46.89313502977527, 46.57289819544464])
        expected['rho_n'] = np.array([265046.1149047851, 265002.17291259766, 265019.84206542966, 264976.8479370117, 264957.45318603516, 264913.6813232422, 264929.9165283203, 264885.221105957, 264865.6319213867, 264819.55115966796, 264773.7620483398, 264792.47628173826, 264714.3625854492, 264731.91021728516, 264688.673046875, 264645.8004394531, 264627.83963623043, 264582.48800048826, 264598.8204223633, 264554.3194335937, 264535.62950439454, 264494.0936279297, 264514.41193847655, 264472.8517578125, 264458.4879760742, 264415.9313232422, 264376.9230834961, 264399.5016845703, 264326.589086914, 264348.0983032226, 264308.9442382812, 264272.6094604492, 264263.83564453124, 264226.60161132814, 264250.9544189453, 264214.2064697266, 264206.84229736327, 264170.67764892575, 264134.75604248047, 264160.66431884764, 264091.7376098633, 264118.6180541992, 264084.32482910156, 264111.91009521484, 264044.6603759765, 264009.6137207031, 264037.4663330078, 264004.1938842773, 263999.26013183594, 263965.0398193359])
        expected['lat_s'] = np.array([-66.14862925743498, -66.11841250054684, -66.0878542342002, -66.0570364206651, -66.03679173996517, -66.00545483205315, -65.97523807516501, -65.94454320503499, -65.92526841119721, -65.89398614479856, -65.86226674629302, -65.83157187616297, -65.81313036540394, -65.78173881597857, -65.74988281368962, -65.71748039626709, -65.70077375355723, -65.66971005321201, -65.63790869243644, -65.60515110517704, -65.58735163220001, -65.56422461167034, -65.53990913822508, -65.51383147597397, -65.50518445648476, -65.47413441651787, -65.44281116898418, -65.39091539167043, -65.345139463853, 49.65650753517379, 49.62992785401868, 49.619689400452835, 49.699841670362936, 49.69489319830926, 49.683309197476916, 49.66435883762472, 49.738743012780745, 49.71803729431185, 49.697867745692804, 49.67828559334237, 49.75322301382117, 49.73867471088904, 49.724584030631306, 49.70744708600375, 49.79239073361661, 49.77787658163033, 49.76261452392994, 49.751481315582815, 49.839170699242025, 49.827019792708576])
        expected['lon_s'] = np.array([155.9441743748198, 155.91697656154483, 155.88958750297306, 155.8617886330511, 155.8617886330511, 155.8334570083739, 155.8245504416962, 155.81505647874988, 155.83323844232046, 155.82217353586503, 155.8099474972507, 155.78027715549618, 155.77948485355245, 155.74768349277687, 155.71396967903368, 155.6780702047561, 155.68985911126356, 155.66625397749198, 155.6382365415166, 155.6048915579886, 155.60620295430923, 155.5572851394736, 155.50023939952567, 155.4320604512306, 155.41116007237034, 155.31481342393815, 155.19508020778784, 154.99062532517266, 154.80366938721107, 200.13059422897823, 200.28615861751445, 200.30585688308076, 200.35985635865887, 200.32032322374286, 200.24577853914133, 200.13284819140435, 200.1538578532913, 200.04600916629678, 199.93831074346403, 199.83074892441468, 199.8507203975478, 199.72171178450458, 199.59285343562314, 199.4629842187445, 199.46186406772063, 199.33265054900232, 199.1906918972928, 199.0499490192555, 199.03570124464684, 198.89446659298932])
        expected['rho_s'] = np.array([6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.198860740661, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 264232.33740234375, 264234.79212646483, 264211.0226196289, 264170.48321533203, 264188.8328857422, 264148.94969482423, 264190.0480957031, 264155.36600341793, 264197.77683105465, 264239.142578125, 264279.29311523435, 264243.4201171875, 264279.6333740234, 264314.8501586914, 264287.58084716793, 264307.0971191406, 264341.90073242184, 264313.0516479492, 264344.6471069336, 264361.5628295898, 264393.81450195314])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_180_90000(self):
        """Test for latitude = -30, longitude = 180, rho = 90000"""
        lat = -30
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-78.65109928784653, -78.65504713718681, -78.65742404301798, -78.65977362809247, -78.65506079756514, -78.65851687328518, -78.66727317580114, -78.67776434636627, -78.68031883711586, -78.68865166790329, -78.69676593263725, -78.71191529221633, -78.72071257586731, -78.73379921831707, -78.74847046465425, -78.76144782407727, -78.78001227824137, -78.80160933639695, -78.82481831919665, -78.84622413205548, -78.86549892589323, -78.88781998410084, -78.90828323085421, -78.9286918360942, -78.9482125167421, -78.96989153716771, -78.99382452001944, -79.0195743331904, -79.04199101404637, -79.06723539321874, -79.09062196093686, -79.12384400105981, -79.15504430518843, -79.18781555282615, -79.21870166825295, -79.25139095362061, -79.28238635207416, -79.31006227859103, -79.33773820510794, -79.36724462232237, -79.39893670007123, -79.43612024991279, -79.47316719597094, -79.51218123651006, -79.55462403201253, -79.59109724218038, -79.6330072829276, -79.67687075777741, -79.72558366693792, -79.76682434914648])
        expected['lon_n'] = np.array([76.56912535075426, 76.3258135219509, 76.06205210677251, 75.79986163510324, 76.06695618259658, 75.82148601401549, 75.53329984236515, 75.26987993664527, 75.51071240677992, 75.21982831040741, 74.92826802530708, 74.66202725145999, 74.90841949557903, 74.6162649840209, 74.34603537969852, 74.05501467954261, 74.26668907211048, 73.93547953887821, 73.62581925247734, 73.29441164375916, 73.50517079097824, 73.21620597776251, 72.90491327615001, 72.59389378210432, 72.82930308203797, 72.53418426838006, 72.22285741582171, 71.93307981009475, 72.15174148618354, 71.8564450876072, 71.53976336673956, 71.2340167787331, 71.44270637863369, 71.13067601659083, 70.79699395487901, 70.48418495127078, 70.7303517991472, 70.40667596456947, 70.08343043190945, 69.7811467498122, 70.02742288071533, 69.67487583651597, 69.32149550923785, 68.9893502400893, 69.21238323724688, 68.85145555093591, 68.47666941079896, 68.12251727214463, 68.33179426831383, 67.94823133509342])
        expected['rho_n'] = np.array([270931.546875, 270861.0646972656, 270851.05136718747, 270841.54842529295, 270787.1799316406, 270715.8228027344, 270702.115234375, 270629.71302490233, 270570.87255859375, 270556.4115600586, 270541.70751953125, 270471.3711669922, 270417.05128173827, 270405.33665771486, 270333.95522460935, 270323.0912475586, 270263.86191406247, 270246.70314941404, 270170.0234008789, 270153.7152832031, 270093.926953125, 270019.5074951172, 270008.4976928711, 269998.1441040039, 269945.1366455078, 269873.65799560543, 269866.90142822266, 269800.3565307617, 269811.17189941404, 269743.3388793945, 269736.1934448242, 269671.32553710934, 269682.69990234374, 269616.08209228516, 269610.61364746094, 269545.4540893555, 269500.5885375976, 269496.7727783203, 269492.00915527344, 269428.8425415039, 269383.53951416013, 269380.2098388672, 269377.48776855465, 269313.9322875976, 269269.2611694336, 269265.7370605469, 269263.4767700195, 269200.65041503904, 269156.4653808594, 269153.50026855466])
        expected['lat_s'] = np.array([46.08871041089344, 46.07909691963665, 46.065610711120456, 46.05685099350991, 46.12672724381373, 46.11711375255694, 46.106639657464726, 46.10092961931859, 46.17421754911283, 46.167605925996256, 46.15715573656614, 46.1522892267825, 46.226700722695206, 46.22102825058951, 46.21144549518398, 46.20664045710288, 46.28508859481506, 46.283206877698724, 46.277479764079665, 46.276380103623296, 46.35951716620069, 46.350699391982204, 46.34309397634139, 46.339313466635794, 46.4209205668391, 46.412410151133265, 46.40490035814084, 46.40195313151398, 46.483881250608285, 46.47583187267142, 46.472461174316024, 46.46709947581756, 46.551008349771166, 46.54470067007266, 46.54308191523937, 46.53748457521454, 46.62008205284751, 46.61212488246444, 46.60878833505489, 46.60158590057511, 46.68428583104563, 46.68389992535752, 46.67963447222085, 46.67575833986687, 46.76201338380037, 46.76141915734258, 46.75836947787817, 46.75990968553601, 46.84340874813941, 46.8440029745972])
        expected['lon_s'] = np.array([197.80369904291663, 197.76755368182893, 197.72974175458373, 197.69377397841444, 197.75879737931297, 197.7226656786036, 197.67277797690582, 197.62469344514895, 197.67732688289306, 197.6290101247044, 197.57910876262827, 197.5232104944609, 197.56770234671436, 197.51158551249353, 197.45384309325024, 197.3978901835695, 197.4259486006799, 197.3537671615312, 197.27993281660338, 197.20773771707633, 197.23688896445395, 197.16611454427428, 197.0957089543098, 197.0266694021793, 197.05888057430508, 196.98805151261206, 196.91246863925667, 196.83849769054544, 196.86503980566013, 196.7891700643596, 196.71498054959494, 196.6268437885451, 196.640258280075, 196.5517253680533, 196.46484536181075, 196.3763397705457, 196.39416656427943, 196.31051040732513, 196.2285208165283, 196.14496028222234, 196.16202209476904, 196.06295703104718, 195.9623893257079, 195.86186260150367, 195.86122056372167, 195.76192327356804, 195.65205285057922, 195.54371238996447, 195.53151367210683, 195.42285902279028])
        expected['rho_s'] = np.array([269832.31655273435, 269860.3635986328, 269827.5286254883, 269854.09311523434, 269791.7771484375, 269819.7512817383, 269786.4302246094, 269812.82458496094, 269750.72735595703, 269777.34045410156, 269742.97431640624, 269768.8582885742, 269705.3271118164, 269732.5235107422, 269698.837890625, 269724.4302124023, 269659.5623046875, 269685.4462768555, 269650.3267089844, 269674.70382080076, 269669.4541137695, 269637.90726318356, 269604.27025146486, 269631.4180419922, 269627.35924072267, 269594.8888305664, 269561.98094482423, 269588.73986816406, 269585.65323486325, 269553.4258666992, 269580.6708740234, 269547.4470336914, 269544.79787597654, 269513.00798339845, 269540.56894531247, 269507.29649658204, 269505.30355224607, 269474.24278564454, 269502.6543945312, 269470.01385498047, 269467.92369384767, 269496.16517333983, 269463.28159179684, 269429.71749267576, 269426.7280761719, 269454.84803466796, 269422.01306152344, 269448.72337646486, 269385.5324584961, 269413.5795043945])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_180_90000(self):
        """Test for latitude = 0, longitude = 180, rho = 90000"""
        lat = 0
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-76.24562960557313, -76.25476839868259, -76.26489073903255, -76.27389292835863, -76.27260885279466, -76.28202085347095, -76.29894606223422, -76.3158985917542, -76.32373964892139, -76.34100636714317, -76.35686606639595, -76.37939203027864, -76.39491022007292, -76.41799625946757, -76.43977090254157, -76.46139528145383, -76.48849747208044, -76.52029883285601, -76.55304275973705, -76.58436600727072, -76.61216487719267, -76.64113853965188, -76.6712460135133, -76.69986450613564, -76.72674813070881, -76.75465628365748, -76.78667621048652, -76.8199802128795, -76.8504974980911, -76.88306384005372, -76.91415486115562, -76.95472618482549, -76.99704603692291, -77.03689336054074, -77.07659041999688, -77.11761253615197, -77.15609382193583, -77.19327737177736, -77.2288899781098, -77.26435232028052, -77.3050329269771, -77.35171043976499, -77.39818304687776, -77.44606267295953, -77.49786282762491, -77.54492283100629, -77.59681860832006, -77.6485094799587, -77.7093254843285, -77.7601011106184])
        expected['lon_n'] = np.array([76.55530787806333, 76.27676593352103, 76.01708214127716, 75.74052778178334, 75.9729522890497, 75.70931381727638, 75.40416145572772, 75.10104449055173, 75.30612775057052, 75.01466308811857, 74.70444955639493, 74.3999528930066, 74.61293185170594, 74.32186334022583, 74.01290656330949, 73.70403174866317, 73.87689700636702, 73.5224238488217, 73.18443876793279, 72.82810779893322, 72.9998529056132, 72.66651235335989, 72.35044534951754, 72.01586170283531, 72.21477730203358, 71.87455191909692, 71.5422154046516, 71.22721391031978, 71.41617792389735, 71.09537076877102, 70.75690074445104, 70.40765268162077, 70.60394548817777, 70.24790138712335, 69.89102400299019, 69.55142699745716, 69.74789055874339, 69.41969996912428, 69.07358013293388, 68.72714610804168, 68.9407192931988, 68.56572824738676, 68.18989708830681, 67.8304788738024, 68.00384956550484, 67.6372186712369, 67.23783701989873, 66.83684344391645, 67.01356775850137, 66.6044872385418])
        expected['rho_n'] = np.array([269610.95390625, 269585.02132568357, 269499.32471923827, 269474.43721923826, 269425.4642578125, 269338.21218261716, 269306.3979858398, 269275.82330322266, 269219.87503662106, 269126.35247802734, 269093.20155029296, 269064.7899414062, 269013.43516845704, 268923.0235473633, 268893.42103271483, 268864.2073852539, 268803.22814941406, 268763.3692626953, 268663.38178710936, 268624.1791137695, 268560.5021118164, 268526.6220581055, 268433.3182373047, 268401.9658203125, 268348.3264526367, 268315.36995849607, 268287.6145629883, 268199.7305786133, 268210.71607666014, 268121.3495361328, 268090.8477661133, 268063.5784545898, 268014.7999267578, 267986.1938842773, 267958.3169677734, 267868.80460205075, 267884.06763916014, 267798.5897705078, 267774.4557006836, 267749.4709838867, 267703.9978271484, 267678.38120117184, 267652.78887939453, 267567.4568359375, 267582.33100585936, 267495.05462646484, 267469.29217529297, 267444.57480468747, 267397.4246582031, 267371.07890625])
        expected['lat_s'] = np.array([47.512449683004256, 47.51116902253488, 47.506234210859546, 47.50566389006385, 47.57343985719788, 47.57219676276895, 47.57017502677462, 47.57256559298413, 47.64348686223095, 47.64521490009096, 47.643340013163794, 47.64173150361426, 47.717003603362386, 47.71880335820868, 47.71680552787645, 47.71528581078613, 47.794802873103414, 47.79670508078726, 47.80283176047276, 47.80540674178986, 47.88531653998442, 47.88385487950204, 47.88740657787044, 47.88682259669641, 47.963436828616445, 47.96211177191746, 47.961066752974446, 47.96385347015581, 48.036468626316804, 48.03852451325698, 48.03701162635582, 48.04080579643976, 48.11448987720586, 48.11755321704861, 48.11688727360453, 48.11658333018647, 48.1927433545267, 48.1900317694262, 48.19164027897574, 48.189598052413906, 48.26581613336208, 48.26541656729563, 48.26529020879599, 48.26946686947345, 48.34415498804754, 48.34751885621377, 48.348253101549545, 48.34930836577632, 48.428955201687835, 48.42913278660625])
        expected['lon_s'] = np.array([195.26210369048485, 195.2367363679074, 195.20923802630895, 195.18393900562322, 195.24166776448817, 195.2162867815324, 195.17851583542222, 195.14291688946813, 195.19022277965962, 195.15448722992213, 195.11667530267692, 195.0723337145852, 195.11504971765447, 195.0727025448004, 195.02819703216858, 194.98371884029348, 195.01251491783424, 194.95428072497072, 194.89812290961487, 194.83977943372463, 194.86809739802348, 194.8116253939658, 194.75744833346928, 194.7010036501683, 194.73081059570623, 194.67413368597346, 194.61289420987512, 194.55345790371769, 194.57606582987043, 194.51645193879455, 194.45477533058934, 194.38441072175988, 194.39566687351206, 194.3250017363591, 194.25223290094178, 194.17945040514613, 194.19649855731447, 194.12784149577752, 194.06117884947818, 193.99252178794123, 194.0089279023276, 193.92543566991336, 193.8419707582558, 193.76041829956586, 193.75955769573042, 193.67773202947367, 193.5862074945955, 193.49465563896064, 193.4871287704953, 193.39530370729364])
        expected['rho_s'] = np.array([268377.78314208984, 268388.3068603515, 268337.7784301758, 268347.2570678711, 268296.38837890624, 268306.8391845703, 268255.84897460934, 268264.93874511716, 268213.4867553711, 268223.7188232422, 268173.3119140625, 268124.8493408203, 268136.97713623045, 268150.61179199215, 268101.93048095703, 268052.5200439453, 268062.1202026367, 268012.56394042965, 268023.1848754883, 267972.6078369141, 267981.7705200195, 267936.29736328125, 267949.78619384766, 267902.7575683594, 267916.0276611328, 267870.11702880857, 267825.27578125, 267841.21933593747, 267795.5760498047, 267812.5889892578, 267768.282434082, 267786.170324707, 267742.66580810543, 267761.47725830076, 267719.2608642578, 267676.3639526367, 267694.2518432617, 267653.4450927734, 267672.5968017578, 267630.52623291017, 267648.26829833986, 267608.4094116211, 267568.0158325195, 267587.9695800781, 267546.6038330078, 267567.52974853513, 267528.18124999997, 267488.1765380859, 267508.44624023436, 267469.6081298828])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_180_90000(self):
        """Test for latitude = 30, longitude = 180, rho = 90000"""
        lat = 30
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([75.04992655787247, 75.05221979388631, 75.05463938839978, 75.05683017157607, 75.05844551131477, 75.06077972846363, 75.06304393617349, 75.06543620993028, 75.06708228552026, 75.0694848045608, 75.07175925755442, 75.07003975743086, 75.06790532331524, 75.06613288922563, 75.06428532305513, 75.06233701159438, 75.06182474740663, 75.06152421908315, 75.06122369075968, 75.06094023790912, 75.05984057745275, 75.05782908674217, 75.0556605016807, 75.05310430338382, 75.05021342581762, 75.0475513595886, 75.04630655761237, 75.04482099146789, 75.0429973309595, 75.04097730251246, 75.03893678349792, 75.04470658579929, 75.04977970880532, 75.05486819973697, 75.05983545481087, 75.06424946456198, 75.06833562523295, 75.07238421986348, 75.07617155975825, 75.07964129585662, 75.08280538098963, 75.08040627704365, 75.0775017390991, 75.07433423887151, 75.0708320593746, 75.0670054458921, 75.06519544576204, 75.0626751059583, 75.05993107745925, 75.05661160552262])
        expected['lon_n'] = np.array([206.09410069520806, 206.06440303269684, 206.03507420040083, 206.00547216053798, 205.98311012119535, 205.95367200587262, 205.90617487038435, 205.85904656511124, 205.81884407165654, 205.77167478524845, 205.72434157430027, 205.6208505479962, 205.52485906940086, 205.42147732612355, 205.31830048852134, 205.21517829243248, 205.07865647130228, 204.93463510246337, 204.79069569589453, 204.64708413840586, 204.5098383172236, 204.34282653163845, 204.17585572718832, 204.00866635668476, 203.84826619421617, 203.68104950295591, 203.48645741350225, 203.29196094669697, 203.10403512187324, 202.90918348523113, 202.7146460372908, 202.47702375606607, 202.24554864509435, 202.0077897600862, 201.77046800718495, 201.53317357504037, 201.3184870690485, 201.0980085626405, 200.8779808487177, 200.65813071971334, 200.4444004402053, 200.18912895016524, 199.9343355733671, 199.6802115551076, 199.43201614104768, 199.17867076435357, 198.88409836582926, 198.58970355222337, 198.30114172016866, 198.00749822737149])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199240493775, 6371.200759506225, 6371.199620246887, 6371.198860740661, 6371.2, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.197721481323, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199620246887])
        expected['lat_s'] = np.array([53.568259325677644, 53.570684042832994, 53.570745514535524, 53.57381226947286, 53.63694370797129, 53.63939233078874, 53.64234638760477, 53.648312557844775, 53.71453124184805, 53.71723258166479, 53.72293237452717, 53.728584356065355, 53.79475864383902, 53.797149210048524, 53.80253481420908, 53.80555034272764, 53.876280366677705, 53.885822140948214, 53.89298017919839, 53.902904444062415, 53.97399305294391, 53.977794053217025, 53.98436469519857, 53.99100022397724, 54.05912794585353, 54.062587436668146, 54.06779887100487, 54.07077683348299, 54.137528272241525, 54.14249040467354, 54.14513710297692, 54.15116816001404, 54.21857871202748, 54.22134152354675, 54.22696276923367, 54.23030273173781, 54.29608086853962, 54.2995232838813, 54.300657095283526, 54.30472788802885, 54.37048553426315, 54.373074175958585, 54.37857589333503, 54.38166655393446, 54.44926152105549, 54.454257804433354, 54.457836823558445, 54.46430842779703, 54.53274009309139, 54.53583416878541])
        expected['lon_s'] = np.array([194.85645875567778, 194.83835875437725, 194.81766328119213, 194.79960426102662, 194.8619912089055, 194.84385022646995, 194.81363346958182, 194.78593022230822, 194.83876856572746, 194.80838788429924, 194.7806026747556, 194.74627414398714, 194.79248720391146, 194.7554402578533, 194.72093414216639, 194.6838325545949, 194.7175190475814, 194.67047270457834, 194.6208308896907, 194.57348401836416, 194.60676070000045, 194.55793850781322, 194.51145224032211, 194.46480204829092, 194.49880272997925, 194.44947510379342, 194.3971558547511, 194.34232309609425, 194.3702858905563, 194.3176661131905, 194.2625464865885, 194.19910768957743, 194.21580067190894, 194.14958881809483, 194.0856582474635, 194.019309789866, 194.03929492337747, 193.9788614096012, 193.91605098999375, 193.85553551394744, 193.87486494929857, 193.79726033994888, 193.721977994917, 193.6442641025406, 193.64837587642094, 193.57261541814717, 193.48701948746856, 193.40356823618936, 193.39930619814726, 193.31327313536178])
        expected['rho_s'] = np.array([260993.70563964843, 260997.98317871094, 260939.79892578124, 260943.5174682617, 260936.7609008789, 260940.98983154297, 260883.2430541992, 260887.30185546874, 260880.59389648438, 260822.9443359375, 260827.24617919922, 260833.4680541992, 260828.60721435546, 260773.1693359375, 260779.22108154296, 260722.71381835936, 260716.9537231445, 260722.81103515625, 260666.2551635742, 260671.65069580078, 260665.50173339844, 260611.47349853514, 260619.5911010742, 260627.53857421875, 260623.6012939453, 260569.79179687498, 260580.38842773438, 260528.353137207, 260526.8948852539, 260537.61303710938, 260485.6749633789, 260498.240234375, 260499.0422729492, 260449.92348632813, 260463.04775390626, 260413.6373168945, 260415.09556884764, 260428.85174560547, 260380.024609375, 260393.1488769531, 260394.48560791014, 260348.21041259766, 260363.76510009766, 260316.34760742186, 260319.3856323242, 260334.77019042967, 260288.59221191405, 260304.36563720703, 260308.54595947266, 260262.78115234376])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_180_90000(self):
        """Test for latitude = 60, longitude = 180, rho = 90000"""
        lat = 60
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([76.10408173267297, 76.10684625173953, 76.10956893589741, 76.11215074740369, 76.10046087863921, 76.10298548731117, 76.10633142623084, 76.10975078968407, 76.09850829831024, 76.10164420891292, 76.10484756763366, 76.10637155359221, 76.09324819887567, 76.09456130274361, 76.09589916604729, 76.09703383122316, 76.0858519577782, 76.08901177404297, 76.09209218935865, 76.09507698202594, 76.08356896704812, 76.08575633512982, 76.08803420321802, 76.09005081657047, 76.07747216944024, 76.07931632051614, 76.08326331608276, 76.08703955692013, 76.07644422597015, 76.07991652338946, 76.08307890097517, 76.09522639241405, 76.09318501962585, 76.10495599688673, 76.11606786089268, 76.12701836167952, 76.12404893693785, 76.13462206977303, 76.14491943372047, 76.1547847882029, 76.15079510395397, 76.15632584963438, 76.16120516602271, 76.16569516162835, 76.15669382607592, 76.16034883105553, 76.16737794948511, 76.17356012445762, 76.1664029399811, 76.17181757244562])
        expected['lon_n'] = np.array([210.36652062483444, 210.3209496026921, 210.27531027865808, 210.22945238857062, 210.2361323135789, 210.1903290650048, 210.12488219237773, 210.05950362164236, 210.0464033188143, 209.98090180467386, 209.91548225280349, 209.78536714911473, 209.70799476619683, 209.57818019083157, 209.44837927584464, 209.31830515329094, 209.2029022770744, 209.03399169889997, 208.8650264792122, 208.6957470708226, 208.58053543990283, 208.38976825638437, 208.19913767664931, 208.00805630442903, 207.87152082292047, 207.68046677145688, 207.46216026520494, 207.24355323062952, 207.0804483132496, 206.8618139579175, 206.64300201766696, 206.376324111713, 206.1658586626282, 205.89930370007932, 205.63206571861338, 205.36486871828248, 205.1691428174266, 204.91654876154118, 204.66400934716913, 204.41118306485194, 204.21634508858816, 203.9237671053015, 203.63105251823143, 203.33840623305306, 203.1048410841953, 202.81280951604225, 202.47729696363288, 202.14119701495488, 201.86483390075784, 201.52930768797012])
        expected['rho_n'] = np.array([6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.2])
        expected['lat_s'] = np.array([63.607727282889485, 63.61192272658717, 63.616282094824925, 63.620808802697354, 63.65649141846879, 63.66083200368633, 63.66785173060581, 63.67501318395057, 63.71338347916041, 63.72023586644522, 63.72724705562824, 63.734806367492155, 63.77318007779658, 63.7802356632092, 63.78758836185072, 63.79532184353847, 63.837266035231515, 63.848252394511476, 63.85968954627667, 63.87123427352128, 63.913164804835986, 63.92195184320321, 63.931257975947354, 63.940465070948534, 63.98023555493822, 63.98908064991339, 63.997493735423554, 64.00612880208175, 64.04487646524322, 64.05300268280824, 64.06148919285198, 64.07065530671815, 64.10971545103416, 64.11848541392845, 64.12745174476139, 64.13662639636401, 64.17392947451603, 64.18095773917197, 64.18836678687414, 64.19577071193444, 64.2331113561269, 64.24197694166958, 64.25124721592057, 64.26039454676652, 64.29947347410274, 64.30838516342232, 64.31840675848201, 64.32880059885149, 64.3687469602123, 64.37840143260411])
        expected['lon_s'] = np.array([198.03858924847336, 198.0241638889463, 198.00841347272026, 197.99345535843793, 198.082042911973, 198.06836887325463, 198.04289226765047, 198.0161725676174, 198.0964136299867, 198.06992615638538, 198.0424141544086, 198.0096975482842, 198.08152381759606, 198.04652592828893, 198.01251158622227, 197.98051898014992, 198.0367587577758, 197.98810049012863, 197.942597769878, 197.8952508985515, 197.9508486383954, 197.901056559346, 197.8549527824484, 197.8046416090221, 197.8622201037253, 197.811266892517, 197.75494515262108, 197.69892394104863, 197.74625715199682, 197.68776341194484, 197.63107284183374, 197.56074921413926, 197.59301502777842, 197.5220220415453, 197.45093343266382, 197.379831163404, 197.41504761876456, 197.34658180252435, 197.28013772227848, 197.21143967960649, 197.2473528142624, 197.15940729850934, 197.07418019804592, 196.98556532375417, 197.00224464570732, 196.91518705454635, 196.8178978400087, 196.72284892751878, 196.7293376072303, 196.63058673221025])
        expected['rho_s'] = np.array([255085.30620117186, 255090.84755859376, 255070.23759765623, 255053.5892211914, 255058.86323242186, 255079.5218017578, 255075.90047607422, 255042.43359375, 255096.65626220702, 255076.07060546873, 255029.69819335936, 255039.17683105468, 255077.4559448242, 255042.26346435546, 255029.26071777343, 255063.86989746094, 255052.44692382813, 255018.15369873046, 255060.1999633789, 255061.58530273437, 255048.87420654297, 255006.00159912108, 255055.36342773438, 255001.89418945313, 255056.48142089843, 254999.43946533202, 255002.6719238281, 255017.44887695313, 255027.60803222656, 254991.92946777344, 255004.2030883789, 255006.02590332032, 254991.61351318358, 254995.42927246093, 254998.97768554688, 255004.17878417968, 254991.15173339844, 254979.1697631836, 255021.50767822264, 255005.90438232422, 255033.8542114258, 254994.8702758789, 255028.28854980468, 254974.89222412108, 254994.0925415039, 255000.26580810547, 254968.50021972656, 255002.11292724608, 255021.0458984375, 254971.2708984375])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_180_90000(self):
        """Test for latitude = 90, longitude = 180, rho = 90000"""
        lat = 90
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([82.34980649276585, 82.35883258775402, 82.36779123462414, 82.37671829186934, 82.3705207489712, 82.37948793357778, 82.3916328636957, 82.40375218060426, 82.4006760341568, 82.41277400672419, 82.42483270570385, 82.44572881569586, 82.45134237741996, 82.47210871381773, 82.49274869171587, 82.51323328281039, 82.52234731648412, 82.54664912955103, 82.57078872562515, 82.5948292839563, 82.60333969966213, 82.62841930052078, 82.65333070797111, 82.6780534314456, 82.68715295096736, 82.71162466498987, 82.73989311041726, 82.76792506054464, 82.78009987274018, 82.80777665303073, 82.83521352292667, 82.87523373570788, 82.89924099686684, 82.93864563884911, 82.97768273127667, 83.01635825056505, 83.03864558471362, 83.0763614624236, 83.11364874576317, 83.15056378379298, 83.17138348104365, 83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544])
        expected['lon_n'] = np.array([239.34851591481595, 239.28043258916927, 239.21185748990234, 239.14295454155524, 239.06115619605518, 238.99288162511172, 238.89602954268094, 238.79876764889997, 238.68896552780285, 238.59211344537206, 238.49471494780767, 238.287869498983, 238.0690302379758, 237.8613924872074, 237.65249798163174, 237.44212815519535, 237.18687032553365, 236.94062834557633, 236.6923919503814, 236.44207917767875, 236.18225878165148, 235.91757529093513, 235.65048757363095, 235.38099562973895, 235.10336210035626, 234.83108343928288, 234.52913443645494, 234.2242347919056, 233.91245031667285, 233.6039443322417, 233.29270627214257, 232.91871243394934, 232.5390358783667, 232.15840309630022, 231.77345363467828, 231.3842421350143, 231.00396452278466, 230.62024449521334, 230.2324536748967, 229.84072866561812, 229.44985059979663, 228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([48.517652038249615, 48.525165246336634, 48.53232328458681, 48.53916030394599, 48.49639990464713, 48.50390969763955, 48.5095480187994, 48.51500192485165, 48.47052714807111, 48.47649331831112, 48.482288733821875, 48.487216715308044, 48.44232871208272, 48.44792263701296, 48.45331848645727, 48.45830452455138, 48.411500653263865, 48.41536654033409, 48.41881578586495, 48.422087446477384, 48.37538944312201, 48.380078367987224, 48.38436431169141, 48.38848974595009, 48.342594289822195, 48.347385667524954, 48.3516818565129, 48.355674102082766, 48.30976498557653, 48.31448806138759, 48.31890036359142, 48.322499873284016, 48.27616728504923, 48.28038151176713, 48.2843976629991, 48.288202078366794, 48.24280181095372, 48.24802690566878, 48.252961717344114, 48.25768479315518, 48.21248601632262, 48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855])
        expected['lon_s'] = np.array([3.5521261090034026, 3.5211221728134223, 3.489967545574878, 3.458780048050954, 3.5427674691800157, 3.511755422140396, 3.4786102216456167, 3.445244320663281, 3.5271468265481967, 3.494132039977882, 3.4609479927821982, 3.4263814726098216, 3.5069271187274316, 3.472532847388186, 3.438059388543251, 3.4035411065818875, 3.4810074046008794, 3.4435687898825122, 3.4061756386108084, 3.368535746755337, 3.4458618124529328, 3.409107497312036, 3.372313908583412, 3.3353341971999053, 3.413228449259081, 3.3763868357628555, 3.3387919804672244, 3.300950171144415, 3.37792661653387, 3.3403176739730758, 3.302593898856861, 3.2626253393812386, 3.3372528400264443, 3.297408931503175, 3.2573996043359443, 3.217312797210316, 3.292921710661888, 3.2540583611713387, 3.215103444457229, 3.1759666739564674, 3.251479324429424, 3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065])
        expected['rho_s'] = np.array([377304.9994140625, 377315.9606079101, 377326.75167236326, 377336.95943603513, 377284.7540161133, 377295.9339477539, 377304.2216796875, 377312.72814941406, 377258.0923095703, 377266.86612548825, 377275.5427246094, 377282.6395507812, 377226.9586303711, 377234.93040771486, 377242.73205566406, 377249.9260986328, 377191.3529785156, 377196.60268554685, 377201.4635253906, 377206.25145263673, 377147.7755493164, 377154.1432495117, 377159.87904052733, 377165.37178955076, 377107.7222290039, 377114.3572753906, 377119.5583740234, 377124.61364746094, 377066.623828125, 377072.55405273434, 377078.1683227539, 377082.00838623045, 377022.75474853517, 377027.34824218747, 377031.45565185545, 377035.73319091793, 376977.743371582, 376983.6006835937, 376989.2878662109, 376994.8292236328, 376936.8394042969, 376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_270_90000(self):
        """Test for latitude = -90, longitude = 270, rho = 90000"""
        lat = -90
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.90688287510915, -72.88874189267358, -72.8686747968921, -72.85101192769844, -72.88099645815478, -72.86271887193584, -72.84415441777173, -72.82800785057384, -72.85974090945771, -72.84296596485618, -72.82429222766535, -72.81035864175851, -72.84452324798693, -72.82994762429811, -72.81564520817611, -72.79945765984317, -72.83621773795619, -72.82401902009855, -72.81206618905102, -72.80037290519198, -72.83520686995902, -72.82410098236858, -72.81326830234494, -72.80268150913142, -72.84099887037519, -72.82978369975805, -72.81889637822104, -72.81034498138018, -72.85094362580674, -72.8417501911839, -72.83282996412785, -72.82743411468354, -72.86945343845747, -72.86344287198787, -72.85766453195004, -72.85213207872232, -72.89609117622052, -72.88967079840072, -72.88348264701267, -72.87751306167809, -72.91960068734372, -72.9163904984338, -72.91345351709069, -72.91068046028767, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99974612706464, -72.99780635334037])
        expected['lon_n'] = np.array([28.701909722686793, 28.64750726594764, 28.593613658301656, 28.539769569527152, 28.67329123006444, 28.618718018596038, 28.547515004045948, 28.47625222534062, 28.59222542235285, 28.5201703417038, 28.44859849693853, 28.36504991546364, 28.46849654553807, 28.38402759607252, 28.299840391910237, 28.216035678341473, 28.29420207075039, 28.18470218552404, 28.07530646068253, 27.965915858482898, 28.043086750820287, 27.934664327935504, 27.826364848455782, 27.718183189739243, 27.795961968912763, 27.68674041389509, 27.568446660112347, 27.44994629310721, 27.51700850547298, 27.39750410065985, 27.27809190340052, 27.13911804435827, 27.185151811816862, 27.04481874512978, 26.90454885769252, 26.764360932525303, 26.817284653309148, 26.684408445648426, 26.551807153101798, 26.419308313392722, 26.47082501520754, 26.3113042395946, 26.1516383224618, 25.99222853742287, 26.015915633464477, 25.855064678510676, 25.67931708097708, 25.503724870247098, 25.51064385187632, 25.333709508974433])
        expected['rho_n'] = np.array([257214.93735351562, 257202.37208251952, 257252.87620849608, 257240.5539794922, 257196.92794189454, 257184.0710205078, 257235.5230102539, 257224.14864501954, 257181.61629638672, 257169.6829345703, 257220.8918823242, 257210.87855224608, 257169.9259765625, 257159.45086669922, 257149.097277832, 257201.91030273438, 257162.5375, 257153.52064208983, 257144.649609375, 257135.90009765624, 257159.49947509766, 257151.30895996094, 257143.19135742186, 257135.24388427733, 257096.86755371094, 257088.45830078123, 257144.67391357422, 257137.96595458983, 257101.07218017578, 257093.92674560545, 257087.02435302734, 257082.60098876953, 257111.15842285156, 257106.24897460936, 257101.4853515625, 257096.8432495117, 257062.33128662108, 257057.17879638672, 257052.19643554688, 257047.38420410155, 257075.91733398437, 257072.9765258789, 257070.06002197266, 257067.4108642578, 257035.18349609376, 257032.02395019532, 257030.27404785156, 257028.6456665039, 257060.654296875, 257058.56413574217])
        expected['lat_s'] = np.array([-68.3394943959866, -68.30527514824485, -68.2709602778547, -68.23645416216777, -68.2116469151023, -68.17697687487532, -68.14398706118416, -68.11077868143954, -68.08737845334309, -68.05406079057175, -68.02046992023364, -67.98901006891657, -67.96737202962595, -67.93565263112043, -67.90382394958817, -67.87180402275914, -67.85184621000437, -67.82172507576462, -67.7914126962281, -67.7609363921515, -67.74045948501981, -67.71996891750976, -67.69928710470296, -67.67850966924777, -67.66795019679094, -67.64704981793071, -67.63134038283968, -67.61556264585695, -67.61003019262924, -67.59412951224144, -67.57810588844859, -67.56609841588772, -67.56437720821688, -67.55228777338596, -67.54010271590664, -67.52782203577897, -67.5294066396664, -67.5205956956371, -67.51167546858105, -67.50270060001165, -67.50416226049404, -67.50283720379505, -67.50144384520436, -67.50009146774872, -67.50918927972316, -67.50768663810575, -67.50597909081327, -67.50410761898067, -67.51286392149663, -67.51101977042072])
        expected['lon_s'] = np.array([152.83723060440772, 152.8151690933886, 152.79298463896438, 152.7708821468102, 152.80068909234816, 152.778627581329, 152.7802258455948, 152.7817967891039, 152.83522252879175, 152.83653392511238, 152.83811852899981, 152.82361120720273, 152.86101332309772, 152.84649234092225, 152.83184841534177, 152.81706788597785, 152.87289785225354, 152.87669543743206, 152.88045204147556, 152.88393543795226, 152.93972442309294, 152.92906932798772, 152.91849619515253, 152.90778645853396, 152.9491637445259, 152.93848132866398, 152.92726615804682, 152.91583242137622, 152.9569091790447, 152.94559838577914, 152.93435589440531, 152.92029936509343, 152.9588216320123, 152.94481974421376, 152.93061295074014, 152.91644713840154, 152.9439181592433, 152.9186464593143, 152.89315619333178, 152.86762494621428, 152.8948364198676, 152.87409996554746, 152.85318592630887, 152.8319030568551, 152.8638820025491, 152.84281769914878, 152.8134342253394, 152.7841873553134, 152.8081476589218, 152.77864124170736])
        expected['rho_s'] = np.array([6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_270_90000(self):
        """Test for latitude = -60, longitude = 270, rho = 90000"""
        lat = -60
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-73.33314132083129, -73.31269173445628, -73.29229678959462, -73.27191550511131, -73.32659799960643, -73.30509656409922, -73.28289844929668, -73.26075497600749, -73.31237754575446, -73.28907294030637, -73.26583663674998, -73.24261399357195, -73.29441414823734, -73.27012599554877, -73.24601542777862, -73.22204146379187, -73.27031724084551, -73.24529142772658, -73.2190771616921, -73.19298583906263, -73.24056493682096, -73.21537519916194, -73.18905165010071, -73.16296032747127, -73.2118508215502, -73.184885234707, -73.15842508186236, -73.13363149517522, -73.18246734774081, -73.15547444014092, -73.12875474010784, -73.10401579493404, -73.15291994939133, -73.12741602303052, -73.10080560602415, -73.07591639668863, -73.12595436254813, -73.10017722862051, -73.07610764198537, -73.05231126291704, -73.10081926640248, -73.07602567971534, -73.05154628173003, -73.02736741206817, -73.07687262317242, -73.05194243270188, -73.02680733655623, -73.00195910835572, -73.05084960243468, -73.02520907229044])
        expected['lon_n'] = np.array([354.43362441040574, 354.6137501591973, 354.79223666258804, 354.9692751658748, 355.22065344808794, 355.3964625173241, 355.5655506804169, 355.7325078244887, 355.9888858051743, 356.15428566611536, 356.3176911118187, 356.4490219891797, 356.66111302328693, 356.7907773344905, 356.9185291927264, 357.0453521452352, 357.2571699717756, 357.3612347339699, 357.4744246288954, 357.5852649387464, 357.7932851801083, 357.8625159775355, 357.941035832234, 358.01805304531507, 358.1835621892828, 358.2590221192331, 358.30483902818554, 358.33855284192873, 358.47313488933463, 358.5162197226191, 358.55788387655616, 358.5541682536477, 358.65197656256225, 358.6468402603064, 358.65066516624165, 358.6432339204247, 358.73571468178665, 358.73503166286963, 358.7233110582539, 358.7105522648843, 358.8106828381167, 358.7616147591193, 358.7116724159082, 358.66074652545655, 358.7122188310418, 358.6601454688096, 358.5885104447945, 358.51600115656566, 358.5451250831866, 358.4714136816638])
        expected['rho_n'] = np.array([256521.2955078125, 256522.3405883789, 256523.2641479492, 256524.21201171874, 256469.4546508789, 256468.6769165039, 256468.89565429688, 256469.13869628907, 256475.33626708985, 256473.95092773438, 256472.37115478516, 256480.65888671874, 256433.4115234375, 256440.45974121094, 256447.1677001953, 256453.65692138672, 256467.80196533204, 256412.72864990233, 256419.2664794922, 256425.5126586914, 256437.7133666992, 256392.1916015625, 256408.2566772461, 256424.27314453124, 256384.51147460938, 256399.31273193358, 256424.68631591796, 256388.1814086914, 256357.63103027342, 256382.12966308594, 256406.4338623047, 256381.0359741211, 256361.78704833984, 256335.51420898436, 256371.41151123046, 256345.3817138672, 256328.32016601562, 256366.50206298826, 256342.78116455078, 256319.1088745117, 256363.51264648436, 256351.04459228515, 256338.84388427733, 256326.81330566405, 256320.9316894531, 256308.26920166015, 256301.77998046874, 256295.58240966796, 256295.6553222656, 256288.94736328124])
        expected['lat_s'] = np.array([-68.30381348776245, -68.2656737114371, -68.22747929359838, -68.18917559273297, -68.15971015665352, -68.12135181427473, -68.08385407573135, -68.04623339378293, -68.01717776905369, -67.9791199549984, -67.94077527299794, -67.90462991191023, -67.87684470236661, -67.84011194501028, -67.80309231970881, -67.76569020381382, -67.73674386211133, -67.69904121789284, -67.66090144156749, -67.6222015897302, -67.59086468181818, -67.56272430243772, -67.53388724376194, -67.50408029822401, -67.48010633423726, -67.44607833179225, -67.41437259366504, -67.37444330777714, -67.31533485069986, 40.303598240500584, 43.36239600745324, 49.5139888079523, 49.61990455141169, 49.66184874310474, 49.709072671026206, 49.757638731119584, 49.86247530469011, 49.906601741822975, 49.95197810357396, 49.99859072956471, 50.107245378881274, 50.15747117494296, 50.20880346165019, 50.26495103172228, 50.38348896476785, 50.43615313836324, 50.49570555773658, 50.556272260201666, 50.6796698728417, 50.74032878286058])
        expected['lon_s'] = np.array([159.84434899468047, 159.78280899025867, 159.72193834437553, 159.6615867928693, 159.66665479323345, 159.60728678896768, 159.5711550882583, 159.535105349819, 159.56355991790127, 159.52848006632408, 159.49409689404223, 159.44582111698858, 159.46113440110776, 159.41406073734802, 159.36783401704537, 159.32286405155, 159.36045741274177, 159.33823197718255, 159.31839710783282, 159.30196367268977, 159.34991160066326, 159.3325902409281, 159.31984510793686, 159.31281001309176, 159.37209605508747, 159.3817676029522, 159.40029107598127, 159.43034390832932, 159.5243956132004, 205.79589463604532, 224.34003581754834, 223.30464744126613, 223.49329726614187, 223.5007831534722, 223.50484028583918, 223.50815975777581, 223.69246558233945, 223.69376331828175, 223.69444633719874, 223.69463758249552, 223.87390272745165, 223.8489998577378, 223.8234412898636, 223.7946042311878, 223.9441580532544, 223.91471993793166, 223.8748316331788, 223.83421932837388, 223.9714924703128, 223.92754703319295])
        expected['rho_s'] = np.array([6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 171311.13760986328, 190872.5910888672, 257498.30001220704, 257465.9754272461, 257380.1816040039, 257350.40895996094, 257315.31369628906, 257278.22548828126, 257250.98048095702, 257219.2148925781, 257183.00163574217, 257128.1956665039, 257075.94163818358, 257019.9690673828, 257020.67388916016, 257001.91104736328, 256935.7064086914, 256919.81146240234, 256900.1979736328, 256859.95021972657, 256834.4551147461])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_270_90000(self):
        """Test for latitude = -30, longitude = 270, rho = 90000"""
        lat = -30
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-68.01682259921685, -68.14460177820945, -68.26351537165937, -68.37592662501945, -68.57065531825651, -68.66775328749742, -68.7654249926286, -68.85634847085981, -69.03068221923522, -69.10902448901524, -69.1827905320514, -69.24575121582058, -69.39357016983797, -69.44770624919951, -69.49871410192117, -69.54659372800296, -69.68213200189268, -69.7251758540421, -69.76510513993, -69.80197450106974, -69.922950811649, -69.94296326591714, -69.96119987100107, -69.9780021363593, -70.07933482288561, -70.09162916339162, -70.09791293742805, -70.10303557930555, -70.19290720840456, -70.19468305758878, -70.19606275580114, -70.19464207645376, -70.27817529000302, -70.27398155385262, -70.26976049694557, -70.26517060982331, -70.34466035138391, -70.33800774713234, -70.33110925607062, -70.32392389706376, -70.40084548749644, -70.39005378860782, -70.37969922182609, -70.36863431537066, -70.44136216965296, -70.42846677249997, -70.40777129931485, -61.079167232803115, -61.38469525475614, -61.78592788735935])
        expected['lon_n'] = np.array([339.9638140122203, 340.54681163901597, 341.10625877355335, 341.66739979500494, 342.2426656476601, 342.77610342183806, 343.3289662540156, 343.8593167826887, 344.42540286109937, 344.94059036981514, 345.45861923722555, 345.90304598613994, 346.40632164494315, 346.8398200911855, 347.2781269906036, 347.70974030463503, 348.21962758655485, 348.6627429391486, 349.0982358006286, 349.5258602841848, 350.0052576016496, 350.34805113571423, 350.6967459532217, 351.0289936752078, 351.432876421209, 351.75859448234854, 352.0215840861507, 352.2912672753395, 352.6280229221778, 352.8927883751642, 353.1434016761903, 353.3600825974198, 353.63859039101624, 353.85092731193356, 354.049767779051, 354.2455756421769, 354.50875649127585, 354.6821066924108, 354.85305266695786, 355.0215944149171, 355.2709782818926, 355.40116168747306, 355.5194878846543, 355.6451360446259, 355.85422179549835, 355.9771105590452, 356.0737713961792, 339.63891557378116, 338.66799052288553, 338.0946644439546])
        expected['rho_n'] = np.array([255098.74642333985, 255126.59903564453, 255100.69075927735, 255140.7926879883, 255160.84365234376, 255148.64294433594, 255189.1580444336, 255172.67979736326, 255194.31053466795, 255182.52299804686, 255234.33955078124, 255192.85228271483, 255248.99498291014, 255209.6707885742, 255232.37091064453, 255255.97028808593, 255293.78762207032, 255296.3638671875, 255299.59632568358, 255303.60651855468, 255281.39248046876, 255272.61866455077, 255325.18864746092, 255316.5849609375, 255342.85780029296, 255332.65003662108, 255296.04791259766, 255320.95971679688, 255319.30703125, 255342.42032470703, 255303.3634765625, 255351.99617919922, 255312.50185546876, 255359.21452636717, 255343.5626220703, 255327.3517211914, 255367.9154296875, 255370.95345458985, 255373.14083251954, 255374.84212646485, 255350.24627685547, 255369.66533203123, 255326.5496826172, 255344.68061523436, 255337.0977050781, 255353.25999755858, 255381.6716064453, 238604.9203613281, 232817.3857055664, 229246.44262695312])
        expected['lat_s'] = np.array([44.16333131028562, 44.34355609682016, 44.534032997298894, 44.724673822317705, 44.97042061356018, 45.17332845832833, 45.40717022984747, 45.649525832167036, 45.93458036208293, 46.18143364397095, 46.43085507698689, 46.66339228237456, 46.94103264194641, 47.180966942205195, 47.4183706573765, 47.6618805616658, 47.98810064170936, 48.276785417169116, 48.56095202249291, 48.849162099805355, 49.17407761371745, 49.422962876883545, 49.666667441564186, 49.913312402682514, 50.194162951164266, 50.43616338364699, 50.65747175803916, 50.87736286817855, 51.125694886021876, 51.34473222251501, 51.55847274730681, 51.76951876247103, 52.01134502512991, 52.21788653053652, 52.42680494177428, 52.630761220580034, 52.849986387275344, 53.035613853443365, 53.2201689799117, 53.40376446480164, 53.61591697061141, 53.79785613462762, 53.97878784574126, 54.16203840607812, 54.36979568515698, 54.54590528271659, 54.730275994077324, 54.913529969508765, 55.12079547496739, 55.300292846355326])
        expected['lon_s'] = np.array([232.46772870343173, 232.5826671267847, 232.6908846439944, 232.79929340650085, 233.06124482154917, 233.15659426236255, 233.2444578158456, 233.32445299140477, 233.5619250084678, 233.63017225865454, 233.69374765944903, 233.76232275871595, 233.98353892555434, 234.03790723134765, 234.09167448049396, 234.137354785663, 234.31862800623517, 234.33264355441202, 234.34537502702491, 234.34821638571967, 234.50063888723767, 234.47268975315396, 234.44493186436702, 234.4089777485761, 234.5238888511724, 234.4783724705434, 234.4128026545113, 234.34400898919094, 234.43200914645737, 234.35133095198117, 234.2721553991224, 234.18442844942274, 234.25005290696822, 234.15519523977508, 234.05320685508846, 233.95310360261274, 234.014821191953, 233.91529167536757, 233.8136038190044, 233.70951173605343, 233.76191294736577, 233.64290373126747, 233.52154493009468, 233.39338326050856, 233.42767081014202, 233.29833434801864, 233.15894384743703, 233.01731304480765, 233.03987998982538, 232.89212933769966])
        expected['rho_s'] = np.array([257315.96990966797, 257171.6029663086, 257071.1780151367, 256897.91337890623, 256805.21716308594, 256663.32924804688, 256537.23906249998, 256456.33038330078, 256335.92738037108, 256231.3464111328, 256115.26955566407, 256017.5909790039, 255941.12996826172, 255879.59173583984, 255751.4843017578, 255673.0546508789, 255608.52700195313, 255564.39057617186, 255449.4560180664, 255385.4387573242, 255345.40974121093, 255318.89385986328, 255227.24272460936, 255189.9600830078, 255181.69665527344, 255136.36932373047, 255114.93302001953, 255090.06982421875, 255038.52061767576, 255069.31403808593, 255036.81932373045, 255001.89418945313, 255003.546875, 254964.66015625, 254983.15565185546, 254938.19288330077, 254966.19132080078, 254950.70954589843, 254932.6515258789, 254911.67700195312, 254934.4500366211, 254902.14975585937, 254867.22462158202, 254890.72678222656, 254900.18111572266, 254860.8083129883, 254864.8428100586, 254866.61701660155, 254857.43002929687, 254856.45786132812])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_270_90000(self):
        """Test for latitude = 0, longitude = 270, rho = 90000"""
        lat = 0
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([63.28579143900281, 63.2998001569905, 63.31382082780925, 63.32796102693847, 63.38323091770221, 63.39707229605524, 63.419237967459225, 63.44141729924155, 63.50460679434792, 63.52641900346236, 63.548466854103154, 63.575837129654694, 63.64458127610358, 63.67175176862189, 63.69929962909184, 63.726526470670805, 63.80103871187377, 63.83467739353608, 63.86843048086699, 63.90208111536035, 63.97639698862467, 64.01019617973247, 64.04364190855074, 64.07737962795602, 64.15172623707161, 64.18547932440252, 64.22626067638937, 64.267274254808, 64.34901112860555, 64.38999226362563, 64.43089143637566, 64.4889429216789, 64.58758792876681, 64.64548573481372, 64.70399313524403, 64.76226830924259, 64.86146997674786, 64.92077480176378, 64.97975519279413, 65.03908904611404, 65.13842390230812, 65.20269256730334, 65.26700904362278, 65.33140406711766, 65.43545004629175, 65.50000216413754, 65.57040433900745, 65.64105923087665, 65.7512182293579, 65.82218047973976])
        expected['lon_n'] = np.array([263.1538288705747, 263.1403870582882, 263.1267813214615, 263.11317558463486, 262.9121767777397, 262.89857104091305, 262.87390039763096, 262.8492570751055, 262.63495305970724, 262.6103370579385, 262.5856390938997, 262.5361065620388, 262.2943725069337, 262.24473069204606, 262.19484299034826, 262.144654760327, 261.9046692336495, 261.85934409831725, 261.81371843466155, 261.7678195634391, 261.52384520628624, 261.4834924486698, 261.44232006835296, 261.40128429181954, 261.15843008569055, 261.1168478940235, 261.0753476646265, 261.0334922653927, 260.7862394174382, 260.74405616912424, 260.70157239248675, 260.65515442688735, 260.3990769745252, 260.35203063152215, 260.3043832318721, 260.25640798314197, 259.99158788864224, 259.94011558305704, 259.88804222082484, 259.8357502925392, 259.5647557070298, 259.49582543792604, 259.4262394706619, 259.3559431637242, 259.0622723501703, 258.9908012506953, 258.91771822657614, 258.8440068250534, 258.54151140709183, 258.46607879789826])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.198860740661, 6371.2, 6371.2, 6371.199620246887, 6371.198860740661, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([-71.89994906690919, -71.89451223632986, -71.88899344348047, -71.8834336694961, -71.84185147782907, -71.83633268497968, -71.83156521293901, -71.82683872203336, -71.78529751150134, -71.78057102059569, -71.77576256742, -71.77447849185606, -71.73573765888375, -71.7343306399147, -71.73307388510744, -71.7315985642467, -71.6907403726317, -71.68803561772037, -71.68528988167401, -71.68265342865439, -71.64104391623067, -71.64597531281143, -71.6509476905272, -71.65587908710793, -71.62127734877265, -71.6261677642184, -71.63456889689752, -71.64281976541488, -71.61101840463931, -71.61937855618339, -71.62772504734914, -71.63758784051066, -71.60677002697557, -71.61679674467715, -71.62676882086535, -71.63674089705358, -71.6106768951808, -71.62585357551657, -71.64094829358231, -71.6561659550531, -71.62956919842506, -71.65189025663267, -71.67410203181353, -71.69638210888613, -71.67639697537467, -71.69886829774401, -71.72118935595162, -71.74356505567258, -71.72311546929754, -71.74568241431527])
        expected['lon_s'] = np.array([176.50442347477053, 176.48126913348418, 176.45799184879277, 176.43501509242486, 176.24717122987116, 176.2239349263148, 176.23539598374205, 176.24689802230438, 176.09310948295237, 176.1045295592446, 176.11601793742858, 176.11470654110795, 175.9457549817985, 175.94408841564103, 175.94291362310378, 175.9410011701362, 175.79924742410174, 175.82690969024029, 175.8543533903254, 175.8816604866271, 175.73854070275866, 175.7625829686371, 175.78744485721595, 175.81215648163305, 175.6633676407535, 175.6877650764688, 175.718145757897, 175.74904553370214, 175.60436846670294, 175.63526824250806, 175.6652117918294, 175.70579677587762, 175.56883782264052, 175.6091359387436, 175.64936575295496, 175.690264925705, 175.5413941225554, 175.5714879360385, 175.60192325898007, 175.6320717139765, 175.48123381634593, 175.52899049902265, 175.57614612505245, 175.62380718508078, 175.48754491113903, 175.53569774478763, 175.57704771002287, 175.61858892055488, 175.47342007993544, 175.51489298857575])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.198101234436, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198860740661, 6371.2, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.2, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.198860740661])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_270_90000(self):
        """Test for latitude = 30, longitude = 270, rho = 90000"""
        lat = 30
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([65.77936373138023, 65.79545053442291, 65.81158514878976, 65.8277265933458, 65.85181496300115, 65.86785395471964, 65.88976861667162, 65.91165083522503, 65.9414748562359, 65.96321876345864, 65.98501389710015, 66.01286740853546, 66.04845269411123, 66.07605178099996, 66.10366623581432, 66.13130288874348, 66.17031180664071, 66.20148649755996, 66.23275510358026, 66.26400151148579, 66.30296603315341, 66.335631412859, 66.36835314162524, 66.4011380496413, 66.44156764688589, 66.4742979133886, 66.51409230304039, 66.55395328703658, 66.60148969611257, 66.64133701973043, 66.68124239995623, 66.73645594165932, 66.79943028580684, 66.85472237468538, 66.91006739753, 66.96552341094862, 67.03036239673955, 67.0874747310319, 67.14465195212138, 67.20189406000796, 67.26681500806893, 67.32728267279106, 67.38786303563448, 67.44851170036961, 67.51679310150223, 67.57736492660919, 67.64327454455253, 67.70930710590092, 67.78295020553199, 67.84907155933959])
        expected['lon_n'] = np.array([261.05196109690837, 261.02652547243923, 261.0008712819167, 260.97508048761074, 260.8796217637706, 260.85426810157156, 260.8201717972348, 260.78602085138476, 260.6816009193536, 260.6478051433404, 260.61365419749035, 260.5513628722598, 260.4179829381478, 260.3556642921606, 260.29304511784994, 260.23007077370244, 260.10332978346366, 260.0483331002667, 259.9928992849629, 259.93732886587566, 259.80905791326285, 259.75591904152014, 259.70237035842723, 259.6484118639841, 259.52101517558503, 259.4670020396286, 259.41189607340493, 259.3563256543177, 259.2261149279806, 259.1703532635966, 259.11418178786244, 259.05459521754324, 258.91900230214014, 258.8589785997141, 258.79849044442443, 258.73745587400117, 258.59440439202444, 258.5283428023721, 258.46165283531604, 258.39444377388315, 258.2483870086716, 258.1679000594922, 258.086620808369, 258.00465853832884, 257.84207271532586, 257.759318143342, 257.67680945816824, 257.59350847105077, 257.42778076102957, 257.34344158515825])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([-72.47346639113681, -72.46764706996396, -72.46117205063078, -72.45502488037778, -72.43689755832057, -72.43070940693252, -72.42774510483275, -72.4245895574362, -72.40968608466724, -72.40644857500064, -72.40333400873914, -72.40389408425108, -72.39267891363392, -72.39360781936102, -72.39404495146792, -72.39485091378998, -72.38405921490136, -72.38568479992381, -72.38685959246106, -72.38825295105173, -72.37736562951474, -72.3844826866299, -72.39165438525839, -72.39889438577862, -72.39399030995455, -72.40101174442131, -72.41128434893301, -72.42129740625626, -72.41952155707204, -72.4296712181787, -72.43984820004201, -72.45188299335956, -72.45193763487293, -72.46395876781216, -72.4758159762113, -72.4879327317989, -72.49219476984098, -72.50858722384902, -72.5251299420188, -72.54129016959504, -72.54574345293389, -72.57116541702467, -72.59673764527722, -72.62203666596292, -72.63500036500761, -72.66036768758505, -72.68598089697258, -72.71149848371175, -72.72448950351313, -72.74996610911728])
        expected['lon_s'] = np.array([169.23931517916805, 169.21303261124183, 169.18549328850835, 169.15845939977342, 169.12877539764057, 169.10123607490706, 169.11951366112604, 169.13669841707778, 169.15208000308866, 169.16948332509384, 169.18685932634236, 169.19659917609877, 169.2032927614854, 169.21340144145702, 169.22248559305314, 169.2326079334031, 169.27901223862415, 169.3295009969689, 169.3798394911519, 169.43032824949663, 169.4765003282859, 169.51224953840176, 169.54883203159633, 169.58567407197938, 169.6168060742163, 169.6532929447625, 169.69449264583602, 169.73569234690956, 169.77163280232216, 169.8124909939372, 169.85433273279267, 169.91043590663517, 169.96040557060297, 170.01708248033574, 170.07332225796162, 170.13009479034278, 170.1629753210072, 170.20295924840846, 170.2434486098083, 170.28298174472434, 170.31564370933535, 170.38308499720003, 170.451701077602, 170.5196614598436, 170.57840108670572, 170.6464161104607, 170.7138164171904, 170.7811484220284, 170.8376340864644, 170.9049797516807])
        expected['rho_s'] = np.array([6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.2, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.198101234436, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_270_90000(self):
        """Test for latitude = 60, longitude = 270, rho = 90000"""
        lat = 60
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([73.28380185549996, 73.2989409697953, 73.31405276333395, 73.32917992479824, 73.33394056664973, 73.34901308660066, 73.36881209745724, 73.38860427812463, 73.39815117503703, 73.41784431796147, 73.43754770616965, 73.46598349123171, 73.48397420950553, 73.5122016737979, 73.54042743054298, 73.56861732879491, 73.5904380756458, 73.62274145532538, 73.65501409915369, 73.68729698826576, 73.70899820680619, 73.74315598284542, 73.77731717397924, 73.81150056322785, 73.8348598101893, 73.86897148245163, 73.90937375893955, 73.94976237504913, 73.97939685831052, 74.01965399327858, 74.05988380748995, 74.11575304735337, 74.16061031472742, 74.21643515836124, 74.27222926614377, 74.32800800600069, 74.37341681115022, 74.42961560764108, 74.48577513054423, 74.54187659683942, 74.58660921326111, 74.64586622695286, 74.70500541988142, 74.76408997129661, 74.8117185879264, 74.87047016761956, 74.93623635159032, 75.00194106385855, 75.05608738850383, 75.12166232717783])
        expected['lon_n'] = np.array([255.69244825898278, 255.6603463698837, 255.6281625185146, 255.5957054595787, 255.5161747368831, 255.48410016854072, 255.43872039169514, 255.3930400865261, 255.3000675515439, 255.25479705772506, 255.20919871482604, 255.11868504794504, 254.98044201914396, 254.8898737107496, 254.79889559100502, 254.70750765991025, 254.56483866852705, 254.47014492587397, 254.37487744733065, 254.27900891214034, 254.13437282627615, 254.04006157421662, 253.9451492655101, 253.8494446548599, 253.70428947461878, 253.6081477318617, 253.50621398868842, 253.40346062281475, 253.25065563070322, 253.14708264212913, 253.0426353893413, 252.92463704124017, 252.7554669158773, 252.63596592615875, 252.5155360307131, 252.3938767012168, 252.21962491511147, 252.09608045340426, 251.97130655764647, 251.84549447313483, 251.66678940369061, 251.51595150606008, 251.36396613664894, 251.21036884259368, 251.00267645031195, 250.8470574202624, 250.68367929531567, 250.51852532118477, 250.2981287770468, 250.1302154064912])
        expected['rho_n'] = np.array([6371.200759506225, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.200759506225, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.200759506225, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([43.06176181604055, 43.06516325024722, 43.06831196745459, 43.0710884393522, 43.039754946534764, 43.043245173200646, 43.04469658839927, 43.04575526772062, 43.01245809551681, 43.01424418998477, 43.01567511461589, 43.015873190101814, 42.98130901780697, 42.982295980142034, 42.982767263194766, 42.98299948962654, 42.946140373770575, 42.9450168076521, 42.94352782641304, 42.94165976967504, 42.904315710388, 42.90346876693092, 42.90213688004277, 42.900586427101175, 42.863129669692825, 42.86209148093898, 42.85979995247244, 42.85723180134452, 42.81833728911588, 42.816288232364876, 42.813880590682444, 42.809761986612926, 42.76891062518708, 42.76532136077824, 42.76127105860042, 42.75681436016699, 42.716202055362096, 42.713090904195155, 42.70949139450256, 42.705570865918965, 42.66431310823751, 42.657342900189505, 42.64994922041297, 42.642114993434966, 42.59650640525221, 42.58910248019191, 42.57971097008314, 42.5698584222054, 42.52159289043551, 42.51215356900255])
        expected['lon_s'] = np.array([337.24621834525567, 337.2166299657712, 337.18701426553, 337.1574532068022, 337.2558079308504, 337.2260829475825, 337.1947733604271, 337.16340913175844, 337.25996068586574, 337.2284871741703, 337.197177587015, 337.16447464126895, 337.25963283678556, 337.22690257028285, 337.1941449830235, 337.16149667879085, 337.2541413646929, 337.2187883055489, 337.18362649170166, 337.14838271558443, 337.2409727599731, 337.2056743423425, 337.17037592471183, 337.1351048278379, 337.22758558919986, 337.19220520929923, 337.1557319991313, 337.11912218518006, 337.21037351249146, 337.17379101929686, 337.1371265638322, 337.0982218063198, 337.1870415862867, 337.1478909419642, 337.108794939155, 337.0697808986159, 337.15928369749975, 337.1208980343643, 337.08248505047214, 337.04415402885, 337.1336021862205, 337.0927576549838, 337.0520497275306, 337.01123251705053, 337.0982491270765, 337.0573772750832, 337.01516670601245, 336.97287417467174, 337.058415463837, 337.0161229324963])
        expected['rho_s'] = np.array([375298.49333496095, 375321.387890625, 375344.16092529293, 375366.18052978517, 375277.3243774414, 375300.0488037109, 375321.60662841797, 375342.6297607422, 375252.0480102539, 375273.72735595703, 375294.92061767576, 375315.4576660156, 375223.7579223633, 375245.2671386719, 375265.82849121094, 375286.5113647461, 375192.9158935547, 375212.7481201172, 375232.26439208986, 375251.5376220703, 375157.1644165039, 375177.2639892578, 375196.9260864258, 375215.956274414, 375121.51015625, 375141.19655761716, 375160.0080078125, 375178.28476562497, 375082.30748291017, 375101.3376708984, 375119.66303710936, 375136.9676269531, 375039.5077880859, 375057.2255493164, 375074.1412719726, 375090.95977783203, 374993.42702636716, 375011.41213378904, 375028.52229003905, 375045.65675048827, 374947.4434814453, 374961.6614379883, 374975.4176147461, 374988.8335327148, 374886.56146240234, 374900.1232055664, 374912.1780883789, 374923.60106201173, 374818.55831298826, 374830.58889160154])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_270_90000(self):
        """Test for latitude = 90, longitude = 270, rho = 90000"""
        lat = 90
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([82.34980649276585, 82.35883258775402, 82.36779123462414, 82.37671829186934, 82.3705207489712, 82.37948793357778, 82.3916328636957, 82.40375218060426, 82.4006760341568, 82.41277400672419, 82.42483270570385, 82.44572881569586, 82.45134237741996, 82.47210871381773, 82.49274869171587, 82.51323328281039, 82.52234731648412, 82.54664912955103, 82.57078872562515, 82.5948292839563, 82.60333969966213, 82.62841930052078, 82.65333070797111, 82.6780534314456, 82.68715295096736, 82.71162466498987, 82.73989311041726, 82.76792506054464, 82.78009987274018, 82.80777665303073, 82.83521352292667, 82.87523373570788, 82.89924099686684, 82.93864563884911, 82.97768273127667, 83.01635825056505, 83.03864558471362, 83.0763614624236, 83.11364874576317, 83.15056378379298, 83.17138348104365, 83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544])
        expected['lon_n'] = np.array([239.34851591481595, 239.28043258916927, 239.21185748990234, 239.14295454155524, 239.06115619605518, 238.99288162511172, 238.89602954268094, 238.79876764889997, 238.68896552780285, 238.59211344537206, 238.49471494780767, 238.287869498983, 238.0690302379758, 237.8613924872074, 237.65249798163174, 237.44212815519535, 237.18687032553365, 236.94062834557633, 236.6923919503814, 236.44207917767875, 236.18225878165148, 235.91757529093513, 235.65048757363095, 235.38099562973895, 235.10336210035626, 234.83108343928288, 234.52913443645494, 234.2242347919056, 233.91245031667285, 233.6039443322417, 233.29270627214257, 232.91871243394934, 232.5390358783667, 232.15840309630022, 231.77345363467828, 231.3842421350143, 231.00396452278466, 230.62024449521334, 230.2324536748967, 229.84072866561812, 229.44985059979663, 228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([48.517652038249615, 48.525165246336634, 48.53232328458681, 48.53916030394599, 48.49639990464713, 48.50390969763955, 48.5095480187994, 48.51500192485165, 48.47052714807111, 48.47649331831112, 48.482288733821875, 48.487216715308044, 48.44232871208272, 48.44792263701296, 48.45331848645727, 48.45830452455138, 48.411500653263865, 48.41536654033409, 48.41881578586495, 48.422087446477384, 48.37538944312201, 48.380078367987224, 48.38436431169141, 48.38848974595009, 48.342594289822195, 48.347385667524954, 48.3516818565129, 48.355674102082766, 48.30976498557653, 48.31448806138759, 48.31890036359142, 48.322499873284016, 48.27616728504923, 48.28038151176713, 48.2843976629991, 48.288202078366794, 48.24280181095372, 48.24802690566878, 48.252961717344114, 48.25768479315518, 48.21248601632262, 48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855])
        expected['lon_s'] = np.array([3.5521261090034026, 3.5211221728134223, 3.489967545574878, 3.458780048050954, 3.5427674691800157, 3.511755422140396, 3.4786102216456167, 3.445244320663281, 3.5271468265481967, 3.494132039977882, 3.4609479927821982, 3.4263814726098216, 3.5069271187274316, 3.472532847388186, 3.438059388543251, 3.4035411065818875, 3.4810074046008794, 3.4435687898825122, 3.4061756386108084, 3.368535746755337, 3.4458618124529328, 3.409107497312036, 3.372313908583412, 3.3353341971999053, 3.413228449259081, 3.3763868357628555, 3.3387919804672244, 3.300950171144415, 3.37792661653387, 3.3403176739730758, 3.302593898856861, 3.2626253393812386, 3.3372528400264443, 3.297408931503175, 3.2573996043359443, 3.217312797210316, 3.292921710661888, 3.2540583611713387, 3.215103444457229, 3.1759666739564674, 3.251479324429424, 3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065])
        expected['rho_s'] = np.array([377304.9994140625, 377315.9606079101, 377326.75167236326, 377336.95943603513, 377284.7540161133, 377295.9339477539, 377304.2216796875, 377312.72814941406, 377258.0923095703, 377266.86612548825, 377275.5427246094, 377282.6395507812, 377226.9586303711, 377234.93040771486, 377242.73205566406, 377249.9260986328, 377191.3529785156, 377196.60268554685, 377201.4635253906, 377206.25145263673, 377147.7755493164, 377154.1432495117, 377159.87904052733, 377165.37178955076, 377107.7222290039, 377114.3572753906, 377119.5583740234, 377124.61364746094, 377066.623828125, 377072.55405273434, 377078.1683227539, 377082.00838623045, 377022.75474853517, 377027.34824218747, 377031.45565185545, 377035.73319091793, 376977.743371582, 376983.6006835937, 376989.2878662109, 376994.8292236328, 376936.8394042969, 376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_0_120000(self):
        """Test for latitude = -90, longitude = 0, rho = 120000"""
        lat = -90
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.8068582532915, -75.79483712035227, -75.78302089308815, -75.77143689225582, -75.79725500731846, -75.78517923286589, -75.77498859062422, -75.76505749557103, -75.79554746002594, -75.78229689303612, -75.77209259041612, -75.76370511811535, -75.79595727137615, -75.78707802545514, -75.77563062840619, -75.76721583534874, -75.80199515860244, -75.79550647889093, -75.78922270485452, -75.78317115724988, -75.81811440504367, -75.80915319685263, -75.80321093227471, -75.79751455450693, -75.83300421743431, -75.82681606604626, -75.82226716005906, -75.81792315974693, -75.8521833886237, -75.84737493544802, -75.84274406719075, -75.84073599157475, -75.8803784095175, -75.87787856028129, -75.87555629596349, -75.87346625807746, -75.91268520429168, -75.90675660009211, -75.90379229799231, -75.90104656194598, -75.94038845156524, -75.939896677945, -75.93958248924318, -75.93944588545978, -75.98164279415212, -75.98095977523514, -75.98188868096224, -75.98302249236446, -76.02673570305254, -76.02726845780782])
        expected['lon_n'] = np.array([29.12900145148769, 29.07033012651728, 29.011949084586597, 28.953841250222712, 29.085185787962057, 29.02635736864074, 28.95007440089545, 28.87401341429819, 28.987090611102335, 28.910336360304317, 28.833732373668038, 28.746510857966975, 28.84814065772218, 28.760003896672337, 28.67224962621601, 28.584560242556805, 28.660604153681142, 28.547062504013432, 28.433664288318298, 28.320303638663596, 28.395159096872348, 28.284089975684186, 28.17301743940144, 28.06216176162484, 28.139071399226477, 28.027148504392063, 27.906619571203432, 27.785996722913715, 27.853204076799347, 27.73142009635073, 27.61010056876567, 27.4698562945378, 27.516363052596414, 27.37486373110855, 27.233531749255356, 27.092170739098187, 27.145879931636582, 27.012221667315895, 26.878566818089794, 26.74513053491713, 26.797386604709608, 26.63698814978832, 26.47674337412336, 26.316554947519048, 26.3411419209838, 26.179698447119502, 26.004160877902883, 25.82866770491587, 25.83702444136538, 25.660194258848335])
        expected['rho_n'] = np.array([259169.2380126953, 259157.98516845703, 259146.90245361326, 259136.08708496092, 259160.19685058593, 259148.8953979492, 259139.34384765624, 259130.05964355468, 259093.5061279297, 259146.1247192383, 259136.62177734374, 259128.7958251953, 259093.84638671874, 259085.6072631836, 259139.8056274414, 259131.80954589843, 259099.02318115233, 259092.77700195313, 259086.7981689453, 259080.9651611328, 259048.32462158203, 259104.88049316406, 259099.2662231445, 259093.7734741211, 259061.69193115234, 259055.8103149414, 259051.50847167967, 259047.35245361328, 259079.1180419922, 259074.59746093748, 259070.2227050781, 259068.30267333984, 259039.98828125, 259037.58216552733, 259035.39478759764, 259033.4018432617, 259004.60136718748, 259064.02513427733, 259061.1329345703, 259058.45947265625, 259029.70760498047, 259029.14860839842, 259028.68682861328, 259028.4680908203, 259002.31677246094, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.40144042968, 258978.8389160156])
        expected['lat_s'] = np.array([-67.90639210071609, -67.87263730583788, -67.83869126566293, -67.80471790473129, -67.78272469560383, -67.74847812710539, -67.71566589833265, -67.68266242426316, -67.66180302653794, -67.62860830717167, -67.59524966326532, -67.56380347232658, -67.54455599924549, -67.51289124225332, -67.4811172022344, -67.44915191691877, -67.43124316091499, -67.40090346062178, -67.37041349616686, -67.33978692792851, -67.32142737943951, -67.30080020814609, -67.28014571609597, -67.25943658253249, -67.25093982720497, -67.2300531087231, -67.214220730227, -67.19829272908254, -67.19468638920077, -67.17867642578625, -67.16262548123672, -67.15031748035236, -67.15027649921734, -67.13795483795462, -67.12553755404355, -67.11307928899745, -67.11631679866403, -67.10727362820293, -67.09816215585013, -67.08906434387569, -67.09209694786716, -67.0904303817097, -67.08875015517387, -67.08706992863804, -67.09754743882485, -67.09583989153234, -67.09387279705138, -67.09185106105704, -67.1020007221637, -67.10000630692605])
        expected['lon_s'] = np.array([155.30156285694832, 155.28167334608526, 155.2615925899254, 155.24130692809047, 155.2706357603865, 155.25040474006494, 155.25226255151918, 155.25406572146005, 155.3051418760734, 155.30658987617744, 155.30791493287643, 155.29123561092325, 155.32388391515593, 155.30713629131105, 155.28980127119758, 155.27238428881404, 155.3217665565132, 155.32130210364966, 155.3204278394359, 155.3190618016019, 155.36699606919706, 155.3526799926967, 155.33768089727934, 155.3223402924035, 155.3569420307388, 155.34127357678278, 155.3246215755863, 155.30775100833637, 155.34102768997266, 155.32381561326423, 155.3061937252056, 155.28450104440162, 155.31351568799585, 155.29139953546334, 155.2689965149857, 155.24610172088782, 155.264147080675, 155.23065183298525, 155.19673311356698, 155.16243190355516, 155.17979424442535, 155.14863492143175, 155.11727069276304, 155.08537370933908, 155.10544080512057, 155.0731613311031, 155.03142887527432, 154.98940955150042, 154.99984608055217, 154.95738962467138])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_0_120000(self):
        """Test for latitude = -60, longitude = 0, rho = 120000"""
        lat = -60
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-62.91464396846064, -62.91266321360135, -62.910887364417135, -62.909330081286384, -62.95316623537951, -62.95111717862852, -62.95095325408843, -62.951021555980134, -62.99674284228419, -62.996305710177324, -62.99610080450222, -62.995390464828546, -63.04161718513117, -63.0409068454575, -63.039950618973705, -63.039704732163585, -63.08840398094577, -63.089510471591325, -63.09132730191055, -63.09333537752653, -63.14172043760689, -63.14150187155346, -63.14150187155346, -63.14172043760689, -63.18849357304313, -63.188206705098, -63.18785153526116, -63.18722315785752, -63.234337802752265, -63.23316301021504, -63.232698557351455, -63.23304006680996, -63.28033229662313, -63.280113730569695, -63.28010007019134, -63.27982686262456, -63.3266682999525, -63.32482414887659, -63.323690337474375, -63.322775092125596, -63.36916573696831, -63.368906189779864, -63.36834611426792, -63.368482718051325, -63.41606181580963, -63.4156110233244, -63.41595253278291, -63.415979853539596, -63.46459714005172, -63.463996083404766])
        expected['lon_n'] = np.array([14.433116914416772, 14.421227262619073, 14.409313705159278, 14.397401855246775, 14.461289737195782, 14.449219085385078, 14.428426282004265, 14.407522488049441, 14.462845312779253, 14.441774179189764, 14.420653526728792, 14.388853873500505, 14.440541330044576, 14.4119962619558, 14.379919986066128, 14.351199040606224, 14.390028666037747, 14.344925511853567, 14.30311536262298, 14.261049935072165, 14.295930857389774, 14.252667585413267, 14.209322351166719, 14.16573635275193, 14.199471510836268, 14.155410814274163, 14.104948522912462, 14.050987467148422, 14.080923332506941, 14.026665163514007, 13.975571079654081, 13.912762367593936, 13.927116863908363, 13.863861628231229, 13.800437345372137, 13.733632972647534, 13.754448827916795, 13.691220912996341, 13.631107564337412, 13.570878956236239, 13.587781966884728, 13.51264732469365, 13.434247852079304, 13.358848540057888, 13.360335813749659, 13.284534374340856, 13.200581104157644, 13.113400569591603, 13.108973753235787, 13.02137999222496])
        expected['rho_n'] = np.array([255322.87974853514, 255326.23372802735, 255329.61201171874, 255333.0389038086, 255312.28311767578, 255315.10240478514, 255319.52576904296, 255324.04635009766, 255304.87033691406, 255309.4638305664, 255314.1302368164, 255257.08828125, 255302.7315673828, 255309.17218017578, 255252.25174560546, 255258.66805419922, 255306.69315185546, 255252.39757080076, 255261.41442871094, 255270.38267822264, 255254.87659912108, 255264.13649902344, 255273.37209472657, 255282.5104736328, 255267.3203491211, 255276.7746826172, 255287.59005126954, 255235.0443725586, 255284.7221557617, 255232.3223022461, 255243.28349609373, 255256.4320678711, 255244.98479003905, 255258.18197021485, 255271.33054199218, 255221.11806640626, 255272.44853515623, 255221.77427978514, 255234.33955078124, 255246.8319091797, 255234.70411376952, 255250.04006347657, 255201.869140625, 255217.08356933593, 255207.75075683594, 255222.98948974608, 255239.63786621092, 255192.87658691406, 255248.2901611328, 255201.50457763672])
        expected['lat_s'] = np.array([-71.0975111224594, -71.0620487802887, -71.0264908154696, -70.99076892611043, -70.96370771661884, -70.92787654423296, -70.89457254183998, -70.86118657717697, -70.83655691502989, -70.80293872393509, -70.76919758943524, -70.73792898341492, -70.71548498180226, -70.68406611162018, -70.65248331689804, -70.62075025801417, -70.6004236150442, -70.57095817896479, -70.5413424787236, -70.51154919356401, -70.49093568264891, -70.4699123603836, -70.44898466076668, -70.42787937623135, -70.41607680934558, -70.39488956254019, -70.37890691988235, -70.36280133381948, -70.35613506918955, -70.34008412464001, -70.32382827441538, -70.31208034904296, -70.3097444243468, -70.29785989519098, -70.28596170565683, -70.27389959158259, -70.27562079925343, -70.2676567986812, -70.25961083583891, -70.25159219375334, -70.25324509953248, -70.25346366558591, -70.25358660899096, -70.25368223163935, -70.26358600593588, -70.26368162858424, -70.26329913799074, -70.26290298701886, -70.27247891223521, -70.27198713861497])
        expected['lon_s'] = np.array([148.73813355515958, 148.72432291265784, 148.71026638334595, 148.69631913706078, 148.74698548032393, 148.73313385668715, 148.74743627280915, 148.76147914174268, 148.83972578887438, 148.85391892196964, 148.8680300927949, 148.86502480956008, 148.92620964414508, 148.9232180212886, 148.92026737956715, 148.91737137935905, 148.99742119643162, 149.01364972589957, 149.03000119877257, 149.04620240748386, 149.12599267736795, 149.12237267710782, 149.1183018843625, 149.11449063880565, 149.1741045298815, 149.1703752465947, 149.1627664158593, 149.15506196247551, 149.21093290988622, 149.20343336217755, 149.1959064937122, 149.1831477003426, 149.23412823230757, 149.22180657104488, 149.2093209852421, 149.19689004095267, 149.23397796814584, 149.2078183436247, 149.18138551153675, 149.1549253586921, 149.1922864934521, 149.1692141144358, 149.14596415050104, 149.12275516770134, 149.16347675553297, 149.14040437651667, 149.10849373271435, 149.07637818323695, 149.1082341855259, 149.07644648512866])
        expected['rho_s'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_0_120000(self):
        """Test for latitude = -30, longitude = 0, rho = 120000"""
        lat = -30
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([1.3084431744428429, 1.3102053632486985, 1.3116806841094188, 1.3129784200517207, 1.2912037769777243, 1.2930615884319678, 1.2931913620261923, 1.293102569566983, 1.2731310964338718, 1.2735955492974256, 1.2738414361075456, 1.273295020973947, 1.2491912833929746, 1.2491776230146314, 1.2489795475287053, 1.2485424154218236, 1.225859357188213, 1.2241859608415524, 1.2223623003331596, 1.220258602068796, 1.1940443360342954, 1.192999317091278, 1.1918996566349023, 1.1905609395575851, 1.1645720697656827, 1.1638070885786505, 1.1623659186637667, 1.1607471638304787, 1.1341299166349472, 1.1331190486377807, 1.1318759542088372, 1.1295673502693688, 1.1052245560674407, 1.1034965182074359, 1.1015567444831476, 1.0995213481104855, 1.0721869310521015, 1.0713399875950103, 1.070206176192798, 1.0690245534663774, 1.041464740165381, 1.0392176079284496, 1.0368611926647873, 1.0344091547527512, 1.0050120205650188, 1.0031951902457905, 1.0005382466586639, 0.9977788502339706, 0.9675142820216536, 0.9654037535681113])
        expected['lon_n'] = np.array([358.8109287249268, 358.80035559209165, 358.78978245925646, 358.7791820056646, 358.8193981594976, 358.8088250266624, 358.79980917695804, 358.7907386857402, 358.83259408497406, 358.8235782352697, 358.81456238556524, 358.80625687553453, 358.8488772559554, 358.8406537081947, 358.83234819816397, 358.8240700088899, 358.8688214083318, 358.86264691732214, 358.8564177847991, 358.85016133151936, 358.894940051718, 358.88824646633134, 358.8814709186747, 358.8746680502614, 358.9190096383531, 358.91220676993976, 358.9059776374167, 358.8996665426236, 358.9446091873623, 358.9382980925692, 358.9319869977761, 358.92701462006033, 358.9733506233897, 358.96832360416056, 358.96326926417476, 358.9581602826756, 359.0038679086013, 358.99807590818517, 358.9922565870123, 358.9863826243261, 359.03206292949517, 359.0280740990199, 359.02400330627455, 359.01987787201585, 359.0674706301525, 359.06329055438044, 359.06003938433554, 359.05670625202055, 359.1052825573977, 359.1019494250827])
        expected['rho_n'] = np.array([382317.3516357422, 382313.90043945314, 382310.9839355469, 382308.16464843747, 382279.3884765625, 382275.7671508789, 382275.03802490234, 382274.77067871095, 382308.09173583984, 382306.6334838867, 382305.9772705078, 382306.46335449215, 382281.3571166992, 382281.01685791014, 382280.96824951173, 382281.3328125, 382318.882800293, 382321.3861328125, 382323.88946533203, 382327.1219238281, 382305.0537109375, 382306.779309082, 382308.48060302733, 382310.47354736325, 382288.25950927736, 382289.32889404293, 382291.6864013672, 382294.1654296875, 382272.8020385742, 382274.55194091797, 382276.5448852539, 382280.4092529297, 382320.4625732422, 382323.42768554686, 382326.6358398437, 382329.8439941406, 382309.8659423828, 382311.3728027344, 382313.3414428711, 382315.4316040039, 382295.2348144531, 382299.3422241211, 382303.5468505859, 382307.94591064454, 382290.88436279295, 382294.28695068357, 382299.1963989258, 382304.1544555664, 382288.38103027345, 382292.26970214845])
        expected['lat_s'] = np.array([-79.6092928661293, -79.58922577034781, -79.56904939153958, -79.54876372970463, -79.54862712592123, -79.52825950181625, -79.51966712384038, -79.51097912321615, -79.52263142594018, -79.51394342531592, -79.50517346242162, -79.4966357259591, -79.50857489662829, -79.50003716016576, -79.49144478218989, -79.48275678156563, -79.49846621665668, -79.49337089553583, -79.48826191403666, -79.48311195140249, -79.49894432989856, -79.48938206506054, -79.47979247946586, -79.47018923349282, -79.48175957394682, -79.47212900721709, -79.46253942162241, -79.45292251527101, -79.46457481799507, -79.45497157202203, -79.4453410052923, -79.43381164597332, -79.44352417497308, -79.43198115527576, -79.42039715444344, -79.40879949323275, -79.41893549396104, -79.4076930025872, -79.39638220932167, -79.38508507643445, -79.3951664356494, -79.39071315231055, -79.38623254821502, -79.38173828374116, -79.39841760569433, -79.39384137895041, -79.3857817557298, -79.37768115137416, -79.39074047306724, -79.38261254795492])
        expected['lon_s'] = np.array([96.63807894370596, 96.72132528931009, 96.80391593675388, 96.8860216407666, 96.85095544956776, 96.93455013481955, 97.02913459444588, 97.12364392199134, 97.10109746754114, 97.19719139897404, 97.29301895302932, 97.37093775108083, 97.331493408624, 97.41129733888643, 97.49034311815099, 97.56874685963356, 97.53947949904006, 97.63015709046115, 97.72008336107353, 97.80928563163391, 97.77997728990537, 97.83957069041374, 97.89835129841087, 97.95623032143756, 97.89492254344752, 97.954761830766, 97.9961049658121, 98.03661481777944, 97.95761684983907, 97.99988889061228, 98.04142327095514, 98.06092346103551, 97.96078605761396, 97.98196647423016, 98.00254583419941, 98.02221677900906, 97.87846178754779, 97.8573906539583, 97.83579359580273, 97.8132949526767, 97.6687066781367, 97.6071871642824, 97.54485485791686, 97.48186685339101, 97.29526608526625, 97.23350068460184, 97.17218607642262, 97.11015429838056, 96.92281586982546, 96.86256677115678])
        expected['rho_s'] = np.array([6371.196202468872, 6371.195822715759, 6371.196202468872, 6371.1984809875485, 6371.1965822219845, 6371.195063209533, 6371.195442962647, 6371.196961975097, 6371.198101234436, 6371.196202468872, 6371.195063209533, 6371.1946834564205, 6371.199240493775, 6371.198101234436, 6371.1965822219845, 6371.195822715759, 6371.196961975097, 6371.199240493775, 6371.197721481323, 6371.196961975097, 6371.195822715759, 6371.197341728211, 6371.2, 6371.198101234436, 6371.195063209533, 6371.195822715759, 6371.1965822219845, 6371.197721481323, 6371.195822715759, 6371.195442962647, 6371.195063209533, 6371.196202468872, 6371.199620246887, 6371.1984809875485, 6371.195822715759, 6371.195442962647, 6371.195063209533, 6371.1965822219845, 6371.198101234436, 6371.199620246887, 6371.195063209533, 6371.1946834564205, 6371.193544197083, 6371.194303703308, 6371.196961975097, 6371.198101234436, 6371.199240493775, 6371.199620246887, 6371.195442962647, 6371.195063209533])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_0_120000(self):
        """Test for latitude = 0, longitude = 0, rho = 120000"""
        lat = 0
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([15.864792069412275, 15.865311163789187, 15.865618522301844, 15.865932711003666, 15.843249652770055, 15.843864369795355, 15.843898520741206, 15.843898520741206, 15.822690783368316, 15.822984481502615, 15.823285009826108, 15.823551387203736, 15.80205678188571, 15.802644178154324, 15.803204253666266, 15.803538932935595, 15.779278101003712, 15.779728893488937, 15.780125044460789, 15.780459723730118, 15.758022552306628, 15.759006099547108, 15.759996476976767, 15.760857080812187, 15.73831062636198, 15.739581041547595, 15.74080364540903, 15.741916966243735, 15.719022172145856, 15.720497493006576, 15.721918172353938, 15.723427644160523, 15.698483793311624, 15.70035526514421, 15.702315529436007, 15.704214322025265, 15.681128282630624, 15.683525679029302, 15.685841113157935, 15.688095075584044, 15.664647036163387, 15.666463866482616, 15.66823288547765, 15.669899451635132, 15.645344921568935, 15.647407638698269, 15.649395223746751, 15.651212054065965, 15.623980089845134, 15.626240882460408])
        expected['lon_n'] = np.array([359.0911304054374, 359.0816774236261, 359.0721151587881, 359.06255289395006, 359.0989441418479, 359.08943651852326, 359.08091244243906, 359.07233372484154, 359.1096538784665, 359.101157123139, 359.0926330470548, 359.0845734238342, 359.12241267183606, 359.1144076901288, 359.1063480669082, 359.0982611229309, 359.13765765406356, 359.1309640686769, 359.12421584177696, 359.1174129733636, 359.1567548629829, 359.1496514662461, 359.1425207487526, 359.1353353897457, 359.1743221095282, 359.1671913920347, 359.16036120286464, 359.1534490514246, 359.19284558255725, 359.18598807263055, 359.1791032419472, 359.1730653547209, 359.21350007460734, 359.20748950813777, 359.20142430015477, 359.19530445065845, 359.235220076168, 359.2287177360781, 359.22216075447494, 359.21557645211504, 359.2555193983813, 359.250246492342, 359.24497358630276, 359.23959139723684, 359.28087306058035, 359.27554551302774, 359.2708736636355, 359.26611985197314, 359.3081938172604, 359.3035219678681])
        expected['rho_n'] = np.array([382328.6044799805, 382329.40651855466, 382328.0454833984, 382327.97257080075, 382282.66954345704, 382282.2320678711, 382282.71815185546, 382283.00980224606, 382300.16856689454, 382300.16856689454, 382301.2622558594, 382302.06429443357, 382319.1501464844, 382319.73344726564, 382320.43826904293, 382319.53901367186, 382276.66640625, 382275.7671508789, 382275.13524169923, 382274.77067871095, 382293.6550415039, 382292.82869873045, 382293.0474365234, 382293.2418701172, 382311.47001953126, 382311.34849853517, 382311.2998901367, 382311.2269775391, 382329.79538574215, 382329.50373535155, 382329.1877807617, 382326.1011474609, 382285.31870117184, 382282.2077636719, 382281.0897705078, 382280.0932983398, 382299.220703125, 382297.8110595703, 382296.57154541014, 382295.2591186523, 382314.5809570312, 382313.92474365234, 382313.1956176758, 382312.22344970703, 382332.88201904297, 382331.6425048828, 382331.30224609375, 382329.38221435546, 382289.2316772461, 382287.23873291013])
        expected['lat_s'] = np.array([-72.04081488835152, -72.0299548875712, -72.01899926414251, -72.00794801806543, -71.99752514939198, -71.98659684671998, -71.98467073337403, -71.98267631813638, -71.9812829595457, -71.97949344998315, -71.97762197815058, -71.97775858193395, -71.97837329895927, -71.9786601669044, -71.97894703484954, -71.97919292165966, -71.98416529937543, -71.98905571482118, -71.99389148875355, -71.99871360230756, -72.00369964040166, -72.00214235727091, -72.00046213073509, -71.99880922495595, -71.99721096069015, -71.99570831907278, -71.99069496022199, -71.98559963910114, -71.9804633368453, -71.97554560064287, -71.97055956254877, -71.96280046765165, -71.95485012745775, -71.94726861747904, -71.93960514523027, -71.93188703146816, -71.9167786530241, -71.90205276517355, -71.88723125467462, -71.87238242341903, -71.85711012043487, -71.83865494929748, -71.82007683475504, -71.80134845605087, -71.78204634145644, -71.7633999250223, -71.74549116901852, -71.72752777150137, -71.70862180787879, -71.6908086745234])
        expected['lon_s'] = np.array([44.47828066758225, 44.50756510364867, 44.536938332174316, 44.56636278711873, 44.52818544475294, 44.55814265445262, 44.5535015409116, 44.54886725755974, 44.47588327118357, 44.47149145954725, 44.466737647884926, 44.44304372165415, 44.35027609234704, 44.32646605290037, 44.30239646626524, 44.27815612490087, 44.18743755234476, 44.166581569714126, 44.14564703990804, 44.12433343460302, 44.032108805334914, 44.039686900219046, 44.04708399509017, 44.05471673148766, 43.990721274059226, 43.998326689700036, 44.000560161558624, 44.00279704851181, 43.93313594916685, 43.93549919461967, 43.93786244007249, 43.93843617596278, 43.86672260477223, 43.86744660482425, 43.868215001105874, 43.868898020022876, 43.782468806265534, 43.76902699397895, 43.755544200557345, 43.74212970902744, 43.65549217450042, 43.59068050946616, 43.52576980668895, 43.46070883974999, 43.32244873547601, 43.25763365534717, 43.20555687802041, 43.153527912017836, 43.027736318073714, 42.9758063898141])
        expected['rho_s'] = np.array([6371.198101234436, 6371.195442962647, 6371.190885925293, 6371.188227653503, 6371.1946834564205, 6371.199620246887, 6371.192784690857, 6371.189366912841, 6371.192784690857, 6371.198860740661, 6371.193923950195, 6371.188607406616, 6371.190885925293, 6371.196202468872, 6371.195822715759, 6371.190885925293, 6371.189366912841, 6371.193923950195, 6371.200759506225, 6371.193544197083, 6371.187847900391, 6371.19050617218, 6371.194303703308, 6371.198101234436, 6371.187468147277, 6371.187468147277, 6371.188607406616, 6371.191265678405, 6371.19050617218, 6371.187468147277, 6371.187088394165, 6371.187088394165, 6371.2, 6371.194303703308, 6371.191265678405, 6371.188227653503, 6371.188987159729, 6371.192404937744, 6371.1965822219845, 6371.1965822219845, 6371.185949134827, 6371.187468147277, 6371.191265678405, 6371.198860740661, 6371.186708641052, 6371.187088394165, 6371.189366912841, 6371.1965822219845, 6371.185949134827, 6371.186708641052])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_0_120000(self):
        """Test for latitude = 30, longitude = 0, rho = 120000"""
        lat = 30
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([71.80060578666419, 71.82453876951591, 71.84851273350266, 71.87245254654354, 71.90394825635377, 71.92771389957083, 71.95561692987762, 71.98351142244796, 72.01901474575368, 72.0466616439666, 72.07426585349721, 72.11131962974453, 72.15597711408536, 72.19265352238104, 72.22927358161607, 72.26581850877022, 72.30580243617146, 72.3377045422373, 72.36951956339122, 72.40125091472781, 72.4408147854951, 72.46763864591304, 72.4944112799122, 72.52108829126298, 72.5557275956387, 72.58204089941617, 72.61145340152954, 72.64075662061619, 72.67814678368013, 72.70711532349745, 72.73596604255158, 72.77029628086736, 72.81275956693733, 72.8465194844574, 72.88008132649156, 72.91338020624266, 72.95358952988653, 72.98497937176461, 73.0161404023055, 73.04706066867816, 73.08635303942594, 73.12364587229422, 73.16069281835236, 73.19752973609353, 73.24285145633115, 73.27902584572283, 73.31678996164383, 73.35429111528178, 73.40043416576711, 73.4371515551978])
        expected['lon_n'] = np.array([336.09899245150365, 336.14819713428443, 336.1973198547951, 336.2465245375759, 336.2595018969989, 336.3089797873465, 336.37662598088633, 336.4443268159395, 336.47544515779805, 336.54355580420145, 336.6117484128748, 336.7405111391079, 336.83277333441646, 336.9627928154568, 337.0931947870907, 337.22406121158815, 337.3358850686796, 337.4858487020964, 337.6363860714035, 337.7874971766008, 337.90169793952344, 338.0485743274354, 338.1958058851842, 338.34347457503986, 338.45387775278397, 338.6027212351769, 338.7549524913981, 338.9077028419963, 339.0229964351861, 339.1768396160515, 339.33120189129374, 339.5376648495249, 339.7068349748878, 339.9149918200331, 340.12377704258205, 340.33329992556133, 340.5055846171858, 340.7167194248092, 340.9283186852962, 341.1405463231868, 341.31526256215574, 341.54052220098276, 341.76654682099684, 341.99330910144124, 342.18291515280083, 342.41156256545617, 342.6685962443021, 342.9265861496318, 343.1476657126868, 343.4079778823343])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.196961975097, 6371.197721481323, 6371.196961975097, 6371.196961975097, 6371.198101234436, 6371.197341728211, 6371.197721481323, 6371.197341728211, 6371.197341728211, 6371.198101234436, 6371.197721481323, 6371.197721481323, 6371.197721481323, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.197721481323, 6371.198101234436, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.197721481323])
        expected['lat_s'] = np.array([24.662485531738895, 24.675633645891168, 24.688522212854977, 24.701144402441173, 24.66391304127542, 24.677074815806037, 24.69001119409404, 24.702667534626073, 24.66536104137947, 24.67858428761261, 24.69152066590061, 24.703965270568375, 24.66635141880913, 24.67937658955634, 24.692121722547583, 24.70459364797202, 24.66725983396873, 24.680640174552792, 24.693733647191706, 24.706560742452993, 24.669131305801315, 24.68208134446766, 24.694737684999694, 24.707134478343264, 24.669185947314674, 24.682142816170185, 24.694416666108708, 24.70642413866959, 24.66799749439909, 24.680565042471912, 24.692852552788764, 24.70485319516048, 24.66634458861995, 24.67890530650361, 24.691206477198804, 24.703234440327194, 24.66444579603069, 24.676815268617588, 24.68891153363768, 24.700748251469307, 24.66191179584861, 24.673687041977715, 24.685195910729192, 24.696424741724684, 24.655614361433862, 24.668728324640284, 24.68016889150006, 24.691329420603864, 24.65039609690797, 24.663441758222703])
        expected['lon_s'] = np.array([1.0109148479907253, 0.997963848829029, 0.9850053791479281, 0.9720708151289222, 1.0076202422115461, 0.9946640136862666, 0.9811678800514387, 0.9676691850956721, 1.0026665407942885, 0.9891548257904166, 0.9756806768269798, 0.9618703544903313, 0.9965278015560338, 0.9827013642418124, 0.9688965914338645, 0.955094593390267, 0.9889408487703248, 0.9743483630520041, 0.9597893879492988, 0.9452109894961412, 0.9790273627985802, 0.9647463977394356, 0.9504748241903997, 0.9362110413258858, 0.9702673250229215, 0.9559896683366561, 0.9416304762671737, 0.9272447104929517, 0.9611642836848813, 0.9467832136657136, 0.9324111082698316, 0.917539758734304, 0.9508647853033251, 0.9359505336420734, 0.9210392701885836, 0.9061559678220084, 0.9397252803755632, 0.9250921336873398, 0.9104901497372045, 0.8958860313529537, 0.929386294962868, 0.9141644713455541, 0.8989766919523845, 0.8837872583727753, 0.9168510835668963, 0.9014208324581913, 0.8858617148897551, 0.8702998225569686, 0.9029138691220737, 0.887136719108726])
        expected['rho_s'] = np.array([382280.91964111326, 382284.78400878905, 382288.52685546875, 382292.17248535156, 382280.74951171875, 382284.66248779296, 382288.3567260742, 382292.0995727539, 382280.77381591796, 382284.51666259766, 382288.3567260742, 382291.90513916017, 382280.555078125, 382284.3222290039, 382288.0407714844, 382291.662097168, 382279.60721435543, 382282.96119384764, 382286.50960693357, 382289.9121948242, 382278.3190917969, 382282.0133300781, 382285.561743164, 382289.03724365233, 382277.4684448242, 382281.13837890624, 382284.61387939454, 382288.0650756836, 382276.447668457, 382280.0203857422, 382283.5687988281, 382287.09290771483, 382275.5484130859, 382279.09682617185, 382282.66954345704, 382286.2422607422, 382274.6248535156, 382278.1975708008, 382281.7945922851, 382285.19718017575, 382273.55546875, 382277.05527343747, 382280.38494873047, 382283.6903198242, 382335.0207885742, 382275.47550048825, 382278.7565673828, 382282.0133300781, 382333.34379882814, 382273.7255981445])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_0_120000(self):
        """Test for latitude = 60, longitude = 0, rho = 120000"""
        lat = 60
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([81.70433824827401, 81.72197721180557, 81.73964861873569, 81.75733624736507, 81.77350245135685, 81.79106542903389, 81.81400035049315, 81.83696259270909, 81.8583684055679, 81.88115903928094, 81.90395650318315, 81.94420510041475, 81.98291861263039, 82.02294352116665, 82.06300258064877, 82.10303005050599, 82.14403509118816, 82.1863464055491, 82.22869272462952, 82.271065510693, 82.31210470232102, 82.35298936091907, 82.39389451008464, 82.43483637151698, 82.4745889262601, 82.51540528296646, 82.56027791826614, 82.60518982715357, 82.64906610860808, 82.69379018729332, 82.73851511975221, 82.7947087936012, 82.85003417965194, 82.9059298864984, 82.96165782682334, 83.0173042317651, 83.07196794324, 83.12685833161798, 83.1815835147954, 83.23616355645298, 83.29040507685981, 83.35204155770366, 83.41354015410363, 83.47484451699908, 83.53621376669165, 83.59706947153599, 83.66387982067612, 83.73045324762941, 83.79745270763217, 83.86343489633543])
        expected['lon_n'] = np.array([305.8166930680918, 305.84152763591396, 305.86592507162925, 305.8902132243178, 305.7857386507733, 305.81070982237884, 305.8498331459447, 305.88879254497044, 305.7979237082526, 305.83805789981557, 305.87783692154176, 305.9650174561078, 305.92083979255614, 306.01056115749344, 306.10047376772746, 306.1914245667154, 306.17309233898305, 306.2927572532417, 306.41348767701083, 306.53533825180386, 306.51869991098573, 306.65150610920745, 306.7855963829931, 306.9203969964525, 306.91332092047236, 307.05164591154346, 307.20313950733436, 307.35561665036573, 307.3622009527256, 307.51883085077236, 307.67688142816644, 307.9191892191618, 308.0126808485209, 308.261928111713, 308.51500028084035, 308.7709957709324, 308.8724923819988, 309.1360284009346, 309.40300683521207, 309.6735369678579, 309.78424067392547, 310.0629670335754, 310.34590078975395, 310.633834244405, 310.7560673097916, 311.0540274821442, 311.4120659984363, 311.7774537982753, 311.97219615189067, 312.3523918018503])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([38.55977581331546, 38.56974105931451, 38.579480909070945, 38.58898170220643, 38.55136443535259, 38.56398662493877, 38.5731595689941, 38.5820866262393, 38.54388196311684, 38.55592700171815, 38.56508628539515, 38.5737503803573, 38.535101754938786, 38.54423713295368, 38.55579039793476, 38.564461323086086, 38.52527311272314, 38.53387573598277, 38.54216758563516, 38.552921718483354, 38.51384962133629, 38.52257860309557, 38.53107194332848, 38.5392681703325, 38.502993035650555, 38.511715187220666, 38.51993190479219, 38.52794371668861, 38.49150807256118, 38.50003556373994, 38.508351319054434, 38.51602162149236, 38.476775354521465, 38.48765584586929, 38.49563692191445, 38.50344041304119, 38.46448784420461, 38.472988014626694, 38.48390607201496, 38.49194520466806, 38.45312923961488, 38.46091565526869, 38.468534731287846, 38.47856827917859, 38.439161502762204, 38.44712550333444, 38.45453284348932, 38.461725032685344, 38.42496153947775, 38.43272746456405])
        expected['lon_s'] = np.array([2.3478179764800573, 2.3284940906576215, 2.3091226069544075, 2.2897154782014626, 2.3408866151329635, 2.3215448000639567, 2.3008363068307127, 2.280072958640697, 2.329861622595505, 2.309159105777785, 2.288484763490391, 2.266948536707106, 2.31589537984671, 2.29446502099556, 2.2729629451581257, 2.251456387009048, 2.2984862948694054, 2.2751725113546244, 2.2518647042553677, 2.2283914785121492, 2.2753887295305377, 2.252512078122638, 2.229561148407514, 2.206600186852047, 2.2539163223273104, 2.2310178996914307, 2.207589497064641, 2.1840573609398315, 2.2308193973186774, 2.207389287144595, 2.183910938759499, 2.1591103084397716, 2.204531066420356, 2.1797169891657, 2.1548980027125775, 2.1300196789910406, 2.176040853288352, 2.1518899447134214, 2.1276587814157435, 2.1034188669381915, 2.1493371615111294, 2.1233640865316863, 2.09737820494755, 2.0712774908079923, 2.11535206119185, 2.089361056965836, 2.0623472319116125, 2.0352557134555798, 2.078260292017273, 2.051272293615848])
        expected['rho_s'] = np.array([382273.50686035154, 382285.780480957, 382297.8353637695, 382309.52568359376, 382329.67386474606, 382280.67659912107, 382292.05096435547, 382303.0850708008, 382322.6499511719, 382273.0450805664, 382284.54096679686, 382295.3077270508, 382318.20228271483, 382329.55234374997, 382279.2669555664, 382289.96080322267, 382310.6436767578, 382321.5319580078, 382332.0313720703, 382281.2355957031, 382301.1407348633, 382312.17484130856, 382322.771472168, 382333.12506103516, 382292.002355957, 382302.8663330078, 382313.3414428711, 382323.52490234375, 382281.7945922851, 382292.53704833984, 382303.0607666015, 382312.8310546875, 382331.6425048828, 382280.89533691405, 382291.00588378904, 382300.8976928711, 382320.1223144531, 382330.79185791017, 382279.9474731445, 382290.1309326172, 382309.4284667969, 382319.53901367186, 382329.3093017578, 382277.444140625, 382296.40141601564, 382306.3661376953, 382316.01490478514, 382325.32341308595, 382282.7667602539, 382292.82869873045])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_0_120000(self):
        """Test for latitude = 90, longitude = 0, rho = 120000"""
        lat = 90
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([78.8984418957194, 78.90707440105666, 78.91558737808343, 78.92395179849576, 78.89862204195877, 78.90718368408338, 78.91759374615212, 78.92781000160318, 78.90424499519298, 78.9146729865083, 78.9249327844153, 78.93787172402425, 78.91678351496184, 78.9297173319289, 78.94248295548765, 78.95502403657744, 78.93649117203826, 78.95209815429173, 78.96743961294123, 78.98257531214198, 78.96340126359446, 78.979382198705, 78.99510017153248, 79.01055518207694, 78.99158603920452, 79.00691042238108, 79.02556537655168, 79.04393602409809, 79.02769895689367, 79.04587409027508, 79.06365221891097, 79.09216569737484, 79.08588192333842, 79.11393863290154, 79.14157101696247, 79.16872528778151, 79.16107889100567, 79.1878233504745, 79.21402054103609, 79.23979852873737, 79.23086805639758, 79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212])
        expected['lon_n'] = np.array([222.03207391586602, 221.9699874963106, 221.90740930313495, 221.84420273255566, 221.79726567257933, 221.73515193226723, 221.64709713348742, 221.55822271200722, 221.48625983891196, 221.39868315337407, 221.31038246778414, 221.13775626670125, 220.9828065951903, 220.81105465832118, 220.63863336291337, 220.46533780329182, 220.27212541205049, 220.0598431326465, 219.84644070221862, 219.6319044603885, 219.4388286729306, 219.20795461860584, 218.9758647906088, 218.74253186818282, 218.5328723814201, 218.29949847785912, 218.03881747799642, 217.77659285538127, 217.54065080069236, 217.27835787618554, 217.01458963081797, 216.69424009836604, 216.40249539815812, 216.0810666958173, 215.75787580467065, 215.43290906433978, 215.1539504782581, 214.84212502189035, 214.5286329993651, 214.21382958051922, 213.935089560491, 213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([73.76546679576927, 73.6725284117329, 73.5846785186282, 73.49383358757244, 73.72931460449239, 73.6331455409786, 73.54314926092722, 73.4522309053379, 73.6852137805689, 73.59044832092957, 73.49639490851119, 73.41110975143982, 73.64449902292645, 73.55596098826288, 73.46222688718821, 73.37188226748916, 73.6103702751912, 73.51583191934176, 73.42088545968942, 73.32817076434837, 73.57029926287802, 73.47759993546262, 73.39053380656517, 73.30273513987925, 73.55243319555656, 73.45981583041119, 73.37157149388192, 73.28712474253118, 73.54256527975319, 73.45781287743708, 73.37191471088771, 73.28943334647064, 73.54950987459179, 73.46764834984191, 73.38075980586288, 73.30005941327188, 73.57014899871629, 73.48558784169694, 73.40441787360051, 73.32345622627376, 73.60148932172288, 73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512])
        expected['lon_s'] = np.array([350.3579412496324, 350.43211710401874, 350.4985885050213, 350.56877552893235, 350.38744766684687, 350.46648661592224, 350.5413454892256, 350.61713326825605, 350.4496570298073, 350.5321657149811, 350.6124887396204, 350.6834680654752, 350.5244612615973, 350.6013145501383, 350.684424291959, 350.7623977315239, 350.6063142486107, 350.7001883685634, 350.7917948657116, 350.8799862682748, 350.724476521252, 350.8160556976435, 350.8970617411998, 350.97855955837645, 350.81900633936493, 350.90911019489573, 350.99768408805244, 351.08030205625295, 350.9240546487997, 351.0074649189439, 351.09259639675895, 351.1788753463546, 351.03057827909527, 351.11713043625764, 351.21089527318355, 351.2948246377047, 351.1359817583669, 351.2243097647135, 351.307774676371, 351.3893544558177, 351.22447368925356, 351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506])
        expected['rho_s'] = np.array([327743.56041259767, 328264.4966186523, 328804.7303588867, 329309.40705566405, 327946.1116088867, 328462.8431884766, 329007.79194335936, 329528.7767578125, 328135.9274047851, 328669.76914062497, 329199.3819458008, 329737.8900878906, 328291.8145385742, 328852.0506347656, 329376.3651245117, 329890.90932617185, 328487.8765136719, 329023.95423583983, 329577.99276123045, 330110.0116821289, 328679.73386230465, 329194.0836303711, 329747.1985961914, 330278.97447509767, 328802.1298095703, 329332.95782470703, 329836.68665771483, 330333.5130981445, 328807.93851318356, 329354.3212158203, 329858.12296142575, 330370.86865234375, 328782.75936279295, 329328.58306884766, 329813.1844970703, 330318.12854003906, 328725.0125854492, 329232.3870483398, 329724.6442993164, 330217.48485107423, 328624.97650146484, 329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_90_120000(self):
        """Test for latitude = -90, longitude = 90, rho = 120000"""
        lat = -90
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.8068582532915, -75.79483712035227, -75.78302089308815, -75.77143689225582, -75.79725500731846, -75.78517923286589, -75.77498859062422, -75.76505749557103, -75.79554746002594, -75.78229689303612, -75.77209259041612, -75.76370511811535, -75.79595727137615, -75.78707802545514, -75.77563062840619, -75.76720217497041, -75.80199515860244, -75.79550647889093, -75.78922270485452, -75.78315749687155, -75.81811440504367, -75.80915319685263, -75.80321093227471, -75.79751455450693, -75.83300421743431, -75.82681606604626, -75.8222534996807, -75.81792315974693, -75.8521833886237, -75.84737493544802, -75.8427304068124, -75.84073599157475, -75.8803784095175, -75.87787856028129, -75.87555629596349, -75.87345259769913, -75.91267154391335, -75.90675660009211, -75.90379229799231, -75.90104656194598, -75.94038845156524, -75.939896677945, -75.93958248924318, -75.93944588545978, -75.9816291337738, -75.98095977523514, -75.98188868096224, -75.98302249236446, -76.02672204267421, -76.02726845780782])
        expected['lon_n'] = np.array([29.128984376014763, 29.070319881233527, 29.011935424208257, 28.9538207596552, 29.085172127583718, 29.0263437082624, 28.95006074051711, 28.873999753919847, 28.98707353562941, 28.91031928483139, 28.8337187132897, 28.746497197588635, 28.848130412438426, 28.759990236293994, 28.67223596583767, 28.584549997273047, 28.660593908397384, 28.5470471360878, 28.433648920392663, 28.32028827073796, 28.395142021399426, 28.284079730400432, 28.173005486570393, 28.062144686151914, 28.13905944639543, 28.02713313646643, 27.906602495730507, 27.785983062535376, 27.853193831515593, 27.731403020877803, 27.610085200840036, 27.469840926612164, 27.516349392218075, 27.374851778277506, 27.233521503971602, 27.092155371172556, 27.14586456371095, 27.012209714484847, 26.87855145016416, 26.745121997180668, 26.79737635942585, 26.636971074315394, 26.4767331288396, 26.316537872046123, 26.34112655305817, 26.179681371646577, 26.004143802429958, 25.82865575208482, 25.837005658345163, 25.660182306017287])
        expected['rho_n'] = np.array([259169.2380126953, 259157.98516845703, 259146.90245361326, 259136.08708496092, 259160.19685058593, 259148.8953979492, 259139.34384765624, 259130.05964355468, 259093.5061279297, 259146.1247192383, 259136.62177734374, 259128.7958251953, 259093.84638671874, 259085.6072631836, 259139.82993164062, 259131.80954589843, 259099.02318115233, 259092.77700195313, 259086.7981689453, 259080.9408569336, 259048.32462158203, 259104.88049316406, 259099.2662231445, 259093.7977783203, 259061.69193115234, 259055.8103149414, 259051.50847167967, 259047.35245361328, 259079.1180419922, 259074.59746093748, 259070.2227050781, 259068.2783691406, 259039.98828125, 259037.58216552733, 259035.39478759764, 259033.3775390625, 259004.60136718748, 259064.02513427733, 259061.1329345703, 259058.435168457, 259029.70760498047, 259029.14860839842, 259028.7111328125, 259028.4680908203, 259002.2924682617, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.35283203123, 258978.8389160156])
        expected['lat_s'] = np.array([-67.90639210071609, -67.87263730583788, -67.83869126566293, -67.80470424435293, -67.78272469560383, -67.74847812710539, -67.715679558711, -67.68266242426316, -67.66180302653794, -67.62859464679332, -67.59526332364368, -67.56380347232658, -67.54461064075886, -67.51287758187496, -67.48113086261276, -67.44915191691877, -67.43124316091499, -67.40090346062178, -67.37041349616686, -67.33977326755016, -67.32142737943951, -67.30082752890274, -67.2801593764743, -67.25943658253249, -67.25093982720497, -67.2300531087231, -67.214220730227, -67.19829272908254, -67.19468638920077, -67.17867642578625, -67.16261182085839, -67.15034480110904, -67.15027649921734, -67.13795483795462, -67.12553755404355, -67.11309294937578, -67.11633045904236, -67.10728728858129, -67.09818947660682, -67.08909166463235, -67.09209694786716, -67.0904303817097, -67.0887638155522, -67.08708358901637, -67.09754743882485, -67.09586721228902, -67.09387279705138, -67.09185106105704, -67.1020007221637, -67.10001996730438])
        expected['lon_s'] = np.array([155.30154919657, 155.28167334608526, 155.26157892954708, 155.24133424884715, 155.2706357603865, 155.25040474006494, 155.25224889114082, 155.25406572146005, 155.3051418760734, 155.3066035365558, 155.30777832909303, 155.29123561092325, 155.32381561326423, 155.30713629131105, 155.2897739504409, 155.27238428881404, 155.3217665565132, 155.32131576402801, 155.3204278394359, 155.3191027827369, 155.36699606919706, 155.35257070966998, 155.337667236901, 155.32232663202515, 155.35695569111712, 155.34127357678278, 155.32463523596462, 155.30775100833637, 155.34102768997266, 155.32378829250754, 155.3061664044489, 155.28437810099658, 155.31351568799585, 155.29139953546334, 155.26898285460737, 155.24607440013114, 155.26402413726993, 155.23056987071521, 155.1967057928103, 155.16237726204182, 155.17979424442535, 155.14863492143175, 155.11722971162803, 155.08529174706905, 155.1054817862556, 155.0730930292114, 155.03142887527432, 154.98940955150042, 154.99983242017385, 154.95726668126633])
        expected['rho_s'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.198860740661, 6371.2, 6371.2, 6371.200759506225, 6371.199620246887, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_90_120000(self):
        """Test for latitude = -60, longitude = 90, rho = 120000"""
        lat = -60
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-65.52012891038873, -65.52544279756302, -65.53130309987088, -65.53697215688197, -65.55526340347927, -65.56087781897702, -65.57064498949015, -65.58102687702856, -65.6038670296131, -65.6138391058013, -65.62364725744945, -65.63565473001032, -65.6604209959408, -65.67210061942151, -65.68358899760548, -65.6956374513014, -65.72679677429497, -65.7446508887854, -65.76234107873572, -65.78060500457633, -65.81214681816346, -65.82841632876642, -65.84487708466617, -65.86120123678248, -65.89136335215727, -65.90771482503027, -65.92565090179073, -65.94381920498296, -65.97570252802859, -65.99317415192547, -66.01120585133432, -66.03321272084008, -66.06915317625271, -66.090859517435, -66.11275710391405, -66.13450442623139, -66.16892857964825, -66.18812141121597, -66.207887978674, -66.22784579142876, -66.26220164295393, -66.28711817304614, -66.31223960881346, -66.33755228987752, -66.37749523614374, -66.40248006812766, -66.4305111644814, -66.45876082688858, -66.50187298092968, -66.52975381312169])
        expected['lon_n'] = np.array([58.85465407792184, 58.79511531892684, 58.7284799933842, 58.669528630657815, 58.75695505203398, 58.689780141546905, 58.621327985685035, 58.54569730100549, 58.62412153305557, 58.54752779170305, 58.478460918815884, 58.399913743360734, 58.47573567333705, 58.39604102610134, 58.32393471903351, 58.244697694472194, 58.30803403864572, 58.21536886217616, 58.13016908246942, 58.03766100035077, 58.10043726901235, 58.01210243247658, 57.923897369535034, 57.84325332600469, 57.91099514219288, 57.82171090936247, 57.731354336832375, 57.641086556761486, 57.707195957738044, 57.623136819622694, 57.5317762092846, 57.434411862666074, 57.49379352731016, 57.39505631266846, 57.296373739540115, 57.20535122356594, 57.26200422763661, 57.17498761761065, 57.08062855422692, 56.98638901915367, 57.05007711806945, 56.94401452554291, 56.83804414057016, 56.73210449144868, 56.78461498578774, 56.677275147886405, 56.56426625297396, 56.45127784862902, 56.497245021743204, 56.38274373049711])
        expected['rho_n'] = np.array([257977.31147460936, 257952.37536621094, 257989.65800781248, 257964.94063720704, 257933.29656982422, 257970.43338623046, 257941.3412597656, 257974.46788330076, 257938.71640625, 257971.5027709961, 257942.11899414062, 257973.81166992188, 257936.84498291014, 257968.19739990233, 257937.37967529296, 257968.8050048828, 257925.47061767578, 257949.94494628906, 257912.34635009765, 257936.9178955078, 257893.29185791014, 257920.05078125, 257946.83400878907, 257911.3741821289, 257870.1542602539, 257896.40279541016, 257921.84929199217, 257947.19857177735, 257904.90926513672, 257867.6995361328, 257892.7328613281, 257914.77677001953, 257869.10917968748, 257890.39965820312, 257911.69013671874, 257870.76186523438, 257889.45179443358, 257850.63798828123, 257873.87280273438, 257897.13192138672, 257853.23853759764, 257870.85908203124, 257888.4310180664, 257906.00295410157, 257856.61682128906, 257873.5325439453, 257887.55606689453, 257901.57958984375, 257848.86378173827, 257862.23109130858])
        expected['lat_s'] = np.array([-65.4030594680147, -65.36853969194948, -65.33392429323584, -65.29933621527891, -65.27795772317677, -65.24334232446313, -65.21039349190698, -65.17743099897251, -65.15771907302786, -65.12481122160673, -65.09188970980728, -65.05929604708797, -65.03985732871013, -65.00733196788252, -64.97482026743324, -64.94233588774068, -64.92685867908142, -64.89839045062081, -64.87008614670029, -64.84191844656314, -64.82660516244397, -64.80876470833189, -64.79107451838158, -64.77350727183631, -64.76853489412053, -64.75121353438539, -64.73979345809312, -64.72856462709763, -64.7295481743381, -64.71857889053106, -64.70759594634566, -64.70083405906738, -64.706202587755, -64.6996865872868, -64.69318424719694, -64.68679119013382, -64.69480983221942, -64.69088930363583, -64.6869551146739, -64.68300726533363, -64.69120349233765, -64.69434537935584, -64.69741896448235, -64.70038326658215, -64.71571021107965, -64.71857889053106, -64.72077821144381, -64.72284092857313, -64.73758047680204, -64.739547571283])
        expected['lon_s'] = np.array([145.44090071447096, 145.44158373338797, 145.44225309192663, 145.44301807311365, 145.48011966068518, 145.4813900758708, 145.50502253039903, 145.5286959660623, 145.5889928760552, 145.61328102874373, 145.6377331059724, 145.64491846497924, 145.68905514739586, 145.69670495926627, 145.70445039378507, 145.71211386603383, 145.77631764423194, 145.80395258961383, 145.8315875349957, 145.85922248037755, 145.92520210775987, 145.93458678767948, 145.9438758449507, 145.95323320411362, 146.00164558495067, 146.01171328378726, 146.01959532208946, 146.02729977547324, 146.07519306193336, 146.0834985719641, 146.09185872350818, 146.094864006743, 146.13885042499788, 146.14211525542115, 146.1449702744942, 146.14738816146038, 146.1807331449884, 146.1713074839338, 146.16108952093543, 146.15013389750675, 146.1841072584384, 146.17550122008416, 146.1655837854093, 146.1541773694954, 146.1908964664734, 146.1777688428886, 146.15546144505936, 146.1311049904791, 146.1594366151563, 146.13330431139184])
        expected['rho_s'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.200759506225, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.200759506225, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_90_120000(self):
        """Test for latitude = -30, longitude = 90, rho = 120000"""
        lat = -30
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-51.78684513116252, -51.91977427278931, -52.04662454605477, -52.17520968736943, -52.302906904092, -52.423213856132605, -52.55846526207719, -52.69173591316249, -52.8250475453828, -52.954001516912626, -53.08116597887994, -53.198317383524, -53.317162675082216, -53.4279073622848, -53.54089235153515, -53.64921915177155, -53.784538859607835, -53.9109383403881, -54.03609472673941, -54.1601173016885, -54.291065688456, -54.39249399763068, -54.49679098625677, -54.59690789911082, -54.705166397455514, -54.80692255571037, -54.89902082647882, -54.99073660665377, -55.09144091577642, -55.18176333736068, -55.275077381801395, -55.36455285992855, -55.463727206677135, -55.55210985453709, -55.643579747901924, -55.73157990516836, -55.81870579822103, -55.8977993888098, -55.979980224903386, -56.05878694754699, -56.14535276508772, -56.23276552608556, -56.31687247552509, -56.4009384438296, -56.49647912993976, -56.58277173991368, -56.66855891588904, -56.75740601661258, -56.85525530666217, -56.94012723728875])
        expected['lon_n'] = np.array([67.02278851388088, 66.88610276821059, 66.75623355133196, 66.62098897557654, 66.60290946484352, 66.47415356879961, 66.32789872810213, 66.1822449440516, 66.15575064026112, 66.01026761093985, 65.86526269486048, 65.72338600542098, 65.7022192491831, 65.56602527713305, 65.42416224807188, 65.28885620061394, 65.24890642415853, 65.09205112986919, 64.93548270352498, 64.77890061680245, 64.74035102912688, 64.60631539685455, 64.46594134903246, 64.3320969620569, 64.31759647044895, 64.1763208376564, 64.04806354542191, 63.91979259280907, 63.912279384722055, 63.78284046976114, 63.64697434679125, 63.517241733696025, 63.509585091636445, 63.37845911995054, 63.240769336472255, 63.10945894967875, 63.11643257282134, 62.99212996011628, 62.861311346943026, 62.73693360215709, 62.743709149813746, 62.60234472456199, 62.467380186562536, 62.33217659194214, 62.322723610130836, 62.17940575077646, 62.03813011798392, 61.88987403185962, 61.875346219495, 61.731994209194774])
        expected['rho_n'] = np.array([255658.7880859375, 255687.34552001953, 255658.9096069336, 255691.5987548828, 255677.67244873047, 255656.1875366211, 255671.8880493164, 255689.1197265625, 255658.39921874998, 255678.1342285156, 255699.13305664062, 255732.45411376952, 255717.19107666015, 255692.1091430664, 255726.81553955076, 255701.66069335936, 255715.34395751954, 255719.91314697266, 255722.63521728516, 255726.7183227539, 255741.42236328125, 255718.77084960937, 255755.83475341796, 255732.50272216796, 255720.71518554687, 255757.48743896483, 255746.72067871093, 255735.6622680664, 255736.14835205078, 255724.21499023438, 255771.51096191406, 255759.52899169922, 255759.04290771484, 255745.74851074218, 255791.56192626953, 255776.95510253907, 255731.94372558594, 255733.66932373046, 255794.72147216796, 255795.42629394532, 255748.32475585936, 255792.75283203126, 255776.4933227539, 255759.40747070312, 255754.59523925782, 255796.2040283203, 255770.5144897461, 255803.9570678711, 255789.81202392577, 255761.81358642576])
        expected['lat_s'] = np.array([-63.63428635979153, -63.5861198657646, -63.53918280578827, -63.49503246299332, -63.47414574451142, -63.439380081636045, -63.4221270237926, -63.4702115555495, 56.0084127085564, 56.05491946661501, 56.107904659101386, 56.15312392650146, 56.30637629600362, 56.355816620310755, 56.41154071865431, 56.47029742098934, 56.64737690541109, 56.72624168466264, 56.806834501774205, 56.89592065911869, 57.10355499479249, 57.1875765668674, 57.272451912588565, 57.365192221139, 57.579766029009534, 57.68422352708112, 57.77519140154194, 57.869663163046965, 58.08918885806576, 58.19377271463699, 58.302403458291465, 58.398190031211726, 58.62183433030003, 58.725598564170866, 58.8320743831422, 58.94130276834905, 59.164865105167316, 59.26379698020037, 59.36434077987754, 59.469693032730405, 59.70389680427555, 59.79508665988439, 59.88733177972, 59.98393797534066, 60.21373285977657, 60.31028441388386, 60.400722948684006, 60.491694238239404, 60.71656114118915, 60.80981371392734])
        expected['lon_s'] = np.array([142.59820330266834, 142.59482918921833, 142.61817477580144, 142.6802338746002, 142.81950143177676, 142.99046106670215, 143.37317022627633, 144.51207694961926, 166.2328981329602, 165.81066949884837, 165.40587150749838, 165.06623352083034, 164.76377908400383, 164.43905823048308, 164.1290086232995, 163.82537939393578, 163.4533116690885, 163.08258266131855, 162.7119219554403, 162.35746245827332, 162.02985926492283, 161.73979479125072, 161.44669771358713, 161.1676435048571, 160.91050054298447, 160.65333026035515, 160.40606375202233, 160.1628816968132, 159.9380728504714, 159.7148349476387, 159.49646013949507, 159.28544827527668, 159.09230418592705, 158.90025292684464, 158.71104302645696, 158.5253301829243, 158.39103500346351, 158.2558792201673, 158.12320962572898, 158.0021923340147, 157.89188477891898, 157.74571873068072, 157.6016563807068, 157.46901410702515, 157.3576273820406, 157.23202020320406, 157.09966479746754, 156.9690715805369, 156.86075844067884, 156.7461751871627])
        expected['rho_s'] = np.array([6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2, 258803.07094726563, 258695.0630859375, 258623.1469604492, 258542.72436523438, 258449.1531982422, 258334.29155273436, 258260.26096191406, 258168.65843505858, 258022.12841796875, 257960.6387939453, 257816.3447631836, 257705.90648193358, 257580.35098876953, 257521.5834350586, 257385.99030761718, 257291.1067138672, 257184.60571289062, 257109.57864990234, 257028.8887084961, 256929.6060546875, 256813.7479370117, 256740.83533935546, 256654.65264892578, 256568.4942626953, 256465.46876220702, 256410.03088378906, 256339.1841430664, 256254.62983398436, 256148.83365478515, 256097.50318603514, 256036.2322998047, 256024.3475463867, 255929.3181274414, 255865.22795410155, 255790.2008911133, 255763.32044677733, 255706.83748779297, 255658.7880859375, 255618.41881103514, 255569.6159790039, 255485.9852294922, 255477.43015136718])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_90_120000(self):
        """Test for latitude = 0, longitude = 90, rho = 120000"""
        lat = 0
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([49.20649027642374, 48.821923305395345, 49.26883282807304, 48.95181301284148, 45.159224117692254, 37.65071178573261, 28.326765908234847, 17.18044359790764, -9.164668370531231, -29.260143588030104, -36.17614725477489, -38.99331341946001, -40.799283737903096, -42.134285192317094, -43.18698126795613, -44.06735167083576, -44.93391143121363, -45.67490499388995, -46.32904587108044, -46.91760327186043, -47.46605380183411, -47.87496356706444, -48.26141567030376, -48.62811486646339, -48.981235646553046, -49.309931670170755, -49.58987380349291, -49.86026733235536, -50.136111352175504, -50.384730237963964, -50.62606814209718, -50.84977049779346, -51.078882363312346, -51.291738378606624, -51.49593371403333, -51.70007440794669, -51.88942091211777, -52.05943798093773, -52.226613691062965, -52.391221250060255, -52.56955748928931, -52.74021659589124, -52.90450996618671, -53.0666449967045, -53.24360153772116, -53.404302228513245, -53.56064525861484, -53.71518511877554, -53.88574860272908, -54.0358898210643])
        expected['lon_n'] = np.array([156.09083219667832, 147.79795801590734, 129.36590023868422, 119.02773957133125, 97.84403080394176, 85.6162231706239, 80.05870314865963, 74.91897896803586, 68.26593758433657, 65.24036868769602, 64.33882469838935, 63.8656906643934, 63.617153740874976, 63.300813529465835, 63.02142464146646, 62.7577725093148, 62.605411479499324, 62.34764014022304, 62.1007697828621, 61.85999195424081, 61.747908549960904, 61.55817955519629, 61.36803391889231, 61.17732137688721, 61.123581448497546, 60.93655037845509, 60.772625838374786, 60.607949977485774, 60.57243299380171, 60.41095366144427, 60.248927913953224, 60.08472333611695, 60.05729329641018, 59.89111479390377, 59.72984036722143, 59.56269197785287, 59.547194278626115, 59.3961856262663, 59.24496523804222, 59.09341700073798, 59.07750265997185, 58.90432321356617, 58.73630739017303, 58.56788175542969, 58.53666779092273, 58.360858721686604, 58.18338308629299, 58.00544299803582, 57.966463108442554, 57.78664471816363])
        expected['rho_n'] = np.array([72615.78869934082, 79931.54709777831, 90330.03797302245, 96743.76424560546, 113115.43740234374, 125146.6722290039, 145569.55159301756, 218311.9833984375, 266010.9430053711, 257999.50120849608, 256150.8265991211, 255647.46232910155, 255361.59633789063, 255247.3666015625, 255103.43713378906, 255038.98239746093, 255022.40693359374, 254984.05490722656, 254934.08547363282, 254900.32694091796, 254902.8302734375, 254870.335559082, 254870.01960449218, 254899.25755615233, 254856.1662109375, 254874.224230957, 254865.57193603515, 254870.4327758789, 254905.23638916016, 254875.73109130858, 254856.65229492186, 254874.15131835936, 254856.74951171875, 254889.778918457, 254870.62720947264, 254916.78088378906, 254915.07958984375, 254902.90318603514, 254895.02862548828, 254891.50451660156, 254906.76755371093, 254941.3524291992, 254920.5966430664, 254903.3649658203, 254904.2885253906, 254954.54960937498, 254942.17877197266, 254933.01608886718, 254940.74482421874, 254938.3873168945])
        expected['lat_s'] = np.array([14.077263261728234, 14.103299942844316, 14.444515703210655, 15.086942805974545, 16.32447746093247, 17.568466644656056, 19.199508988265933, 19.78258857733158, 15.894093580951619, 14.603017073089958, 14.825285089060515, 8.180610687094486, 4.044589635193262, 0.4565478302129975, -1.2915298048250463, 0.6445351267392567, 4.751582516181287, 11.700439392833843, 17.539479321818533, 23.493232277913577, 29.28950839250482, 33.9764456726701, 38.42559674707056, 44.330592397775476, 52.44171524473242, 62.427229830143034, 64.13853372668973, 63.72523898001226, 64.56208912504786, 65.85969017111417, 66.24328554773314, 66.54192703145547, 66.8943511322498, 67.17264548243467, 67.39461126253445, 67.60062855092225, 67.86251166407888, 67.99742497565956, 68.12429061685067, 68.24332886125295, 68.45996367870553, 68.54903959076626, 68.6329604175509, 68.70910848906009, 68.88436772806803, 68.94914865725102, 69.00202968935255, 69.05108923061346, 69.19889110915796, 69.23947438565887])
        expected['lon_s'] = np.array([139.52818353490935, 139.8178655179879, 140.25040773774649, 140.80745064569604, 141.7654393183037, 142.5742703198166, 143.5799883543443, 143.56489363627858, 139.59341184148295, 138.28032163392635, 138.44693726853964, 132.68037247683796, 128.86039793821158, 124.8022137220918, 122.18166138285467, 118.06655437019204, 115.27920881142319, 112.46421464689413, 110.6402877611231, 109.09039489504215, 107.6828568316426, 107.06992931590399, 106.99762493335024, 107.64182788529833, 110.58233360601554, 122.51379982181406, 127.66993694701335, 127.5308059936202, 129.58400183926105, 133.6271732995851, 134.9768050192013, 136.08186132501766, 136.8794771559134, 137.93893511883078, 138.78044174533304, 139.57334474570146, 140.24032637853153, 140.89498635009892, 141.51968911196664, 142.11866938142006, 142.65232572165152, 143.16122945633083, 143.6543144728924, 144.1140681663043, 144.53870102700398, 144.97193992605787, 145.35346063271646, 145.721962998817, 146.0680350236832, 146.4175904450261])
        expected['rho_s'] = np.array([221564.29842529297, 220469.71020507812, 220001.6356323242, 220177.5251220703, 221390.49909667968, 223276.65078125, 226313.26604003905, 224474.2401977539, 205091.47119140625, 199491.18823852538, 199066.32653198243, 185953.5912963867, 182548.46361694337, 185233.9439575195, 191998.44666137695, 205204.53432617188, 218527.8289916992, 225831.16794433593, 230561.2025878906, 230946.76440429688, 220912.02232666014, 219277.20036621092, 213185.88752441405, 213534.55556640626, 219246.11529541016, 248026.44318847655, 255124.4845703125, 246547.09519042968, 247594.29022216797, 256187.50163574217, 256140.40009765624, 256139.1848876953, 255664.06209716797, 256144.94498291015, 256102.46124267578, 256097.77053222657, 256156.46517333985, 256115.26955566407, 256095.07276611327, 256095.0970703125, 256086.17742919922, 256077.8653930664, 256099.3989135742, 256078.15704345703, 256088.77797851563, 256093.97907714843, 256073.9524169922, 256065.3973388672, 256088.00024414062, 256100.22525634765])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_90_120000(self):
        """Test for latitude = 30, longitude = 90, rho = 120000"""
        lat = 30
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([78.69128396574021, 78.68101221500216, 78.67095732277025, 78.66081619940007, 78.6351065135905, 78.62493550814271, 78.6131269648414, 78.60142428947223, 78.574055721468, 78.56246489044648, 78.55074087073615, 78.53444830824473, 78.50230116914243, 78.48593347457013, 78.46957687905524, 78.45332359015156, 78.41807298384512, 78.39850876074127, 78.37876866026627, 78.35919163055773, 78.32437730258454, 78.30284342367878, 78.28123868156038, 78.2595622224557, 78.22314963021671, 78.2014578031864, 78.17962766482539, 78.15772837079903, 78.12160862292072, 78.09946173453694, 78.07701602537699, 78.06014801944801, 78.02943692761869, 78.01183126125933, 77.99379273166133, 77.97514973032177, 77.9456262376344, -51.77685939459596, -51.91788914057838, -52.06132311314866, -52.22688689862974, -52.37685053204655, -52.525133938927524, -52.67554836482955, -52.844390641112255, -52.98660884001026, -53.133362284517176, -53.27502040790324, -53.44289279732379, -53.584004505576274])
        expected['lon_n'] = np.array([184.05933669685666, 184.07448605643575, 184.08986764244662, 184.10459353029717, 184.30898011102062, 184.32302297995417, 184.31022320544957, 184.29667211013626, 184.4715659340236, 184.45790555568357, 184.44324796972472, 184.33840456596502, 184.42185581724425, 184.3168211681878, 184.2118821417797, 184.1070660587767, 184.1305755698999, 183.96557185993072, 183.7998031687745, 183.63320119453957, 183.6551670829103, 183.45418193639352, 183.25303286533665, 183.05177451125306, 183.0383463593448, 182.83707434488286, 182.59308632735167, 182.34959008344072, 182.2934732492199, 182.05116545822452, 181.81015540317145, 181.4994910789626, 181.37482646623153, 181.06935308579187, 180.7695077812283, 180.47899251507098, 180.39948911313203, 54.938035777634354, 54.86339888547904, 54.78318514386641, 54.85603594155377, 54.751370122712494, 54.64572758681991, 54.53450137128083, 54.58141794068965, 54.4716670460113, 54.34394592362665, 54.22019314114977, 54.24570389769977, 54.114174944852834])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.199240493775, 254966.06979980468, 254928.08233642578, 254946.57783203124, 254951.63310546873, 254931.4849243164, 254909.36810302734, 254945.3626220703, 254909.14936523436, 254883.2653930664, 254908.07998046876, 254871.96394042968, 254884.74794921876, 254908.68758544922])
        expected['lat_s'] = np.array([-14.925646067491854, -16.22067042469378, -16.216367405516664, -17.3814952250833, -17.66684686822809, -14.423531540847534, -8.019348099557646, -10.483748653989949, 12.584498097865264, 18.26181280768239, 22.827876230105957, 26.748466285395807, 30.633846715514245, 34.61261290705968, 38.88780247839449, 43.68629040808442, 50.43321615702013, 58.72306114889421, 64.35387934593648, 66.57578769426581, 67.84559157795746, 68.52190495674108, 69.04021898454938, 69.45447678789982, 69.87411678031624, 70.16254371613482, 70.37686822210065, 70.56229932033004, 70.80878377200288, 70.95605972598128, 71.0895062544377, 71.18485740279837, 71.35481983010496, 71.43332943951967, 71.50510448241275, 71.56877721340291, 71.695963873485, 71.73526478196925, 71.77119157700352, 71.80387915482392, 71.91547932322004, 71.93162589041796, 71.94555093858833, 71.95868709990954, 72.0489378045075, 72.05596265406886, 72.05795023911733, 72.05721599378157, 72.13705065989527, 72.13266909354269])
        expected['lon_s'] = np.array([99.29184649287687, 96.41234802182622, 95.00018592035858, 96.31499050539685, 95.05519626393385, 92.82048131091074, 91.2763258037326, 91.16591579579935, 89.48400220325121, 89.25952120598957, 89.05225911562553, 88.9397044282929, 88.75222939595439, 88.64634097325168, 88.58905617668279, 88.69167976896223, 89.44350601166221, 95.29311224329291, 107.93839469906028, 116.00327716533624, 121.00300295854225, 124.12666703204253, 126.67738751720545, 128.83753046487035, 130.64399255693368, 132.32353607383982, 133.65090137676174, 134.85439436889632, 135.92502652129582, 136.98009950276605, 137.9802714236877, 138.7991291429022, 139.5490975741479, 140.2996670620406, 141.01823028348264, 141.689159765653, 142.2483336926236, 142.81212482747313, 143.35587618729784, 143.88119969674187, 144.35883482540086, 144.8110343295907, 145.2467184363675, 145.68468382632705, 146.04952521103246, 146.44573082440655, 146.8078947749573, 147.13988294975496, 147.44270621679664, 147.75578842797168])
        expected['rho_s'] = np.array([146642.26603393554, 162472.83049926758, 174005.91430664062, 168248.4560974121, 179255.02588500976, 203733.52266845704, 229206.7538696289, 234666.8137451172, 256314.51538085938, 257645.21889648438, 258994.56373291014, 260271.31192626953, 261775.47451171876, 263318.0377319336, 264970.6017578125, 266737.15247802733, 268610.2041992187, 266921.74287109374, 262246.4169799805, 260347.28685302733, 259507.6496826172, 259067.67076416014, 258776.67658691405, 258553.87999267576, 258378.8168457031, 258234.49851074218, 258158.37775878905, 258040.01630859374, 257988.17545166015, 257916.11350097656, 257883.78891601562, 257783.9715698242, 257778.64895019532, 257726.5407470703, 257694.48350830076, 257623.24790039062, 257626.26162109376, 257570.53209228514, 257525.86097412108, 257491.4219238281, 257479.7559082031, 257430.92877197266, 257390.63240966797, 257417.02677001953, 257347.63828125, 257329.750390625, 257336.21530761718, 257290.7421508789, 257265.0769165039, 257230.80799560546])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_90_120000(self):
        """Test for latitude = 60, longitude = 90, rho = 120000"""
        lat = 60
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([81.22530293083517, 81.22124921356276, 81.21671567550116, 81.21200882138938, 81.19306273040542, 81.18876654141746, 81.18253570134712, 81.17604531408833, 81.155106515414, 81.14919071781912, 81.14281132113433, 81.13010033908894, 81.10291447864499, 81.09072856739205, 81.07803380704593, 81.0651708532915, 81.03383821424772, 81.01718535927758, 81.00006719767023, 80.98245128602713, 80.95077372243026, 80.93050940493647, 80.90976697420079, 80.88854643022322, 80.85351353619657, 80.83244923279625, 80.80962957077924, 80.78632155023656, 80.74953073627229, 80.72623637610796, 80.70234266684396, 80.68149607572344, 80.64749624780876, 80.62640376987812, 80.60472731077344, 80.58241991294418, 80.5504230380036, 80.53082039508567, 80.51065511533486, 80.48999976951109, 80.457675899264, 80.4281575292185, 80.39814738555276, 80.36751227957795, 80.32572518223581, 80.2950730007881, 80.26080764926942, 80.22590196751608, 80.18037619537701, 80.14500947585468])
        expected['lon_n'] = np.array([180.9367654536236, 180.87552597752526, 180.81414989764352, 180.75427645937918, 180.9025735266385, 180.84062371086648, 180.74512400589137, 180.65123622556038, 180.7647266488093, 180.66955479291434, 180.5746971257212, 180.37505069628173, 180.38345182896086, 180.18369611649464, 179.984404856892, 179.78734023895882, 179.72984370652566, 179.46195002689942, 179.195736573809, 178.9317224416313, 178.8775590415131, 178.59314996447378, 178.3107353026721, 178.03064290518822, 177.96075640960063, 177.68089623854854, 177.37001334828625, 177.06142540158507, 176.96394494175064, 176.6563951838033, 176.35068957693187, 175.96981090805528, 175.79958893356024, 175.4203631704628, 175.04427929438353, 174.6711597204041, 174.53626348429634, 174.1923634595862, 173.8514140765975, 173.513606580627, 173.38663336395646, 173.00615084605172, 172.6291517246237, 172.25651026388616, 172.0944162145034, 171.72535377289094, 171.28917789249394, 170.8581792954878, 170.63662161919092, 170.21028121119872])
        expected['rho_n'] = np.array([6371.199620246887, 6371.2, 6371.198860740661, 6371.199620246887, 6371.200759506225, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.200759506225, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([18.67480702603639, 18.73980310617823, 18.805454884480383, 18.870894926918282, 18.955862480193247, 19.020599013146622, 19.136678078090995, 19.25303035060216, 19.385877529958904, 19.499757273990525, 19.613971697291475, 19.7532938959814, 19.905853001282807, 20.04290757716828, 20.17882834165154, 20.314516879703007, 20.52168334741866, 20.715763172684575, 20.908231073306368, 21.099087049284037, 21.291753025391756, 21.466339490766444, 21.639348182442873, 21.811359666500465, 21.980550282430855, 22.147978709555375, 22.33221623222731, 22.514370547202375, 22.689742484331617, 22.868433893397494, 23.046585717518937, 23.25902509127384, 23.46018782270906, 23.669157460365597, 23.878407135778104, 24.089610245293244, 24.277563390873652, 24.48224366973143, 24.700714100523456, 24.942830646222063, 25.248802630471133, 25.733691420028677, 26.19315141530626, 26.538711175984716, 26.8048631573726, 27.03818924960941, 27.271201153144396, 27.48376347030436, 27.65023567094508, 27.83525183518239])
        expected['lon_s'] = np.array([42.978654578698, 42.878401062060554, 42.77610190376669, 42.67368663225693, 42.563017077135214, 42.46142825851503, 42.28233045319354, 42.10105040243223, 41.909941709455275, 41.729798885190775, 41.54714255131171, 41.32342995033169, 41.08760059376408, 40.86283955874647, 40.63729646706889, 40.40928767710094, 40.062307237075125, 39.7272181563943, 39.38933211324835, 39.04850225857013, 38.69335632739156, 38.37198568165871, 38.04855914898568, 37.722284427428754, 37.38230151620762, 37.05674054941896, 36.69346669312807, 36.32954055377146, 35.95180718700766, 35.588639198648906, 35.22387294602438, 34.78430587670445, 34.33442522173781, 33.89865573759641, 33.463661479925804, 33.02928876946967, 32.62803906139352, 32.24031654296858, 31.85719757204422, 31.48196405451665, 31.13810159584694, 30.797985495936985, 30.63244561611797, 30.541914873764036, 30.566455743451893, 30.51061211679787, 30.456284792139588, 30.405666260200622, 30.45178369747655, 30.405119845067023])
        expected['rho_s'] = np.array([253933.77324218748, 254463.45895996093, 255006.75502929688, 255554.64459228516, 256922.04744873047, 257466.97189941406, 258598.55111083985, 259751.7367553711, 261753.98959960937, 262911.91456298827, 264092.9771240234, 265602.4866333008, 268000.84931640624, 269540.42312011716, 271095.5030029297, 272678.0952392578, 276033.4600708008, 278517.0818847656, 281048.4857543945, 283628.8625854492, 287202.0659545898, 289690.20834960934, 292221.41778564453, 294804.5895996094, 298424.0924682617, 301073.71196289064, 304100.4111083984, 307173.4826660156, 311360.5371948242, 314533.0372314453, 317767.99906005856, 321794.3542236328, 327027.8260498047, 331219.5955932617, 335501.9225830078, 339879.2789916992, 345126.28825683595, 349228.6669555664, 353371.1475830078, 357519.3153930664, 362647.5500366211, 366706.1325683594, 368688.6260986328, 369703.5451538086, 370524.2736572265, 371052.0150390625, 371560.8963623047, 372007.7290649414, 372483.3136352539, 372855.5081420898])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_90_120000(self):
        """Test for latitude = 90, longitude = 90, rho = 120000"""
        lat = 90
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([78.8984418957194, 78.90707440105666, 78.91558737808343, 78.92395179849576, 78.89862204195877, 78.90718368408338, 78.91759374615212, 78.92781000160318, 78.90424499519298, 78.9146729865083, 78.9249327844153, 78.93787172402425, 78.91678351496184, 78.9297173319289, 78.94248295548765, 78.95502403657744, 78.93649117203826, 78.95209815429173, 78.96743961294123, 78.98257531214198, 78.96340126359446, 78.979382198705, 78.99510017153248, 79.01055518207694, 78.99158603920452, 79.00691042238108, 79.02556537655168, 79.04393602409809, 79.02769895689367, 79.04587409027508, 79.06365221891097, 79.09216569737484, 79.08588192333842, 79.11393863290154, 79.14157101696247, 79.16872528778151, 79.16107889100567, 79.1878233504745, 79.21402054103609, 79.23979852873737, 79.23086805639758, 79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212])
        expected['lon_n'] = np.array([222.03207391586602, 221.9699874963106, 221.90740930313495, 221.84420273255566, 221.79726567257933, 221.73515193226723, 221.64709713348742, 221.55822271200722, 221.48625983891196, 221.39868315337407, 221.31038246778414, 221.13775626670125, 220.9828065951903, 220.81105465832118, 220.63863336291337, 220.46533780329182, 220.27212541205049, 220.0598431326465, 219.84644070221862, 219.6319044603885, 219.4388286729306, 219.20795461860584, 218.9758647906088, 218.74253186818282, 218.5328723814201, 218.29949847785912, 218.03881747799642, 217.77659285538127, 217.54065080069236, 217.27835787618554, 217.01458963081797, 216.69424009836604, 216.40249539815812, 216.0810666958173, 215.75787580467065, 215.43290906433978, 215.1539504782581, 214.84212502189035, 214.5286329993651, 214.21382958051922, 213.935089560491, 213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([73.76546679576927, 73.6725284117329, 73.5846785186282, 73.49383358757244, 73.72931460449239, 73.6331455409786, 73.54314926092722, 73.4522309053379, 73.6852137805689, 73.59044832092957, 73.49639490851119, 73.41110975143982, 73.64449902292645, 73.55596098826288, 73.46222688718821, 73.37188226748916, 73.6103702751912, 73.51583191934176, 73.42088545968942, 73.32817076434837, 73.57029926287802, 73.47759993546262, 73.39053380656517, 73.30273513987925, 73.55243319555656, 73.45981583041119, 73.37157149388192, 73.28712474253118, 73.54256527975319, 73.45781287743708, 73.37191471088771, 73.28943334647064, 73.54950987459179, 73.46764834984191, 73.38075980586288, 73.30005941327188, 73.57014899871629, 73.48558784169694, 73.40441787360051, 73.32345622627376, 73.60148932172288, 73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512])
        expected['lon_s'] = np.array([350.3579412496324, 350.43211710401874, 350.4985885050213, 350.56877552893235, 350.38744766684687, 350.46648661592224, 350.5413454892256, 350.61713326825605, 350.4496570298073, 350.5321657149811, 350.6124887396204, 350.6834680654752, 350.5244612615973, 350.6013145501383, 350.684424291959, 350.7623977315239, 350.6063142486107, 350.7001883685634, 350.7917948657116, 350.8799862682748, 350.724476521252, 350.8160556976435, 350.8970617411998, 350.97855955837645, 350.81900633936493, 350.90911019489573, 350.99768408805244, 351.08030205625295, 350.9240546487997, 351.0074649189439, 351.09259639675895, 351.1788753463546, 351.03057827909527, 351.11713043625764, 351.21089527318355, 351.2948246377047, 351.1359817583669, 351.2243097647135, 351.307774676371, 351.3893544558177, 351.22447368925356, 351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506])
        expected['rho_s'] = np.array([327743.56041259767, 328264.4966186523, 328804.7303588867, 329309.40705566405, 327946.1116088867, 328462.8431884766, 329007.79194335936, 329528.7767578125, 328135.9274047851, 328669.76914062497, 329199.3819458008, 329737.8900878906, 328291.8145385742, 328852.0506347656, 329376.3651245117, 329890.90932617185, 328487.8765136719, 329023.95423583983, 329577.99276123045, 330110.0116821289, 328679.73386230465, 329194.0836303711, 329747.1985961914, 330278.97447509767, 328802.1298095703, 329332.95782470703, 329836.68665771483, 330333.5130981445, 328807.93851318356, 329354.3212158203, 329858.12296142575, 330370.86865234375, 328782.75936279295, 329328.58306884766, 329813.1844970703, 330318.12854003906, 328725.0125854492, 329232.3870483398, 329724.6442993164, 330217.48485107423, 328624.97650146484, 329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_180_120000(self):
        """Test for latitude = -90, longitude = 180, rho = 120000"""
        lat = -90
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.8068582532915, -75.79483712035227, -75.78302089308815, -75.77142323187746, -75.79725500731846, -75.78517923286589, -75.77498859062422, -75.76504383519267, -75.79554746002594, -75.78228323265779, -75.77209259041612, -75.76370511811535, -75.79594361099782, -75.78707802545514, -75.77563062840619, -75.76720217497041, -75.80199515860244, -75.7954928185126, -75.7892090444762, -75.78315749687155, -75.81811440504367, -75.80915319685263, -75.80321093227471, -75.79751455450693, -75.83300421743431, -75.82681606604626, -75.8222534996807, -75.81792315974693, -75.8521833886237, -75.84737493544802, -75.8427304068124, -75.84073599157475, -75.8803784095175, -75.87787856028129, -75.87555629596349, -75.87345259769913, -75.91267154391335, -75.90674293971375, -75.90379229799231, -75.90104656194598, -75.94038845156524, -75.939896677945, -75.93958248924318, -75.93944588545978, -75.9816291337738, -75.98094611485678, -75.98188868096224, -75.98302249236446, -76.02672204267421, -76.02726845780782])
        expected['lon_n'] = np.array([29.12898779110935, 29.070326711422695, 29.01193883930284, 28.953827589844373, 29.08517895777289, 29.026347123356985, 28.950064155611695, 28.873996338825265, 28.987076950723996, 28.910322699925977, 28.833722128384284, 28.74649378249405, 28.84813382753301, 28.759997066483166, 28.672246211121426, 28.584555119914924, 28.660593908397384, 28.547050551182387, 28.43365233548725, 28.32029339337984, 28.395147144041303, 28.28408485304231, 28.173005486570393, 28.062153223888377, 28.1390662765846, 28.0271399666556, 27.90660932591968, 27.78598647762996, 27.853195539062884, 27.731404728425094, 27.610090323481913, 27.469851171895918, 27.516351099765366, 27.374858608466674, 27.233521503971602, 27.092157078719847, 27.145866271258242, 27.012213129579433, 26.878556572806037, 26.745121997180668, 26.797379774520437, 26.636977904504565, 26.4767331288396, 26.316542994688003, 26.341128260605462, 26.179686494288458, 26.004148925071835, 25.828660874726697, 25.83701078098704, 25.660187428659164])
        expected['rho_n'] = np.array([259169.2380126953, 259157.98516845703, 259146.90245361326, 259136.0627807617, 259160.22115478516, 259148.8953979492, 259139.34384765624, 259130.03533935547, 259093.5061279297, 259146.10041503905, 259136.62177734374, 259128.7958251953, 259093.84638671874, 259085.6072631836, 259139.78132324218, 259131.80954589843, 259099.02318115233, 259092.77700195313, 259086.7981689453, 259080.9408569336, 259048.32462158203, 259104.90479736327, 259099.2662231445, 259093.7977783203, 259061.69193115234, 259055.8103149414, 259051.48416748046, 259047.35245361328, 259079.1180419922, 259074.59746093748, 259070.1984008789, 259068.2783691406, 259039.98828125, 259037.58216552733, 259035.39478759764, 259033.3775390625, 259004.60136718748, 259064.04943847656, 259061.15723876952, 259058.48377685546, 259029.70760498047, 259029.14860839842, 259028.68682861328, 259028.4680908203, 259002.2924682617, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.35283203123, 258978.8146118164])
        expected['lat_s'] = np.array([-67.90639210071609, -67.87263730583788, -67.83869126566293, -67.80471790473129, -67.78272469560383, -67.74846446672706, -67.715679558711, -67.68266242426316, -67.66180302653794, -67.62860830717167, -67.59524966326532, -67.56380347232658, -67.54455599924549, -67.51287758187496, -67.48113086261276, -67.4491655772971, -67.43124316091499, -67.40090346062178, -67.3703998357885, -67.33978692792851, -67.32142737943951, -67.30081386852441, -67.2801593764743, -67.25943658253249, -67.25095348758333, -67.2300531087231, -67.21420706984867, -67.19829272908254, -67.19468638920077, -67.17869008616458, -67.16261182085839, -67.15033114073069, -67.15027649921734, -67.13795483795462, -67.12552389366522, -67.11309294937578, -67.11633045904236, -67.10728728858129, -67.09817581622849, -67.08905068349733, -67.09211060824552, -67.0904303817097, -67.0887638155522, -67.08708358901637, -67.09756109920318, -67.09583989153234, -67.09387279705138, -67.09185106105704, -67.1020007221637, -67.10004728806106])
        expected['lon_s'] = np.array([155.30156285694832, 155.28167334608526, 155.26160625030374, 155.24132058846882, 155.2706357603865, 155.25040474006494, 155.2522352307625, 155.25406572146005, 155.30511455531672, 155.3066035365558, 155.30791493287643, 155.29123561092325, 155.32388391515593, 155.30713629131105, 155.28976029006253, 155.27237062843568, 155.3217665565132, 155.32130210364966, 155.32044149981425, 155.3190618016019, 155.36699606919706, 155.35257070966998, 155.33768089727934, 155.3223402924035, 155.3569420307388, 155.34125991640445, 155.32477183974802, 155.3077646687147, 155.341041350351, 155.32378829250754, 155.30620738558395, 155.28450104440162, 155.31351568799585, 155.29139953546334, 155.2689965149857, 155.24611538126615, 155.26402413726993, 155.23055621033689, 155.19671945318862, 155.16255484696023, 155.17969862177696, 155.1486622421884, 155.11724337200636, 155.08529174706905, 155.10539982398555, 155.07314767072475, 155.03142887527432, 154.98940955150042, 154.99983242017385, 154.95721203975296])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_180_120000(self):
        """Test for latitude = -60, longitude = 180, rho = 120000"""
        lat = -60
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-81.4980314169346, -81.50229345497667, -81.5044517947544, -81.50886409695823, -81.5056129269133, -81.509738361172, -81.5171149654756, -81.52690945674539, -81.53133541932758, -81.53858908022613, -81.54808304317243, -81.55891572319607, -81.56739881814522, -81.5780675736288, -81.59097663116012, -81.60170002815704, -81.61791489724664, -81.63800931378483, -81.65594539054527, -81.67620373162353, -81.69285573282002, -81.71001316801511, -81.72950652790632, -81.74681422726312, -81.76327498316286, -81.78261807889234, -81.80261687278212, -81.82495159136806, -81.8446908380694, -81.8645530281758, -81.88675114297834, -81.91252827690599, -81.9385376372654, -81.96640480907902, -81.9920589996016, -82.02006277519865, -82.0441733429688, -82.06736866539018, -82.09292723326436, -82.1162181783341, -82.14077953858947, -82.17378301265896, -82.204477882789, -82.23757697950688, -82.27017064222619, -82.30070158781615, -82.33766657160425, -82.3723229514529, -82.4094928409161, -82.44623925865076])
        expected['lon_n'] = np.array([76.546455952899, 76.37006631758341, 76.16607588783181, 75.99278715839942, 76.31164087942312, 76.13370079116595, 75.91147375633041, 75.722837591833, 76.04999682288745, 75.82565925959838, 75.63148381168408, 75.4059646256686, 75.73456502663791, 75.50584931209087, 75.30657854305575, 75.07841607383148, 75.3838006618119, 75.15586358883024, 74.89982028741397, 74.6731399692396, 74.97931685916376, 74.73195472818257, 74.51431075028012, 74.26824635524123, 74.58895788771919, 74.36734556990895, 74.11522279707627, 73.89294795091655, 74.21406246455552, 73.95784157822084, 73.73142080723491, 73.46375252385128, 73.77475835751865, 73.53152849098532, 73.25864877326498, 73.0156579633526, 73.34039247725168, 73.07388532602695, 72.83768372414957, 72.57155906351836, 72.89867048324862, 72.63604287947246, 72.34285700934966, 72.07970348100741, 72.38773135219665, 72.08824121746993, 71.80691938593544, 71.49401475967882, 71.79227546035493, 71.50379559057028])
        expected['rho_n'] = np.array([272483.73455810547, 272445.23670654296, 272466.64870605466, 272429.244543457, 272434.5914672851, 272395.7047485351, 272417.2625732422, 272380.8548828125, 272326.09752197267, 272347.242175293, 272309.1817993164, 272333.0728271484, 272281.3534912109, 272304.4910888672, 272268.42365722655, 272292.12025146483, 272240.93560791016, 272204.50361328124, 272228.2731201172, 272192.81329345703, 272141.6043457031, 272167.02653808595, 272132.02849121095, 272157.86385498045, 272108.74506835936, 272074.23310546874, 272102.1829345703, 272071.0492553711, 272024.6039306641, 272052.0190673828, 272020.3020874023, 272050.9982910156, 272006.9104736328, 271976.91909179685, 272007.0076904297, 271978.06138916017, 271934.02218017576, 271963.67330322263, 271934.3138305664, 271965.13155517576, 271921.0923461914, 271892.9723876953, 271924.9081054687, 271897.8818359375, 271855.93278808595, 271887.30950927734, 271860.52628173825, 271893.9688598633, 271852.84615478513, 271825.62545166013])
        expected['lat_s'] = np.array([43.814260832373776, 43.7871962077876, 43.760742885132146, 43.73486671346155, 43.79567247254759, 43.76860443286683, 43.74489684625772, 43.72168103326884, 43.78553647181929, 43.76140199838705, 43.73770465706169, 43.71717310841663, 43.78394845283726, 43.76244360223548, 43.74141686487559, 43.72104582567603, 43.791509472248464, 43.77349143321797, 43.75593101686187, 43.738698449585925, 43.809824624507854, 43.79179975528819, 43.77470037670106, 43.758034715126236, 43.829591191965875, 43.81196588881265, 43.79718194435416, 43.78288635842132, 43.85715100526687, 43.841748928688496, 43.82695473894625, 43.81594788909877, 43.89375057393438, 43.881667969292636, 43.86998834581191, 43.858728778965144, 43.93560255807364, 43.9222871042867, 43.90944634864707, 43.897083706249354, 43.97416922122212, 43.96542316398992, 43.95712448414835, 43.94915706848153, 44.031057866819154, 44.0220044510743, 44.015669450619114, 44.00971011056828, 44.09389902227785, 44.08670000289266])
        expected['lon_s'] = np.array([200.65998853116758, 200.6143628675119, 200.5689011283963, 200.5236442949558, 200.5990769041494, 200.55342391973704, 200.49402859471462, 200.43472889234056, 200.4957361420071, 200.43613591130958, 200.37665862401713, 200.3085343172354, 200.36019786811738, 200.2917730330122, 200.22347114131207, 200.15534683453035, 200.1878175538446, 200.1007599626836, 200.01379799417103, 199.9267540633884, 199.95851444302895, 199.87540470120823, 199.7925681669543, 199.70974529307873, 199.74500272957434, 199.6618110254836, 199.5731824908135, 199.4845539561434, 199.5134866374676, 199.42447561220396, 199.33556020958875, 199.232642919175, 199.24650820319013, 199.14305815802112, 199.0396081128521, 198.93628101108814, 198.9553782200075, 198.85781579790304, 198.76032167769029, 198.6629231801259, 198.6811188040748, 198.5639537390524, 198.44689795705673, 198.32984217506106, 198.32761553339162, 198.21014994004574, 198.0823571006748, 197.95455060092553, 197.94067165653206, 197.8124280246759])
        expected['rho_s'] = np.array([265276.2756713867, 265338.39720458986, 265399.9597412109, 265461.4979736328, 265337.5951660156, 265398.74453125, 265457.29334716796, 265514.845690918, 265388.4638549805, 265447.2071044922, 265505.3913574219, 265559.5897216797, 265430.07264404296, 265485.8264770508, 265540.72966308595, 265594.2718139648, 265462.5673583984, 265516.182421875, 265568.6794921875, 265620.37452392577, 265487.0173828125, 265538.2749389648, 265586.8104248047, 265635.953515625, 265500.7735595703, 265551.3262939453, 265597.91744384763, 265643.512121582, 265504.34627685545, 265551.6665527344, 265597.82022705075, 265640.30396728514, 265498.27022705076, 265542.406652832, 265585.5466064453, 265625.7943603516, 265483.9550537109, 265528.6747802734, 265572.27651367185, 265614.90607910155, 265472.82373046875, 265513.1687011719, 265552.32276611327, 265589.41097412107, 265442.6136108398, 265482.6183227539, 265520.04678955075, 265556.74613037106, 265408.56342773436, 265446.5994995117])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_180_120000(self):
        """Test for latitude = -30, longitude = 180, rho = 120000"""
        lat = -30
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-79.10753350932183, -79.12986822790776, -79.15217562573702, -79.1744147216746, -79.14365154965284, -79.16591796634708, -79.19590249680346, -79.22508106493774, -79.20306053505362, -79.23300408437495, -79.26208702986088, -79.29606039079252, -79.27849314424725, -79.31235722215217, -79.3457158660585, -79.37974386850351, -79.37280439630678, -79.41672251266996, -79.46137828946351, -79.50614334928378, -79.49949074503218, -79.54134614426602, -79.58404848695693, -79.62666886737782, -79.61739347048493, -79.66013679431086, -79.70606298629005, -79.75196185751253, -79.7460195929346, -79.79202774718382, -79.8380632221897, -79.89077862220387, -79.89301892425163, -79.94662224885789, -80.00022557346415, -80.05313221877506, -80.05193010548115, -80.10207735436737, -80.15150060320158, -80.20163419170947, -80.20056868219896, -80.26100219597524, -80.32071170969948, -80.38122718574579, -80.39035231847694, -80.45019843598459, -80.51624636525861, -80.58238991718102, -80.59620055968279, -80.6624124134969])
        expected['lon_n'] = np.array([104.87760251686998, 104.71471616554352, 104.55294313505176, 104.39218780274635, 104.71146499549859, 104.54807321017354, 104.38505708525285, 104.25292707575896, 104.54609245531424, 104.38300119831268, 104.25276998140805, 104.09897461186686, 104.39779538805493, 104.2467457545601, 104.11731366978835, 103.96183124352218, 104.271730586544, 104.14442269060413, 103.98872852847369, 103.8316136869959, 104.14546770954715, 104.02566619150512, 103.87346908622973, 103.72504907556535, 104.07957004443486, 103.92759833540208, 103.78163036264974, 103.63936435242753, 104.00633675615398, 103.85972674561967, 103.7140114898666, 103.60830748227148, 103.96154437557703, 103.82118398813328, 103.68268824233293, 103.57643098941504, 103.9347222227064, 103.79477847680201, 103.68731228040103, 103.55069483662244, 103.91451169295233, 103.78322862691552, 103.68408160092362, 103.55361132739803, 103.93563746805518, 103.83418183812381, 103.70816484793708, 103.58106868786147, 104.00971769979314, 103.88072957731745])
        expected['rho_n'] = np.array([281771.389855957, 281765.0464599609, 281759.14053955075, 281753.7450073242, 281689.4604003906, 281683.14130859374, 281674.6834472656, 281609.621105957, 281601.11463623046, 281593.09425048827, 281528.78533935547, 281523.36550292966, 281514.39725341793, 281510.14401855465, 281445.2275024414, 281439.90488281247, 281431.2039794922, 281364.00286865234, 281355.5693115234, 281346.89271240233, 281338.3133300781, 281275.7300170898, 281270.2129638672, 281266.71315917966, 281203.9111083984, 281198.75861816405, 281195.5018554687, 281194.1894287109, 281133.6233642578, 281130.3909057617, 281127.9361816406, 281069.6061035156, 281069.2901489258, 281066.90833740233, 281065.7660400391, 281007.654699707, 281008.26230468747, 281007.8977416992, 280947.81776123046, 280948.4253662109, 280948.4253662109, 280948.4010620117, 280891.1160644531, 280892.5986206055, 280894.2026977539, 280836.2857910156, 280837.8412597656, 280839.59116210934, 280783.4241577148, 280783.8130249023])
        expected['lat_s'] = np.array([40.46382081805074, 40.445392967670045, 40.427361268261215, 40.41500887114725, 40.48385717798097, 40.4651287992768, 40.44922811888901, 40.43924579741704, 40.510614444054504, 40.49440299005948, 40.48386742326473, 40.46946596939976, 40.54229286142502, 40.52697957730585, 40.51744121812993, 40.503015858602865, 40.578755826309134, 40.57234910886766, 40.56127737222307, 40.54914012606796, 40.629760263936205, 40.616308206365865, 40.605106696127045, 40.59182197819137, 40.672254285857434, 40.65904811509722, 40.646931359509615, 40.6354327360419, 40.716735892827145, 40.70384732586333, 40.69166568347861, 40.68679234350581, 40.76493995289451, 40.75413459362755, 40.74357512117071, 40.738585667982015, 40.81549018294177, 40.80329488017872, 40.79679937027804, 40.785328067567, 40.862536525944826, 40.858465733199495, 40.849323524995434, 40.840307675291015, 40.9202823602827, 40.91583249203843, 40.907588453710225, 40.900105981474475, 40.986473723529286, 40.97798721348555])
        expected['lon_s'] = np.array([195.71159843976338, 195.67854032418052, 195.64555051048936, 195.61437752711743, 195.67080855004008, 195.63765481180883, 195.59366839355394, 195.55155344713165, 195.59676929943714, 195.55270091891222, 195.5104493687065, 195.46038408209031, 195.499220537711, 195.44895034541972, 195.40051064382598, 195.3503497345614, 195.37381826454958, 195.31050241094357, 195.2455746326934, 195.180114099688, 195.20449787502497, 195.14189236109263, 195.07992888494226, 195.0171457860915, 195.04405673142136, 194.98138291559732, 194.91411921265103, 194.8469101512181, 194.8686164924004, 194.80105226113065, 194.7335973128876, 194.6570445526701, 194.66573255329436, 194.58730832124428, 194.50882944768082, 194.4320034798965, 194.44495351856287, 194.37124211704008, 194.29921094205312, 194.22554052166538, 194.23795780557646, 194.15090021441546, 194.06225801936705, 193.97356118280527, 193.97036465427368, 193.88308849705928, 193.78631837689852, 193.68967120014284, 193.67926199184774, 193.58239624903862])
        expected['rho_s'] = np.array([281252.2278564453, 281237.28077392577, 281221.3129150391, 281262.26549072267, 281225.73627929686, 281211.421105957, 281195.5018554687, 281235.82252197264, 281198.53988037107, 281183.17962646484, 281224.7884155273, 281207.96990966797, 281170.201184082, 281155.32701416017, 281197.1545410156, 281180.1659057617, 281142.37287597655, 281183.37406005856, 281165.0243896484, 281149.0565307617, 281170.2740966797, 281156.00753173826, 281136.41834716796, 281121.5684814453, 281142.32426757814, 281127.2556640625, 281111.50654296874, 281094.201953125, 281115.03065185546, 281100.9099121094, 281085.087878418, 281126.81818847655, 281089.70567626954, 281075.0016357422, 281059.5198608398, 281101.0557373047, 281064.7938720703, 281050.77034912107, 281093.52143554686, 281077.6021850586, 281040.8299316406, 281083.67823486327, 281067.95341796876, 281051.8154296875, 281014.7515258789, 281057.867175293, 281042.5312255859, 281025.2509399414, 281046.6386352539, 281031.32698974607])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_180_120000(self):
        """Test for latitude = 0, longitude = 180, rho = 120000"""
        lat = 0
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-76.4772276599499, -76.50307309576922, -76.5289321919669, -76.55472298627288, -76.52950592785717, -76.55458638248948, -76.59018532844357, -76.62577061401936, -76.61088080162872, -76.64638412493446, -76.68127273121488, -76.72206262093817, -76.71292382782872, -76.75364541566032, -76.79433968273526, -76.83455583656828, -76.83930964823062, -76.89350036910551, -76.94767742960204, -77.0019227919903, -77.00720935840789, -77.05836747529128, -77.10907479968944, -77.16031487884288, -77.16300597337587, -77.21419141101595, -77.26920175459122, -77.32430772081489, -77.33078274014807, -77.38583406485836, -77.440899049947, -77.5054716583603, -77.52231490485357, -77.58688751326687, -77.65152842357185, -77.7161829942552, -77.72883250459807, -77.78893816929417, -77.84907115474698, -77.90884897036293, -77.92183999016427, -77.99493667466177, -78.06804701953757, -78.14121200592675, -78.16749457385296, -78.24060491872879, -78.32075035844971, -78.40095043968401, -78.43454131002213, -78.51468674974305])
        expected['lon_n'] = np.array([97.13352037553119, 96.95056009823405, 96.76895219839258, 96.58884694016852, 96.88371986701631, 96.72267766676575, 96.53345410599971, 96.34542582833843, 96.63781256632834, 96.44773523191606, 96.28329159745883, 96.09630833874056, 96.39752651132729, 96.2120732149831, 96.02906512636179, 95.8664588127913, 96.16387940019949, 95.9645334990835, 95.76645118296396, 95.56857377251953, 95.87086428480595, 95.67892913893942, 95.51270282510882, 95.32373881153124, 95.63887690964647, 95.44440776359787, 95.25688491993516, 95.07048222729634, 95.42320685641414, 95.23242601251735, 95.0420208290249, 94.85583670243952, 95.19114434917378, 95.00081429776222, 94.81105115205175, 94.6206459685593, 94.96751029536924, 94.77809548930644, 94.58872849456783, 94.4241482563272, 94.77645624390563, 94.58419324895894, 94.39262010311843, 94.20110159879127, 94.56112086994264, 94.36409040295528, 94.16827570964018, 93.97194192194817, 94.33959734459162, 94.13939766982938])
        expected['rho_n'] = np.array([278606.83729248046, 278580.12697753904, 278554.60756835935, 278529.9874145508, 278542.0909057617, 278457.41507568356, 278426.4758300781, 278396.4115356445, 278403.7514038086, 278372.8121582031, 278285.60869140626, 278255.9575683594, 278264.99873046874, 278236.1496459961, 278208.6859008789, 278122.260168457, 278124.88502197264, 278087.45655517577, 278051.3162109375, 278015.75916748046, 278018.9187133789, 277987.22603759763, 277898.95318603516, 277870.03118896484, 277877.88144531247, 277846.60194091796, 277818.5548950195, 277791.77166748047, 277744.96177978517, 277716.2585205078, 277688.4545166016, 277661.76850585936, 277672.3165283203, 277642.3737548828, 277613.16010742186, 277585.2345825195, 277598.6018920898, 277573.1553955078, 277548.70537109376, 277466.5571777344, 277480.7022216797, 277453.74886474607, 277428.1079345703, 277403.31765136716, 277415.59127197263, 277388.6865234375, 277362.29216308595, 277336.8699707031, 277348.1228149414, 277321.4368041992])
        expected['lat_s'] = np.array([42.53400334415742, 42.5213435885308, 42.508776040457974, 42.49660464335701, 42.56253304432056, 42.549835722653505, 42.539464080398844, 42.52905145700916, 42.5970084241562, 42.586281612064695, 42.57593046037754, 42.56593106343264, 42.629308388741194, 42.62383740721601, 42.61371506686605, 42.603821537853285, 42.67016999545079, 42.662393825080734, 42.660054485290004, 42.65277350363477, 42.71882826309796, 42.70882203596389, 42.702377752481986, 42.69687603510554, 42.76231607754343, 42.75304409574514, 42.744263887567094, 42.73568516996956, 42.805913175015625, 42.79650800452852, 42.78746483406742, 42.77985941842661, 42.846474253401745, 42.83815849808725, 42.835026856352805, 42.82720628975314, 42.892765860501505, 42.883316293784794, 42.87419116105366, 42.865366556646, 42.935980467380176, 42.9282077121047, 42.92075597572022, 42.91356037142961, 42.98101873476724, 42.97313669646505, 42.9664294507001, 42.96498486569064, 43.03313990832361, 43.025906737992564])
        expected['lon_s'] = np.array([192.70407490291, 192.6790217690344, 192.65391399364543, 192.62886085976984, 192.67661754244656, 192.65153708781426, 192.6176866702877, 192.5836996489777, 192.62241316119335, 192.58845346064004, 192.55453474122174, 192.515534361061, 192.54672100481125, 192.50990628518488, 192.47087858426744, 192.4318235625933, 192.45074318659422, 192.39961239046752, 192.35068091525355, 192.29942717572177, 192.31777306383242, 192.26826785272817, 192.21965056621602, 192.171962185431, 192.19209758310419, 192.14266067389164, 192.08928957571717, 192.03585017565098, 192.05364964862804, 192.0000599844001, 191.94648398055054, 191.88411069305, 191.8905310708698, 191.8279801984508, 191.76745106202617, 191.70483188771547, 191.71472200163367, 191.65598237477155, 191.5972564082878, 191.53853044180403, 191.54999149923128, 191.47894387148483, 191.40796454563005, 191.33695789901859, 191.33366574783867, 191.26253615782213, 191.1849861899858, 191.10938965625212, 191.09902142909203, 191.02132119709398])
        expected['rho_s'] = np.array([277997.09354248043, 277965.20643310546, 277933.1491943359, 277900.265612793, 277953.1758544922, 277921.3616577148, 277889.15859375, 277857.10135498043, 277910.37615966797, 277878.9022216797, 277846.67485351564, 277816.1001708984, 277812.7947998047, 277841.5709716797, 277811.2879394531, 277780.47021484375, 277776.99471435545, 277747.70815429685, 277775.8524169922, 277745.39925537107, 277742.8473144531, 277717.0362548828, 277683.52076416014, 277717.0848632812, 277714.8245727539, 277687.28791503905, 277660.48038330075, 277633.1867675781, 277691.83280029293, 277666.2161743164, 277639.7732055664, 277614.3510131836, 277615.9793945312, 277591.91823730466, 277625.846899414, 277600.5462280273, 277602.5634765625, 277578.67244873045, 277554.0279907226, 277528.6544067383, 277589.1475585937, 277566.95782470703, 277543.8688354492, 277520.19654541014, 277523.8907836914, 277501.3850952148, 277478.73358154297, 277513.87745361327, 277518.54385986325, 277496.57286376954])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_180_120000(self):
        """Test for latitude = 30, longitude = 180, rho = 120000"""
        lat = 30
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-72.17965897379955, -72.21961558044413, -72.25958584746704, -72.29935120881484, -72.31731460633199, -72.35582321287251, -72.40760970715957, -72.45900005047471, -72.48953099606467, -72.53967824495092, -72.58955228627036, -72.64665266773164, -72.68375425530317, -72.73958422157884, -72.79523660293611, -72.85067041823993, -72.90378196922595, -72.97462469129732, -73.04512590391019, -73.11536756933461, -73.16764583724188, -73.23025135117422, -73.2926929405665, -73.35495694504033, -73.40063725020937, -73.46172646214595, -73.52572533466898, -73.58957394303025, -73.63804096538067, -73.70090602650146, -73.7635115404338, -73.83804256465697, -73.89857170108164, -73.972105517686, -74.0455163908853, -74.11873601878781, -74.1744293812801, -74.24187066914482, -74.30920267398281, -74.3763980750374, -74.43209143752966, -74.51341166978784, -74.5944860152359, -74.67542375690056, -74.74625281859358, -74.82613871112605, -74.91446671747266, -75.00261713890083, -75.08218884273148, -75.16917813200078])
        expected['lon_n'] = np.array([81.20026681558971, 80.96445453449502, 80.72904523456135, 80.49365642519521, 80.67449251366047, 80.43499876060314, 80.1748027041715, 79.91303570423077, 80.07512292342435, 79.80956516849425, 79.54376835694319, 79.29078498027509, 79.4690802383691, 79.21100837076935, 78.95258816352191, 78.6937103336001, 78.83734238165628, 78.53464888820884, 78.23041860219813, 77.92534137273033, 78.07062632656567, 77.78223524924023, 77.49292209637683, 77.20285762270473, 77.37406997462944, 77.07764659484005, 76.78948091375722, 76.49986040238116, 76.68183030224864, 76.38674563953658, 76.08989878801866, 75.7882298229467, 75.96831459060326, 75.65959004011869, 75.34954043293513, 75.0378652407291, 75.23877525516502, 74.93895044116898, 74.63780057047394, 74.33570813367345, 74.53867403504955, 74.21553095522708, 73.8903866299778, 73.56358939894936, 73.75469126173715, 73.42045595470258, 73.07475959024072, 72.72657020673181, 72.91047988032358, 72.55384155281136])
        expected['rho_n'] = np.array([239238.70096435546, 239149.11568603516, 239058.9714111328, 238969.31322021483, 239062.93299560546, 238970.50412597656, 238871.24577636717, 238770.35904541015, 238852.9933227539, 238750.67264404296, 238648.570703125, 238550.35743408202, 238635.6165649414, 238535.1430053711, 238436.34643554688, 238336.4561767578, 238405.9418823242, 238289.47615966797, 238173.54512939454, 238055.49963378906, 238123.40556640623, 238013.8422363281, 237904.83790283202, 237796.36826171874, 237872.2702758789, 237761.88060302733, 237653.87274169922, 237546.05931396483, 237623.41958007813, 237514.3909423828, 237404.75469970703, 237295.4587158203, 237371.84681396483, 237261.14118652343, 237150.1439086914, 237040.45905761718, 237122.2912963867, 237017.2728515625, 236913.17796630858, 236808.8400390625, 236891.49862060547, 236782.05681152342, 236674.34060058594, 236566.25982666016, 236645.22416992186, 236536.09831542967, 236425.61142578124, 236316.38835449217, 236393.1653198242, 236281.73056640624])
        expected['lat_s'] = np.array([50.32868352686767, 50.32126252633445, 50.31760496003391, 50.310720129350536, 50.36420392564632, 50.35677951001852, 50.354849981577985, 50.349682943470874, 50.40504504178841, 50.39931109798019, 50.397303022364206, 50.39211549368958, 50.44413421440839, 50.44187342179312, 50.43639561007877, 50.43460951561081, 50.48940470822724, 50.48620817969567, 50.48669995331591, 50.48407374558005, 50.54243429694322, 50.53725359845777, 50.53283105097018, 50.532072899972306, 50.58500003585074, 50.58380816784057, 50.57865137501721, 50.573781450138995, 50.63012368060243, 50.62470051040144, 50.62284952913637, 50.6185806609051, 50.67205762701172, 50.67059938162392, 50.66585923033993, 50.661371796055235, 50.717365686871, 50.711232176996326, 50.708889422111014, 50.70325451604575, 50.75574793491189, 50.75435799141579, 50.749669066550574, 50.74523285868465, 50.80247667411853, 50.79739842847062, 50.79322176779316, 50.7927573149296, 50.84707097920954, 50.84245377133061])
        expected['lon_s'] = np.array([191.77157649628484, 191.7509493249914, 191.73290396520423, 191.71218117126242, 191.75791611794483, 191.73711136173296, 191.71123860515695, 191.68240154648117, 191.71996758691623, 191.69107588672708, 191.66502554523265, 191.63135271262448, 191.66105037513572, 191.63010961819555, 191.59632750256065, 191.56529112297213, 191.58407414318967, 191.53941836639612, 191.49749466527058, 191.452715945072, 191.47382122960732, 191.4302719434593, 191.3868182799597, 191.34606937137139, 191.36533050483084, 191.32448597359416, 191.27677027205246, 191.22899992899738, 191.24670377932605, 191.1988378136226, 191.15356731980376, 191.0973412025562, 191.10352935394425, 191.0497621047979, 190.99324911960522, 190.93670881365586, 190.94862066356836, 190.89545447106897, 190.84489741083254, 190.7916765768198, 190.80048752084912, 190.73837378053702, 190.67365090796199, 190.60887339387358, 190.60832697873997, 190.5434128608682, 190.47251549728344, 190.40407700179992, 190.39439179355685, 190.323289524297])
        expected['rho_s'] = np.array([265160.5633789062, 265111.56611328124, 265123.6209960937, 265074.23486328125, 265124.64177246095, 265075.984765625, 265089.1576416016, 265040.91380615236, 265091.66097412107, 265043.63587646483, 265056.4198852539, 265009.658605957, 265001.2007446289, 265016.26934814453, 264969.92124023434, 264984.3822387695, 264977.0909790039, 264932.66290283203, 264948.8251953125, 264903.23051757814, 264956.6997558594, 264913.7542358398, 264869.7879394531, 264886.6064453125, 264880.992175293, 264898.3939819336, 264856.85810546874, 264814.8604492187, 264871.99962158204, 264830.7310913086, 264850.41749267577, 264810.8016479492, 264809.3190917969, 264831.60604248045, 264792.47628173826, 264752.90904541017, 264812.4543334961, 264773.22735595703, 264794.66365966795, 264754.99920654297, 264753.46804199216, 264778.2097167969, 264741.38885498047, 264704.2520385742, 264766.51939697267, 264729.99018554686, 264694.4331420898, 264719.6122924805, 264722.09132080077, 264686.8259277344])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_180_120000(self):
        """Test for latitude = 60, longitude = 180, rho = 120000"""
        lat = 60
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([76.45915338554431, 76.4611836592751, 76.46300048959432, 76.46490952746734, 76.45599015418495, 76.45798713074353, 76.45997898466024, 76.46203401782677, 76.45333406437146, 76.45531994187263, 76.45727849861714, 76.45648705044707, 76.44479718168259, 76.44384607784066, 76.44284118625902, 76.44177994561673, 76.43107020899815, 76.4312281571227, 76.4310582561671, 76.4308422514346, 76.41966976949976, 76.41798954296394, 76.41623247679995, 76.41426623609263, 76.4015415936689, 76.39900161707129, 76.39814784342505, 76.39683730087805, 76.3849707009688, 76.38341512538533, 76.38152230921159, 76.38745774360032, 76.38284736591056, 76.38818601252058, 76.39321303174971, 76.39776791415247, 76.39254964962657, 76.39702683862751, 76.4012461879873, 76.4052427024254, 76.39891709348032, 76.3971796641102, 76.3947856828061, 76.39215264488107, 76.37963205435878, 76.37605645032829, 76.37476383702786, 76.3727856434895, 76.36106674642105, 76.35818867545953])
        expected['lon_n'] = np.array([205.99014521604047, 205.9348616648984, 205.8791683024061, 205.82351592104885, 205.80829825957807, 205.75315131221936, 205.67718594827048, 205.6011522824299, 205.56618171387944, 205.49042125560567, 205.4147290992236, 205.27472388161667, 205.17587738394823, 205.03645956260993, 204.89698709975826, 204.75755561804164, 204.61555598519706, 204.43261619846746, 204.24947150606272, 204.06617654949625, 203.92434084119176, 203.72138860019402, 203.51839537806123, 203.31518358987503, 203.15410040848943, 202.95087495992487, 202.72285592467318, 202.49463198374636, 202.30900110248376, 202.08132357669055, 201.8534274848439, 201.5787445971827, 201.34699627864416, 201.07285980611653, 200.79884627699394, 200.5248054271147, 200.31171718538863, 200.0558719594583, 199.8004502052565, 199.54531531899985, 199.33375703964788, 199.0425450941952, 198.75164733744435, 198.4615282222589, 198.21575069516516, 197.92702493857038, 197.5932472542102, 197.25979741893016, 196.97081211514694, 196.63900152526773])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.199240493775, 6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.2, 6371.198860740661, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.200759506225, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2])
        expected['lat_s'] = np.array([62.87688508886707, 62.87156095640904, 62.86695057871928, 62.86258779538694, 62.90390873231823, 62.89857606212374, 62.895517844922864, 62.892727712646916, 62.93576131951258, 62.931960319239465, 62.928885026565666, 62.926241743356876, 62.96911654832434, 62.965971246211545, 62.96305475543595, 62.960368783544844, 63.00568879623517, 63.00486405089289, 63.004259579151345, 63.0033579941809, 63.049253450308804, 63.04662895012022, 63.04472844998367, 63.043046515900556, 63.08724979266158, 63.085055594390724, 63.08283236781588, 63.08081916955802, 63.12482437083312, 63.122292078198335, 63.11947462516571, 63.118014672230615, 63.162648250909356, 63.16063163755691, 63.15884042044708, 63.15727630712714, 63.20093829139645, 63.197857876080775, 63.19448717772537, 63.191857554894916, 63.23552295425881, 63.23360025600745, 63.231892708714945, 63.229864142531454, 63.275255872208064, 63.27265698522888, 63.27150268325914, 63.27056011715368, 63.31606112985701, 63.31451921465188])
        expected['lon_s'] = np.array([194.6204893802322, 194.60009443537052, 194.58318288698558, 194.5662030367089, 194.62921836199146, 194.60839994540126, 194.5821993397451, 194.5556845453871, 194.6093288511284, 194.5793716414287, 194.55278854517903, 194.52014024094635, 194.56389443276947, 194.53105488324005, 194.49816069219725, 194.46519819926277, 194.4963985033914, 194.45093676427578, 194.4053794025118, 194.3562293612444, 194.39024370331106, 194.34166739793392, 194.2963695833584, 194.2509761461345, 194.28187592193964, 194.2361956167706, 194.18480527345542, 194.1332646659785, 194.1579762903956, 194.10617613573024, 194.05098820723651, 193.98849197633092, 194.0019747697525, 193.93905506711835, 193.8759987607008, 193.81284683163486, 193.8295124932097, 193.76996690402552, 193.7071291636614, 193.64743331031548, 193.66342961335167, 193.58913081556025, 193.5147773762555, 193.43717276690583, 193.44117525775945, 193.36325645970794, 193.28102098210098, 193.19866256108898, 193.19134059829872, 193.1085996866932])
        expected['rho_s'] = np.array([255350.41640625, 255291.81898193358, 255296.4853881836, 255301.1517944336, 255345.48265380858, 255286.7880126953, 255292.98558349608, 255299.25606689454, 255345.16669921874, 255288.17335205077, 255294.5410522461, 255302.1725708008, 255286.10749511718, 255293.9091430664, 255301.63787841797, 255309.39091796873, 255295.65904541014, 255305.91541748046, 255316.19609374998, 255263.0428100586, 255312.6962890625, 255259.78604736328, 255270.0667236328, 255280.2987915039, 255266.73704833983, 255277.16354980468, 255288.80526123047, 255300.34975585938, 255288.0761352539, 255299.81506347656, 255248.16864013672, 255261.92481689452, 255251.74135742188, 255265.2301879883, 255278.67041015625, 255292.06202392577, 255281.05222167968, 255293.93344726562, 255243.45362548827, 255256.23763427735, 255245.3007446289, 255260.976953125, 255276.50733642577, 255228.70097656248, 255283.94442138672, 255236.16236572264, 255253.00517578123, 255269.75076904296, 255263.01850585936, 255279.86131591795])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_180_120000(self):
        """Test for latitude = 90, longitude = 180, rho = 120000"""
        lat = 90
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([78.8984418957194, 78.90707440105666, 78.91558737808343, 78.92395179849576, 78.89862204195877, 78.90718368408338, 78.91759374615212, 78.92781000160318, 78.90424499519298, 78.9146729865083, 78.9249327844153, 78.93787172402425, 78.91678351496184, 78.9297173319289, 78.94248295548765, 78.95502403657744, 78.93649117203826, 78.95209815429173, 78.96743961294123, 78.98257531214198, 78.96340126359446, 78.979382198705, 78.99510017153248, 79.01055518207694, 78.99158603920452, 79.00691042238108, 79.02556537655168, 79.04393602409809, 79.02769895689367, 79.04587409027508, 79.06365221891097, 79.09216569737484, 79.08588192333842, 79.11393863290154, 79.14157101696247, 79.16872528778151, 79.16107889100567, 79.1878233504745, 79.21402054103609, 79.23979852873737, 79.23086805639758, 79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212])
        expected['lon_n'] = np.array([222.03207391586602, 221.9699874963106, 221.90740930313495, 221.84420273255566, 221.79726567257933, 221.73515193226723, 221.64709713348742, 221.55822271200722, 221.48625983891196, 221.39868315337407, 221.31038246778414, 221.13775626670125, 220.9828065951903, 220.81105465832118, 220.63863336291337, 220.46533780329182, 220.27212541205049, 220.0598431326465, 219.84644070221862, 219.6319044603885, 219.4388286729306, 219.20795461860584, 218.9758647906088, 218.74253186818282, 218.5328723814201, 218.29949847785912, 218.03881747799642, 217.77659285538127, 217.54065080069236, 217.27835787618554, 217.01458963081797, 216.69424009836604, 216.40249539815812, 216.0810666958173, 215.75787580467065, 215.43290906433978, 215.1539504782581, 214.84212502189035, 214.5286329993651, 214.21382958051922, 213.935089560491, 213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([73.76546679576927, 73.6725284117329, 73.5846785186282, 73.49383358757244, 73.72931460449239, 73.6331455409786, 73.54314926092722, 73.4522309053379, 73.6852137805689, 73.59044832092957, 73.49639490851119, 73.41110975143982, 73.64449902292645, 73.55596098826288, 73.46222688718821, 73.37188226748916, 73.6103702751912, 73.51583191934176, 73.42088545968942, 73.32817076434837, 73.57029926287802, 73.47759993546262, 73.39053380656517, 73.30273513987925, 73.55243319555656, 73.45981583041119, 73.37157149388192, 73.28712474253118, 73.54256527975319, 73.45781287743708, 73.37191471088771, 73.28943334647064, 73.54950987459179, 73.46764834984191, 73.38075980586288, 73.30005941327188, 73.57014899871629, 73.48558784169694, 73.40441787360051, 73.32345622627376, 73.60148932172288, 73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512])
        expected['lon_s'] = np.array([350.3579412496324, 350.43211710401874, 350.4985885050213, 350.56877552893235, 350.38744766684687, 350.46648661592224, 350.5413454892256, 350.61713326825605, 350.4496570298073, 350.5321657149811, 350.6124887396204, 350.6834680654752, 350.5244612615973, 350.6013145501383, 350.684424291959, 350.7623977315239, 350.6063142486107, 350.7001883685634, 350.7917948657116, 350.8799862682748, 350.724476521252, 350.8160556976435, 350.8970617411998, 350.97855955837645, 350.81900633936493, 350.90911019489573, 350.99768408805244, 351.08030205625295, 350.9240546487997, 351.0074649189439, 351.09259639675895, 351.1788753463546, 351.03057827909527, 351.11713043625764, 351.21089527318355, 351.2948246377047, 351.1359817583669, 351.2243097647135, 351.307774676371, 351.3893544558177, 351.22447368925356, 351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506])
        expected['rho_s'] = np.array([327743.56041259767, 328264.4966186523, 328804.7303588867, 329309.40705566405, 327946.1116088867, 328462.8431884766, 329007.79194335936, 329528.7767578125, 328135.9274047851, 328669.76914062497, 329199.3819458008, 329737.8900878906, 328291.8145385742, 328852.0506347656, 329376.3651245117, 329890.90932617185, 328487.8765136719, 329023.95423583983, 329577.99276123045, 330110.0116821289, 328679.73386230465, 329194.0836303711, 329747.1985961914, 330278.97447509767, 328802.1298095703, 329332.95782470703, 329836.68665771483, 330333.5130981445, 328807.93851318356, 329354.3212158203, 329858.12296142575, 330370.86865234375, 328782.75936279295, 329328.58306884766, 329813.1844970703, 330318.12854003906, 328725.0125854492, 329232.3870483398, 329724.6442993164, 330217.48485107423, 328624.97650146484, 329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90S_270_120000(self):
        """Test for latitude = -90, longitude = 270, rho = 120000"""
        lat = -90
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.8068582532915, -75.79483712035227, -75.78302089308815, -75.77143689225582, -75.79725500731846, -75.78517923286589, -75.77498859062422, -75.76505749557103, -75.79554746002594, -75.78229689303612, -75.77209259041612, -75.76370511811535, -75.79594361099782, -75.78707802545514, -75.77561696802786, -75.76720217497041, -75.80199515860244, -75.7954928185126, -75.7892090444762, -75.78315749687155, -75.81811440504367, -75.80915319685263, -75.80321093227471, -75.79751455450693, -75.83300421743431, -75.82681606604626, -75.8222534996807, -75.81792315974693, -75.8521833886237, -75.84737493544802, -75.8427304068124, -75.84073599157475, -75.8803784095175, -75.87787856028129, -75.87555629596349, -75.87345259769913, -75.91267154391335, -75.90675660009211, -75.90379229799231, -75.90104656194598, -75.94038845156524, -75.939896677945, -75.93958248924318, -75.93944588545978, -75.9816291337738, -75.98095977523514, -75.98188868096224, -75.98302249236446, -76.02672204267421, -76.02726845780782])
        expected['lon_n'] = np.array([29.12900828167686, 29.07033695670645, 29.011952499681183, 28.9538446653173, 29.085189203056643, 29.02635736864074, 28.950077815990035, 28.874016829392772, 28.987097441291503, 28.910336360304317, 28.833735788762624, 28.74650744287239, 28.84814748791135, 28.760010726861506, 28.672253041310597, 28.584567072745973, 28.66061098387031, 28.547064211560727, 28.43366599586559, 28.320305346210887, 28.395159096872348, 28.284100220967943, 28.173020854496023, 28.062166884266716, 28.13907993696294, 28.02715191948665, 27.906621278750727, 27.7860001380083, 27.853207491893933, 27.73142009635073, 27.61010056876567, 27.469863124726967, 27.516363052596414, 27.37486885375043, 27.233538579444527, 27.092169031550895, 27.145881639183877, 27.01222508241048, 26.878568525637085, 26.745135657559008, 26.79739514244607, 26.636993272430196, 26.476748496765236, 26.316560070160925, 26.341145336078387, 26.17970186221409, 26.00416600054476, 25.828671120010455, 25.83702614891267, 25.660199381490212])
        expected['rho_n'] = np.array([259169.2380126953, 259157.98516845703, 259146.90245361326, 259136.08708496092, 259160.22115478516, 259148.87109375, 259139.34384765624, 259130.05964355468, 259093.5061279297, 259146.1247192383, 259136.62177734374, 259128.7958251953, 259093.84638671874, 259085.6072631836, 259139.78132324218, 259131.80954589843, 259099.02318115233, 259092.77700195313, 259086.7981689453, 259080.9408569336, 259048.32462158203, 259104.88049316406, 259099.2662231445, 259093.7734741211, 259061.69193115234, 259055.8103149414, 259051.48416748046, 259047.35245361328, 259079.1180419922, 259074.59746093748, 259070.1984008789, 259068.2783691406, 259039.98828125, 259037.58216552733, 259035.39478759764, 259033.3775390625, 259004.60136718748, 259064.02513427733, 259061.15723876952, 259058.48377685546, 259029.70760498047, 259029.14860839842, 259028.7111328125, 259028.4680908203, 259002.2924682617, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.35283203123, 258978.8146118164])
        expected['lat_s'] = np.array([-67.90639210071609, -67.87263730583788, -67.83869126566293, -67.80470424435293, -67.78272469560383, -67.74846446672706, -67.71565223795432, -67.68266242426316, -67.66180302653794, -67.62860830717167, -67.59526332364368, -67.56380347232658, -67.54455599924549, -67.51287758187496, -67.4811172022344, -67.44915191691877, -67.43124316091499, -67.40090346062178, -67.37041349616686, -67.33978692792851, -67.32142737943951, -67.30081386852441, -67.28011839533929, -67.25942292215413, -67.25093982720497, -67.2300531087231, -67.214220730227, -67.19829272908254, -67.19467272882241, -67.17867642578625, -67.16261182085839, -67.15035846148737, -67.15027649921734, -67.13795483795462, -67.12552389366522, -67.11309294937578, -67.11634411942072, -67.1072599678246, -67.09816215585013, -67.08906434387569, -67.09213792900218, -67.0904303817097, -67.08875015517387, -67.08708358901637, -67.09753377844649, -67.09583989153234, -67.0938864574297, -67.09185106105704, -67.10204170329871, -67.10000630692605])
        expected['lon_s'] = np.array([155.30154919657, 155.28167334608526, 155.26157892954708, 155.24130692809047, 155.2706357603865, 155.25040474006494, 155.25233085341088, 155.25406572146005, 155.30512821569508, 155.30658987617744, 155.3077646687147, 155.29123561092325, 155.32388391515593, 155.30713629131105, 155.28976029006253, 155.27238428881404, 155.3217665565132, 155.32130210364966, 155.3204278394359, 155.3190618016019, 155.36699606919706, 155.35257070966998, 155.33777651992773, 155.3223402924035, 155.35695569111712, 155.34127357678278, 155.32463523596462, 155.3077646687147, 155.341082331486, 155.32380195288587, 155.30620738558395, 155.28447372364496, 155.31352934837417, 155.29139953546334, 155.26898285460737, 155.24610172088782, 155.2640104768916, 155.2305971914719, 155.19673311356698, 155.16251386582522, 155.1796849613986, 155.1486212610534, 155.11727069276304, 155.08530540744738, 155.10545446549892, 155.07314767072475, 155.03140155451763, 154.98939589112206, 154.99972313714713, 154.95738962467138])
        expected['rho_s'] = np.array([6371.2, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.198860740661, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_270_120000(self):
        """Test for latitude = -60, longitude = 270, rho = 120000"""
        lat = -60
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-74.57817552349789, -74.57629039128699, -74.57395446659083, -74.57172782492142, -74.63602722576792, -74.63259847080457, -74.62821348935742, -74.62366458337019, -74.685887606709, -74.67971311569931, -74.67371620960805, -74.66531507692892, -74.72395908114265, -74.71441047668299, -74.70483455146663, -74.69562745646544, -74.75166232841622, -74.73928602564015, -74.72682776059406, -74.7142465521429, -74.76920225420483, -74.75433976257088, -74.73901281807338, -74.72371319433253, -74.77668814153515, -74.76050059320224, -74.74300164854864, -74.72501093027483, -74.77552700937625, -74.75673032878038, -74.73864398785818, -74.71837198640159, -74.76604670680828, -74.74440866751766, -74.72346730752241, -74.7026762116889, -74.75070610193242, -74.72995598723392, -74.70936979707551, -74.6882781729185, -74.73595289332519, -74.71189696706841, -74.68863334275534, -74.66490526557874, -74.70924685367044, -74.68474013492843, -74.65868979343401, -74.63276239534466, -74.6755467003056, -74.64888164178586])
        expected['lon_n'] = np.array([335.9678254986827, 336.1695073244949, 336.3876362458284, 336.5853565619219, 336.78690178395067, 336.98372051507374, 337.1762772081547, 337.366593599188, 337.5630571604742, 337.76943815643534, 337.9550553773196, 338.11176723763634, 338.27735834387414, 338.4316113360897, 338.58400651685105, 338.71602041312906, 338.8727869149592, 339.01512805726225, 339.1544912370872, 339.2907671714073, 339.441249899201, 339.5304794905181, 339.6352272716294, 339.73759814690953, 339.8389581541925, 339.93958050104516, 339.9944405804587, 340.06470956663975, 340.1357162132512, 340.2044005955449, 340.2537965236224, 340.27068075125067, 340.3065255840149, 340.3391465674909, 340.3525610590208, 340.3641450598531, 340.39291381663725, 340.4007821945611, 340.40712061011084, 340.42881329091483, 340.4538937455471, 340.4261904982735, 340.3801003817543, 340.3492552474625, 340.33835426554714, 340.3064163009882, 340.2566652030738, 340.2052202182453, 340.1567258751382, 340.1044066260959])
        expected['rho_n'] = np.array([257992.33146972655, 257973.90888671874, 258015.76071777343, 257998.6262573242, 257988.3698852539, 257970.7493408203, 257954.36831054688, 257938.497668457, 257930.06411132813, 257974.00610351562, 257958.59724121093, 257955.09743652344, 257959.39927978514, 257956.0939086914, 257953.2503173828, 257891.1287841797, 257897.83674316405, 257897.496484375, 257897.98256835938, 257899.12486572264, 257907.65563964844, 257861.23461914062, 257875.40396728515, 257890.2538330078, 257852.16915283204, 257867.11623535157, 257834.13543701172, 257861.8665283203, 257836.66307373045, 257864.66151123046, 257832.94453125, 257815.27537841795, 257804.87318115233, 257847.62426757812, 257830.90297851563, 257814.8135986328, 257807.81398925782, 257793.6932495117, 257780.2044189453, 257827.69482421875, 257821.8132080078, 257822.54233398437, 257763.38591308592, 257765.4517700195, 257774.614453125, 257776.87474365233, 257786.62072753906, 257797.12014160157, 257754.44196777343, 257765.23303222656])
        expected['lat_s'] = np.array([44.420675762738774, 44.38099919385017, 44.34334777605048, 44.30771467915052, 44.32644647294928, 44.29336445170432, 44.26850256312547, 44.24587756149981, 44.27877516763718, 44.2594764681373, 44.24273225938702, 44.22430099391174, 44.262259770224084, 44.246997712523694, 44.23391448516853, 44.223481371211335, 44.28274692263954, 44.28883945137919, 44.29786554636736, 44.309941320819945, 44.380845514593844, 44.397938062991805, 44.41739385684259, 44.439298273510815, 44.520003788743686, 44.54525841319981, 44.57533173611537, 44.60742679502526, 44.69840491476983, 44.73290420026757, 44.76946278780006, 44.81148894176315, 44.91273283583025, 44.95682170692268, 45.002816200793546, 45.050706072159095, 45.15254419268398, 45.19676966755981, 45.24262414255269, 45.29032276862148, 45.397853851819576, 45.45101321412978, 45.505634236922376, 45.56180912775115, 45.677867702128005, 45.73532325342615, 45.79689399369923, 45.86002201710307, 45.98313276179797, 46.04721018149644])
        expected['lon_s'] = np.array([222.99917406082648, 223.11370267282925, 223.22724773759154, 223.34028736835523, 223.63253750256175, 223.74266747273904, 223.85137676356896, 223.95988114872378, 224.24721354672786, 224.35308147886306, 224.45879914683653, 224.55253666300578, 224.82489728634923, 224.917009217496, 225.0093943562096, 225.1019844005983, 225.3716402690304, 225.46164850191283, 225.55194360274038, 225.64264851491816, 225.9107471002195, 225.97429518025731, 226.03829405278032, 226.1025934536268, 226.34363082943656, 226.4068100792592, 226.4457558179066, 226.4850020848775, 226.70004376070617, 226.7380059521131, 226.77628233222185, 226.79949131502156, 226.99751215943857, 227.01924582137755, 227.04130733239668, 227.06386061703608, 227.25957285751363, 227.2795579910251, 227.30017150194018, 227.32115384307045, 227.5154044230656, 227.5137924984215, 227.51248110210085, 227.51134729069864, 227.68233424638072, 227.67934262352426, 227.6696574152812, 227.6600405089298, 227.82168376582732, 227.8101680668867])
        expected['rho_s'] = np.array([254385.05361328126, 254538.36450195312, 254683.36335449218, 254820.41473388672, 254953.2371826172, 255074.8310913086, 255179.5821899414, 255275.58377685546, 255362.86015624998, 255440.41485595703, 255508.75826416016, 255569.4701538086, 255621.65126953126, 255666.34669189452, 255700.49409179686, 255728.24948730468, 255714.85787353516, 255696.82415771484, 255667.27025146483, 255626.22045898438, 255565.94604492188, 255502.2447387695, 255428.82175292968, 255346.527734375, 255244.74174804686, 255144.92440185547, 255031.6911376953, 254909.9757080078, 254767.57740478514, 254632.13010253906, 254489.0755859375, 254330.85524902344, 254149.13275146484, 253978.1284057617, 253799.27380371094, 253612.61755371094, 253419.8123413086, 253240.54456787108, 253054.22857666016, 252861.08310546874, 252640.9356689453, 252424.5553833008, 252201.58865966796, 251972.0841064453, 251715.1644165039, 251475.03892822264, 251220.59826660156, 250959.74129638672, 250670.0109375, 250398.67885742188])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_270_120000(self):
        """Test for latitude = -30, longitude = 270, rho = 120000"""
        lat = -30
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-63.257560445930324, -63.987898913501425, -64.60028001410643, -65.12109559869825, -65.65503880687481, -66.05276072224467, -66.43033357956298, -66.76643352824095, -67.14845966889811, -67.41963183932597, -67.66751306468407, -67.8766397966915, -68.15304389202359, -68.32987748963521, -68.49436210522745, -68.64774083322928, -68.88465277478033, -69.02659776611154, -69.15728660569056, -69.28017536923744, -69.47639304371356, -69.55873780434723, -69.63738060245075, -69.71049094732658, -69.86210748652252, -69.92702160439433, -69.9756115701498, -70.02156508288564, -70.14756841269403, -70.18756600047362, -70.22469490880181, -70.2548570241766, -70.36424933392351, -70.38986254331107, -70.41337205443426, -70.43551552772342, -70.53619251608941, -70.55240738517904, -70.56822610329678, -70.5826378024455, -70.67827411120402, -70.685022338104, -70.69104656495196, -70.69641509363959, -70.78258476020847, -70.78535781701149, -70.78487970376958, -70.78418302447423, -70.86432846419518, -70.86099533188019])
        expected['lon_n'] = np.array([320.2464785577409, 321.3452374291425, 322.35493063376714, 323.27558549237153, 324.1015739290795, 324.9129457609636, 325.72590219673526, 326.4956098746823, 327.19209792472685, 327.8914546542228, 328.570894552099, 329.15984810385083, 329.72153554043604, 330.27210342905244, 330.80996716581257, 331.3357551281202, 331.8755586386046, 332.4113459978571, 332.91975795891614, 333.4299457691594, 333.91543561536395, 334.30601315286196, 334.7026558983429, 335.0769775856163, 335.4509714238095, 335.8288174886946, 336.1287720762849, 336.42247021059546, 336.7224794396991, 337.0085550828959, 337.27493246052643, 337.51286893045295, 337.73299226702414, 337.9642078308074, 338.17640814794134, 338.3841278609798, 338.5749360256333, 338.74599128320705, 338.9274284283193, 339.09140760991295, 339.27407418907575, 339.4026729907688, 339.5276927733367, 339.64948870661635, 339.77822411209274, 339.8959219318704, 339.98258337205954, 340.0795993790304, 340.185603914949, 340.2651619584013])
        expected['rho_n'] = np.array([255061.14782714844, 255072.3763671875, 255145.58061523436, 255173.53044433592, 255226.82955322266, 255255.21685791016, 255303.58221435547, 255351.6073120117, 255366.2627441406, 255388.86564941405, 255443.96326904296, 255460.66025390624, 255530.00013427733, 255529.00366210938, 255545.79786376952, 255578.53562011718, 255612.85314941406, 255632.4909423828, 255608.01661376952, 255656.6007080078, 255689.0954223633, 255657.42705078123, 255692.95979003905, 255676.65167236328, 255698.35532226562, 255753.9633300781, 255746.283203125, 255743.99860839843, 255777.83005371093, 255783.71166992188, 255733.9609741211, 255784.2463623047, 255750.85239257812, 255807.5783935547, 255807.8700439453, 255811.394152832, 255815.45295410155, 255788.2565551758, 255823.74068603516, 255801.42943115233, 255814.50509033204, 255822.1123046875, 255831.78537597656, 255843.96177978514, 255831.9555053711, 255847.70462646484, 255814.74813232422, 255844.5450805664, 255850.69404296874, 255823.060168457])
        expected['lat_s'] = np.array([33.3712157801261, 33.533863074831615, 33.74698546750352, 33.99865744785098, 34.34821628428848, 34.696566177242886, 35.142287247004994, 35.63725398068289, 36.21431617308976, 36.79196917685983, 37.39021861607208, 37.94949841097482, 38.56911951210004, 39.163673818971304, 39.77553582519939, 40.3870255861177, 41.13410484734453, 41.8329766334094, 42.522200777271635, 43.19317124057701, 43.87180517613113, 44.41919361168888, 44.94490644191559, 45.460233969509304, 45.97512436499613, 46.451717889806694, 46.8704016557393, 47.27738530762368, 47.68454654442648, 48.06269996782424, 48.43577173047951, 48.78897447283921, 49.14244700767112, 49.47734825814977, 49.804763621298086, 50.12473749334568, 50.41743500494282, 50.70019459129759, 50.973115290152954, 51.245195875740414, 51.50987253626758, 51.77380470617521, 52.0290010641344, 52.28412229001272, 52.5325089493694, 52.77390149501599, 53.0223052298456, 53.26288498298096, 53.50530888719222, 53.737391885000086])
        expected['lon_s'] = np.array([235.0752217209758, 235.36375623227383, 235.65308304551556, 235.949267368684, 236.400387702985, 236.6935940636753, 236.99625340617692, 237.2929021822089, 237.73776606323017, 238.01171129046105, 238.27929078138547, 238.54430212118197, 238.94602652740545, 239.18543831819272, 239.40769267378494, 239.61964710410876, 239.94154025931314, 240.10371627096592, 240.24474601694834, 240.36809923335878, 240.60521608058494, 240.6501860460803, 240.68775208651536, 240.70870710688897, 240.85225036248596, 240.8535617588066, 240.81965669976665, 240.7775554137227, 240.86279617456447, 240.80968462357845, 240.74460658116655, 240.67450151952553, 240.73430665589817, 240.65106031029407, 240.56234981335393, 240.46833908961787, 240.52166920665735, 240.4262651243306, 240.3318992307577, 240.22802571386015, 240.27086466033447, 240.15434163309405, 240.0395124927678, 239.91553089895373, 239.94110312720625, 239.81428017469744, 239.67926099518465, 239.5463455139362, 239.5547056654803, 239.413921806308])
        expected['rho_s'] = np.array([262176.591015625, 261800.0460571289, 261450.3329345703, 261013.02747802733, 260658.08895263672, 260198.9340209961, 259748.13973388672, 259312.04948730467, 258821.4692260742, 258431.21669921876, 257969.9473022461, 257638.55954589843, 257269.13571777343, 256937.99100341796, 256673.24536132812, 256366.4291503906, 256121.97751464843, 255885.1087890625, 255694.8312133789, 255502.3662597656, 255359.57908935545, 255292.30506591796, 255176.27681884766, 255125.5296508789, 255088.24700927734, 255004.66486816405, 254970.71190185545, 254946.82087402343, 254948.1819091797, 254889.4872680664, 254897.6777832031, 254871.74520263672, 254880.08154296875, 254872.7659790039, 254871.76950683593, 254878.03999023436, 254878.28303222655, 254909.36810302734, 254886.86241455076, 254928.05803222655, 254901.3963256836, 254937.92553710938, 254919.86751708985, 254964.24698486328, 254947.4284790039, 254944.92514648437, 254977.4684692383, 254954.45239257812, 254994.40849609373, 254981.45435791014])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_0N_270_120000(self):
        """Test for latitude = 0, longitude = 270, rho = 120000"""
        lat = 0
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-61.879597101636904, -62.259724449704805, -62.62400575889828, -62.97189461408371, -63.40699132459184, -63.72290123408328, -64.04424797415405, -64.35201629815481, -64.74236160922104, -65.01813732714947, -65.28213779894881, -65.52092121233247, -65.847923349036, -66.06430374194198, -66.27049349260633, -66.46705267654096, -66.76525873570372, -66.95537022106186, -67.13564623401516, -67.30645560477885, -67.56094845325353, -67.68146031096921, -67.79765548912948, -67.90777179892842, -68.10501400178006, -68.20406540512357, -68.2833912221441, -68.35865990679764, -68.52162822039415, -68.58937003658235, -68.65504913564118, -68.7113298944021, -68.85502341416083, -68.90586734234239, -68.95417044015272, -69.00106651899404, -69.130949396251, -69.16947166316987, -69.2066552130114, -69.24243174388394, -69.36461016775712, -69.39216315086895, -69.41818617160672, -69.4434032300224, -69.55424353987337, -69.57571765462387, -69.59435041067968, -69.61205426100835, -69.71532672125895, -69.7298203826777])
        expected['lon_n'] = np.array([335.14907706249494, 335.4017667410287, 335.6616417785694, 335.9224730425938, 336.20980544059796, 336.48178357334785, 336.76206721612846, 337.05502769000867, 337.36388884427663, 337.6593628277714, 337.9662568875584, 338.2263231703958, 338.533545079263, 338.80929347643473, 339.0851238358765, 339.36095419531836, 339.7060426729441, 340.0124722798675, 340.3176451319837, 340.6212607009691, 340.9585354421843, 341.2022092710137, 341.4564015911649, 341.69693353297606, 341.9876263840518, 342.22826760888967, 342.43456664258076, 342.6270413733917, 342.873283353349, 343.06578540491665, 343.2684234572126, 343.41120173162255, 343.61102574598044, 343.7653333797094, 343.9060625973683, 344.0572556648357, 344.22689024306214, 344.3488227801252, 344.46927999632754, 344.58872634453274, 344.7585794888126, 344.85534960897337, 344.93892380365764, 345.0330711311771, 345.16716140496277, 345.2606257135652, 345.3296652656957, 345.397010930912, 345.51765939241113, 345.596233888623])
        expected['rho_n'] = np.array([255133.67155761717, 255074.8310913086, 255033.24660644532, 254976.05882568358, 254975.06235351562, 254924.67974853516, 254879.9114135742, 254893.32733154297, 254871.76950683593, 254862.24226074218, 254907.2293334961, 254854.92669677734, 254899.79224853514, 254900.8859375, 254900.32694091796, 254898.4555175781, 254957.9278930664, 254974.3818359375, 254991.880859375, 255011.27561035156, 255020.2681640625, 255000.11998291014, 255042.16624755858, 255025.78521728516, 255062.7762084961, 255050.4782836914, 255093.95849609375, 255079.08432617187, 255120.37716064454, 255109.10001220703, 255160.4061767578, 255128.34893798828, 255155.42381591795, 255187.57827148438, 255161.13530273436, 255197.1055175781, 255203.20587158203, 255211.37208251952, 255220.92363281248, 255231.73900146483, 255245.5680908203, 255281.29526367187, 255257.84171142578, 255296.84995117187, 255280.34739990233, 255322.78253173828, 255312.33172607422, 255303.67943115233, 255302.3183959961, 255357.63475341795])
        expected['lat_s'] = np.array([-23.028156045905405, -12.601961070340153, -7.533632857110533, 2.5901828137090916, 12.303046492736556, 19.554419277918143, 27.04019049503622, 33.25083369600463, 37.26576573920528, 39.71440904722236, 41.456421474277434, 42.73544952844319, 43.81066473777577, 44.71280295354022, 45.499111406265015, 46.20255258888463, 46.93035364117909, 47.579074763263144, 48.17533320233233, 48.72197714727055, 49.235002901113965, 49.64485181735516, 50.03646754360702, 50.4071214192961, 50.762082935367076, 51.09954892187906, 51.39599962242513, 51.679438812602314, 51.95799783261753, 52.22521190842136, 52.481610379674464, 52.73385951100679, 52.975907754813704, 53.216712904191674, 53.44894275106669, 53.67676371083247, 53.88042287640932, 54.08028445680765, 54.27340122540059, 54.467156616680924, 54.65821408323911, 54.852133399059525, 55.03997726161322, 55.22887638839367, 55.416167005624594, 55.5960810185519, 55.784064899983576, 55.96986995107002, 56.15195596415339, 56.332710090348606])
        expected['lon_s'] = np.array([242.64984686900658, 248.28581844377763, 249.6560636743089, 252.18162174256952, 253.5284257638693, 253.46823813690315, 252.51340501169204, 251.05289200108987, 249.92290550480297, 249.07328461356676, 248.41974479302326, 247.93769736216043, 247.62736088703173, 247.2500885580369, 246.90754091078242, 246.58605756692825, 246.34541634209037, 246.0061745063942, 245.67944557725744, 245.3693003474255, 245.19597746704727, 244.92823405158276, 244.66294950421948, 244.40534208948327, 244.28756230743556, 244.03856093105358, 243.79879397042944, 243.56580255746198, 243.47078096572878, 243.23855453394833, 243.01266651771766, 242.7812050671243, 242.70290377847925, 242.47065002594215, 242.24424291533455, 242.01758991791687, 241.96065346099562, 241.75093933271955, 241.54690792183294, 241.33662005766658, 241.2811316008494, 241.06371301918955, 240.85159466432563, 240.63270076180507, 240.57188475743527, 240.35665183630982, 240.13065453705244, 239.90359172828454, 239.84414176174874, 239.6141283112594])
        expected['rho_s'] = np.array([223675.31256103516, 248053.88262939453, 261716.75556640624, 265075.7417236328, 259314.11534423826, 256973.52374267578, 255674.90177001953, 255136.7824951172, 254957.6848510742, 254890.75108642576, 254849.8471191406, 254887.73736572266, 254883.67856445312, 254928.95728759764, 254928.47120361327, 254950.70954589843, 254973.55549316405, 254986.87419433592, 255027.7781616211, 255024.47279052733, 255045.73896484374, 255058.03688964844, 255102.22192382812, 255116.4398803711, 255128.3732421875, 255132.38343505858, 255163.0067260742, 255149.17763671876, 255186.48458251954, 255199.2928955078, 255164.0031982422, 255205.1259033203, 255179.50927734375, 255238.8115234375, 255245.51948242186, 255258.81387939453, 255259.08122558592, 255277.79545898436, 255241.16903076172, 255268.14669189454, 255291.16276855467, 255314.5191040039, 255281.5869140625, 255311.5539916992, 255340.08712158204, 255318.33486328126, 255339.4066040039, 255363.34624023436, 255327.86210937498, 255358.50970458984])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30N_270_120000(self):
        """Test for latitude = 30, longitude = 270, rho = 120000"""
        lat = 30
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([67.85860308832635, 67.88037602385305, 67.90208065748806, 67.92394238547398, 67.93536416931353, 67.95715759540775, 67.98462007851307, 68.01188790122706, 68.02888653452393, 68.05610142327185, 68.08292869878437, 68.1127561348898, 68.13166722115429, 68.16058794964741, 68.18934304605317, 68.21765759525745, 68.24351669145511, 68.27933932610455, 68.31417329087161, 68.3480681046278, 68.36853135138116, 68.40107720277626, 68.43179768611569, 68.46053229195394, 68.47159890595665, 68.49188456779159, 68.51556141854944, 68.53009093846134, 68.49336671884149, -67.38333621407651, -66.96841588237658, -67.03476433997409, -67.20307386150154, -67.27182654568688, -67.34045628646717, -67.40888112157236, -67.56618037815775, -67.62640898625892, -67.68648733019836, -67.7462651458143, -67.89831881711714, -67.95223633042522, -68.0055118059513, -68.05813158331708, -68.20047272562016, -68.2495681253742, -68.29718820426754, -68.34389303781208, -68.47822919840789, -68.52105448450388])
        expected['lon_n'] = np.array([242.65897200173774, 242.55050859771794, 242.4412802125111, 242.33166933671072, 242.26686450186563, 242.15687113547176, 242.02482991843706, 241.89196907870198, 241.80492514791933, 241.6725560818045, 241.53950399677265, 241.36842141844215, 241.2440300132779, 241.0739309821879, 240.9039685548813, 240.73441593892488, 240.595817740287, 240.4108015760497, 240.22758858175328, 240.04699838009813, 239.9250112015217, 239.75330024578759, 239.58628846020244, 239.42605222227394, 239.34236874456295, 239.21163892384888, 239.08648253749757, 238.99422034218904, 239.1038312179894, 342.6798250752976, 345.0306122630759, 345.1189675901792, 344.94916908741266, 344.67964982276396, 344.4076170485007, 344.13279755705605, 343.96122320510534, 343.69143073288984, 343.4194526001399, 343.14493363701877, 342.96330524660976, 342.6715742067802, 342.37828588381984, 342.0830577871352, 341.87951814986883, 341.5819951096231, 341.2650470113778, 340.94670555454184, 340.7181127833999, 340.3983233264599])
        expected['rho_n'] = np.array([6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.198860740661, 6371.2, 6371.2, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.198860740661, 217148.05529785156, 241707.1326538086, 243694.6571533203, 243104.13802490235, 242582.18104248046, 242040.63487548826, 241480.0342163086, 240889.46647949217, 240354.33662109374, 239804.06524658203, 239238.33640136718, 238608.29864501953, 238047.74659423827, 237475.21257324217, 236891.35279541014, 236264.25584716795, 235663.57756347655, 235050.43122558593, 234428.4624633789, 233783.28319091795, 233148.74915771483])
        expected['lat_s'] = np.array([51.19711134398352, 51.39575373561501, 51.586070126648245, 51.77220644190943, 51.96248868199682, 52.13928129847343, 52.32875416114416, 52.518394363449566, 52.7100563017493, 52.88752852204832, 53.06178713834286, 53.228402772956144, 53.39665765297024, 53.55743347584317, 53.71591435515498, 53.86869544160441, 54.052943209560084, 54.2308969581956, 54.40296991395573, 54.576115209415555, 54.748741410498454, 54.892202703825404, 55.03429795931835, 55.174900818477646, 55.3154217153669, 55.452486536536135, 55.575382130272175, 55.70066146002855, 55.826032997338714, 55.94552715686809, 56.06736065618819, 56.18923172154873, 56.31128720201686, 56.427899021716485, 56.54705167178736, 56.66561692558961, 56.771478027535636, 56.87823046916835, 56.98150634451353, 57.08740159740541, 57.19089945389861, 57.30254231097705, 57.41375828123237, 57.52459859108334, 57.63630291986431, 57.745538135260325, 57.859598879304954, 57.973328359174836, 58.08823263158196, 58.200397998131905])
        expected['lon_s'] = np.array([248.9195233949714, 248.7387419480195, 248.56500925629106, 248.3924240363432, 248.32884863554872, 248.15842175537855, 247.97733978010316, 247.7907663327351, 247.7184756105597, 247.53955197506204, 247.36120207545466, 247.19200462933512, 247.14124266342358, 246.9720998588174, 246.80325758253466, 246.64108157088188, 246.5662773390919, 246.36847506072834, 246.17717512245463, 245.97959141014448, 245.909868839097, 245.73397780759083, 245.55797749305793, 245.38208646155178, 245.33777219421674, 245.16122546455023, 244.9973555659833, 244.82687404429979, 244.79146634364244, 244.62650361480829, 244.45484730058752, 244.28155174096597, 244.2465811724155, 244.07847655656315, 243.90345978927073, 243.72808785214147, 243.7110943414865, 243.5444377257382, 243.38422880856638, 243.21697113617108, 243.20101581426994, 243.02474229217026, 242.8479769964503, 242.67071992711016, 242.64083101930217, 242.46171613850777, 242.27724638940407, 242.09212094214004, 242.05725965661628, 241.8700031903312])
        expected['rho_s'] = np.array([255883.62623291014, 255947.10880126953, 255957.3408691406, 255971.29147949218, 255921.80812988282, 255943.05, 255939.64741210936, 255998.85244140626, 255998.8038330078, 256004.73405761717, 256012.63292236326, 256032.82971191406, 255998.97396240235, 256023.93437499998, 256049.45378417967, 256015.7681640625, 256002.98415527344, 256042.4784790039, 256021.77130126953, 256060.02611083983, 256050.83912353514, 256069.57766113282, 256087.24681396483, 256104.08962402344, 256076.91752929686, 256092.32639160156, 256066.39381103514, 256098.8642211914, 256090.98966064453, 256061.60588378905, 256090.62509765624, 256117.0194580078, 256106.39852294922, 256070.0637451172, 256091.91322021483, 256111.86696777344, 256062.6995727539, 256104.38127441405, 256084.0872680664, 256122.12333984373, 256070.1123535156, 256089.72584228514, 256107.37069091797, 256122.87677001953, 256111.47810058593, 256123.7517211914, 256121.58864746094, 256116.9951538086, 256088.82658691405, 256080.44163818358])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60N_270_120000(self):
        """Test for latitude = 60, longitude = 270, rho = 120000"""
        lat = 60
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([71.84224945003375, 71.85619669631892, 71.87015248034055, 71.88409801907842, 71.86700034803859, 71.88087929243206, 71.89789670874913, 71.91487997412037, 71.90065269007924, 71.9175608233696, 71.93443480571412, 71.95505002417651, 71.94419343849077, 71.96459350599432, 71.98494576217365, 72.00526386740715, 71.9987649424119, 72.02380612345644, 72.04877387996743, 72.07375529685675, 72.06644016425567, 72.09387361905702, 72.12124218706126, 72.1485885569507, 72.14334809431001, 72.1704998038081, 72.2045312213477, 72.2385336105833, 72.2392166295003, 72.27291507531785, 72.30655717207476, 72.35451193023741, 72.36913707279771, 72.41676227433291, 72.46428673057787, 72.51167458303941, 72.52501906513032, 72.5720568703969, 72.61892392093424, 72.6654904431481, 72.67758329307361, 72.72554488142544, 72.77332546976426, 72.82088919959693, 72.83358652126398, 72.8805731001118, 72.9339373680971, 72.98709160776544, 73.0045939675136, 73.05722740525772])
        expected['lon_n'] = np.array([239.47932769780004, 239.43971260061397, 239.39987893737444, 239.35977206656813, 239.26073432360295, 239.22103726414684, 239.1642100902523, 239.10691846349425, 238.99042275701052, 238.93356826235933, 238.87657716392474, 238.77111904313975, 238.60593774825216, 238.5006981935206, 238.39502150668216, 238.2889623292502, 238.10859069364852, 237.98805151517612, 237.86680199703005, 237.74506070526377, 237.5628312582078, 237.43486083391846, 237.30618006995542, 237.17684360783204, 236.98647257528546, 236.85664433954184, 236.71310108394488, 236.5687382056475, 236.3627123795232, 236.21739327474202, 236.07155507558392, 235.8991611009328, 235.6642572349977, 235.49041526024254, 235.31548045522018, 235.13931621614722, 234.9026638217846, 234.72802954508572, 234.55230243811963, 234.3752639348329, 234.1347593137784, 233.92523643079912, 233.71426554771574, 233.50198326831176, 233.22538792768293, 233.0115210443915, 232.7786389144507, 232.5439536145691, 232.24394438546545, 232.00677289672592])
        expected['rho_n'] = np.array([6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([68.19033513103011, 68.20945283051698, 68.22899058663779, 68.2461872954206, 68.24377794619087, 68.26302371172468, 68.28557358126947, 68.3079834319363, 68.31122606424475, 68.3347868017867, 68.35740326567591, 68.37991727672757, 68.38591076772425, 68.41023990154784, 68.43356499756344, 68.45422973489731, 68.47084075495877, 68.50015592687647, 68.53037097621731, 68.55812203481509, 68.57567562098201, 68.60291270784472, 68.6285003040229, 68.65279187180604, 68.66973415604225, 68.69686537497284, 68.72252810323187, 68.74628179361788, 68.76513994591629, 68.79189208934794, 68.81610169486105, 68.84304849868404, 68.86826555709973, 68.89614638929172, 68.92370961768731, 68.94916231762936, 68.97429570622771, 68.99933517972498, 69.02380262487925, 69.04682548502407, 69.073036335964, 69.10274936640084, 69.13134053826653, 69.15818318170467, 69.19331596724793, 69.22205398818076, 69.25177555635406, 69.28039404897642, 69.32131029719937, 69.35234667678792])
        expected['lon_s'] = np.array([267.22120652131736, 267.629405946874, 268.0199561636153, 268.4199866829246, 267.31163822592833, 267.7164771984133, 268.1253050013736, 268.52375091679545, 267.42048412054163, 267.8318254331165, 268.2374567075452, 268.64347047256746, 267.53848246864277, 267.9508892907281, 268.36023618806536, 268.77859893510697, 267.66178104353986, 268.094869678432, 268.5076316703542, 268.9297100403043, 267.8231374324922, 268.2360360281978, 268.65221311470503, 269.0661772199212, 267.9475561584132, 268.36130169757587, 268.7715501798835, 269.18524107753285, 268.0650080913807, 268.47599423411873, 268.8968704907749, 269.3055890107085, 268.1749194955046, 268.5998119033927, 269.01292906515175, 269.42749422701485, 268.2916337680417, 268.70849387346595, 269.1165566952392, 269.5227617055582, 268.39288449229804, 268.81138384312305, 269.2252386653124, 269.6392027705286, 268.4998998962138, 268.9232077002145, 269.348154749616, 269.7651787795803, 268.62756979217966, 269.0523529170411])
        expected['rho_s'] = np.array([271209.2223510742, 271492.90096435545, 271835.7359985351, 272134.53182373045, 271281.5030395508, 271585.69439697266, 271890.7607055664, 272233.49852294923, 271363.65123291017, 271679.0954345703, 272004.38283691404, 272331.7847045898, 271454.4760253906, 271776.9927490234, 272109.40128173825, 272390.4793457031, 271583.5799316406, 271869.275793457, 272232.4291381836, 272550.35236816405, 271677.63718261715, 272019.40283203125, 272340.023828125, 272659.21087646484, 271781.05155029293, 272131.95557861327, 272463.36763916013, 272773.0274414062, 271870.24796142575, 272215.6106323242, 272518.125, 272857.58175048826, 271992.935559082, 272299.24138183595, 272637.0454467773, 272945.02825927734, 272081.86462402344, 272401.00306396483, 272739.9737304687, 273070.77818603517, 272170.5020385742, 272500.6016723633, 272835.2947998047, 273154.2145019531, 272299.3385986328, 272619.789465332, 272931.8796875, 273246.64337158203, 272404.57578124997, 272739.779296875])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_90N_270_120000(self):
        """Test for latitude = 90, longitude = 270, rho = 120000"""
        lat = 90
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([78.8984418957194, 78.90707440105666, 78.91558737808343, 78.92395179849576, 78.89862204195877, 78.90718368408338, 78.91759374615212, 78.92781000160318, 78.90424499519298, 78.9146729865083, 78.9249327844153, 78.93787172402425, 78.91678351496184, 78.9297173319289, 78.94248295548765, 78.95502403657744, 78.93649117203826, 78.95209815429173, 78.96743961294123, 78.98257531214198, 78.96340126359446, 78.979382198705, 78.99510017153248, 79.01055518207694, 78.99158603920452, 79.00691042238108, 79.02556537655168, 79.04393602409809, 79.02769895689367, 79.04587409027508, 79.06365221891097, 79.09216569737484, 79.08588192333842, 79.11393863290154, 79.14157101696247, 79.16872528778151, 79.16107889100567, 79.1878233504745, 79.21402054103609, 79.23979852873737, 79.23086805639758, 79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212])
        expected['lon_n'] = np.array([222.03207391586602, 221.9699874963106, 221.90740930313495, 221.84420273255566, 221.79726567257933, 221.73515193226723, 221.64709713348742, 221.55822271200722, 221.48625983891196, 221.39868315337407, 221.31038246778414, 221.13775626670125, 220.9828065951903, 220.81105465832118, 220.63863336291337, 220.46533780329182, 220.27212541205049, 220.0598431326465, 219.84644070221862, 219.6319044603885, 219.4388286729306, 219.20795461860584, 218.9758647906088, 218.74253186818282, 218.5328723814201, 218.29949847785912, 218.03881747799642, 217.77659285538127, 217.54065080069236, 217.27835787618554, 217.01458963081797, 216.69424009836604, 216.40249539815812, 216.0810666958173, 215.75787580467065, 215.43290906433978, 215.1539504782581, 214.84212502189035, 214.5286329993651, 214.21382958051922, 213.935089560491, 213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225])
        expected['lat_s'] = np.array([73.76546679576927, 73.6725284117329, 73.5846785186282, 73.49383358757244, 73.72931460449239, 73.6331455409786, 73.54314926092722, 73.4522309053379, 73.6852137805689, 73.59044832092957, 73.49639490851119, 73.41110975143982, 73.64449902292645, 73.55596098826288, 73.46222688718821, 73.37188226748916, 73.6103702751912, 73.51583191934176, 73.42088545968942, 73.32817076434837, 73.57029926287802, 73.47759993546262, 73.39053380656517, 73.30273513987925, 73.55243319555656, 73.45981583041119, 73.37157149388192, 73.28712474253118, 73.54256527975319, 73.45781287743708, 73.37191471088771, 73.28943334647064, 73.54950987459179, 73.46764834984191, 73.38075980586288, 73.30005941327188, 73.57014899871629, 73.48558784169694, 73.40441787360051, 73.32345622627376, 73.60148932172288, 73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512])
        expected['lon_s'] = np.array([350.3579412496324, 350.43211710401874, 350.4985885050213, 350.56877552893235, 350.38744766684687, 350.46648661592224, 350.5413454892256, 350.61713326825605, 350.4496570298073, 350.5321657149811, 350.6124887396204, 350.6834680654752, 350.5244612615973, 350.6013145501383, 350.684424291959, 350.7623977315239, 350.6063142486107, 350.7001883685634, 350.7917948657116, 350.8799862682748, 350.724476521252, 350.8160556976435, 350.8970617411998, 350.97855955837645, 350.81900633936493, 350.90911019489573, 350.99768408805244, 351.08030205625295, 350.9240546487997, 351.0074649189439, 351.09259639675895, 351.1788753463546, 351.03057827909527, 351.11713043625764, 351.21089527318355, 351.2948246377047, 351.1359817583669, 351.2243097647135, 351.307774676371, 351.3893544558177, 351.22447368925356, 351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506])
        expected['rho_s'] = np.array([327743.56041259767, 328264.4966186523, 328804.7303588867, 329309.40705566405, 327946.1116088867, 328462.8431884766, 329007.79194335936, 329528.7767578125, 328135.9274047851, 328669.76914062497, 329199.3819458008, 329737.8900878906, 328291.8145385742, 328852.0506347656, 329376.3651245117, 329890.90932617185, 328487.8765136719, 329023.95423583983, 329577.99276123045, 330110.0116821289, 328679.73386230465, 329194.0836303711, 329747.1985961914, 330278.97447509767, 328802.1298095703, 329332.95782470703, 329836.68665771483, 330333.5130981445, 328807.93851318356, 329354.3212158203, 329858.12296142575, 330370.86865234375, 328782.75936279295, 329328.58306884766, 329813.1844970703, 330318.12854003906, 328725.0125854492, 329232.3870483398, 329724.6442993164, 330217.48485107423, 328624.97650146484, 329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984])

        years = np.arange(1965, 2015)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])


class Trace2006to2015(unittest.TestCase):
    """
    The returns from 2006-2015 should be similar but not identical between
    IGRF-11 and IGRF-13 since the former had provisional versions. These tests
    look for a relative tolerance of 5% or an absolute tolerance of 0.1 to allow
    easy comparison of the current version of the code with the previous version
    (2010-30-11).
    """

    def test_90S_0_30000(self):
        """Test for latitude = -90, longitude = 0, rho = 30000"""
        lat = -90
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-7.328262729146587, -7.34104201308368, -7.35437454234355, -7.436842246382284, -7.449539568049332, -7.465699795625582, -7.482447419470461, -7.568473652066771, -7.584729502291395, -7.601490786514603])
        expected['lon_n'] = np.array([351.367552491987, 351.3807757382201, 351.3939716636966, 351.52339008809, 351.53664065507985, 351.55464503373196, 351.5726494123841, 351.70714949752005, 351.72534512146893, 351.74340414163447])
        expected['rho_n'] = np.array([293212.1784667969, 293222.92092285154, 293234.46541748045, 293308.08283691405, 293319.1655517578, 293333.9425048828, 293349.740234375, 293427.4164550781, 293443.287097168, 293459.44938964845])
        expected['lat_s'] = np.array([-78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.46619240663597, -78.4636788970214, -78.4600588967613, -78.4564388965012])
        expected['lon_s'] = np.array([136.39912361196477, 136.33206481469358, 136.26503333817908, 136.20222291857164, 136.13491823449033, 136.05294230407182, 135.97082976986994, 135.8930065944668, 135.81082575837323, 135.72845367698287])
        expected['rho_s'] = np.array([6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198860740661])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_0_30000(self):
        """Test for latitude = -60, longitude = 0, rho = 30000"""
        lat = -60
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([5.023184988526836, 5.00661494960039, 4.989792193674646, 4.9314828687302565, 4.9156641506125, 4.89793980971632, 4.880072034847558, 4.820205426772404, 4.803280218009107, 4.786115952624868])
        expected['lon_n'] = np.array([355.5093792046827, 355.5164006391495, 355.52328546983284, 355.59806238086617, 355.60497453230624, 355.6145641179009, 355.6240170997122, 355.70166269019694, 355.7111429927649, 355.7205140123062])
        expected['rho_n'] = np.array([320921.90638427733, 320930.0239868164, 320938.2631103516, 320973.8444580078, 320981.2572387695, 320990.5900512695, 320999.9957763672, 321037.4485473633, 321046.14945068356, 321054.7045288086])
        expected['lat_s'] = np.array([-79.77171476459219, -79.7569342352283, -79.74211272472937, -79.72767370482396, -79.71278389243332, -79.69486147605122, -79.6768980785341, -79.6593308319888, -79.6413127929583, -79.62326743317115])
        expected['lon_s'] = np.array([59.71415142269791, 59.67220723100486, 59.630235718555134, 59.58353771519975, 59.541812089560146, 59.52018088045872, 59.498720426086535, 59.47232857513361, 59.45105253586902, 59.429776496604426])
        expected['rho_s'] = np.array([6371.192404937744, 6371.192025184631, 6371.193164443969, 6371.193923950195, 6371.194303703308, 6371.194303703308, 6371.1946834564205, 6371.195822715759, 6371.1965822219845, 6371.197341728211])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_0_30000(self):
        """Test for latitude = -30, longitude = 0, rho = 30000"""
        lat = -30
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([64.72819761811527, 64.79961407607692, 64.87112444913967, 64.94139002022618, 65.01223957248672, 65.08555482303763, 65.15839537544123, 65.23060072025203, 65.3029904801704, 65.37505068346132])
        expected['lon_n'] = np.array([339.16484580386896, 339.3510094398868, 339.53802001936174, 339.73535784486177, 339.92460872638446, 340.13708225108525, 340.3507852098366, 340.5752798674766, 340.79166026038257, 341.0092427665825])
        expected['rho_n'] = np.array([6371.196961975097, 6371.195442962647, 6371.199620246887, 6371.195442962647, 6371.198101234436, 6371.198101234436, 6371.196202468872, 6371.196202468872, 6371.1984809875485, 6371.2])
        expected['lat_s'] = np.array([-69.4400564373291, -69.41762609609478, -69.39516843410377, -69.37198677206075, -69.34939250628634, -69.32884729726294, -69.30828842786121, -69.28701921878579, -69.26640570787069, -69.24581951771228])
        expected['lon_s'] = np.array([31.191967882736247, 31.155887408445654, 31.11978644358755, 31.082862440934463, 31.04669317418466, 31.024949266961926, 31.003215605022945, 30.980624754343125, 30.958809130134107, 30.936966185168405])
        expected['rho_s'] = np.array([6371.185949134827, 6371.183290863037, 6371.184050369263, 6371.1844301223755, 6371.185189628601, 6371.187088394165, 6371.19050617218, 6371.194303703308, 6371.2, 6371.195822715759])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_0_30000(self):
        """Test for latitude = 0, longitude = 0, rho = 30000"""
        lat = 0
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([61.92215270894623, 61.96862702360629, 62.015024498638184, 62.0605511245509, 62.10611190140947, 62.152852593447165, 62.199506200572934, 62.245025996296484, 62.2907131316547, 62.33631488964829])
        expected['lon_n'] = np.array([347.0189076512233, 347.1772314361842, 347.33596503249527, 347.49521772318326, 347.6548802252215, 347.8329569172621, 348.0115253829229, 348.19074954674403, 348.3704654841854, 348.550673195247])
        expected['rho_n'] = np.array([6371.194303703308, 6371.194303703308, 6371.196961975097, 6371.197341728211, 6371.1946834564205, 6371.1946834564205, 6371.1984809875485, 6371.195822715759, 6371.1946834564205, 6371.195442962647])
        expected['lat_s'] = np.array([-63.422249967197644, -63.39206053106619, -63.36178913266471, -63.33169531918162, -63.30121901510503, -63.27582437177094, -63.250238483140066, -63.22503508510272, -63.19940821533683, -63.173685722922556])
        expected['lon_s'] = np.array([29.303803067821242, 29.23081908144507, 29.157787283744707, 29.082224900956856, 29.008906235311354, 28.94519081563889, 28.88113388650793, 28.81476152324833, 28.75057140542855, 28.686166136649913])
        expected['rho_s'] = np.array([6371.186328887939, 6371.191645431519, 6371.181392097473, 6371.187847900391, 6371.189746665955, 6371.18101234436, 6371.189746665955, 6371.188607406616, 6371.18101234436, 6371.190126419067])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_0_30000(self):
        """Test for latitude = 30, longitude = 0, rho = 30000"""
        lat = 30
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([66.59947308276013, 66.62879166977241, 66.65790022846771, 66.68693707017673, 66.71570241186623, 66.7449322064193, 66.77389220850016, 66.80271219170302, 66.83118725035281, 66.8593651957737])
        expected['lon_n'] = np.array([349.7228429398479, 349.8761943470931, 350.02984628266165, 350.18546531271124, 350.3396363426568, 350.5120029965512, 350.6847794617958, 350.85963230454814, 351.0331191094665, 351.206933763465])
        expected['rho_n'] = np.array([6371.196202468872, 6371.195822715759, 6371.196961975097, 6371.1965822219845, 6371.197721481323, 6371.197721481323, 6371.198860740661, 6371.198860740661, 6371.2, 6371.198860740661])
        expected['lat_s'] = np.array([-66.89929436797604, -66.88297021585973, -66.86645481844661, -66.85538991199121, -66.83857398625463, -66.82524145699477, -66.81179964470817, -66.80369904035254, -66.79001134125585, -66.7761733779974])
        expected['lon_s'] = np.array([42.698835388780914, 42.5738395118751, 42.44860799344291, 42.30981854950826, 42.18424210652299, 42.0651611734384, 41.945981202610845, 41.81330819307794, 41.693728656183936, 41.57388274191231])
        expected['rho_s'] = np.array([6371.193923950195, 6371.185949134827, 6371.187847900391, 6371.1844301223755, 6371.192404937744, 6371.188227653503, 6371.184809875488, 6371.187468147277, 6371.185949134827, 6371.1984809875485])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_0_30000(self):
        """Test for latitude = 60, longitude = 0, rho = 30000"""
        lat = 60
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([77.05780153704936, 77.08237655768306, 77.10676887075647, 77.13087858475296, 77.15481498269928, 77.17950526277522, 77.20393428811542, 77.22800045965596, 77.25188819250444, 77.27551467061716])
        expected['lon_n'] = np.array([345.6836329892425, 345.905177005161, 346.1272674362131, 346.3483196785114, 346.57177614739754, 346.82615971284554, 347.0814175425072, 347.33582842871186, 347.5926981830177, 347.85038756002393])
        expected['rho_n'] = np.array([6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887])
        expected['lat_s'] = np.array([30.298402464996578, 30.3024049558502, 30.306141069326202, 30.24423906487837, 30.248808461433114, 30.25230551828816, 30.255584009089766, 30.193682004641943, 30.19782109927897, 30.201659665592516])
        expected['lon_s'] = np.array([4.958679344501999, 4.911492555734143, 4.864199045260505, 4.87509789274173, 4.827863719536507, 4.776903251252218, 4.725902228719733, 4.73304703347839, 4.682092968496448, 4.6310471228475345])
        expected['rho_s'] = np.array([343078.7566894531, 343083.8119628906, 343088.79432373046, 343067.7225830078, 343073.06950683595, 343078.1004760742, 343082.86409912107, 343061.3548828125, 343066.4344604492, 343071.44112548826])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_0_30000(self):
        """Test for latitude = 90, longitude = 0, rho = 30000"""
        lat = 90
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361, 87.37246976875205])
        expected['lon_n'] = np.array([257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423, 252.70249196867468])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395, 47.362253823155676])
        expected['lon_s'] = np.array([8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431, 8.388868220687279])
        expected['rho_s'] = np.array([327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766, 327358.6305053711])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_90_30000(self):
        """Test for latitude = -90, longitude = 90, rho = 30000"""
        lat = -90
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-7.328262729146587, -7.341035182894515, -7.35437454234355, -7.436835416193105, -7.449532737860167, -7.4656793050580745, -7.482440589281282, -7.568473652066771, -7.584729502291395, -7.601483956325438])
        expected['lon_n'] = np.array([351.367552491987, 351.3807757382201, 351.3939443429399, 351.52339008809, 351.53661333432314, 351.5546177129753, 351.5726494123841, 351.70714949752005, 351.72534512146893, 351.74337682087776])
        expected['rho_n'] = np.array([293212.2513793945, 293222.92092285154, 293234.46541748045, 293308.10714111326, 293319.1655517578, 293333.9425048828, 293349.740234375, 293427.4164550781, 293443.287097168, 293459.44938964845])
        expected['lat_s'] = np.array([-78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.4662060670143, -78.4636788970214, -78.46007255713963, -78.45645255687953])
        expected['lon_s'] = np.array([136.39912361196477, 136.33206481469358, 136.26501967780072, 136.20220925819328, 136.13491823449033, 136.05295596445018, 135.9708161094916, 135.8930339152235, 135.81082575837323, 135.72845367698287])
        expected['rho_s'] = np.array([6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_90_30000(self):
        """Test for latitude = -60, longitude = 90, rho = 30000"""
        lat = -60
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([9.196724269538947, 9.267321104800189, 9.33799307214231, 9.316805825336942, 9.388823339945546, 9.466270854944327, 9.543643237862227, 9.52881489717413, 9.607683091520272, 9.68659226700143])
        expected['lon_n'] = np.array([28.246515397512656, 28.26356013458642, 28.280560475430583, 28.36844110438655, 28.385740266006902, 28.406198390118384, 28.42648405195332, 28.51767903020404, 28.538290833571843, 28.55888897656131])
        expected['rho_n'] = np.array([293541.91353759763, 293480.73986816406, 293419.1044189453, 293478.5767944336, 293415.7261352539, 293349.4728881836, 293282.58773193357, 293337.39370117185, 293269.29333496094, 293200.63397216797])
        expected['lat_s'] = np.array([-66.62855932965508, -66.62911940516702, -66.62963849954394, -66.63079963170284, -66.63129140532308, -66.62972046181397, -66.62813585792654, -66.62720695219943, -66.62558136717695, -66.62394212177617])
        expected['lon_s'] = np.array([110.18343006791929, 110.14947719755516, 110.11557896870438, 110.07986390953438, 110.045740284441, 110.00766197981818, 109.96945390160113, 109.92956559684825, 109.89126872617199, 109.8529581951174])
        expected['rho_s'] = np.array([6371.198101234436, 6371.197721481323, 6371.1984809875485, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.1984809875485])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_90_30000(self):
        """Test for latitude = -30, longitude = 90, rho = 30000"""
        lat = -30
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([73.20848365197494, 73.14552979839493, 73.08243763348423, 73.02056465734037, 72.95685777540437, 72.88176496812196, 72.80632723628646, 72.73206600453551, 72.65609381039745, 72.58008575776626])
        expected['lon_n'] = np.array([67.12017335106692, 67.35732434923894, 67.59254240387584, 67.85375615849381, 68.08438432600845, 68.34354219368625, 68.60042560837043, 68.8817269493374, 69.13271591076786, 69.38143724939387])
        expected['rho_n'] = np.array([6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.2, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775])
        expected['lat_s'] = np.array([-55.033452609723014, -55.04503661055537, -55.0566206113877, -55.067904083896565, -55.07943344321555, -55.092929897015495, -55.106385369680424, -55.11955397440019, -55.132968465930105, -55.14632831594665])
        expected['lon_s'] = np.array([101.76375342520701, 101.73564036658324, 101.7075204777703, 101.67851949455442, 101.65033130384978, 101.62210896219929, 101.5938661299813, 101.5647763543062, 101.5365130315207, 101.50822921816767])
        expected['rho_s'] = np.array([6371.197721481323, 6371.196961975097, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.199240493775, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_90_30000(self):
        """Test for latitude = 0, longitude = 90, rho = 30000"""
        lat = 0
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([66.42878153275963, 66.3723044060601, 66.31563432651649, 66.25819926578586, 66.20174092210652, 66.1335517285277, 66.0649476009568, 65.99583291674546, 65.92753102504531, 65.8594306239257])
        expected['lon_n'] = np.array([87.46719077531903, 87.49640349439916, 87.52544545875006, 87.55734927236318, 87.58583799139132, 87.62183308831727, 87.65778037391905, 87.69635045216212, 87.73158739809021, 87.76647600437063])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([-48.46500994908348, -48.492371686898565, -48.519774405848665, -48.54639848323336, -48.57389682483185, -48.60709154419811, -48.64012233902429, -48.6724154734201, -48.705528230516336, -48.73864098761254])
        expected['lon_s'] = np.array([100.75301520163936, 100.72568761477014, 100.69828489582005, 100.66996693152117, 100.64244126916603, 100.6161860219965, 100.58982149180025, 100.56254171625521, 100.53611571435644, 100.5096350709443])
        expected['rho_s'] = np.array([6371.198101234436, 6371.196961975097, 6371.197341728211, 6371.197341728211, 6371.198860740661, 6371.198860740661, 6371.196961975097, 6371.196202468872, 6371.198101234436, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_90_30000(self):
        """Test for latitude = 30, longitude = 90, rho = 30000"""
        lat = 30
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([68.19646864090478, 68.15335819441096, 68.11033141773446, 68.06702801839657, 68.02417541154392, 67.97124998321277, 67.9181042812809, 67.86472464536996, 67.81183678307927, 67.75901893022755])
        expected['lon_n'] = np.array([89.44825982332453, 89.46269884322994, 89.4770354102978, 89.49281314728053, 89.50686284640325, 89.52664990442878, 89.54640964169762, 89.56765153001636, 89.58706975782671, 89.60626941958361])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([-49.87939186203138, -49.92008612910632, -49.96042522634443, -50.000327191475634, -50.04088485476717, -50.08798583928359, -50.135441993636846, -50.18185995923625, -50.22901558526601, -50.27587068297228])
        expected['lon_s'] = np.array([103.33584440609052, 103.31247149875072, 103.28900296876256, 103.26239938194537, 103.23882839911965, 103.2173406239908, 103.19585284886193, 103.17110024330981, 103.14943488326253, 103.1276875609452])
        expected['rho_s'] = np.array([6371.196961975097, 6371.198860740661, 6371.197721481323, 6371.197341728211, 6371.199620246887, 6371.196961975097, 6371.2, 6371.1965822219845, 6371.199240493775, 6371.197721481323])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_90_30000(self):
        """Test for latitude = 60, longitude = 90, rho = 30000"""
        lat = 60
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([77.97526584353767, 77.93784579839611, 77.90049747024084, 77.86388936383686, 77.8266375121036, 77.78063277294899, 77.73464852436187, 77.68942840318816, 77.64351672536098, 77.59764858998976])
        expected['lon_n'] = np.array([88.9332362391489, 88.97704507248535, 89.02062167939003, 89.06923213571302, 89.1124672331592, 89.1617811989667, 89.21064437228897, 89.26469848938045, 89.31309720983916, 89.36125004348774])
        expected['rho_n'] = np.array([6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-58.711891968368406, -58.760386311475486, -58.80862110739412, -58.8529763558642, -58.90130677443119, -58.954022174445356, -59.00660097067612, -59.05497237037815, -59.107387242068825, -59.1598021137595])
        expected['lon_s'] = np.array([118.489711910031, 118.4443184728071, 118.39839228082793, 118.3119357463139, 118.26570902601127, 118.2164360413388, 118.16656200001935, 118.07588440859827, 118.02539565025354, 117.9740872692084])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_90_30000(self):
        """Test for latitude = 90, longitude = 90, rho = 30000"""
        lat = 90
        lon = 90
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361, 87.37246976875205])
        expected['lon_n'] = np.array([257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423, 252.70249196867468])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395, 47.362253823155676])
        expected['lon_s'] = np.array([8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431, 8.388868220687279])
        expected['rho_s'] = np.array([327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766, 327358.6305053711])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_180_30000(self):
        """Test for latitude = -90, longitude = 180, rho = 30000"""
        lat = -90
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-7.328255898957423, -7.341035182894515, -7.35437454234355, -7.436835416193105, -7.449532737860167, -7.4656793050580745, -7.482447419470461, -7.568473652066771, -7.584729502291395, -7.601483956325438])
        expected['lon_n'] = np.array([351.367552491987, 351.3807757382201, 351.3939716636966, 351.52339008809, 351.53661333432314, 351.55464503373196, 351.5726494123841, 351.70714949752005, 351.72534512146893, 351.74340414163447])
        expected['rho_n'] = np.array([293212.2270751953, 293222.92092285154, 293234.46541748045, 293308.10714111326, 293319.1655517578, 293333.9911132812, 293349.740234375, 293427.4164550781, 293443.287097168, 293459.44938964845])
        expected['lat_s'] = np.array([-78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.4662060670143, -78.4636788970214, -78.46007255713963, -78.4564388965012])
        expected['lon_s'] = np.array([136.3991372723431, 136.33206481469358, 136.2650469985574, 136.20218193743662, 136.1348772533553, 136.05294230407182, 135.9708161094916, 135.8930065944668, 135.81081209799487, 135.72842635622618])
        expected['rho_s'] = np.array([6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_180_30000(self):
        """Test for latitude = -60, longitude = 180, rho = 30000"""
        lat = -60
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-19.84995974207176, -15.442957765039452, -20.41038359366047, -15.057229661852148, -18.920630543221478, -18.60549927529543, -17.955661417282073, -17.45035719229537, -17.152246755780993, -15.685921254195136])
        expected['lon_n'] = np.array([280.8162062906075, 279.02147846353824, 282.16907551989027, 278.38001441744734, 280.4186346393994, 282.2353829963527, 281.1595462398057, 280.70970998106867, 280.2280450407994, 278.09033243436875])
        expected['rho_n'] = np.array([100107.1251586914, 100451.6675628662, 101081.86329650879, 99949.54888305663, 100092.79783325196, 102052.20237426758, 101322.57816162109, 101055.38387145996, 100761.29090881348, 99364.28553771973])
        expected['lat_s'] = np.array([-67.6949977459042, -67.69905487827117, -67.70312567101652, -67.70675933165495, -67.71083012440027, -67.7140129925535, -67.71719586070674, -67.71991427599639, -67.72309714414962, -67.72630733305954])
        expected['lon_s'] = np.array([159.614731695163, 159.58758852340137, 159.56036338936968, 159.53341146290484, 159.50604972508975, 159.46909840168, 159.43207877637852, 159.39529137750884, 159.3580805069106, 159.32078767404232])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_180_30000(self):
        """Test for latitude = -30, longitude = 180, rho = 30000"""
        lat = -30
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([67.31325858687772, 67.32042174776977, 67.32725876712897, 67.33543962620735, 67.34163631533184, 67.35037383482758, 67.35864860900705, 67.36818184554109, 67.37564041211475, 67.38270795035842])
        expected['lon_n'] = np.array([206.17359043676868, 205.96470959157134, 205.7558560671307, 205.5392434477929, 205.33044456486562, 205.0919753401838, 204.85331487020522, 204.60671772041107, 204.367756722109, 204.12874108229357])
        expected['rho_n'] = np.array([6371.197721481323, 6371.196202468872, 6371.196961975097, 6371.196202468872, 6371.1965822219845, 6371.196202468872, 6371.197341728211, 6371.197341728211, 6371.198101234436, 6371.1984809875485])
        expected['lat_s'] = np.array([-58.072176450705, -58.080345356952336, -58.088555244334714, -58.09651924490694, -58.10472913228929, -58.11269313286152, -58.12067079381211, -58.128416228330906, -58.13640754965982, -58.144426191745396])
        expected['lon_s'] = np.array([163.73643667056388, 163.7281858020465, 163.71986663163742, 163.71232610279372, 163.7039932720063, 163.68978647853268, 163.67553870392405, 163.6620422501241, 163.6477125132454, 163.63335545561003])
        expected['rho_s'] = np.array([6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.196961975097, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.197721481323, 6371.198101234436])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_180_30000(self):
        """Test for latitude = 0, longitude = 180, rho = 30000"""
        lat = 0
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([64.34580947743211, 64.34064585441958, 64.33502460873265, 64.32928041964067, 64.3234235324274, 64.31748639049135, 64.31114455984701, 64.30449537068999, 64.29764981359435, 64.29072570932325])
        expected['lon_n'] = np.array([195.61801118775588, 195.47169487535587, 195.32552882711758, 195.17889832601574, 195.03330601366775, 194.86175898247373, 194.69037587581977, 194.51846001441055, 194.34754136062014, 194.17693689553155])
        expected['rho_n'] = np.array([6371.195822715759, 6371.197721481323, 6371.197341728211, 6371.195442962647, 6371.195442962647, 6371.196961975097, 6371.197341728211, 6371.195063209533, 6371.195442962647, 6371.1984809875485])
        expected['lat_s'] = np.array([-53.00132106748248, -53.01582838927959, -53.03034937145503, -53.044761070603755, -53.05939133580594, -53.07400794062977, -53.08872016810196, -53.10335043330414, -53.118049000398, -53.132638284465145])
        expected['lon_s'] = np.array([162.99402242854018, 162.99721895707174, 163.0004154856033, 163.0045135991053, 163.00761450498848, 163.00648069358627, 163.005264919914, 163.00497805196886, 163.00374861791826, 163.00254650462435])
        expected['rho_s'] = np.array([6371.2, 6371.197721481323, 6371.197341728211, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.197721481323, 6371.1984809875485, 6371.199240493775, 6371.197721481323])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_180_30000(self):
        """Test for latitude = 30, longitude = 180, rho = 30000"""
        lat = 30
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([67.65240138483095, 67.64507088430423, 67.63761914791975, 67.63006666624501, 67.62236221286123, 67.61422404246517, 67.60586901356295, 67.59735518276253, 67.58864839911806, 67.57976573810245])
        expected['lon_n'] = np.array([191.68300260312813, 191.54624855556614, 191.40983601746262, 191.27351910200753, 191.13769396017264, 190.9738923634974, 190.8103503140106, 190.64691754755054, 190.48404485660242, 190.32150001473445])
        expected['rho_n'] = np.array([6371.199620246887, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.197721481323, 6371.196202468872, 6371.196202468872, 6371.195822715759, 6371.196202468872, 6371.196961975097])
        expected['lat_s'] = np.array([-54.97110664297915, -54.99737555052701, -55.023671778831556, -55.04987238448771, -55.07625057506232, -55.10339374682394, -55.130591560098935, -55.15768009034721, -55.184905224378866, -55.212157679167234])
        expected['lon_s'] = np.array([157.8915842505955, 157.90330485521125, 157.91498447869196, 157.92904100800385, 157.94069331072788, 157.9502145944309, 157.95969489699885, 157.97157942615468, 157.9810050872093, 157.9903897671289])
        expected['rho_s'] = np.array([6371.197341728211, 6371.1965822219845, 6371.197341728211, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197341728211, 6371.197721481323, 6371.198101234436])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_180_30000(self):
        """Test for latitude = 60, longitude = 180, rho = 30000"""
        lat = 60
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([76.64369655918159, 76.637977983299, 76.63207243098788, 76.62644435511179, 76.6201776565483, 76.61457263256065, 76.60870123119538, 76.60301680625864, 76.59661691900634, 76.58995577701828])
        expected['lon_n'] = np.array([192.58984681923073, 192.38600665364086, 192.18269924280625, 191.9771105487889, 191.77481400595144, 191.53205542247085, 191.2898295937455, 191.04528150070237, 190.80408020035253, 190.56341165475797])
        expected['rho_n'] = np.array([6371.199240493775, 6371.198101234436, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.199620246887, 6371.1984809875485, 6371.199240493775, 6371.2])
        expected['lat_s'] = np.array([60.71036957470653, 60.868396246438536, 61.02502955957985, 61.27223801130471, 61.42542720155705, 61.580477618358216, 61.72810191198428, 61.96771519335209, 62.10208208979917, 62.230873844336216])
        expected['lon_s'] = np.array([189.27933609079557, 189.21850642604744, 189.15762211978594, 189.10519358771694, 189.04850301760584, 188.99403908916415, 188.94188376466192, 188.89187311955908, 188.84413009726072, 188.80178292440664])
        expected['rho_s'] = np.array([256276.9896972656, 256207.16373291015, 256111.45379638672, 256012.82735595704, 255958.19151611326, 255916.6313354492, 255846.17346191406, 255784.85396728516, 255703.7508544922, 255664.01348876953])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_180_30000(self):
        """Test for latitude = 90, longitude = 180, rho = 30000"""
        lat = 90
        lon = 180
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361, 87.37246976875205])
        expected['lon_n'] = np.array([257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423, 252.70249196867468])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395, 47.362253823155676])
        expected['lon_s'] = np.array([8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431, 8.388868220687279])
        expected['rho_s'] = np.array([327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766, 327358.6305053711])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_270_30000(self):
        """Test for latitude = -90, longitude = 270, rho = 30000"""
        lat = -90
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-7.328296880092438, -7.3410488432728584, -7.354381372532714, -7.436842246382284, -7.449539568049332, -7.465699795625582, -7.482440589281282, -7.568480482255936, -7.584743162669739, -7.601490786514603])
        expected['lon_n'] = np.array([351.36760713350037, 351.3808030589768, 351.3939716636966, 351.52339008809, 351.53664065507985, 351.55464503373196, 351.5726494123841, 351.7071768182767, 351.72534512146893, 351.74340414163447])
        expected['rho_n'] = np.array([293212.27568359376, 293222.92092285154, 293234.46541748045, 293308.10714111326, 293319.189855957, 293334.01541748043, 293349.740234375, 293427.4407592773, 293443.287097168, 293459.44938964845])
        expected['lat_s'] = np.array([-78.47447059591002, -78.4739515015331, -78.47339142602115, -78.4738558788847, -78.47328214299444, -78.4697577653827, -78.46619240663597, -78.4636788970214, -78.46007255713963, -78.4564388965012])
        expected['lon_s'] = np.array([136.39912361196477, 136.33205115431522, 136.2650060174224, 136.20218193743662, 136.13490457411197, 136.0529286436935, 135.9708161094916, 135.8930065944668, 135.81079843761654, 135.72844001660454])
        expected['rho_s'] = np.array([6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.198101234436, 6371.199240493775, 6371.199240493775])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_270_30000(self):
        """Test for latitude = -60, longitude = 270, rho = 30000"""
        lat = -60
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([62.84160374670937, 62.947288971284266, 63.05283417698118, 63.14973236318886, 63.25580178590457, 63.36890971855999, 63.482335255011805, 63.586949847434305, 63.70081422354029, 63.81478617512571])
        expected['lon_n'] = np.array([287.6067530426775, 287.54872375548905, 287.4896562795468, 287.3904272912848, 287.33029430583207, 287.27152735821323, 287.211858825624, 287.1114550448248, 287.0504477951583, 286.9882930737112])
        expected['rho_n'] = np.array([6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.199620246887, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.198860740661])
        expected['lat_s'] = np.array([-80.5906134649417, -80.59947905048438, -80.60834463602706, -80.61812546691851, -80.62704569397457, -80.63655331729922, -80.64612924251557, -80.656552111189, -80.66612803640538, -80.67581324464845])
        expected['lon_s'] = np.array([212.62475874935907, 212.6057298423314, 212.5866053126554, 212.5826847840718, 212.563328027964, 212.52248349672732, 212.4815570032206, 212.45569790702294, 212.41411571535588, 212.3723149576354])
        expected['rho_s'] = np.array([6371.195063209533, 6371.1946834564205, 6371.195442962647, 6371.195063209533, 6371.195063209533, 6371.195063209533, 6371.195822715759, 6371.1946834564205, 6371.1946834564205, 6371.195442962647])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_270_30000(self):
        """Test for latitude = -30, longitude = 270, rho = 30000"""
        lat = -30
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([53.231551490163525, 53.31734208123347, 53.40374397423413, 53.48904962187301, 53.57550274129244, 53.667857144154766, 53.76033790551674, 53.85240544043393, 53.9454326169295, 54.038575906640965])
        expected['lon_n'] = np.array([273.0465929021512, 272.9890007470696, 272.9311353844213, 272.86947243659444, 272.8111426210825, 272.75368706978435, 272.6959856316761, 272.63432268384923, 272.5761567928774, 272.51774501509544])
        expected['rho_n'] = np.array([6371.197341728211, 6371.1965822219845, 6371.2, 6371.197721481323, 6371.197341728211, 6371.197341728211, 6371.197721481323, 6371.197341728211, 6371.198101234436, 6371.197341728211])
        expected['lat_s'] = np.array([-73.49665604956141, -73.50403265386501, -73.51132729589858, -73.51855363604045, -73.5258346176957, -73.53379861826792, -73.54178993959684, -73.54978126092576, -73.55784088414637, -73.565941488502])
        expected['lon_s'] = np.array([237.50966506570845, 237.57370491936646, 237.63837315042815, 237.70539096656432, 237.77068757502965, 237.8345635041476, 237.898657999319, 237.96504743805153, 238.02971566911322, 238.09465710774168])
        expected['rho_s'] = np.array([6371.1965822219845, 6371.1984809875485, 6371.1984809875485, 6371.195822715759, 6371.194303703308, 6371.192784690857, 6371.191645431519, 6371.192025184631, 6371.193544197083, 6371.1946834564205])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_270_30000(self):
        """Test for latitude = 0, longitude = 270, rho = 30000"""
        lat = 0
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([51.77671778185622, 51.844722560327455, 51.912898093527936, 51.98092677766127, 52.049266235401824, 52.12186090099531, 52.19447264206171, 52.267190251060256, 52.34029035065232, 52.41386514839169])
        expected['lon_n'] = np.array([270.93786761855813, 270.88601282237937, 270.83393946014723, 270.7809098714312, 270.7283720563355, 270.6763533356167, 270.6241160488444, 270.57092253558835, 270.5182207959525, 270.4653004902633])
        expected['rho_n'] = np.array([6371.196202468872, 6371.1965822219845, 6371.199620246887, 6371.196961975097, 6371.196202468872, 6371.197341728211, 6371.197721481323, 6371.196202468872, 6371.197341728211, 6371.2])
        expected['lat_s'] = np.array([-71.68388286270499, -71.68593191945601, -71.68751652334345, -71.69019395749808, -71.6919698066823, -71.6931036180845, -71.69431939175678, -71.69649139191284, -71.69737931650494, -71.69841750525879])
        expected['lon_s'] = np.array([237.07788782713692, 237.1830727403551, 237.28973297443403, 237.3990706426676, 237.50581283901656, 237.61665314886753, 237.72752077947516, 237.84172154239778, 237.95406449386616, 238.06635280382116])
        expected['rho_s'] = np.array([6371.192025184631, 6371.199620246887, 6371.192025184631, 6371.193164443969, 6371.196961975097, 6371.191645431519, 6371.193164443969, 6371.198101234436, 6371.192784690857, 6371.193164443969])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_270_30000(self):
        """Test for latitude = 30, longitude = 270, rho = 30000"""
        lat = 30
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([58.67394867366723, 58.734494885564814, 58.7951981918133, 58.855761479183805, 58.91656723826984, 58.98045682776614, 59.0445376625592, 59.10862191244685, 59.173112558590105, 59.23779445003012])
        expected['lon_n'] = np.array([270.7505018692463, 270.70943877195623, 270.6681844293693, 270.6255367281918, 270.5838725742547, 270.54433943933867, 270.50461505912585, 270.46352464107906, 270.42341777027275, 270.3831469749264])
        expected['rho_n'] = np.array([6371.198860740661, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.1984809875485])
        expected['lat_s'] = np.array([-73.95274876157816, -73.96974227223316, -73.98664016023977, -74.0148351811336, -74.03167842762684, -74.04945057984722, -74.0671954113109, -74.09607345112173, -74.11369533918037, -74.13131722723898])
        expected['lon_s'] = np.array([209.9287328197933, 210.09648226580882, 210.2658163157118, 210.47405512312713, 210.64450932405396, 210.82519514835747, 211.00685085952313, 211.22776649803802, 211.41088386968607, 211.59528531689807])
        expected['rho_s'] = np.array([6371.1965822219845, 6371.197341728211, 6371.195822715759, 6371.1965822219845, 6371.196961975097, 6371.1965822219845, 6371.196961975097, 6371.2, 6371.199620246887, 6371.200759506225])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_270_30000(self):
        """Test for latitude = 60, longitude = 270, rho = 30000"""
        lat = 60
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([71.54002894718633, 71.59650265879128, 71.65299173832187, 71.70856386495639, 71.76508880298012, 71.82615240170733, 71.88728088723164, 71.94760341043391, 72.00890777332935, 72.0703060513259])
        expected['lon_n'] = np.array([267.88613909739644, 267.83313682943714, 267.7797520708843, 267.7234713121234, 267.66940353465355, 267.61926994614566, 267.5688358293143, 267.5154237500048, 267.46419733122974, 267.412424497321])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887])
        expected['lat_s'] = np.array([30.94342503964507, 30.867637260614607, 30.791145972099635, 30.66079181178994, 30.58405463646485, 30.495596856524017, 30.406141868964355, 30.261854122747835, 30.17176392759537, 30.08090192106669])
        expected['lon_s'] = np.array([329.4653214842905, 329.4280832929356, 329.39122759217423, 329.4767142398261, 329.43996782209143, 329.4016368004693, 329.36368826944073, 329.4476722754752, 329.4100242727701, 329.3726494776318])
        expected['rho_s'] = np.array([318595.45982666017, 318549.11171875, 318501.9858764648, 318364.1324584961, 318315.76710205077, 318256.07598876953, 318195.55853271484, 318042.8552490234, 317979.98028564453, 317915.98732910154])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_270_30000(self):
        """Test for latitude = 90, longitude = 270, rho = 30000"""
        lat = 90
        lon = 270
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([86.94608645863413, 86.99034736511629, 87.03440208526287, 87.0773409232538, 87.12094378369811, 87.17196444302445, 87.22273729454997, 87.27233193844367, 87.32254386068361, 87.37246976875205])
        expected['lon_n'] = np.array([257.06793907479664, 256.6556688564947, 256.2315960713069, 255.8019771725131, 255.35298785723316, 254.858673406621, 254.3467370679502, 253.82359921904057, 253.27311329269423, 252.70249196867468])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([47.56281891303852, 47.5564224408808, 47.54955810076493, 47.47685415214474, 47.470590868675835, 47.462193151091306, 47.453358301399895, 47.379093654554346, 47.3709042577395, 47.362253823155676])
        expected['lon_s'] = np.array([8.947699784425737, 8.864264754845799, 8.780735810164774, 8.808962420683498, 8.72570070715375, 8.63603483750347, 8.546060755566891, 8.568022375069422, 8.47848884033431, 8.388868220687279])
        expected['rho_s'] = np.array([327456.8923828125, 327457.3784667969, 327457.4270751953, 327413.0962158203, 327413.46077880857, 327411.56505126954, 327409.3290649414, 327362.90804443357, 327360.91510009766, 327358.6305053711])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_0_60000(self):
        """Test for latitude = -90, longitude = 0, rho = 60000"""
        lat = -90
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-32.80199282365358, -33.378474449980985, -33.885055920342495, -34.155941222825206, -34.57930366833928, -34.911032295948445, -35.23009775283643, -35.431274144649976, -35.724740052528745, -36.005105657579435])
        expected['lon_n'] = np.array([359.98451914887505, 0.10806184359147995, 0.20914364739781677, 0.3554047846759409, 0.4288793707335895, 0.45909527384807963, 0.48788185315703886, 0.6056764161609894, 0.6296286089197548, 0.6516987643970643])
        expected['rho_n'] = np.array([301707.56547851564, 302019.8744384766, 302077.815649414, 302133.20491943357, 301862.86931152345, 300665.0125488281, 299566.8273071289, 299156.49951171875, 298128.33466796874, 297178.1863037109])
        expected['lat_s'] = np.array([-74.65020669848485, -74.65431847236522, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67629802111432, -74.67860662505376, -74.68829183329686, -74.69058677685797, -74.69288172041911])
        expected['lon_s'] = np.array([142.25000025878109, 142.20713399155008, 142.164103799779, 142.13929655271352, 142.09623904018576, 142.0419526966625, 141.98754340973417, 141.95116582221468, 141.89661993150295, 141.84196475776452])
        expected['rho_s'] = np.array([6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_0_60000(self):
        """Test for latitude = -60, longitude = 0, rho = 60000"""
        lat = -60
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-1.131204774437336, -1.1366689257733498, -1.142481416757036, -1.1904020239738458, -1.195285609230396, -1.2017879493202486, -1.2084268931935043, -1.2576998778659743, -1.2634235763904513, -1.269352180590019])
        expected['lon_n'] = np.array([357.4764736856464, 357.4758179874861, 357.4751076478124, 357.5489556531186, 357.54819067193154, 357.54939278522545, 357.5504309739793, 357.6262460737664, 357.62731158327693, 357.6282131682474])
        expected['rho_n'] = np.array([340407.72519531247, 340418.7349975586, 340430.1822753906, 340484.7208984375, 340495.05018310546, 340507.9071044922, 340520.83693847654, 340577.24698486325, 340589.107434082, 340601.210925293])
        expected['lat_s'] = np.array([-80.11049214742482, -80.10613448673436, -80.10174950528722, -80.10397614695663, -80.09955018437446, -80.09113539131701, -80.08270693788123, -80.08084912642698, -80.07235237109947, -80.06381463463697])
        expected['lon_s'] = np.array([99.28698339818783, 99.20334773180102, 99.1196915748467, 99.01081152928752, 98.92700510817147, 98.84362898897312, 98.75976109615453, 98.65088105059537, 98.56740930874864, 98.48386926501021])
        expected['rho_s'] = np.array([6371.199240493775, 6371.198860740661, 6371.199240493775, 6371.197341728211, 6371.197341728211, 6371.196202468872, 6371.195822715759, 6371.194303703308, 6371.195442962647, 6371.195063209533])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_0_60000(self):
        """Test for latitude = -30, longitude = 0, rho = 60000"""
        lat = -30
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([10.34895669195258, 10.352221522375842, 10.35519948485397, 10.311636538327633, 10.315365821614463, 10.318733104875278, 10.32196378435269, 10.277881743449427, 10.281877404113885, 10.285538385509014])
        expected['lon_n'] = np.array([358.45715224667686, 358.45160613307075, 358.445950736438, 358.505127495407, 358.4995267402876, 358.49504613619206, 358.49053821133987, 358.5509444043594, 358.5464638002639, 358.5419558754117])
        expected['rho_n'] = np.array([360496.7499267578, 360492.93416748045, 360489.3614501953, 360525.0400146484, 360520.7867797851, 360517.3112792969, 360517.1654541016, 360552.64958496095, 360549.87890625, 360547.0353149414])
        expected['lat_s'] = np.array([-75.65831529922207, -75.64316593964298, -75.62794827817217, -75.61247106951294, -75.59722608728546, -75.58080631252076, -75.56434555662102, -75.54759793277614, -75.53108253536305, -75.51448517567994])
        expected['lon_s'] = np.array([50.062803842430164, 50.002291781478434, 49.941673852594576, 49.85246475184504, 49.79214052109548, 49.7480858009489, 49.70414377892362, 49.63142958501967, 49.58774369508827, 49.54403048440019])
        expected['rho_s'] = np.array([6371.1965822219845, 6371.198101234436, 6371.195063209533, 6371.19050617218, 6371.189366912841, 6371.187847900391, 6371.188607406616, 6371.190126419067, 6371.192404937744, 6371.195822715759])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_0_60000(self):
        """Test for latitude = 0, longitude = 0, rho = 60000"""
        lat = 0
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([66.31638906241977, 66.36427551869073, 66.41203732400935, 66.45838698771705, 66.50558871752372, 66.55438529650158, 66.6029411113112, 66.64962033164636, 66.69745726904584, 66.74505856491895])
        expected['lon_n'] = np.array([344.1436712182147, 344.32371500473624, 344.5043052063914, 344.6959056729886, 344.877834591721, 345.0817293988243, 345.28633454560116, 345.50244173094035, 345.70874076463144, 345.91569549648284])
        expected['rho_n'] = np.array([6371.199240493775, 6371.196202468872, 6371.1965822219845, 6371.197341728211, 6371.195822715759, 6371.195822715759, 6371.1965822219845, 6371.1965822219845, 6371.1965822219845, 6371.196961975097])
        expected['lat_s'] = np.array([-68.84308424349163, -68.81796280772434, -68.79269110779529, -68.77011050239923, -68.74472951944347, -68.72156151777878, -68.69828423308738, -68.67758875990222, -68.65428415445413, -68.63095222824938])
        expected['lon_s'] = np.array([38.04237782778682, 37.960483859638366, 37.87849768393612, 37.783325828041164, 37.70128159573097, 37.63044911894335, 37.55955858554779, 37.4753047870411, 37.40431863099715, 37.333349550426135])
        expected['rho_s'] = np.array([6371.193544197083, 6371.184050369263, 6371.195063209533, 6371.1844301223755, 6371.198860740661, 6371.184050369263, 6371.195442962647, 6371.185569381713, 6371.19050617218, 6371.187088394165])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_0_60000(self):
        """Test for latitude = 30, longitude = 0, rho = 60000"""
        lat = 30
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([71.46872518734597, 71.504773218238, 71.54059243779284, 71.5766148554755, 71.61192693348445, 71.64843770969276, 71.68464795757758, 71.7210904318942, 71.75669791558477, 71.79196559736411])
        expected['lon_n'] = np.array([344.1609106156798, 344.3615815734948, 344.5628809087134, 344.7579237906523, 344.9604798806782, 345.1883623121465, 345.4170370455585, 345.6395372879609, 345.869823946017, 346.1009302267736])
        expected['rho_n'] = np.array([6371.196961975097, 6371.1965822219845, 6371.197341728211, 6371.198101234436, 6371.197721481323, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198860740661, 6371.2])
        expected['lat_s'] = np.array([22.323289174982094, 22.339258157261582, 22.354885630082578, 22.304635928358792, 22.32098057104264, 22.336874421241248, 22.3524199317922, 22.302026796095845, 22.318275816131305, 22.334190156897435])
        expected['lon_s'] = np.array([1.3078755715687183, 1.287955324854376, 1.2680517267261358, 1.3001213858700496, 1.2801794746494337, 1.2595116423860926, 1.2388618460910283, 1.2700979019910834, 1.249434552039385, 1.2287728029132734])
        expected['rho_s'] = np.array([377097.80611572263, 377099.7990600586, 377101.8892211914, 377095.61873779295, 377097.83041992184, 377100.0177978515, 377101.9378295898, 377095.6916503906, 377097.8790283203, 377099.9934936523])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_0_60000(self):
        """Test for latitude = 60, longitude = 0, rho = 60000"""
        lat = 60
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([80.33396665924309, 80.37470959141585, 80.41518529243736, 80.45583260196175, 80.49567565671137, 80.53807149466328, 80.58004983730218, 80.62217588278187, 80.66342083385865, 80.70425341226428])
        expected['lon_n'] = np.array([333.4914994339596, 333.7873012665345, 334.08526143888713, 334.3653811571277, 334.66795853735925, 335.0177735058906, 335.3707030406835, 335.70622925347124, 336.0649234679236, 336.4266776071242])
        expected['rho_n'] = np.array([6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887])
        expected['lat_s'] = np.array([34.06346569779065, 34.07144677383581, 34.079192208354605, 34.026893449879815, 34.03533897878854, 34.043026356699386, 34.05049516855679, 33.998029070447345, 34.00615699555966, 34.01404244895644])
        expected['lon_s'] = np.array([3.2620834065592805, 3.2280131423187877, 3.1938643309028394, 3.23045941725871, 3.196343689571554, 3.160354995947934, 3.124282419063569, 3.158981701037938, 3.1229959956220794, 3.0869394269935824])
        expected['rho_s'] = np.array([369586.47182617185, 369593.83599853516, 369601.1272583008, 369569.8477539062, 369577.600793457, 369584.8677490234, 369591.91596679686, 369560.63646240236, 369568.29228515626, 369575.7293701172])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_0_60000(self):
        """Test for latitude = 90, longitude = 0, rho = 60000"""
        lat = 90
        lon = 0
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089, 85.75967897126492])
        expected['lon_n'] = np.array([246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524, 241.7348747277917])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269, 46.93261784888895])
        expected['lon_s'] = np.array([5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653, 4.87329514968767])
        expected['rho_s'] = np.array([353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582, 352904.7256958008])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_90_60000(self):
        """Test for latitude = -90, longitude = 90, rho = 60000"""
        lat = -90
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-32.78376987894798, -33.37848811035933, -33.88487833542408, -34.17342650710043, -34.57963151741943, -34.91092301292173, -35.234182205960096, -35.42688916320283, -35.724740052528745, -36.00440897828409])
        expected['lon_n'] = np.array([359.97714254457145, 0.10806322430354849, 0.20905967075558, 0.36276919602466917, 0.4290091976886726, 0.4590617098716114, 0.4896805941471323, 0.6038198320063625, 0.6296228459476426, 0.6514934851959937])
        expected['rho_n'] = np.array([301575.0832885742, 302019.8987426758, 302076.1872680664, 302272.5651977539, 301865.37264404295, 300664.91533203126, 299602.5787841797, 299121.40424804686, 298128.33466796874, 297177.01970214845])
        expected['lat_s'] = np.array([-74.65022035886321, -74.65433213274355, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67629802111432, -74.67860662505376, -74.68829183329686, -74.69058677685797, -74.69288172041911])
        expected['lon_s'] = np.array([142.24998659840273, 142.20713399155008, 142.16411746015734, 142.13929655271352, 142.09623904018576, 142.0419526966625, 141.98752974935584, 141.95116582221468, 141.89661993150295, 141.84196475776452])
        expected['rho_s'] = np.array([6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_90_60000(self):
        """Test for latitude = -60, longitude = 90, rho = 60000"""
        lat = -60
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([3.2164838398425815, 3.2679766359953106, 3.3191962245812334, 3.2838226748697394, 3.3361009427770085, 3.3914664561891357, 3.4464221582510532, 3.414600306907971, 3.470628348669578, 3.526478805512781])
        expected['lon_n'] = np.array([25.574671752703313, 25.584645536438824, 25.594438320161327, 25.676786495889587, 25.687069345685043, 25.700424073059708, 25.713701960806215, 25.79927569337001, 25.812983883034228, 25.826674997225517])
        expected['rho_n'] = np.array([313004.25449218747, 312956.205090332, 312907.985559082, 312992.92873535154, 312943.93146972655, 312892.8926513672, 312842.1454833984, 312924.63393554685, 312873.0361206055, 312821.21956787107])
        expected['lat_s'] = np.array([-68.24503287976532, -68.25127567266671, -68.25749114481141, -68.26788669272818, -68.27411582525124, -68.2787057123735, -68.28329559949574, -68.29202458125502, -68.29660080799891, -68.30117703474284])
        expected['lon_s'] = np.array([122.99217261090341, 122.96485185422335, 122.93740815413824, 122.89962354764974, 122.87216618718628, 122.83882120365828, 122.80525765407684, 122.76146248111871, 122.7279399126723, 122.69419877817244])
        expected['rho_s'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_90_60000(self):
        """Test for latitude = -30, longitude = 90, rho = 60000"""
        lat = -30
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([80.67822270756372, 80.60428505602468, 80.53035081958024, 80.45930319183375, 80.38532285161241, 80.29802620383047, 80.21083627775432, 80.12619742733317, 80.03874368520033, 79.95127628268915])
        expected['lon_n'] = np.array([76.16100788746766, 76.36654193997168, 76.56831255824304, 76.83639065297686, 77.02903613851707, 77.23407158721169, 77.43370435627281, 77.6991528281762, 77.88929846448018, 78.07492934574279])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-60.903726993782186, -60.91722344758213, -60.930665259868704, -60.94089688324539, -60.954352355910316, -60.96787613046695, -60.98137258426689, -60.99160420764355, -61.00504601993015, -61.01847417183839])
        expected['lon_s'] = np.array([112.77599229193771, 112.73223126992544, 112.68837462526479, 112.63489424406359, 112.59083269372783, 112.5425978978092, 112.49412404526961, 112.43613573921621, 112.38753894327157, 112.33886701524607])
        expected['rho_s'] = np.array([6371.199620246887, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.199240493775, 6371.198860740661])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_90_60000(self):
        """Test for latitude = 0, longitude = 90, rho = 60000"""
        lat = 0
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([73.84107186723942, 73.78273009889647, 73.72480497209288, 73.6618699015331, 73.60373303886524, 73.53431270368853, 73.46462769868147, 73.39059869336228, 73.32104346194946, 73.25168801356986])
        expected['lon_n'] = np.array([75.74088978180934, 75.88058764090361, 76.0187555376238, 76.20261739989138, 76.33782099451179, 76.4913090055403, 76.64390226178756, 76.83951887961673, 76.98792522990277, 77.13513629708405])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([-55.122395333094914, -55.14616439140656, -55.17004273274492, -55.18782854534365, -55.21144733949356, -55.23812605839163, -55.26469549426298, -55.28536364669142, -55.31178281840104, -55.338352254272394])
        expected['lon_s'] = np.array([107.28261457646902, 107.25916653704837, 107.23566385611436, 107.19961411767503, 107.1759748329576, 107.15277268034708, 107.12943392395314, 107.09356860062141, 107.07020935365996, 107.04678180480683])
        expected['rho_s'] = np.array([6371.198101234436, 6371.198101234436, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.1984809875485, 6371.199240493775, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_90_60000(self):
        """Test for latitude = 30, longitude = 90, rho = 60000"""
        lat = 30
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([74.7888494070379, 74.74065900734888, 74.69263765484182, 74.64313073619027, 74.59493179876478, 74.53649953041533, 74.47822947905865, 74.41863607855029, 74.36024649888314, 74.30202084375608])
        expected['lon_n'] = np.array([86.94894334185513, 87.00125576070826, 87.05303542480613, 87.1209616561019, 87.17185339560767, 87.2304973998214, 87.28851985682066, 87.36219027720841, 87.41909941337296, 87.47546896459308])
        expected['rho_n'] = np.array([6371.199240493775, 6371.198860740661, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887])
        expected['lat_s'] = np.array([-55.66747174961861, -55.70623990334761, -55.74489877404989, -55.780197191680514, -55.81881508124775, -55.86155840507371, -55.904247087386295, -55.943315769438755, -55.98585418758958, -56.02841992649712])
        expected['lon_s'] = np.array([114.26511696497306, 114.25952304004282, 114.25373103962664, 114.2287803585886, 114.22280394306483, 114.21908148996717, 114.21500386703268, 114.19155582761202, 114.18738258202914, 114.18303175152785])
        expected['rho_s'] = np.array([6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.1984809875485, 6371.199240493775, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_90_60000(self):
        """Test for latitude = 60, longitude = 90, rho = 60000"""
        lat = 60
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([83.10793358497517, 83.05703928414846, 83.00626536540588, 82.95781413786793, 82.90702357053924, 82.84628824777901, 82.78566733068739, 82.72754668971882, 82.66705768065553, 82.60668819990273])
        expected['lon_n'] = np.array([102.62878322321079, 102.64954699828763, 102.67031760355364, 102.71114847441197, 102.73089455130248, 102.73835994806531, 102.74598926936821, 102.77322806377822, 102.779928479354, 102.78669719682149])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-25.31661092731818, -24.73112711166469, -24.143887937394496, -23.436526226191305, -22.84904116511099, -22.194872967163846, -21.5403427691907, -20.7723016574011, -20.11984100674647, -19.467544280631913])
        expected['lon_s'] = np.array([81.31414655962134, 80.52826499371966, 79.74961659814905, 78.89609566889673, 78.13868233145568, 77.30942906432443, 76.49036643943485, 75.61134158382171, 74.81804243248142, 74.03451045165423])
        expected['rho_s'] = np.array([79735.57626342773, 80613.67482910155, 81501.89001770019, 82586.74440612792, 83498.65618896484, 84530.41197814941, 85577.91688842773, 86842.54336242676, 87927.1486328125, 89030.77193908692])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_90_60000(self):
        """Test for latitude = 90, longitude = 90, rho = 60000"""
        lat = 90
        lon = 90
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089, 85.75967897126492])
        expected['lon_n'] = np.array([246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524, 241.7348747277917])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269, 46.93261784888895])
        expected['lon_s'] = np.array([5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653, 4.87329514968767])
        expected['rho_s'] = np.array([353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582, 352904.7256958008])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_180_60000(self):
        """Test for latitude = -90, longitude = 180, rho = 60000"""
        lat = -90
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-32.783797199704665, -33.37848811035933, -33.88485101466739, -34.166828544362204, -34.57930366833928, -34.91100497519177, -35.232351715262524, -35.42892455957549, -35.724726392150416, -36.00509199720109])
        expected['lon_n'] = np.array([359.97714254457145, 0.1080659857276856, 0.20905609557843632, 0.3600146821180235, 0.42887651592795983, 0.4590881768546452, 0.48887826036306736, 0.6046723249921447, 0.6296240732472591, 0.6516928413423934])
        expected['rho_n'] = np.array([301575.01037597656, 302019.8987426758, 302076.2115722656, 302220.52990722656, 301862.89361572266, 300664.9882446289, 299586.6595336914, 299137.15336914064, 298128.33466796874, 297178.2106079101])
        expected['lat_s'] = np.array([-74.65022035886321, -74.65431847236522, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67629802111432, -74.67862028543212, -74.68830549367519, -74.69060043723633, -74.69288172041911])
        expected['lon_s'] = np.array([142.24998659840273, 142.20713399155008, 142.16409013940066, 142.13928289233516, 142.09623904018576, 142.0419526966625, 141.98750242859916, 141.95112484107966, 141.89660627112463, 141.8419510973862])
        expected['rho_s'] = np.array([6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_180_60000(self):
        """Test for latitude = -60, longitude = 180, rho = 60000"""
        lat = -60
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-72.3120963418061, -72.24647188426061, -72.17845686050563, -72.19769067320837, -72.12927949848154, -72.06250756915549, -71.99543511150597, -72.02063850954332, -71.95195412724965, -71.88339268836108])
        expected['lon_n'] = np.array([16.21003160813061, 15.906436529712712, 15.606386319474053, 15.705716053026256, 15.403082323734038, 15.097190594207724, 14.79015310044814, 14.882568975013, 14.576848000215936, 14.269165907353432])
        expected['rho_n'] = np.array([256269.16374511717, 256198.87600097657, 256192.14373779297, 256178.89794921875, 256129.05003662108, 256127.15430908202, 256109.02337646484, 256042.7701293945, 256033.72896728516, 255995.2797241211])
        expected['lat_s'] = np.array([-67.8976221378218, -67.9061598742843, -67.91467028999014, -67.92587180022898, -67.93442319706983, -67.94229157499367, -67.95014629253919, -67.96062380272599, -67.96851950140652, -67.97638787933039])
        expected['lon_s'] = np.array([154.4672962309663, 154.44171034233543, 154.41606981219118, 154.41208098171592, 154.38627652703158, 154.3520436189115, 154.31768776738633, 154.30503825704346, 154.2705184809782, 154.2358757615079])
        expected['rho_s'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_180_60000(self):
        """Test for latitude = -30, longitude = 180, rho = 60000"""
        lat = -30
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([71.74795356589985, 71.76550544451949, 71.78270386084958, 71.806273136128, 71.82247946748114, 71.84325007274717, 71.86358696100088, 71.89004540629821, 71.90915286050132, 71.92778220146253])
        expected['lon_n'] = np.array([211.43152470135786, 211.171198871332, 210.910627154496, 210.6212456997409, 210.35982703944782, 210.0625908671472, 209.764808279713, 209.43760123733438, 209.13817940449934, 208.8382384772874])
        expected['rho_n'] = np.array([6371.1984809875485, 6371.197721481323, 6371.198860740661, 6371.199240493775, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.199240493775])
        expected['lat_s'] = np.array([-63.46767072517824, -63.481071556329795, -63.49445872710302, -63.50803714317303, -63.521506276216286, -63.53467488093608, -63.54782982527752, -63.56120333567239, -63.57446756304057, -63.58777277154374])
        expected['lon_s'] = np.array([157.06690721020817, 157.05327415262482, 157.03964109504147, 157.036649472185, 157.0228934711966, 157.00312690373858, 156.9833056947672, 156.9741532412794, 156.95415444738958, 156.93400538933804])
        expected['rho_s'] = np.array([6371.199620246887, 6371.1984809875485, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_180_60000(self):
        """Test for latitude = 0, longitude = 180, rho = 60000"""
        lat = 0
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([71.38739812489862, 71.39018996472187, 71.39305864417328, 71.3966069274471, 71.39878063515046, 71.40323904113119, 71.40727056028878, 71.41219171158578, 71.41536262690795, 71.41831497617669])
        expected['lon_n'] = np.array([202.58951697169618, 202.36615612545842, 202.14323241132755, 201.9171121686651, 201.6944889828577, 201.43641711525794, 201.1783452476582, 200.91703587039186, 200.6589366820354, 200.40102873897573])
        expected['rho_n'] = np.array([6371.2, 6371.197721481323, 6371.198860740661, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.199620246887])
        expected['lat_s'] = np.array([-60.814470081708436, -60.832693026414034, -60.85086132960629, -60.86958970831046, -60.88777167188101, -60.90626782415342, -60.924750316047465, -60.94360163815671, -60.962248054590845, -60.98067590497155])
        expected['lon_s'] = np.array([156.83364258967387, 156.83003624979213, 156.82647089104537, 156.8335469670255, 156.82996794790043, 156.8218673435448, 156.8136847769191, 156.81647149410048, 156.80811134255637, 156.7999560966874])
        expected['rho_s'] = np.array([6371.199620246887, 6371.198860740661, 6371.1984809875485, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.198860740661, 6371.200759506225, 6371.199240493775])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_180_60000(self):
        """Test for latitude = 30, longitude = 180, rho = 60000"""
        lat = 30
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([73.32781388696424, 73.32569140567965, 73.32316252813945, 73.32106907515885, 73.31810306551176, 73.31684631070448, 73.3152941502156, 73.3137898010509, 73.31127629143634, 73.30845030066725])
        expected['lon_n'] = np.array([198.786262736158, 198.56596181466838, 198.34601606301564, 198.13158910421225, 197.91235369223318, 197.6556068813324, 197.39917425913345, 197.14787793919035, 196.89177316607154, 196.6360235627896])
        expected['rho_n'] = np.array([6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.2, 6371.198860740661, 6371.199240493775, 6371.198860740661])
        expected['lat_s'] = np.array([-62.085609267382836, -62.110880967311886, -62.136166327619264, -62.161588291710075, -62.1869965954225, -62.21289667275519, -62.23886505197959, -62.2650110161224, -62.29112965950853, -62.31713901986794])
        expected['lon_s'] = np.array([153.31437395944647, 153.31142331772503, 153.3083360722202, 153.32273411099058, 153.31957856359404, 153.31299426123414, 153.30616407206412, 153.31696943133107, 153.310016298756, 153.3030221850459])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_180_60000(self):
        """Test for latitude = 60, longitude = 180, rho = 60000"""
        lat = 60
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([78.14202320457623, 78.14387247829401, 78.14542036991467, 78.1437281905478, 78.14468014816337, 78.1481379314307, 78.15117309674312, 78.15096306842614, 78.15320507802119, 78.1550142243776])
        expected['lon_n'] = np.array([201.89869797866277, 201.6146167507036, 201.3306311453928, 201.06982720212503, 200.78659291762293, 200.4556292712008, 200.12465196440033, 199.81699292342626, 199.48634346570594, 199.155748649499])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([81.36583065791359, 81.39706852808264, 81.42676533682021, 81.51654646591274, 81.54890363333203, 81.5697288801114, 81.59715550472359, 81.67991178425476, 81.70342300292522, 81.72548109884978])
        expected['lon_s'] = np.array([196.4284541139129, 196.47672989096657, 196.52622144169246, 196.48101924976532, 196.5241177434281, 196.56777631260283, 196.5988263525697, 196.53671261225762, 196.57160121853804, 196.60881208913625])
        expected['rho_s'] = np.array([263486.36861572263, 263546.01112060546, 263617.95155029296, 263655.8661010742, 263684.9339233398, 263794.4243408203, 263799.06644287106, 263814.0135253906, 263869.2326660156, 263948.36713867186])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_180_60000(self):
        """Test for latitude = 90, longitude = 180, rho = 60000"""
        lat = 90
        lon = 180
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089, 85.75967897126492])
        expected['lon_n'] = np.array([246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524, 241.7348747277917])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269, 46.93261784888895])
        expected['lon_s'] = np.array([5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653, 4.87329514968767])
        expected['rho_s'] = np.array([353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582, 352904.7256958008])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_270_60000(self):
        """Test for latitude = -90, longitude = 270, rho = 60000"""
        lat = -90
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([-32.795791011887204, -33.37848811035933, -33.88485101466739, -34.1580585814679, -34.57930366833928, -34.911032295948445, -35.23022069624149, -35.43126048427165, -35.724740052528745, -36.00470950660757])
        expected['lon_n'] = np.array([359.98195099774716, 0.10806736643975413, 0.2090603244260279, 0.3563294482152578, 0.4288829192303067, 0.4590981019732828, 0.4879725132461002, 0.6056791375644869, 0.6296300496627828, 0.6515275827809909])
        expected['rho_n'] = np.array([301659.7591186523, 302019.8987426758, 302076.1629638672, 302150.77685546875, 301862.8450073242, 300665.0125488281, 299569.28203125, 299156.52381591796, 298128.33466796874, 297174.85662841797])
        expected['lat_s'] = np.array([-74.65022035886321, -74.65431847236522, -74.65844390662389, -74.66989130367284, -74.67398941717485, -74.67628436073596, -74.67860662505376, -74.68829183329686, -74.69060043723633, -74.69288172041911])
        expected['lon_s'] = np.array([142.2499729380244, 142.20713399155008, 142.164103799779, 142.13928289233516, 142.09623904018576, 142.0419526966625, 141.9875160889775, 141.95116582221468, 141.89660627112463, 141.84196475776452])
        expected['rho_s'] = np.array([6371.199240493775, 6371.198860740661, 6371.198860740661, 6371.2, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_270_60000(self):
        """Test for latitude = -60, longitude = 270, rho = 60000"""
        lat = -60
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([34.63732794657137, 34.569138752992544, 34.503183031272314, 34.50214142742389, 34.439532498396964, 34.37195802184345, 34.30645992279761, 34.29881011092719, 34.23583576677968, 34.17402255479106])
        expected['lon_n'] = np.array([285.8944246177553, 286.12547625699847, 286.36106314185054, 286.4460580158822, 286.6873822596371, 286.96613594004367, 287.2492336207623, 287.3826955171444, 287.6735796135169, 287.969190200795])
        expected['rho_n'] = np.array([84873.69056396485, 85917.67744140625, 86967.49732666016, 87378.04993591308, 88432.4086303711, 89607.55919494628, 90790.96103515624, 91327.05698547362, 92515.59308776855, 93711.48728637696])
        expected['lat_s'] = np.array([-77.35003021322916, -77.36026183660584, -77.37057542225256, -77.38138078151951, -77.39169436716622, -77.40163912259777, -77.411611198786, -77.42214335048615, -77.43214274743104, -77.44219678588931])
        expected['lon_s'] = np.array([174.18679002522347, 174.15250247558998, 174.11801002028145, 174.12060549216605, 174.08573054626396, 174.03668978802327, 173.9875534071342, 173.97543665154657, 173.9258494781723, 173.87603007836623])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.197721481323, 6371.197341728211, 6371.197721481323, 6371.1984809875485, 6371.198860740661, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_270_60000(self):
        """Test for latitude = -30, longitude = 270, rho = 60000"""
        lat = -30
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([60.96954793295184, 61.0640811661594, 61.158873946555424, 61.24675799060598, 61.341854714420066, 61.44362453305325, 61.54515017242362, 61.639686820725764, 61.74166496012864, 61.84406315616547])
        expected['lon_n'] = np.array([281.49581011302377, 281.4816579610635, 281.46717796002304, 281.3940129736339, 281.37953297259344, 281.3707083681858, 281.36161055621136, 281.2927349286209, 281.2838283619432, 281.274457342402])
        expected['rho_n'] = np.array([6371.199620246887, 6371.198101234436, 6371.198101234436, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.198860740661, 6371.198860740661])
        expected['lat_s'] = np.array([-77.37718704536914, -77.39236372570491, -77.40748576452731, -77.42136470892078, -77.43659603076989, -77.4528382206162, -77.46914871235418, -77.48406584550148, -77.50033535610447, -77.51671414973416])
        expected['lon_s'] = np.array([209.40651021623248, 209.45750440857577, 209.50948214815958, 209.57110411485144, 209.62325943935366, 209.6678469142555, 209.71233876650896, 209.76763597802938, 209.8132070001717, 209.85881900344904])
        expected['rho_s'] = np.array([6371.196961975097, 6371.1984809875485, 6371.197721481323, 6371.196202468872, 6371.195822715759, 6371.1965822219845, 6371.199620246887, 6371.1965822219845, 6371.195822715759, 6371.196202468872])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_270_60000(self):
        """Test for latitude = 0, longitude = 270, rho = 60000"""
        lat = 0
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([58.829724798067716, 58.90233995422871, 58.97548103495579, 59.05158300268807, 59.124946064563176, 59.20297756073599, 59.280961245584606, 59.36234294954531, 59.44073986083872, 59.51964562122529])
        expected['lon_n'] = np.array([272.71396268957153, 272.66904736558956, 272.62394079631076, 272.56184071637705, 272.5163243357481, 272.473294143977, 272.42996342388244, 272.3694206270794, 272.32576205790474, 272.28188492267657])
        expected['rho_n'] = np.array([6371.2, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.197721481323, 6371.198860740661, 6371.1984809875485, 6371.199620246887])
        expected['lat_s'] = np.array([-74.61169809194433, -74.62475741363738, -74.6378030749521, -74.65703688765487, -74.67004156783457, -74.68386587071467, -74.69763553208142, -74.71766164672789, -74.73136300620294, -74.74525561097477])
        expected['lon_s'] = np.array([211.10791033848264, 211.21336845926766, 211.31978280653644, 211.43045919184732, 211.53766584105986, 211.64603362243128, 211.75546691331323, 211.86861582710364, 211.97892338219935, 212.08868452216146])
        expected['rho_s'] = np.array([6371.196202468872, 6371.1965822219845, 6371.196961975097, 6371.196961975097, 6371.196202468872, 6371.196961975097, 6371.196202468872, 6371.198101234436, 6371.195822715759, 6371.199240493775])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_270_60000(self):
        """Test for latitude = 30, longitude = 270, rho = 60000"""
        lat = 30
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([64.44818547535414, 64.51303641397611, 64.57803420166525, 64.64333764031974, 64.70861375821755, 64.77805287641449, 64.84758932480709, 64.91733409396939, 64.98708227822627, 65.05702170777991])
        expected['lon_n'] = np.array([268.73166187513067, 268.68210202251305, 268.6322689623286, 268.57841975091225, 268.5280675963509, 268.4816769515082, 268.43490381607194, 268.3840598878904, 268.33682229959055, 268.28914757918386])
        expected['rho_n'] = np.array([6371.199240493775, 6371.1984809875485, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199240493775])
        expected['lat_s'] = np.array([-74.64313062250471, -74.65893568024413, -74.67479537949691, -74.69797704153993, -74.71380942003603, -74.72934127020864, -74.7449004411379, -74.76795915977587, -74.78354565146185, -74.79922776579619])
        expected['lon_s'] = np.array([183.4888792973772, 183.57477575637927, 183.66063123424632, 183.76289282649975, 183.8501416629575, 183.93647525406647, 184.02279518479708, 184.1272697583416, 184.21476448160945, 184.30253241244412])
        expected['rho_s'] = np.array([6371.198860740661, 6371.198101234436, 6371.1984809875485, 6371.2, 6371.198101234436, 6371.197721481323, 6371.199620246887, 6371.196961975097, 6371.1984809875485, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_270_60000(self):
        """Test for latitude = 60, longitude = 270, rho = 60000"""
        lat = 60
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([74.21708744142697, 74.2791021439961, 74.34111343147065, 74.39956106774574, 74.46150917597046, 74.52975471860994, 74.59804294993172, 74.66279314326344, 74.73112064817296, 74.79947888893373])
        expected['lon_n'] = np.array([262.83964016875416, 262.7439628788606, 262.64743864551, 262.53613388279547, 262.43810700782745, 262.3405719064797, 262.2420532578914, 262.12834426858905, 262.02799512930324, 261.9264711974801])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2])
        expected['lat_s'] = np.array([39.50468467366253, 39.48803267246604, 39.47080352028468, 39.40632311942518, 39.38963696728284, 39.3696006073526, 39.348990511532094, 39.280579336805246, 39.26038929761869, 39.239789447081925])
        expected['lon_s'] = np.array([335.0638089808965, 335.0205055815586, 334.97725682373414, 335.06348113181633, 335.02009577020846, 334.97471599336285, 334.92941817878733, 335.0134568263352, 334.96813169100295, 334.92272459340074])
        expected['rho_s'] = np.array([350511.5884155273, 350516.69229736325, 350521.4802246094, 350431.3116455078, 350436.41552734375, 350438.79733886715, 350440.6444580078, 350347.21911621094, 350349.45510253904, 350351.25361328124])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_270_60000(self):
        """Test for latitude = 90, longitude = 270, rho = 60000"""
        lat = 90
        lon = 270
        rho = 60000

        expected = {}
        expected['lat_n'] = np.array([85.315567264826, 85.36310324701518, 85.41027338719692, 85.45207841378564, 85.49850790532926, 85.55267941629711, 85.60638476685409, 85.6546507255108, 85.7074314392089, 85.75967897126492])
        expected['lon_n'] = np.array([246.5008168060865, 246.0383037162499, 245.56641960687207, 245.08562893081654, 244.5955218767331, 244.04976244129242, 243.49061583507847, 242.92048628467919, 242.3351117520524, 241.7348747277917])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([47.05908221646632, 47.0583343107522, 47.057306367282116, 46.99760709884162, 46.997531966760754, 46.996083966656705, 46.99447887220175, 46.93424001881683, 46.9335399244269, 46.93261784888895])
        expected['lon_s'] = np.array([5.141578576983424, 5.087981228792687, 5.034293380595448, 5.0954479062159805, 5.041853546233006, 4.985267136506743, 4.928502714975236, 4.986595181413487, 4.930053594803653, 4.87329514968767])
        expected['rho_s'] = np.array([353011.3239135742, 353011.37252197263, 353010.81352539064, 352959.4830566406, 352960.13927001954, 352958.8511474609, 352957.8060668945, 352905.7950805664, 352905.308996582, 352904.7256958008])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_0_90000(self):
        """Test for latitude = -90, longitude = 0, rho = 90000"""
        lat = -90
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.9163904984338, -72.91345351709069, -72.91069412066602, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99974612706464, -72.99780635334037, -72.99611246642618])
        expected['lon_n'] = np.array([26.311299116952725, 26.151631492272628, 25.992225122328286, 26.015910510822597, 25.85506126341609, 25.679313665882493, 25.503724870247098, 25.510640436781735, 25.33370780142714, 25.156565137755567])
        expected['rho_n'] = np.array([257072.95222167968, 257070.06002197266, 257067.4108642578, 257035.20780029296, 257032.02395019532, 257030.27404785156, 257028.6213623047, 257060.654296875, 257058.53983154296, 257056.47397460937])
        expected['lat_s'] = np.array([-67.50285086417338, -67.50144384520436, -67.50003682623534, -67.50916195896647, -67.50771395886244, -67.50597909081327, -67.50410761898067, -67.51287758187496, -67.5110470911774, -67.50910731745313])
        expected['lon_s'] = np.array([152.87409996554746, 152.85318592630887, 152.8320669813952, 152.8639776251975, 152.84269475574374, 152.8135025272311, 152.7841873553134, 152.80809301740845, 152.77857293981566, 152.7489708999528])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_0_90000(self):
        """Test for latitude = -60, longitude = 0, rho = 90000"""
        lat = -60
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-16.47693912235266, -17.77632114024506, -18.983837113800774, -18.18404928274896, -19.355044234812624, -20.43765653901633, -21.480578613953085, -20.85218071974441, -21.848076942245626, -22.745461346347753])
        expected['lon_n'] = np.array([0.028635141208455332, 0.24032732962075956, 0.4409943387326203, 0.3520257086666358, 0.5454451933780253, 0.7204271160334945, 0.8945325866802334, 0.8304130119079751, 0.9936463154999347, 1.1384130287320011])
        expected['rho_n'] = np.array([347685.6662597656, 346238.3025878906, 345217.0158325195, 346000.6075195312, 345072.9162353515, 344028.6134033203, 343446.38200683595, 343929.7925292969, 343331.54466552736, 342765.0380859375])
        expected['lat_s'] = np.array([-75.7420534184464, -75.74066005985574, -75.73936232391344, -75.76847259015602, -75.76698360891697, -75.76352753319694, -75.76024904239534, -75.78737855377861, -75.78388149692358, -75.78041176082522])
        expected['lon_s'] = np.array([135.53718105946584, 135.50611735912062, 135.47482143234362, 135.42228361724787, 135.39145214333445, 135.3500748573425, 135.30827409962203, 135.2445484346658, 135.20326677132223, 135.16157529662848])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.198101234436, 6371.198860740661, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_0_90000(self):
        """Test for latitude = -30, longitude = 0, rho = 90000"""
        lat = -30
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([5.708218811333268, 5.709728283139839, 5.71106700021717, 5.67455963910345, 5.676567714719425, 5.6779405827426075, 5.679204167739059, 5.644916618105597, 5.646890542775722, 5.648748354229966])
        expected['lon_n'] = np.array([358.7649205706776, 358.7598935514485, 358.754811890706, 358.8085791398523, 358.8034974791098, 358.7994540071212, 358.7953285728625, 358.8502432937894, 358.8461451802874, 358.8419377837587])
        expected['rho_n'] = np.array([382301.5052978516, 382299.9012207031, 382298.56448974606, 382279.36417236325, 382277.29831542965, 382275.9858886719, 382274.7463745117, 382317.0599853516, 382315.1156494141, 382313.3657470703])
        expected['lat_s'] = np.array([-78.6087930961275, -78.59883468031762, -78.58875332110267, -78.58477815100574, -78.5747377729258, -78.56177407388111, -78.54871475218806, -78.54172063847795, -78.52867497716323, -78.51556101395681])
        expected['lon_s'] = np.array([70.58165175072685, 70.51074072676379, 70.43926962728877, 70.28041991776179, 70.20981625231136, 70.15161621039368, 70.09299952693664, 69.94665589377993, 69.88886566321246, 69.83100713075328])
        expected['rho_s'] = np.array([6371.191645431519, 6371.191645431519, 6371.192404937744, 6371.199620246887, 6371.197721481323, 6371.1965822219845, 6371.195442962647, 6371.191265678405, 6371.192025184631, 6371.192025184631])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_0_90000(self):
        """Test for latitude = 0, longitude = 0, rho = 90000"""
        lat = 0
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([19.597442639500045, 19.6679711728696, 19.736594083460716, 19.665566946281757, 19.73494800787074, 19.80391925810953, 19.870875602543165, 19.803290880705887, 19.870943904434867, 19.93681424879047])
        expected['lon_n'] = np.array([358.29055027244186, 358.2782832526925, 358.26661728959016, 358.3150569911839, 358.30360959413497, 358.29371948021674, 358.2844031021889, 358.33330725664615, 358.3242914069417, 358.3157400101009])
        expected['rho_n'] = np.array([206840.88745117188, 208341.6960571289, 209828.52974853516, 208668.56323242188, 210174.59724121093, 211699.9530883789, 213207.83421630858, 212092.66033935547, 213620.37369384765, 215137.85498046875])
        expected['lat_s'] = np.array([-70.32011265150689, -70.29769597065092, -70.2751700067682, -70.25488434493326, -70.23235838105057, -70.21129407765025, -70.1901478119799, -70.17111890495224, -70.15005460155191, -70.1288946755032])
        expected['lon_s'] = np.array([40.256236798179295, 40.18549652894548, 40.114725523860386, 40.00030619488434, 39.92962056716387, 39.87097997804473, 39.81228816250681, 39.70988996646998, 39.65125620754, 39.592543901434574])
        expected['rho_s'] = np.array([6371.193164443969, 6371.185569381713, 6371.185569381713, 6371.198860740661, 6371.189746665955, 6371.184809875488, 6371.186708641052, 6371.198101234436, 6371.188607406616, 6371.185189628601])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_0_90000(self):
        """Test for latitude = 30, longitude = 0, rho = 90000"""
        lat = 30
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([72.71998681498833, 72.75759042146385, 72.79496009396027, 72.83570729500127, 72.87250323160742, 72.91078131926346, 72.948746925765, 72.99016690043923, 73.02746656349667, 73.06441788690645])
        expected['lon_n'] = np.array([342.207831682893, 342.4253322268229, 342.6435977519398, 342.84148199257345, 343.06138676309115, 343.3089401393691, 343.55747706288753, 343.78574198494937, 344.03630064446213, 344.28787017197203])
        expected['rho_n'] = np.array([6371.197721481323, 6371.197341728211, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.2, 6371.199620246887])
        expected['lat_s'] = np.array([24.092854585149, 24.10631005781393, 24.1194581719662, 24.07380518755383, 24.087588509298925, 24.10096201969381, 24.114028171576038, 24.06823858338028, 24.081939942855314, 24.095320283439378])
        expected['lon_s'] = np.array([1.0033409152531216, 0.9865257365684906, 0.9697136528133273, 1.004407598702407, 0.9875750243797338, 0.9703104405920572, 0.9530483114036136, 0.9872190007692468, 0.9699592194583304, 0.9526915407411862])
        expected['rho_s'] = np.array([382297.73814697267, 382300.60604248045, 382303.3767211914, 382293.2418701172, 382296.20698242186, 382299.0991821289, 382301.6997314453, 382291.6377929687, 382294.50568847655, 382297.4708007812])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_0_90000(self):
        """Test for latitude = 60, longitude = 0, rho = 90000"""
        lat = 60
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([81.99164076420048, 82.04278778202648, 82.09373074795101, 82.14635308663773, 82.19667365157416, 82.25103000453642, 82.30501667350984, 82.36091664922452, 82.41410162474412, 82.46699460967669])
        expected['lon_n'] = np.array([323.3445157614753, 323.6611360106404, 323.98108939212045, 324.23254963660366, 324.56039871676427, 324.94510229157606, 325.3350241309138, 325.6557424935809, 326.05645603180716, 326.46184141942575])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([36.0948869003575, 36.103305108509545, 36.11143986381103, 36.06449597364553, 36.07568382350601, 36.08393810711797, 36.091854296366016, 36.04480453826838, 36.05342423700094, 36.06179804892338])
        expected['lon_s'] = np.array([2.5580776661160836, 2.5287518220278105, 2.4993591701517186, 2.542740689778231, 2.5134280791814754, 2.4827793128283355, 2.4520639521307883, 2.494123189822669, 2.463498329131624, 2.432767387065033])
        expected['rho_s'] = np.array([382286.63112792966, 382295.2834228515, 382303.86280517577, 382326.7330566406, 382274.1630737305, 382282.81536865234, 382291.12740478513, 382313.8761352539, 382322.8200805664, 382331.6425048828])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_0_90000(self):
        """Test for latitude = 90, longitude = 0, rho = 90000"""
        lat = 90
        lon = 0
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544, 83.52580242396243])
        expected['lon_n'] = np.array([228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622, 224.2913502291445])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855, 48.150225426943365])
        expected['lon_s'] = np.array([3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065, 3.054661447079983])
        expected['rho_s'] = np.array([376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015, 376847.2055175781])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_90_90000(self):
        """Test for latitude = -90, longitude = 90, rho = 90000"""
        lat = -90
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.9163904984338, -72.91345351709069, -72.91068046028767, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99974612706464, -72.99780635334037, -72.99611246642618])
        expected['lon_n'] = np.array([26.311299116952725, 26.151628077178042, 25.992218292139118, 26.01590538818072, 25.855054433226922, 25.67930512814603, 25.503711209868758, 25.510626776403395, 25.33370097123797, 25.156563430208273])
        expected['rho_n'] = np.array([257072.95222167968, 257070.08432617187, 257067.4108642578, 257035.20780029296, 257032.02395019532, 257030.27404785156, 257028.6456665039, 257060.654296875, 257058.56413574217, 257056.47397460937])
        expected['lat_s'] = np.array([-67.50285086417338, -67.50145750558272, -67.5000504866137, -67.50916195896647, -67.50771395886244, -67.50597909081327, -67.50412127935903, -67.51287758187496, -67.5110470911774, -67.50912097783146])
        expected['lon_s'] = np.array([152.87409996554746, 152.85318592630887, 152.83205332101684, 152.8639776251975, 152.84277671801377, 152.81351618760945, 152.7841873553134, 152.80807935703012, 152.77857293981566, 152.74882063579108])
        expected['rho_s'] = np.array([6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_90_90000(self):
        """Test for latitude = -60, longitude = 90, rho = 90000"""
        lat = -60
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-66.56767502339358, -66.59259155348579, -66.6177403100098, -66.67090650250915, -66.69552250427787, -66.72336235533487, -66.75142077244527, -66.80792009725963, -66.83543209923644, -66.86312168613165])
        expected['lon_n'] = np.array([48.22212933281426, 48.07720637900493, 47.932088764804256, 47.96552254079146, 47.81861541702825, 47.66339936813971, 47.50795109281938, 47.531566471874704, 47.37415793226259, 47.216551317164544])
        expected['rho_n'] = np.array([256462.52795410156, 256480.245715332, 256497.8662597656, 256441.77216796874, 256458.63928222656, 256472.8086303711, 256486.85645751952, 256427.50560302733, 256440.75139160155, 256453.87565917967])
        expected['lat_s'] = np.array([-65.90827490054221, -65.9118539196673, -65.91535097652235, -65.9290523359974, -65.93265867587917, -65.93575958176234, -65.93894244991557, -65.95228863955379, -65.95549882846367, -65.9586816966169])
        expected['lon_s'] = np.array([146.26688915117893, 146.26568703788502, 146.2645532264828, 146.30313013491502, 146.30233783297132, 146.29436017202073, 146.28617760539507, 146.31797896617064, 146.3103564750569, 146.30270666318648])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.200759506225, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_90_90000(self):
        """Test for latitude = -30, longitude = 90, rho = 90000"""
        lat = -30
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-59.40422726339759, -59.47509730622562, -59.54797542466966, -59.64123482759703, -59.71132622885969, -59.78584359270454, -59.860456579197745, -59.95617485022632, -60.029886251749076, -60.1016988606826])
        expected['lon_n'] = np.array([54.777437539679845, 54.61512150914908, 54.446730025351584, 54.4369321189872, 54.27210599393646, 54.09547388690534, 53.91833976097023, 53.90098083519464, 53.72152444494173, 53.54736145129557])
        expected['rho_n'] = np.array([255386.36231689452, 255359.74921875, 255394.16396484376, 255367.1619995117, 255338.1184814453, 255365.5822265625, 255392.14671630858, 255357.3187988281, 255382.42503662108, 255344.7049194336])
        expected['lat_s'] = np.array([-64.6106755620232, -64.62618009143912, -64.64164363972003, -64.66528975462663, -64.68080794442088, -64.6958616813516, -64.7107788144989, -64.73423368410872, -64.74920545876938, -64.76393134661996])
        expected['lon_s'] = np.array([145.54858547692538, 145.56496427055507, 145.5806737056461, 145.63616216246328, 145.65195355982436, 145.65912525845286, 145.66547733438097, 145.71443613035163, 145.72046035719958, 145.72547371605037])
        expected['rho_s'] = np.array([6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_90_90000(self):
        """Test for latitude = 0, longitude = 90, rho = 90000"""
        lat = 0
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([80.63331165244995, 80.56841461005106, 80.50370710540166, 80.42437616573925, 80.35977623656927, 80.28303991501782, 80.20652728216169, 80.1150411670976, 80.03837570875879, 79.96217128818897])
        expected['lon_n'] = np.array([118.22437272115435, 118.2125838146469, 118.20218826673015, 117.80346914374148, 117.79471284122553, 117.74963359270345, 117.70616626882548, 117.28550857822273, 117.24475966963443, 117.20634668574229])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([-61.27459260533553, -61.30076589023503, -61.3269391751345, -61.32676159021608, -61.35300317700728, -61.379490650608574, -61.40585518080482, -61.405896161939864, -61.43241095629784, -61.45893941103418])
        expected['lon_s'] = np.array([127.12764724767104, 127.1283849081014, 127.12920453080179, 127.01503108863587, 127.01549554149942, 127.0114247487541, 127.00734029563043, 126.88744315494002, 126.88301719235785, 126.87871417318074])
        expected['rho_s'] = np.array([6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_90_90000(self):
        """Test for latitude = 30, longitude = 90, rho = 90000"""
        lat = 30
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([78.46452936925861, 78.40950878039956, 78.35497142742432, 78.29733146101857, 78.24271812218879, 78.17672824952272, 78.11111830612923, 78.04279080121971, 77.97696741441464, 77.91157774462188])
        expected['lon_n'] = np.array([96.71586796816324, 96.79125276603267, 96.86606382801182, 96.97208202430876, 97.04450252007841, 97.11995561983954, 97.19501939881798, 97.2999720856044, 97.37236526061736, 97.44458768090108])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-59.18497819104016, -59.21937502370034, -59.253853818630574, -59.28143412249909, -59.315789974024256, -59.35257737289393, -59.38925548873692, -59.41911707578819, -59.455767870874496, -59.492377684825755])
        expected['lon_s'] = np.array([121.61852594616546, 121.60951009646105, 121.60042594486492, 121.56850164068429, 121.55907597962967, 121.54936345062991, 121.53989680844028, 121.50737144761267, 121.49731740915442, 121.48735899334454])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_90_90000(self):
        """Test for latitude = 60, longitude = 90, rho = 90000"""
        lat = 60
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([84.1078801096542, 84.05589041723935, 84.00385376727397, 83.95519678028727, 83.90316354541646, 83.84263014012357, 83.78196610746282, 83.72482090288509, 83.66411375465519, 83.60330842245597])
        expected['lon_n'] = np.array([134.02831030953996, 133.78556538643772, 133.54824363353646, 133.4354635499612, 133.2037971936927, 132.91612328623012, 132.63512930377578, 132.47456521676713, 132.20266904628727, 131.93645559319685])
        expected['rho_n'] = np.array([6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2])
        expected['lat_s'] = np.array([15.369842411207301, 15.731487267381127, 16.08887008551288, 16.43903339369193, 16.786751494148106, 17.167220351674487, 17.541425925731986, 17.90029772510279, 18.26177865673654, 18.616115210498464])
        expected['lon_s'] = np.array([44.37171947124629, 43.938729874097085, 43.50225346537659, 43.088436209227616, 42.64571359251114, 42.1506888022253, 41.651709332410014, 41.173821486751734, 40.666785808810424, 40.15665947026968])
        expected['rho_s'] = np.array([220195.51022949218, 223172.75032958985, 226203.1437133789, 229782.5689575195, 232920.60563964842, 236487.68435058594, 240124.6862426758, 244362.90111083983, 248157.12686767578, 252022.66114501952])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_90_90000(self):
        """Test for latitude = 90, longitude = 90, rho = 90000"""
        lat = 90
        lon = 90
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544, 83.52580242396243])
        expected['lon_n'] = np.array([228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622, 224.2913502291445])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855, 48.150225426943365])
        expected['lon_s'] = np.array([3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065, 3.054661447079983])
        expected['rho_s'] = np.array([376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015, 376847.2055175781])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_180_90000(self):
        """Test for latitude = -90, longitude = 180, rho = 90000"""
        lat = -90
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.9163904984338, -72.91345351709069, -72.91069412066602, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99973246668631, -72.99780635334037, -72.99609880604785])
        expected['lon_n'] = np.array([26.31129740940543, 26.151633199819923, 25.992225122328286, 26.015907095728014, 25.855057848321504, 25.679308543240616, 25.50372145515251, 25.510635314139858, 25.33370097123797, 25.15656172266098])
        expected['rho_n'] = np.array([257072.95222167968, 257070.06002197266, 257067.4108642578, 257035.20780029296, 257032.04825439453, 257030.27404785156, 257028.6456665039, 257060.654296875, 257058.56413574217, 257056.49827880858])
        expected['lat_s'] = np.array([-67.50283720379505, -67.50145750558272, -67.5000504866137, -67.50916195896647, -67.50768663810575, -67.50597909081327, -67.50412127935903, -67.51286392149663, -67.51103343079907, -67.50912097783146])
        expected['lon_s'] = np.array([152.8741136259258, 152.85318592630887, 152.83205332101684, 152.8639776251975, 152.84281769914878, 152.8134342253394, 152.7841873553134, 152.8081476589218, 152.77857293981566, 152.74882063579108])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_180_90000(self):
        """Test for latitude = -60, longitude = 180, rho = 90000"""
        lat = -60
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-78.73452321836908, -78.7465443513083, -78.7556558236611, -78.8019371854771, -78.8133709221477, -78.82552865887033, -78.84067801844941, -78.88805221053263, -78.90246390968136, -78.91693025034346])
        expected['lon_n'] = np.array([48.240471805830325, 47.9468385583169, 47.642399951536504, 47.754336506749254, 47.45703886274611, 47.13029968832563, 46.814287325996645, 46.89313502977527, 46.57289819544464, 46.252531587519776])
        expected['rho_n'] = np.array([264118.6180541992, 264084.32482910156, 264111.91009521484, 264044.6603759765, 264009.6137207031, 264037.4663330078, 264004.1938842773, 263999.26013183594, 263965.0398193359, 263931.18406982423])
        expected['lat_s'] = np.array([49.73867471088904, 49.724584030631306, 49.70744708600375, 49.79239073361661, 49.77787658163033, 49.76261452392994, 49.751481315582815, 49.839170699242025, 49.827019792708576, 49.81531284847117])
        expected['lon_s'] = np.array([199.72171178450458, 199.59285343562314, 199.4629842187445, 199.46186406772063, 199.33265054900232, 199.1906918972928, 199.0499490192555, 199.03570124464684, 198.89446659298932, 198.7533002432235])
        expected['rho_s'] = np.array([264279.6333740234, 264314.8501586914, 264287.58084716793, 264307.0971191406, 264341.90073242184, 264313.0516479492, 264344.6471069336, 264361.5628295898, 264393.81450195314, 264424.85096435546])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_180_90000(self):
        """Test for latitude = -30, longitude = 180, rho = 90000"""
        lat = -30
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-79.43612024991279, -79.47316719597094, -79.51218123651006, -79.55462403201253, -79.59109724218038, -79.6330072829276, -79.67687075777741, -79.72558366693792, -79.76682434914648, -79.80798306908497])
        expected['lon_n'] = np.array([69.67487583651597, 69.32149550923785, 68.9893502400893, 69.21238323724688, 68.85145555093591, 68.47666941079896, 68.12251727214463, 68.33179426831383, 67.94823133509342, 67.56336383574154])
        expected['rho_n'] = np.array([269380.2098388672, 269377.48776855465, 269313.9322875976, 269269.2611694336, 269265.7370605469, 269263.4767700195, 269200.65041503904, 269156.4653808594, 269153.50026855466, 269150.3407226562])
        expected['lat_s'] = np.array([46.68389992535752, 46.67963447222085, 46.67575833986687, 46.76201338380037, 46.76141915734258, 46.75836947787817, 46.75990968553601, 46.84340874813941, 46.8440029745972, 46.84064252152556])
        expected['lon_s'] = np.array([196.06295703104718, 195.9623893257079, 195.86186260150367, 195.86122056372167, 195.76192327356804, 195.65205285057922, 195.54371238996447, 195.53151367210683, 195.42285902279028, 195.31266075072128])
        expected['rho_s'] = np.array([269496.16517333983, 269463.28159179684, 269429.71749267576, 269426.7280761719, 269454.84803466796, 269422.01306152344, 269448.72337646486, 269385.5324584961, 269413.5795043945, 269380.67161865236])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_180_90000(self):
        """Test for latitude = 0, longitude = 180, rho = 90000"""
        lat = 0
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-77.35171043976499, -77.39818304687776, -77.44606267295953, -77.49786282762491, -77.54492283100629, -77.59681860832006, -77.6485094799587, -77.7093254843285, -77.7601011106184, -77.81065817085482])
        expected['lon_n'] = np.array([68.56572824738676, 68.18989708830681, 67.8304788738024, 68.00384956550484, 67.6372186712369, 67.23783701989873, 66.83684344391645, 67.01356775850137, 66.6044872385418, 66.19376747318141])
        expected['rho_n'] = np.array([267678.38120117184, 267652.78887939453, 267567.4568359375, 267582.33100585936, 267495.05462646484, 267469.29217529297, 267444.57480468747, 267397.4246582031, 267371.07890625, 267344.7817626953])
        expected['lat_s'] = np.array([48.26541656729563, 48.26529020879599, 48.26946686947345, 48.34415498804754, 48.34751885621377, 48.348253101549545, 48.34930836577632, 48.428955201687835, 48.42913278660625, 48.433606560512615])
        expected['lon_s'] = np.array([193.92543566991336, 193.8419707582558, 193.76041829956586, 193.75955769573042, 193.67773202947367, 193.5862074945955, 193.49465563896064, 193.4871287704953, 193.39530370729364, 193.30530913478955])
        expected['rho_s'] = np.array([267608.4094116211, 267568.0158325195, 267587.9695800781, 267546.6038330078, 267567.52974853513, 267528.18124999997, 267488.1765380859, 267508.44624023436, 267469.6081298828, 267490.6069580078])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_180_90000(self):
        """Test for latitude = 30, longitude = 180, rho = 90000"""
        lat = 30
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([75.08040627704365, 75.0775017390991, 75.07433423887151, 75.0708320593746, 75.0670054458921, 75.06519544576204, 75.0626751059583, 75.05993107745925, 75.05661160552262, 75.05305136941776])
        expected['lon_n'] = np.array([200.18912895016524, 199.9343355733671, 199.6802115551076, 199.43201614104768, 199.17867076435357, 198.88409836582926, 198.58970355222337, 198.30114172016866, 198.00749822737149, 197.71463337613966])
        expected['rho_n'] = np.array([6371.199240493775, 6371.197721481323, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([54.373074175958585, 54.37857589333503, 54.38166655393446, 54.44926152105549, 54.454257804433354, 54.457836823558445, 54.46430842779703, 54.53274009309139, 54.53583416878541, 54.54180033902541])
        expected['lon_s'] = np.array([193.79726033994888, 193.721977994917, 193.6442641025406, 193.64837587642094, 193.57261541814717, 193.48701948746856, 193.40356823618936, 193.39930619814726, 193.31327313536178, 193.22934377084067])
        expected['rho_s'] = np.array([260348.21041259766, 260363.76510009766, 260316.34760742186, 260319.3856323242, 260334.77019042967, 260288.59221191405, 260304.36563720703, 260308.54595947266, 260262.78115234376, 260278.99205322264])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_180_90000(self):
        """Test for latitude = 60, longitude = 180, rho = 90000"""
        lat = 60
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([76.15632584963438, 76.16120516602271, 76.16569516162835, 76.15669382607592, 76.16034883105553, 76.16737794948511, 76.17356012445762, 76.1664029399811, 76.17181757244562, 76.17675067657366])
        expected['lon_n'] = np.array([203.9237671053015, 203.63105251823143, 203.33840623305306, 203.1048410841953, 202.81280951604225, 202.47729696363288, 202.14119701495488, 201.86483390075784, 201.52930768797012, 201.19372683366905])
        expected['rho_n'] = np.array([6371.199620246887, 6371.198860740661, 6371.199240493775, 6371.2, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.1984809875485, 6371.2, 6371.2])
        expected['lat_s'] = np.array([64.24197694166958, 64.25124721592057, 64.26039454676652, 64.29947347410274, 64.30838516342232, 64.31840675848201, 64.32880059885149, 64.3687469602123, 64.37840143260411, 64.38842302766382])
        expected['lon_s'] = np.array([197.15940729850934, 197.07418019804592, 196.98556532375417, 197.00224464570732, 196.91518705454635, 196.8178978400087, 196.72284892751878, 196.7293376072303, 196.63058673221025, 196.5338029516712])
        expected['rho_s'] = np.array([254994.8702758789, 255028.28854980468, 254974.89222412108, 254994.0925415039, 255000.26580810547, 254968.50021972656, 255002.11292724608, 255021.0458984375, 254971.2708984375, 254982.1591796875])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_180_90000(self):
        """Test for latitude = 90, longitude = 180, rho = 90000"""
        lat = 90
        lon = 180
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544, 83.52580242396243])
        expected['lon_n'] = np.array([228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622, 224.2913502291445])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855, 48.150225426943365])
        expected['lon_s'] = np.array([3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065, 3.054661447079983])
        expected['rho_s'] = np.array([376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015, 376847.2055175781])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_270_90000(self):
        """Test for latitude = -90, longitude = 270, rho = 90000"""
        lat = -90
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-72.9163904984338, -72.91345351709069, -72.91068046028767, -72.95828687880265, -72.95487178421766, -72.95346476524864, -72.9522626519547, -72.99974612706464, -72.99780635334037, -72.99609880604785])
        expected['lon_n'] = np.array([26.3113042395946, 26.1516383224618, 25.99222853742287, 26.015915633464477, 25.855064678510676, 25.67931708097708, 25.503724870247098, 25.51064385187632, 25.333709508974433, 25.15656855285015])
        expected['rho_n'] = np.array([257072.9765258789, 257070.06002197266, 257067.4108642578, 257035.18349609376, 257032.02395019532, 257030.27404785156, 257028.6456665039, 257060.654296875, 257058.56413574217, 257056.49827880858])
        expected['lat_s'] = np.array([-67.50283720379505, -67.50144384520436, -67.50009146774872, -67.50918927972316, -67.50768663810575, -67.50597909081327, -67.50410761898067, -67.51286392149663, -67.51101977042072, -67.50910731745313])
        expected['lon_s'] = np.array([152.87409996554746, 152.85318592630887, 152.8319030568551, 152.8638820025491, 152.84281769914878, 152.8134342253394, 152.7841873553134, 152.8081476589218, 152.77864124170736, 152.7489708999528])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_270_90000(self):
        """Test for latitude = -60, longitude = 270, rho = 90000"""
        lat = -60
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-73.07602567971534, -73.05154628173003, -73.02736741206817, -73.07687262317242, -73.05194243270188, -73.02680733655623, -73.00195910835572, -73.05084960243468, -73.02520907229044, -72.99989639122637])
        expected['lon_n'] = np.array([358.7616147591193, 358.7116724159082, 358.66074652545655, 358.7122188310418, 358.6601454688096, 358.5885104447945, 358.51600115656566, 358.5451250831866, 358.4714136816638, 358.39669141214387])
        expected['rho_n'] = np.array([256351.04459228515, 256338.84388427733, 256326.81330566405, 256320.9316894531, 256308.26920166015, 256301.77998046874, 256295.58240966796, 256295.6553222656, 256288.94736328124, 256282.45814208983])
        expected['lat_s'] = np.array([50.15747117494296, 50.20880346165019, 50.26495103172228, 50.38348896476785, 50.43615313836324, 50.49570555773658, 50.556272260201666, 50.6796698728417, 50.74032878286058, 50.80194391936327])
        expected['lon_s'] = np.array([223.8489998577378, 223.8234412898636, 223.7946042311878, 223.9441580532544, 223.91471993793166, 223.8748316331788, 223.83421932837388, 223.9714924703128, 223.92754703319295, 223.88287759602105])
        expected['rho_s'] = np.array([257075.94163818358, 257019.9690673828, 257020.67388916016, 257001.91104736328, 256935.7064086914, 256919.81146240234, 256900.1979736328, 256859.95021972657, 256834.4551147461, 256805.0713378906])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_270_90000(self):
        """Test for latitude = -30, longitude = 270, rho = 90000"""
        lat = -30
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([-70.39005378860782, -70.37969922182609, -70.36863431537066, -70.44136216965296, -70.42846677249997, -70.40777129931485, -61.079167232803115, -61.38469525475614, -61.78592788735935, -61.96489250399205])
        expected['lon_n'] = np.array([355.40116168747306, 355.5194878846543, 355.6451360446259, 355.85422179549835, 355.9771105590452, 356.0737713961792, 339.63891557378116, 338.66799052288553, 338.0946644439546, 337.2956415940899])
        expected['rho_n'] = np.array([255369.66533203123, 255326.5496826172, 255344.68061523436, 255337.0977050781, 255353.25999755858, 255381.6716064453, 238604.9203613281, 232817.3857055664, 229246.44262695312, 224981.25009765624])
        expected['lat_s'] = np.array([53.79785613462762, 53.97878784574126, 54.16203840607812, 54.36979568515698, 54.54590528271659, 54.730275994077324, 54.913529969508765, 55.12079547496739, 55.300292846355326, 55.47863933086811])
        expected['lon_s'] = np.array([233.64290373126747, 233.52154493009468, 233.39338326050856, 233.42767081014202, 233.29833434801864, 233.15894384743703, 233.01731304480765, 233.03987998982538, 232.89212933769966, 232.74208374201282])
        expected['rho_s'] = np.array([254902.14975585937, 254867.22462158202, 254890.72678222656, 254900.18111572266, 254860.8083129883, 254864.8428100586, 254866.61701660155, 254857.43002929687, 254856.45786132812, 254853.0795776367])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_270_90000(self):
        """Test for latitude = 0, longitude = 270, rho = 90000"""
        lat = 0
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([65.20269256730334, 65.26700904362278, 65.33140406711766, 65.43545004629175, 65.50000216413754, 65.57040433900745, 65.64105923087665, 65.7512182293579, 65.82218047973976, 65.89306076785155])
        expected['lon_n'] = np.array([259.49582543792604, 259.4262394706619, 259.3559431637242, 259.0622723501703, 258.9908012506953, 258.91771822657614, 258.8440068250534, 258.54151140709183, 258.46607879789826, 258.39004513205765])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199240493775, 6371.2, 6371.198860740661, 6371.198860740661, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-71.65189025663267, -71.67410203181353, -71.69638210888613, -71.67639697537467, -71.69886829774401, -71.72118935595162, -71.74356505567258, -71.72311546929754, -71.74568241431527, -71.76829034046801])
        expected['lon_s'] = np.array([175.52899049902265, 175.57614612505245, 175.62380718508078, 175.48754491113903, 175.53569774478763, 175.57704771002287, 175.61858892055488, 175.47342007993544, 175.51489298857575, 175.5565434821345])
        expected['rho_s'] = np.array([6371.1984809875485, 6371.198101234436, 6371.199240493775, 6371.199240493775, 6371.1984809875485, 6371.1984809875485, 6371.198860740661, 6371.198101234436, 6371.198860740661, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_270_90000(self):
        """Test for latitude = 30, longitude = 270, rho = 90000"""
        lat = 30
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([67.32728267279106, 67.38786303563448, 67.44851170036961, 67.51679310150223, 67.57736492660919, 67.64327454455253, 67.70930710590092, 67.78295020553199, 67.84907155933959, 67.9152373093768])
        expected['lon_n'] = np.array([258.1679000594922, 258.086620808369, 258.00465853832884, 257.84207271532586, 257.759318143342, 257.67680945816824, 257.59350847105077, 257.42778076102957, 257.34344158515825, 257.25841939036997])
        expected['rho_n'] = np.array([6371.199620246887, 6371.198860740661, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.2, 6371.2])
        expected['lat_s'] = np.array([-72.57116541702467, -72.59673764527722, -72.62203666596292, -72.63500036500761, -72.66036768758505, -72.68598089697258, -72.71149848371175, -72.72448950351313, -72.74996610911728, -72.775306110938])
        expected['lon_s'] = np.array([170.38308499720003, 170.451701077602, 170.5196614598436, 170.57840108670572, 170.6464161104607, 170.7138164171904, 170.7811484220284, 170.8376340864644, 170.9049797516807, 170.9723117565187])
        expected['rho_s'] = np.array([6371.199620246887, 6371.198101234436, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.1984809875485, 6371.198860740661, 6371.199240493775, 6371.2, 6371.198860740661])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_270_90000(self):
        """Test for latitude = 60, longitude = 270, rho = 90000"""
        lat = 60
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([74.64586622695286, 74.70500541988142, 74.76408997129661, 74.8117185879264, 74.87047016761956, 74.93623635159032, 75.00194106385855, 75.05608738850383, 75.12166232717783, 75.18718774698034])
        expected['lon_n'] = np.array([251.51595150606008, 251.36396613664894, 251.21036884259368, 251.00267645031195, 250.8470574202624, 250.68367929531567, 250.51852532118477, 250.2981287770468, 250.1302154064912, 249.96044422448136])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.2, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([42.657342900189505, 42.64994922041297, 42.642114993434966, 42.59650640525221, 42.58910248019191, 42.57971097008314, 42.5698584222054, 42.52159289043551, 42.51215356900255, 42.5021883230035])
        expected['lon_s'] = np.array([337.0927576549838, 337.0520497275306, 337.01123251705053, 337.0982491270765, 337.0573772750832, 337.01516670601245, 336.97287417467174, 337.058415463837, 337.0161229324963, 336.9737757596422])
        expected['rho_s'] = np.array([374961.6614379883, 374975.4176147461, 374988.8335327148, 374886.56146240234, 374900.1232055664, 374912.1780883789, 374923.60106201173, 374818.55831298826, 374830.58889160154, 374841.76882324216])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_270_90000(self):
        """Test for latitude = 90, longitude = 270, rho = 90000"""
        lat = 90
        lon = 270
        rho = 90000

        expected = {}
        expected['lat_n'] = np.array([83.20892305449568, 83.24590511375672, 83.28235527203613, 83.30259952584923, 83.33807808971922, 83.3800875950962, 83.4213923103282, 83.44636006683919, 83.48642126075544, 83.52580242396243])
        expected['lon_n'] = np.array([228.98235147186594, 228.50986630584114, 228.0323814413439, 227.55902201110533, 227.07388733473766, 226.52755416340668, 225.97411759533887, 225.42768880135952, 224.86302340229622, 224.2913502291445])
        expected['rho_n'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199240493775, 6371.2, 6371.200759506225])
        expected['lat_s'] = np.array([48.21641337509538, 48.22013924328762, 48.22368752656144, 48.17738225908334, 48.18170576882795, 48.18521648606134, 48.1885291278088, 48.14211799239856, 48.146287822886855, 48.150225426943365])
        expected['lon_s'] = np.array([3.2098501752118427, 3.1681896498127613, 3.1263327564750423, 3.199079607220967, 3.157353554694536, 3.114017711958098, 3.0704630897248073, 3.1414865983657734, 3.0981686848759065, 3.054661447079983])
        expected['rho_s'] = np.array([376940.9225097656, 376944.51953125, 376948.189465332, 376888.303918457, 376893.04323730466, 376896.2027832031, 376899.21650390624, 376839.20943603513, 376843.3654541015, 376847.2055175781])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_0_120000(self):
        """Test for latitude = -90, longitude = 0, rho = 120000"""
        lat = -90
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.939896677945, -75.93958248924318, -75.93944588545978, -75.98164279415212, -75.98095977523514, -75.98188868096224, -75.98302249236446, -76.02673570305254, -76.02726845780782, -76.02524672181349])
        expected['lon_n'] = np.array([26.63698814978832, 26.47674337412336, 26.316554947519048, 26.3411419209838, 26.179698447119502, 26.004160877902883, 25.82866770491587, 25.83702444136538, 25.660194258848335, 25.48340505746631])
        expected['rho_n'] = np.array([259029.14860839842, 259028.68682861328, 259028.4680908203, 259002.31677246094, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.40144042968, 258978.8389160156, 259041.88400878906])
        expected['lat_s'] = np.array([-67.0904303817097, -67.08875015517387, -67.08706992863804, -67.09754743882485, -67.09583989153234, -67.09387279705138, -67.09185106105704, -67.1020007221637, -67.10000630692605, -67.0980118916884])
        expected['lon_s'] = np.array([155.14863492143175, 155.11727069276304, 155.08537370933908, 155.10544080512057, 155.0731613311031, 155.03142887527432, 154.98940955150042, 154.99984608055217, 154.95738962467138, 154.91433211214363])
        expected['rho_s'] = np.array([6371.2, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_0_120000(self):
        """Test for latitude = -60, longitude = 0, rho = 120000"""
        lat = -60
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-63.368906189779864, -63.36834611426792, -63.368482718051325, -63.41606181580963, -63.4156110233244, -63.41595253278291, -63.415979853539596, -63.46459714005172, -63.463996083404766, -63.46410536643148])
        expected['lon_n'] = np.array([13.51264732469365, 13.434247852079304, 13.358848540057888, 13.360335813749659, 13.284534374340856, 13.200581104157644, 13.113400569591603, 13.108973753235787, 13.02137999222496, 12.936650641797305])
        expected['rho_n'] = np.array([255250.04006347657, 255201.869140625, 255217.08356933593, 255207.75075683594, 255222.98948974608, 255239.63786621092, 255192.87658691406, 255248.2901611328, 255201.50457763672, 255217.98282470703])
        expected['lat_s'] = np.array([-70.25346366558591, -70.25358660899096, -70.25368223163935, -70.26358600593588, -70.26368162858424, -70.26329913799074, -70.26290298701886, -70.27247891223521, -70.27198713861497, -70.27159098764312])
        expected['lon_s'] = np.array([149.1692141144358, 149.14596415050104, 149.12275516770134, 149.16347675553297, 149.14040437651667, 149.10849373271435, 149.07637818323695, 149.1082341855259, 149.07644648512866, 149.04416701111117])
        expected['rho_s'] = np.array([6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_0_120000(self):
        """Test for latitude = -30, longitude = 0, rho = 120000"""
        lat = -30
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([1.0392176079284496, 1.0368611926647873, 1.0344091547527512, 1.0050120205650188, 1.0031951902457905, 1.0005382466586639, 0.9977788502339706, 0.9675142820216536, 0.9654037535681113, 0.9632112628445384])
        expected['lon_n'] = np.array([359.0280740990199, 359.02400330627455, 359.01987787201585, 359.0674706301525, 359.06329055438044, 359.06003938433554, 359.05670625202055, 359.1052825573977, 359.1019494250827, 359.09847968898436])
        expected['rho_n'] = np.array([382299.3422241211, 382303.5468505859, 382307.94591064454, 382290.88436279295, 382294.28695068357, 382299.1963989258, 382304.1544555664, 382288.38103027345, 382292.26970214845, 382296.23128662107])
        expected['lat_s'] = np.array([-79.39071315231055, -79.38623254821502, -79.38173828374116, -79.39841760569433, -79.39384137895041, -79.3857817557298, -79.37768115137416, -79.39074047306724, -79.38261254795492, -79.37444364170759])
        expected['lon_s'] = np.array([97.6071871642824, 97.54485485791686, 97.48186685339101, 97.29526608526625, 97.23350068460184, 97.17218607642262, 97.11015429838056, 96.92281586982546, 96.86256677115678, 96.80127265354508])
        expected['rho_s'] = np.array([6371.1946834564205, 6371.193544197083, 6371.194303703308, 6371.196961975097, 6371.198101234436, 6371.199240493775, 6371.199620246887, 6371.195442962647, 6371.195063209533, 6371.195063209533])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_0_120000(self):
        """Test for latitude = 0, longitude = 0, rho = 120000"""
        lat = 0
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([15.666463866482616, 15.66823288547765, 15.669899451635132, 15.645344921568935, 15.647407638698269, 15.649395223746751, 15.651212054065965, 15.623980089845134, 15.626240882460408, 15.628419712805638])
        expected['lon_n'] = np.array([359.250246492342, 359.24497358630276, 359.23959139723684, 359.28087306058035, 359.27554551302774, 359.2708736636355, 359.26611985197314, 359.3081938172604, 359.3035219678681, 359.2987408354491])
        expected['rho_n'] = np.array([382313.92474365234, 382313.1956176758, 382312.22344970703, 382332.88201904297, 382331.6425048828, 382331.30224609375, 382329.38221435546, 382289.2316772461, 382287.23873291013, 382285.3916137695])
        expected['lat_s'] = np.array([-71.83865494929748, -71.82007683475504, -71.80134845605087, -71.78204634145644, -71.7633999250223, -71.74549116901852, -71.72752777150137, -71.70862180787879, -71.6908086745234, -71.6728589373846])
        expected['lon_s'] = np.array([43.59068050946616, 43.52576980668895, 43.46070883974999, 43.32244873547601, 43.25763365534717, 43.20555687802041, 43.153527912017836, 43.027736318073714, 42.9758063898141, 42.92387304645991])
        expected['rho_s'] = np.array([6371.187468147277, 6371.191265678405, 6371.198860740661, 6371.186708641052, 6371.187088394165, 6371.189366912841, 6371.1965822219845, 6371.185949134827, 6371.186708641052, 6371.188607406616])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_0_120000(self):
        """Test for latitude = 30, longitude = 0, rho = 120000"""
        lat = 30
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([73.12364587229422, 73.16069281835236, 73.19752973609353, 73.24285145633115, 73.27902584572283, 73.31678996164383, 73.35429111528178, 73.40043416576711, 73.4371515551978, 73.47355304837939])
        expected['lon_n'] = np.array([341.54052220098276, 341.76654682099684, 341.99330910144124, 342.18291515280083, 342.41156256545617, 342.6685962443021, 342.9265861496318, 343.1476657126868, 343.4079778823343, 343.669300919979])
        expected['rho_n'] = np.array([6371.199240493775, 6371.197721481323, 6371.198101234436, 6371.197721481323, 6371.1984809875485, 6371.198101234436, 6371.198101234436, 6371.197721481323, 6371.197721481323, 6371.198101234436])
        expected['lat_s'] = np.array([24.673687041977715, 24.685195910729192, 24.696424741724684, 24.655614361433862, 24.668728324640284, 24.68016889150006, 24.691329420603864, 24.65039609690797, 24.663441758222703, 24.674889155271643])
        expected['lon_s'] = np.array([0.9141644713455541, 0.8989766919523845, 0.8837872583727753, 0.9168510835668963, 0.9014208324581913, 0.8858617148897551, 0.8702998225569686, 0.9029138691220737, 0.887136719108726, 0.8715710915162372])
        expected['rho_s'] = np.array([382277.05527343747, 382280.38494873047, 382283.6903198242, 382335.0207885742, 382275.47550048825, 382278.7565673828, 382282.0133300781, 382333.34379882814, 382273.7255981445, 382277.10388183594])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_0_120000(self):
        """Test for latitude = 60, longitude = 0, rho = 120000"""
        lat = 60
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([83.35204155770366, 83.41354015410363, 83.47484451699908, 83.53621376669165, 83.59706947153599, 83.66387982067612, 83.73045324762941, 83.79745270763217, 83.86343489633543, 83.92916522181305])
        expected['lon_n'] = np.array([310.0629670335754, 310.34590078975395, 310.633834244405, 310.7560673097916, 311.0540274821442, 311.4120659984363, 311.7774537982753, 311.97219615189067, 312.3523918018503, 312.74001869762685])
        expected['rho_n'] = np.array([6371.2, 6371.198860740661, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([38.46091565526869, 38.468534731287846, 38.47856827917859, 38.439161502762204, 38.44712550333444, 38.45453284348932, 38.461725032685344, 38.42496153947775, 38.43272746456405, 38.44030555944818])
        expected['lon_s'] = np.array([2.1233640865316863, 2.09737820494755, 2.0712774908079923, 2.11535206119185, 2.089361056965836, 2.0623472319116125, 2.0352557134555798, 2.078260292017273, 2.051272293615848, 2.024199771623445])
        expected['rho_s'] = np.array([382319.53901367186, 382329.3093017578, 382277.444140625, 382296.40141601564, 382306.3661376953, 382316.01490478514, 382325.32341308595, 382282.7667602539, 382292.82869873045, 382302.64759521483])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_0_120000(self):
        """Test for latitude = 90, longitude = 0, rho = 120000"""
        lat = 90
        lon = 0
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212, 79.40655674354959])
        expected['lon_n'] = np.array([213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833, 210.06037788585613])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512, 73.59206536821557])
        expected['lon_s'] = np.array([351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506, 351.6299410391422])
        expected['rho_s'] = np.array([329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984, 328907.9989013672])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_90_120000(self):
        """Test for latitude = -90, longitude = 90, rho = 120000"""
        lat = -90
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.939896677945, -75.93958248924318, -75.93944588545978, -75.9816291337738, -75.98095977523514, -75.98188868096224, -75.98302249236446, -76.02672204267421, -76.02726845780782, -76.02524672181349])
        expected['lon_n'] = np.array([26.636971074315394, 26.4767331288396, 26.316537872046123, 26.34112655305817, 26.179681371646577, 26.004143802429958, 25.82865575208482, 25.837005658345163, 25.660182306017287, 25.483389689540676])
        expected['rho_n'] = np.array([259029.14860839842, 259028.7111328125, 259028.4680908203, 259002.2924682617, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.35283203123, 258978.8389160156, 259041.88400878906])
        expected['lat_s'] = np.array([-67.0904303817097, -67.0887638155522, -67.08708358901637, -67.09754743882485, -67.09586721228902, -67.09387279705138, -67.09185106105704, -67.1020007221637, -67.10001996730438, -67.09802555206673])
        expected['lon_s'] = np.array([155.14863492143175, 155.11722971162803, 155.08529174706905, 155.1054817862556, 155.0730930292114, 155.03142887527432, 154.98940955150042, 154.99983242017385, 154.95726668126633, 154.91426381025192])
        expected['rho_s'] = np.array([6371.199240493775, 6371.199240493775, 6371.2, 6371.198860740661, 6371.2, 6371.2, 6371.200759506225, 6371.199620246887, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_90_120000(self):
        """Test for latitude = -60, longitude = 90, rho = 120000"""
        lat = -60
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-66.28711817304614, -66.31223960881346, -66.33755228987752, -66.37749523614374, -66.40248006812766, -66.4305111644814, -66.45876082688858, -66.50187298092968, -66.52975381312169, -66.55786687174546])
        expected['lon_n'] = np.array([56.94401452554291, 56.83804414057016, 56.73210449144868, 56.78461498578774, 56.677275147886405, 56.56426625297396, 56.45127784862902, 56.497245021743204, 56.38274373049711, 56.26821170339976])
        expected['rho_n'] = np.array([257870.85908203124, 257888.4310180664, 257906.00295410157, 257856.61682128906, 257873.5325439453, 257887.55606689453, 257901.57958984375, 257848.86378173827, 257862.23109130858, 257875.40396728515])
        expected['lat_s'] = np.array([-64.69434537935584, -64.69741896448235, -64.70038326658215, -64.71571021107965, -64.71857889053106, -64.72077821144381, -64.72284092857313, -64.73758047680204, -64.739547571283, -64.74139172235891])
        expected['lon_s'] = np.array([146.17550122008416, 146.1655837854093, 146.1541773694954, 146.1908964664734, 146.1777688428886, 146.15546144505936, 146.1311049904791, 146.1594366151563, 146.13330431139184, 146.1054098188215])
        expected['rho_s'] = np.array([6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.2, 6371.2])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_90_120000(self):
        """Test for latitude = -30, longitude = 90, rho = 120000"""
        lat = -30
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-56.23276552608556, -56.31687247552509, -56.4009384438296, -56.49647912993976, -56.58277173991368, -56.66855891588904, -56.75740601661258, -56.85525530666217, -56.94012723728875, -57.02801811152847])
        expected['lon_n'] = np.array([62.60234472456199, 62.467380186562536, 62.33217659194214, 62.322723610130836, 62.17940575077646, 62.03813011798392, 61.88987403185962, 61.875346219495, 61.731994209194774, 61.58179151915702])
        expected['rho_n'] = np.array([255792.75283203126, 255776.4933227539, 255759.40747070312, 255754.59523925782, 255796.2040283203, 255770.5144897461, 255803.9570678711, 255789.81202392577, 255761.81358642576, 255792.89865722656])
        expected['lat_s'] = np.array([59.79508665988439, 59.88733177972, 59.98393797534066, 60.21373285977657, 60.31028441388386, 60.400722948684006, 60.491694238239404, 60.71656114118915, 60.80981371392734, 60.90309019232761])
        expected['lon_s'] = np.array([157.74571873068072, 157.6016563807068, 157.46901410702515, 157.3576273820406, 157.23202020320406, 157.09966479746754, 156.9690715805369, 156.86075844067884, 156.7461751871627, 156.633135556399])
        expected['rho_s'] = np.array([255865.22795410155, 255790.2008911133, 255763.32044677733, 255706.83748779297, 255658.7880859375, 255618.41881103514, 255569.6159790039, 255485.9852294922, 255477.43015136718, 255460.56303710936])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_90_120000(self):
        """Test for latitude = 0, longitude = 90, rho = 120000"""
        lat = 0
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-52.74021659589124, -52.90450996618671, -53.0666449967045, -53.24360153772116, -53.404302228513245, -53.56064525861484, -53.71518511877554, -53.88574860272908, -54.0358898210643, -54.18432349210704])
        expected['lon_n'] = np.array([58.90432321356617, 58.73630739017303, 58.56788175542969, 58.53666779092273, 58.360858721686604, 58.18338308629299, 58.00544299803582, 57.966463108442554, 57.78664471816363, 57.60636187502114])
        expected['rho_n'] = np.array([254941.3524291992, 254920.5966430664, 254903.3649658203, 254904.2885253906, 254954.54960937498, 254942.17877197266, 254933.01608886718, 254940.74482421874, 254938.3873168945, 254939.04353027343])
        expected['lat_s'] = np.array([68.54903959076626, 68.6329604175509, 68.70910848906009, 68.88436772806803, 68.94914865725102, 69.00202968935255, 69.05108923061346, 69.19889110915796, 69.23947438565887, 69.274863303296])
        expected['lon_s'] = np.array([143.16122945633083, 143.6543144728924, 144.1140681663043, 144.53870102700398, 144.97193992605787, 145.35346063271646, 145.721962998817, 146.0680350236832, 146.4175904450261, 146.74043982671427])
        expected['rho_s'] = np.array([256077.8653930664, 256099.3989135742, 256078.15704345703, 256088.77797851563, 256093.97907714843, 256073.9524169922, 256065.3973388672, 256088.00024414062, 256100.22525634765, 256064.2793457031])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_90_120000(self):
        """Test for latitude = 30, longitude = 90, rho = 120000"""
        lat = 30
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-52.37685053204655, -52.525133938927524, -52.67554836482955, -52.844390641112255, -52.98660884001026, -53.133362284517176, -53.27502040790324, -53.44289279732379, -53.584004505576274, -53.72017115686964])
        expected['lon_n'] = np.array([54.751370122712494, 54.64572758681991, 54.53450137128083, 54.58141794068965, 54.4716670460113, 54.34394592362665, 54.22019314114977, 54.24570389769977, 54.114174944852834, 53.98652212435988])
        expected['rho_n'] = np.array([254931.4849243164, 254909.36810302734, 254945.3626220703, 254909.14936523436, 254883.2653930664, 254908.07998046876, 254871.96394042968, 254884.74794921876, 254908.68758544922, 254871.28342285156])
        expected['lat_s'] = np.array([71.93162589041796, 71.94555093858833, 71.95868709990954, 72.0489378045075, 72.05596265406886, 72.05795023911733, 72.05721599378157, 72.13705065989527, 72.13266909354269, 72.12705980068682])
        expected['lon_s'] = np.array([144.8110343295907, 145.2467184363675, 145.68468382632705, 146.04952521103246, 146.44573082440655, 146.8078947749573, 147.13988294975496, 147.44270621679664, 147.75578842797168, 148.05740958171944])
        expected['rho_s'] = np.array([257430.92877197266, 257390.63240966797, 257417.02677001953, 257347.63828125, 257329.750390625, 257336.21530761718, 257290.7421508789, 257265.0769165039, 257230.80799560546, 257202.6151245117])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_90_120000(self):
        """Test for latitude = 60, longitude = 90, rho = 120000"""
        lat = 60
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([80.4281575292185, 80.39814738555276, 80.36751227957795, 80.32572518223581, 80.2950730007881, 80.26080764926942, 80.22590196751608, 80.18037619537701, 80.14500947585468, 80.10894095439514])
        expected['lon_n'] = np.array([173.00615084605172, 172.6291517246237, 172.25651026388616, 172.0944162145034, 171.72535377289094, 171.28917789249394, 170.8581792954878, 170.63662161919092, 170.21028121119872, 169.78869461486886])
        expected['rho_n'] = np.array([6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.199620246887, 6371.2, 6371.2])
        expected['lat_s'] = np.array([25.733691420028677, 26.19315141530626, 26.538711175984716, 26.8048631573726, 27.03818924960941, 27.271201153144396, 27.48376347030436, 27.65023567094508, 27.83525183518239, 28.011798564848874])
        expected['lon_s'] = np.array([30.797985495936985, 30.63244561611797, 30.541914873764036, 30.566455743451893, 30.51061211679787, 30.456284792139588, 30.405666260200622, 30.45178369747655, 30.405119845067023, 30.359316596492917])
        expected['rho_s'] = np.array([366706.1325683594, 368688.6260986328, 369703.5451538086, 370524.2736572265, 371052.0150390625, 371560.8963623047, 372007.7290649414, 372483.3136352539, 372855.5081420898, 373205.9017822266])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_90_120000(self):
        """Test for latitude = 90, longitude = 90, rho = 120000"""
        lat = 90
        lon = 90
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212, 79.40655674354959])
        expected['lon_n'] = np.array([213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833, 210.06037788585613])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512, 73.59206536821557])
        expected['lon_s'] = np.array([351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506, 351.6299410391422])
        expected['rho_s'] = np.array([329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984, 328907.9989013672])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_180_120000(self):
        """Test for latitude = -90, longitude = 180, rho = 120000"""
        lat = -90
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.939896677945, -75.93958248924318, -75.93944588545978, -75.9816291337738, -75.98094611485678, -75.98188868096224, -75.98302249236446, -76.02672204267421, -76.02726845780782, -76.02523306143513])
        expected['lon_n'] = np.array([26.636977904504565, 26.4767331288396, 26.316542994688003, 26.341128260605462, 26.179686494288458, 26.004148925071835, 25.828660874726697, 25.83701078098704, 25.660187428659164, 25.48339822727714])
        expected['rho_n'] = np.array([259029.14860839842, 259028.68682861328, 259028.4680908203, 259002.2924682617, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.35283203123, 258978.8146118164, 259041.88400878906])
        expected['lat_s'] = np.array([-67.0904303817097, -67.0887638155522, -67.08708358901637, -67.09756109920318, -67.09583989153234, -67.09387279705138, -67.09185106105704, -67.1020007221637, -67.10004728806106, -67.0980118916884])
        expected['lon_s'] = np.array([155.1486622421884, 155.11724337200636, 155.08529174706905, 155.10539982398555, 155.07314767072475, 155.03142887527432, 154.98940955150042, 154.99983242017385, 154.95721203975296, 154.91426381025192])
        expected['rho_s'] = np.array([6371.199620246887, 6371.198860740661, 6371.199620246887, 6371.199620246887, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.2, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_180_120000(self):
        """Test for latitude = -60, longitude = 180, rho = 120000"""
        lat = -60
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-82.17378301265896, -82.204477882789, -82.23757697950688, -82.27017064222619, -82.30070158781615, -82.33766657160425, -82.3723229514529, -82.4094928409161, -82.44623925865076, -82.48067707244599])
        expected['lon_n'] = np.array([72.63604287947246, 72.34285700934966, 72.07970348100741, 72.38773135219665, 72.08824121746993, 71.80691938593544, 71.49401475967882, 71.79227546035493, 71.50379559057028, 71.18337775622663])
        expected['rho_n'] = np.array([271892.9723876953, 271924.9081054687, 271897.8818359375, 271855.93278808595, 271887.30950927734, 271860.52628173825, 271893.9688598633, 271852.84615478513, 271825.62545166013, 271858.5090332031])
        expected['lat_s'] = np.array([43.96542316398992, 43.95712448414835, 43.94915706848153, 44.031057866819154, 44.0220044510743, 44.015669450619114, 44.00971011056828, 44.09389902227785, 44.08670000289266, 44.080023492978974])
        expected['lon_s'] = np.array([198.5639537390524, 198.44689795705673, 198.32984217506106, 198.32761553339162, 198.21014994004574, 198.0823571006748, 197.95455060092553, 197.94067165653206, 197.8124280246759, 197.68421171357642])
        expected['rho_s'] = np.array([265513.1687011719, 265552.32276611327, 265589.41097412107, 265442.6136108398, 265482.6183227539, 265520.04678955075, 265556.74613037106, 265408.56342773436, 265446.5994995117, 265483.8578369141])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_180_120000(self):
        """Test for latitude = -30, longitude = 180, rho = 120000"""
        lat = -30
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-80.26100219597524, -80.32071170969948, -80.38122718574579, -80.39035231847694, -80.45019843598459, -80.51624636525861, -80.58238991718102, -80.59620055968279, -80.6624124134969, -80.72870622958104])
        expected['lon_n'] = np.array([103.78322862691552, 103.68408160092362, 103.55361132739803, 103.93563746805518, 103.83418183812381, 103.70816484793708, 103.58106868786147, 104.00971769979314, 103.88072957731745, 103.75213077562445])
        expected['rho_n'] = np.array([280948.4010620117, 280891.1160644531, 280892.5986206055, 280894.2026977539, 280836.2857910156, 280837.8412597656, 280839.59116210934, 280783.4241577148, 280783.8130249023, 280784.8094970703])
        expected['lat_s'] = np.array([40.858465733199495, 40.849323524995434, 40.840307675291015, 40.9202823602827, 40.91583249203843, 40.907588453710225, 40.900105981474475, 40.986473723529286, 40.97798721348555, 40.969579250617265])
        expected['lon_s'] = np.array([194.15090021441546, 194.06225801936705, 193.97356118280527, 193.97036465427368, 193.88308849705928, 193.78631837689852, 193.68967120014284, 193.67926199184774, 193.58239624903862, 193.48543488358112])
        expected['rho_s'] = np.array([281083.67823486327, 281067.95341796876, 281051.8154296875, 281014.7515258789, 281057.867175293, 281042.5312255859, 281025.2509399414, 281046.6386352539, 281031.32698974607, 281015.62647705077])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_180_120000(self):
        """Test for latitude = 0, longitude = 180, rho = 120000"""
        lat = 0
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-77.99493667466177, -78.06804701953757, -78.14121200592675, -78.16749457385296, -78.24060491872879, -78.32075035844971, -78.40095043968401, -78.43454131002213, -78.51468674974305, -78.59490049135567])
        expected['lon_n'] = np.array([94.58419324895894, 94.39262010311843, 94.20110159879127, 94.56112086994264, 94.36409040295528, 94.16827570964018, 93.97194192194817, 94.33959734459162, 94.13939766982938, 93.93762705155802])
        expected['rho_n'] = np.array([277453.74886474607, 277428.1079345703, 277403.31765136716, 277415.59127197263, 277388.6865234375, 277362.29216308595, 277336.8699707031, 277348.1228149414, 277321.4368041992, 277295.2125732422])
        expected['lat_s'] = np.array([42.9282077121047, 42.92075597572022, 42.91356037142961, 42.98101873476724, 42.97313669646505, 42.9664294507001, 42.96498486569064, 43.03313990832361, 43.025906737992564, 43.01885456767453])
        expected['lon_s'] = np.array([191.47894387148483, 191.40796454563005, 191.33695789901859, 191.33366574783867, 191.26253615782213, 191.1849861899858, 191.10938965625212, 191.09902142909203, 191.02132119709398, 190.9435526632042])
        expected['rho_s'] = np.array([277566.95782470703, 277543.8688354492, 277520.19654541014, 277523.8907836914, 277501.3850952148, 277478.73358154297, 277513.87745361327, 277518.54385986325, 277496.57286376954, 277473.9942626953])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_180_120000(self):
        """Test for latitude = 30, longitude = 180, rho = 120000"""
        lat = 30
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-74.51341166978784, -74.5944860152359, -74.67542375690056, -74.74625281859358, -74.82613871112605, -74.91446671747266, -75.00261713890083, -75.08218884273148, -75.16917813200078, -75.25598983635163])
        expected['lon_n'] = np.array([74.21553095522708, 73.8903866299778, 73.56358939894936, 73.75469126173715, 73.42045595470258, 73.07475959024072, 72.72657020673181, 72.91047988032358, 72.55384155281136, 72.19473069681959])
        expected['rho_n'] = np.array([236782.05681152342, 236674.34060058594, 236566.25982666016, 236645.22416992186, 236536.09831542967, 236425.61142578124, 236316.38835449217, 236393.1653198242, 236281.73056640624, 236171.21937255858])
        expected['lat_s'] = np.array([50.75435799141579, 50.749669066550574, 50.74523285868465, 50.80247667411853, 50.79739842847062, 50.79322176779316, 50.7927573149296, 50.84707097920954, 50.84245377133061, 50.84157267692768])
        expected['lon_s'] = np.array([190.73837378053702, 190.67365090796199, 190.60887339387358, 190.60832697873997, 190.5434128608682, 190.47251549728344, 190.40407700179992, 190.39439179355685, 190.323289524297, 190.25461880238169])
        expected['rho_s'] = np.array([264778.2097167969, 264741.38885498047, 264704.2520385742, 264766.51939697267, 264729.99018554686, 264694.4331420898, 264719.6122924805, 264722.09132080077, 264686.8259277344, 264712.2724243164])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_180_120000(self):
        """Test for latitude = 60, longitude = 180, rho = 120000"""
        lat = 60
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([76.3971796641102, 76.3947856828061, 76.39215264488107, 76.37963205435878, 76.37605645032829, 76.37476383702786, 76.3727856434895, 76.36106674642105, 76.35818867545953, 76.3548359063507])
        expected['lon_n'] = np.array([199.0425450941952, 198.75164733744435, 198.4615282222589, 198.21575069516516, 197.92702493857038, 197.5932472542102, 197.25979741893016, 196.97081211514694, 196.63900152526773, 196.3077646712788])
        expected['rho_n'] = np.array([6371.199620246887, 6371.198860740661, 6371.198860740661, 6371.200759506225, 6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.2, 6371.2])
        expected['lat_s'] = np.array([63.23360025600745, 63.231892708714945, 63.229864142531454, 63.275255872208064, 63.27265698522888, 63.27150268325914, 63.27056011715368, 63.31606112985701, 63.31451921465188, 63.31264603527201])
        expected['lon_s'] = np.array([193.58913081556025, 193.5147773762555, 193.43717276690583, 193.44117525775945, 193.36325645970794, 193.28102098210098, 193.19866256108898, 193.19134059829872, 193.1085996866932, 193.0227032276911])
        expected['rho_s'] = np.array([255260.976953125, 255276.50733642577, 255228.70097656248, 255283.94442138672, 255236.16236572264, 255253.00517578123, 255269.75076904296, 255263.01850585936, 255279.86131591795, 255233.2215576172])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_180_120000(self):
        """Test for latitude = 90, longitude = 180, rho = 120000"""
        lat = 90
        lon = 180
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212, 79.40655674354959])
        expected['lon_n'] = np.array([213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833, 210.06037788585613])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512, 73.59206536821557])
        expected['lon_s'] = np.array([351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506, 351.6299410391422])
        expected['rho_s'] = np.array([329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984, 328907.9989013672])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90S_270_120000(self):
        """Test for latitude = -90, longitude = 270, rho = 120000"""
        lat = -90
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-75.939896677945, -75.93958248924318, -75.93944588545978, -75.9816291337738, -75.98095977523514, -75.98188868096224, -75.98302249236446, -76.02672204267421, -76.02726845780782, -76.02524672181349])
        expected['lon_n'] = np.array([26.636993272430196, 26.476748496765236, 26.316560070160925, 26.341145336078387, 26.17970186221409, 26.00416600054476, 25.828671120010455, 25.83702614891267, 25.660199381490212, 25.483410180108187])
        expected['rho_n'] = np.array([259029.14860839842, 259028.7111328125, 259028.4680908203, 259002.2924682617, 259001.58764648438, 259002.31677246094, 259003.2889404297, 258978.35283203123, 258978.8146118164, 259041.88400878906])
        expected['lat_s'] = np.array([-67.0904303817097, -67.08875015517387, -67.08708358901637, -67.09753377844649, -67.09583989153234, -67.0938864574297, -67.09185106105704, -67.10204170329871, -67.10000630692605, -67.0980118916884])
        expected['lon_s'] = np.array([155.1486212610534, 155.11727069276304, 155.08530540744738, 155.10545446549892, 155.07314767072475, 155.03140155451763, 154.98939589112206, 154.99972313714713, 154.95738962467138, 154.91433211214363])
        expected['rho_s'] = np.array([6371.2, 6371.198860740661, 6371.2, 6371.199620246887, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199240493775, 6371.199620246887])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60S_270_120000(self):
        """Test for latitude = -60, longitude = 270, rho = 120000"""
        lat = -60
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-74.71189696706841, -74.68863334275534, -74.66490526557874, -74.70924685367044, -74.68474013492843, -74.65868979343401, -74.63276239534466, -74.6755467003056, -74.64888164178586, -74.62235318704955])
        expected['lon_n'] = np.array([340.4261904982735, 340.3801003817543, 340.3492552474625, 340.33835426554714, 340.3064163009882, 340.2566652030738, 340.2052202182453, 340.1567258751382, 340.1044066260959, 340.05063937694956])
        expected['rho_n'] = np.array([257822.54233398437, 257763.38591308592, 257765.4517700195, 257774.614453125, 257776.87474365233, 257786.62072753906, 257797.12014160157, 257754.44196777343, 257765.23303222656, 257776.65600585938])
        expected['lat_s'] = np.array([45.45101321412978, 45.505634236922376, 45.56180912775115, 45.677867702128005, 45.73532325342615, 45.79689399369923, 45.86002201710307, 45.98313276179797, 46.04721018149644, 46.11273901639355])
        expected['lon_s'] = np.array([227.5137924984215, 227.51248110210085, 227.51134729069864, 227.68233424638072, 227.67934262352426, 227.6696574152812, 227.6600405089298, 227.82168376582732, 227.8101680668867, 227.7985977264327])
        expected['rho_s'] = np.array([252424.5553833008, 252201.58865966796, 251972.0841064453, 251715.1644165039, 251475.03892822264, 251220.59826660156, 250959.74129638672, 250670.0109375, 250398.67885742188, 250120.59020996094])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30S_270_120000(self):
        """Test for latitude = -30, longitude = 270, rho = 120000"""
        lat = -30
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-70.685022338104, -70.69104656495196, -70.69641509363959, -70.78258476020847, -70.78535781701149, -70.78487970376958, -70.78418302447423, -70.86432846419518, -70.86099533188019, -70.85755291653851])
        expected['lon_n'] = np.array([339.4026729907688, 339.5276927733367, 339.64948870661635, 339.77822411209274, 339.8959219318704, 339.98258337205954, 340.0795993790304, 340.185603914949, 340.2651619584013, 340.355019927122])
        expected['rho_n'] = np.array([255822.1123046875, 255831.78537597656, 255843.96177978514, 255831.9555053711, 255847.70462646484, 255814.74813232422, 255844.5450805664, 255850.69404296874, 255823.060168457, 255858.56860351562])
        expected['lat_s'] = np.array([51.77380470617521, 52.0290010641344, 52.28412229001272, 52.5325089493694, 52.77390149501599, 53.0223052298456, 53.26288498298096, 53.50530888719222, 53.737391885000086, 53.96970369414514])
        expected['lon_s'] = np.array([240.15434163309405, 240.0395124927678, 239.91553089895373, 239.94110312720625, 239.81428017469744, 239.67926099518465, 239.5463455139362, 239.5547056654803, 239.413921806308, 239.2638215691078])
        expected['rho_s'] = np.array([254937.92553710938, 254919.86751708985, 254964.24698486328, 254947.4284790039, 254944.92514648437, 254977.4684692383, 254954.45239257812, 254994.40849609373, 254981.45435791014, 255032.12861328124])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_0N_270_120000(self):
        """Test for latitude = 0, longitude = 270, rho = 120000"""
        lat = 0
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-69.39216315086895, -69.41818617160672, -69.4434032300224, -69.55424353987337, -69.57571765462387, -69.59435041067968, -69.61205426100835, -69.71532672125895, -69.7298203826777, -69.74323487420762])
        expected['lon_n'] = np.array([344.85534960897337, 344.93892380365764, 345.0330711311771, 345.16716140496277, 345.2606257135652, 345.3296652656957, 345.397010930912, 345.51765939241113, 345.596233888623, 345.6613665725482])
        expected['rho_n'] = np.array([255281.29526367187, 255257.84171142578, 255296.84995117187, 255280.34739990233, 255322.78253173828, 255312.33172607422, 255303.67943115233, 255302.3183959961, 255357.63475341795, 255354.08634033203])
        expected['lat_s'] = np.array([54.852133399059525, 55.03997726161322, 55.22887638839367, 55.416167005624594, 55.5960810185519, 55.784064899983576, 55.96986995107002, 56.15195596415339, 56.332710090348606, 56.51145272583325])
        expected['lon_s'] = np.array([241.06371301918955, 240.85159466432563, 240.63270076180507, 240.57188475743527, 240.35665183630982, 240.13065453705244, 239.90359172828454, 239.84414176174874, 239.6141283112594, 239.3829673889895])
        expected['rho_s'] = np.array([255314.5191040039, 255281.5869140625, 255311.5539916992, 255340.08712158204, 255318.33486328126, 255339.4066040039, 255363.34624023436, 255327.86210937498, 255358.50970458984, 255391.70924072264])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_30N_270_120000(self):
        """Test for latitude = 30, longitude = 270, rho = 120000"""
        lat = 30
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([-67.95223633042522, -68.0055118059513, -68.05813158331708, -68.20047272562016, -68.2495681253742, -68.29718820426754, -68.34389303781208, -68.47822919840789, -68.52105448450388, -68.562800600711])
        expected['lon_n'] = np.array([342.6715742067802, 342.37828588381984, 342.0830577871352, 341.87951814986883, 341.5819951096231, 341.2650470113778, 340.94670555454184, 340.7181127833999, 340.3983233264599, 340.07733175622593])
        expected['rho_n'] = np.array([238047.74659423827, 237475.21257324217, 236891.35279541014, 236264.25584716795, 235663.57756347655, 235050.43122558593, 234428.4624633789, 233783.28319091795, 233148.74915771483, 232507.38564453126])
        expected['lat_s'] = np.array([57.30254231097705, 57.41375828123237, 57.52459859108334, 57.63630291986431, 57.745538135260325, 57.859598879304954, 57.973328359174836, 58.08823263158196, 58.200397998131905, 58.31489587428342])
        expected['lon_s'] = np.array([243.02474229217026, 242.8479769964503, 242.67071992711016, 242.64083101930217, 242.46171613850777, 242.27724638940407, 242.09212094214004, 242.05725965661628, 241.8700031903312, 241.67537011974252])
        expected['rho_s'] = np.array([256089.72584228514, 256107.37069091797, 256122.87677001953, 256111.47810058593, 256123.7517211914, 256121.58864746094, 256116.9951538086, 256088.82658691405, 256080.44163818358, 256130.02220458983])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_60N_270_120000(self):
        """Test for latitude = 60, longitude = 270, rho = 120000"""
        lat = 60
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([72.72554488142544, 72.77332546976426, 72.82088919959693, 72.83358652126398, 72.8805731001118, 72.9339373680971, 72.98709160776544, 73.0045939675136, 73.05722740525772, 73.10965081468485])
        expected['lon_n'] = np.array([233.92523643079912, 233.71426554771574, 233.50198326831176, 233.22538792768293, 233.0115210443915, 232.7786389144507, 232.5439536145691, 232.24394438546545, 232.00677289672592, 231.76771627577548])
        expected['rho_n'] = np.array([6371.199240493775, 6371.199240493775, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.2, 6371.199240493775, 6371.199620246887, 6371.2])
        expected['lat_s'] = np.array([69.10274936640084, 69.13134053826653, 69.15818318170467, 69.19331596724793, 69.22205398818076, 69.25177555635406, 69.28039404897642, 69.32131029719937, 69.35234667678792, 69.38197432986014])
        expected['lon_s'] = np.array([268.81138384312305, 269.2252386653124, 269.6392027705286, 268.4998998962138, 268.9232077002145, 269.348154749616, 269.7651787795803, 268.62756979217966, 269.0523529170411, 269.47205438116004])
        expected['rho_s'] = np.array([272500.6016723633, 272835.2947998047, 273154.2145019531, 272299.3385986328, 272619.789465332, 272931.8796875, 273246.64337158203, 272404.57578124997, 272739.779296875, 273069.39284667966])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)

    def test_90N_270_120000(self):
        """Test for latitude = 90, longitude = 270, rho = 120000"""
        lat = 90
        lon = 270
        rho = 120000

        expected = {}
        expected['lat_n'] = np.array([79.2549726477522, 79.27849838057466, 79.30147855203717, 79.28958804646582, 79.3118664159911, 79.33895580001303, 79.36520507076703, 79.35667928713556, 79.38197147763212, 79.40655674354959])
        expected['lon_n'] = np.array([213.56871821341153, 213.20076226244458, 212.83119438683354, 212.50171972165046, 212.13194694036432, 211.71182200451685, 211.28910159678475, 210.91084572054945, 210.48671829384833, 210.06037788585613])
        expected['rho_n'] = np.array([6371.199620246887, 6371.199240493775, 6371.2, 6371.2, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.199620246887, 6371.200759506225, 6371.2])
        expected['lat_s'] = np.array([73.52105018386764, 73.4465123294553, 73.37440772993477, 73.66336229786674, 73.58166640520422, 73.51485861738503, 73.4402951497633, 73.73628310499309, 73.6631198261512, 73.59206536821557])
        expected['lon_s'] = np.array([351.32075203579404, 351.40749543825325, 351.4905778593173, 351.3307241119823, 351.4288056284637, 351.5112596721241, 351.6045054146731, 351.4476296298162, 351.54054752328506, 351.6299410391422])
        expected['rho_s'] = np.array([329082.13848876953, 329560.51804199215, 330022.1033935547, 328363.94940185547, 328805.70252685546, 329295.82100830076, 329704.0343383789, 328014.74666748045, 328468.3845458984, 328907.9989013672])

        years = np.arange(2006, 2016)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt], rtol=0.05, atol=0.1)


class Trace2016to2025(unittest.TestCase):
    """
    The 2020-01-01 version of Geopack-2008 is based on IGRF-13 which has
    provisional IGRF coefficients up to 2025. These tests allow for easy
    benchmarking of future Geopack-2008 releases with new IGRF versions.
    """
    
    def test_90S_0_30000(self):
        """Test for latitude = -90, longitude = 0, rho = 30000"""
        lat = -90
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([-7.61653086306697, -7.7008905295057986, -7.715281738087015, -7.730335475017725, -7.745853664812003, -7.82690751969254, -7.838006577093807, -7.849508615656106, -7.86157072973036])
        expected['lon_n'] = np.array([351.76154512407, 351.8964276997994, 351.9147052860184, 351.93284626845394, 351.9510418924028, 352.08100673192985, 352.09428461967633, 352.10748054515284, 352.1206764706293])
        expected['rho_n'] = np.array([293477.33728027344, 293557.4439208984, 293574.7971191406, 293593.38983154297, 293612.3471069336, 293690.6309326172, 293706.3314453125, 293722.5666503906, 293739.62819824216])
        expected['lat_s'] = np.array([-78.45104304705688, -78.44667172598807, -78.44122123503038, -78.43574342331604, -78.43025195122337, -78.42275240351469, -78.41414636516046, -78.4055266664279, -78.39686598656033])
        expected['lon_s'] = np.array([135.65456469054166, 135.5850880063043, 135.5111307179714, 135.43715976926015, 135.36313417903557, 135.2849558337956, 135.20252911089187, 135.12000676533978, 135.03753906130106])
        expected['rho_s'] = np.array([6371.198101234436, 6371.1984809875485, 6371.1984809875485, 6371.199240493775, 6371.1984809875485, 6371.198860740661, 6371.198860740661, 6371.198860740661, 6371.198860740661])

        years = np.arange(2016, 2025)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_60S_0_30000(self):
        """Test for latitude = -60, longitude = 0, rho = 30000"""
        lat = -60
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([4.77276976298667, 4.716905645765124, 4.704433720340688, 4.691832021322014, 4.678936624169026, 4.626316846803249, 4.61752639334145, 4.608544694582875, 4.599317109014194])
        expected['lon_n'] = np.array([355.73040412622436, 355.8085414903293, 355.8184862457608, 355.82821243513894, 355.837938624517, 355.91394496960095, 355.9214854984446, 355.92894406501824, 355.9363206693219])
        expected['rho_n'] = np.array([321064.2317749023, 321102.0491088867, 321110.79862060543, 321119.59674072266, 321128.5892944336, 321165.09420166013, 321172.1667236328, 321179.2149414062, 321186.8707641601])
        expected['lat_s'] = np.array([-79.60886939440076, -79.59488116698057, -79.58045580745349, -79.56601678754811, -79.55156410726434, -79.53629180428021, -79.52058236918916, -79.5048592737198, -79.48912251787209])
        expected['lon_s'] = np.array([59.42140951487116, 59.408254570529714, 59.39996955106649, 59.391670871224925, 59.38351562535593, 59.38900709744862, 59.39936166423036, 59.40969574044459, 59.41997517514546])
        expected['rho_s'] = np.array([6371.197721481323, 6371.197341728211, 6371.198860740661, 6371.2, 6371.2, 6371.199240493775, 6371.199240493775, 6371.198860740661, 6371.197721481323])

        years = np.arange(2016, 2025)
        dates = np.array([dt.datetime(i, 1, 1) for i in years])

        trace = Trace(lat=np.ones_like(dates, dtype=float) * lat,
                      lon=np.ones_like(dates, dtype=float) * lon,
                      rho=np.ones_like(dates, dtype=float) * rho,
                      coords='geo', datetime=dates)

        for cnt, year in enumerate(years):
            for coord in expected:
                with self.subTest(coord=coord, year=year):
                    np.testing.assert_allclose(getattr(trace, coord)[cnt], expected[coord][cnt])

    def test_30S_0_30000(self):
        """Test for latitude = -30, longitude = 0, rho = 30000"""
        lat = -30
        lon = 0
        rho = 30000

        expected = {}
        expected['lat_n'] = np.array([65.43263600835371, 65.48940341809298, 65.54645086558821, 65.60246695451877, 65.6581